/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-get.
 *
 *   Swac-get is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-get is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with swac-get; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <string>
#include <string.h>
#include <list>
#include <iostream>

#include <bzlib.h> 
#include <libxml/uri.h>

#include "global.hh"
#include "basic.hh"
#include "http.hh"

#include "swac_index.hh"



void Tswac_index::show_consts() {
	for (tconsts::iterator it = consts.begin(); it != consts.end(); it++)  {
		std::cout << *(it) << "=";
		std::cout << *(++it) << std::endl;
	}
}

void Tswac_index::groups_clear() {
	groups.clear();
	consts.clear();	
}

void Tswac_index::insert(char const **attr) {
	std::string fields("`filename`, `packages_idx`");
	std::string values(db.str(filename) + ", '" + packages_idx + "'");
	std::list<std::string> alphaidx;
	std::string field;
	std::string value;
	std::string lang;
	
	for (tconsts::iterator it = consts.begin(); it != consts.end(); it++) {
		field = *it;
		value = *(++it);		

		fields += ", `" + field + "`";
		values += ", " + db.str(value);

		if (field == "swac_alphaidx") 
			string_split(value, '|', &alphaidx);

		if (field == "swac_lang") 
			lang = value;
	}

	if (attr != NULL) {
		for (int i = 0; attr[i]; i += 2) {
			if (check_field(attr[i])) {
				if (strcmp(attr[i], "swac_lang") == 0)
					lang = attr[i + 1];

				if (strcmp(attr[i], "swac_alphaidx") == 0)
					string_split(attr[i + 1], '|', &alphaidx);
				fields += ", `";
				fields += attr[i];
				fields += "`";
				values += ", " + db.str(attr[i + 1]);
			}
		}
	}

	db.exec("INSERT INTO `sounds` (" + fields + ") VALUES (" + values + ");");
	std::string idx(int2string(db.lastInsertRowId()));

	for (std::list<std::string>::iterator it = alphaidx.begin(); it != alphaidx.end(); it++)
		db.exec("INSERT INTO `alphaidx` (`str`, `sounds_idx`, `packages_idx`, `lang`) VALUES (" + db.str(*it) +", " + idx + ", " + packages_idx + ", " + db.str(lang) + ");");

}

void Tswac_index::group_push(char const **attr) {
	if (attr == NULL) 
		return;

	bool first = true;
	for (int i = 0; attr[i]; i += 2) {
		if (check_field(attr[i])) {
			consts.push_back(attr[i]);
			if (first) {
				groups.push_back(--(consts.end()));
				first = false;
			}
			consts.push_back(attr[i + 1]);
		}
	}
	//show_consts();
}

void Tswac_index::group_pop() {
	if (!groups.empty()) {
		consts.erase(groups.back(), consts.end());
		groups.pop_back();

		//show_consts();
	}
}

void element_start(void *user_data, xmlChar const *name, xmlChar const **attrs) {
	Tswac_index* index = (Tswac_index*) user_data;
	
	if (strcmp((char*) name, "index") == 0) {
		index->groups_clear();
		index->finished = false;
	}

	if (strcmp((char*) name, "group") == 0)
		index->group_push((const char**) attrs);

	if (strcmp((char*) name, "file") == 0 && attrs != NULL) {
		for (int i = 0; attrs[i]; i += 2) {
			if (strcmp((char*) attrs[i], "path") == 0) {
				index->filename = (char*) attrs[i + 1];
			}
		}
	}
	if (strcmp((char*) name, "tag") == 0 && index->filename.size() != 0)
		index->insert((const char**) attrs);
}

void element_end(void *user_data, xmlChar const *name) {
	Tswac_index* index = (Tswac_index*) user_data;

	if (strcmp((char*) name, "group") == 0)
		index->group_pop();

	if (strcmp((char*) name, "file") == 0)
		index->filename = "";

	if (strcmp((char*) name, "index") == 0)
		index->finished = true;
}

void print_location(Tswac_index *index) {
	fprintf(stderr, "line: %i\n", index->parser_ctxt->input->line);
	fprintf(stderr, "column: %i\n", index->parser_ctxt->input->col);
	fprintf(stderr, "file: %s\n", index->parser_ctxt->input->filename);
}

static void parser_warning(void *ctx, char const *msg, ...) {
	va_list args;
	va_start(args, msg);
	fprintf(stderr, "XML warning: ");
	vfprintf(stderr, msg, args);
	va_end(args);
	print_location((Tswac_index*) ctx);
}

static void parser_error(void *ctx, char const *msg, ...) {
	va_list args;
	va_start(args, msg);
	fprintf(stderr, "XML error: ");
	vfprintf(stderr, msg, args);
	va_end(args);
	print_location((Tswac_index*) ctx);
}

static void parser_fatal_error(void *ctx, char const *msg, ...) {
	va_list args;
	va_start(args, msg);
	fprintf(stderr, "XML fatalError: ");
	vfprintf(stderr, msg, args);
	va_end(args);
	print_location((Tswac_index*) ctx);
}


static xmlSAXHandler sax_handler = {
    NULL, /* internalSubset */
    NULL, /* isStandalone */
    NULL, /* hasInternalSubset */
    NULL, /* hasExternalSubset */
    NULL, /* resolveEntity */
    NULL, /* getEntity */
    NULL, /* entityDecl */
    NULL, /* notationDecl */
    NULL, /* attributeDecl */
    NULL, /* elementDecl */
    NULL, /* unparsedEntityDecl */
    NULL, /* setDocumentLocator */
    NULL, /* startDocument */
    NULL, /* endDocument */
    element_start, /* startElement */
    element_end, /* endElement */
    NULL, /* reference */
    NULL, /* characters */
    NULL, /* ignorableWhitespace */
    NULL, /* processingInstruction */
    NULL, /* comment */
    parser_warning, /* xmlParserWarning */
    parser_error, /* xmlParserError */
    parser_fatal_error, /* xmlParserFatalError */
    NULL, /* getParameterEntity */
    NULL, /* cdataBlock; */
    NULL, /* externalSubset; */
    1,
    NULL,
    NULL, /* startElementNs */
    NULL, /* endElementNs */
    NULL  /* xmlStructuredErrorFunc */
};


bool Tswac_index::read_stream_bz2(FILE *f, std::string src_url) {
	BZFILE* bz_file;
	int     len;
	char    buffer[BUFSIZ];
	int     bzerror;

	bz_file = BZ2_bzReadOpen (&bzerror, f, 0, 0, NULL, 0);
	if (bzerror != BZ_OK) {
		BZ2_bzReadClose (&bzerror, bz_file);
		print_error(std::string(_("Not a valid bz2 file:")) + " \"" + src_url + "\"");
		return false;
	}
	bzerror = BZ_OK;

	len = BZ2_bzRead(&bzerror, bz_file, buffer, 4);
	if (len > 0) {
		parser_ctxt = xmlCreatePushParserCtxt(&sax_handler, this, buffer, len, src_url.c_str());
		while ((len = BZ2_bzRead(&bzerror, bz_file, buffer, sizeof(buffer))) > 0) {
			xmlParseChunk(parser_ctxt, buffer, len, 0);
		}
		xmlParseChunk(parser_ctxt, buffer, 0, 1);
		xmlFreeParserCtxt(parser_ctxt);
	}

	return true;
}

/*
bool Tswac_index::read_stream(FILE * f) {
	char buf[BUFSIZ];
	int done;

	XML_Parser parser = XML_ParserCreate(NULL);
	XML_SetUserData(parser, this);
	XML_SetElementHandler(parser, startElement, endElement);
	do {
		size_t len = fread(buf, 1, sizeof(buf), f);
		done = len < sizeof(buf);
		if (XML_Parse(parser, buf, len, done) == XML_STATUS_ERROR) {
			fprintf(stderr,
				"%s at line %lu\n",
				XML_ErrorString(XML_GetErrorCode(parser)),
				XML_GetCurrentLineNumber(parser));
			return false;
		}
	} while (!done);
	XML_ParserFree(parser);

	return true;
}
*/


bool Tswac_index::read_file(int idx, std::string const filename) {
	finished = false;
	packages_idx = int2string(idx);
	FILE * f = fopen(filename.c_str(), "r");
	read_stream_bz2(f, filename.c_str());
	fclose(f);
	return finished;
}


bool Tswac_index::read_url(int idx, std::string const url) {
	finished = false;
	packages_idx = int2string(idx);

	xmlURIPtr uri = xmlParseURI(url.c_str());
	if (uri != NULL) {
		FILE * f = http_fopen(uri);
		xmlFreeURI(uri);
		read_stream_bz2(f, url.c_str());
		fclose(f);
	}
	return finished;
}



