/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtQuteCom.h"

#include "ui_QuteComWindow.h"

#include <presentation/PFactory.h>
#include <presentation/qt/chat/QtChatWindow.h>

#include <cutil/global.h>

#include "QtSystray.h"
#include "QtHttpProxyLogin.h"
#include "QtDialpad.h"
#include "QtIdle.h"
#include "QtLanguage.h"
#include "QtToolBar.h"
#include "QtBrowserWidget.h"
#include "QtIMAccountMonitor.h"
#include "callbar/QtCallBar.h"
#include "callbar/QtPhoneComboBox.h"
#include "contactlist/QtContactList.h"
#include "filetransfer/QtFileTransfer.h"
#include "history/QtHistoryWidget.h"
#include "imaccount/QtIMAccountManager.h"
#include "phonecall/QtContactCallListWidget.h"
#include "phonecall/QtPhoneCall.h"
#include "profile/QtProfileDetails.h"
#include "profilebar/QtProfileBar.h"
#include "statusbar/QtStatusBar.h"
#include "webservices/sms/QtSms.h"

#include "webdirectory/QtWebDirectory.h"

#include <control/CQuteCom.h>
#include <control/history/CHistory.h>
#include <control/phoneline/CPhoneLine.h>
#include <control/profile/CUserProfile.h>
#include <control/profile/CUserProfileHandler.h>

#include <model/QuteCom.h>
#include <model/commandserver/ContactInfo.h>
#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/connect/ConnectHandler.h>
#include <model/contactlist/ContactList.h>
#include <model/contactlist/Contact.h>
#include <model/history/History.h>
#include <model/phonecall/PhoneCall.h>
#include <model/phonecall/SipAddress.h>
#include <model/phonecall/ConferenceCall.h>
#include <model/phoneline/IPhoneLine.h>
#include <model/profile/AvatarList.h>
#include <model/profile/UserProfile.h>

#include <imwrapper/EnumPresenceState.h>
#include <imwrapper/IMWrapperFactory.h>

#include <qtutil/CloseEventFilter.h>
#include <qtutil/Widget.h>
#include <qtutil/SafeConnect.h>
#include <qtutil/LanguageChangeEventFilter.h>

#include <cutil/global.h>
#include <thread/Thread.h>
#include <util/Logger.h>
#include <util/SafeDelete.h>
#include <webcam/WebcamDriver.h>

#include <QtGui/QtGui>

#if defined(OS_MACOSX)
	#include <Carbon/Carbon.h>
#endif

static const char * CSS_DIR = "css";

using namespace std;

QtQuteCom::QtQuteCom(CQuteCom & cQuteCom)
	: QObjectThreadSafe(NULL),
	_cQuteCom(cQuteCom) {

	_qutecomPhoneWindow = NULL;
	_qtSystray = NULL;
	_qtSms = NULL;
	_qtContactList = NULL;
	_qtHistoryWidget = NULL;
	_qtContactCallListWidget = NULL;
	_qtIdle = NULL;
	_qtLanguage = NULL;
	_qtStatusBar = NULL;
	_qtFileTransfer = NULL;
	_qtBrowserWidget = NULL;
	_chatWindow = NULL;
	_activeTabBeforeCall = NULL;
	_qtWebDirectory = NULL;
	_want_to_exit = false;

	NetworkProxyDiscovery::getInstance().proxyNeedsAuthenticationEvent +=
		boost::bind(&QtQuteCom::proxyNeedsAuthenticationEventHandler, this, _1, _2);
	NetworkProxyDiscovery::getInstance().wrongProxyAuthenticationEvent +=
		boost::bind(&QtQuteCom::wrongProxyAuthenticationEventHandler, this, _1, _2);

	//Check if the event has not already been sent
	if (NetworkProxyDiscovery::getInstance().getState() ==
		NetworkProxyDiscovery::NetworkProxyDiscoveryStateNeedsAuthentication) {
		proxyNeedsAuthenticationEventHandler(NetworkProxyDiscovery::getInstance(),
			NetworkProxyDiscovery::getInstance().getNetworkProxy());
	}

	qRegisterMetaType<QVariant>("QVariant");

	//Initialize libwebcam for the Qt thread.
	WebcamDriver::apiInitialize();
	////

	initUi();

	typedef PostEvent0<void ()> MyPostEvent;
	MyPostEvent * event = new MyPostEvent(boost::bind(&QtQuteCom::initThreadSafe, this));
	postEvent(event);
}

QtQuteCom::~QtQuteCom() {
	//TODO delete created objects
}

void QtQuteCom::terminatedEventHandler() {
	if(_want_to_exit)
		QCoreApplication::exit(EXIT_SUCCESS);
}

void QtQuteCom::initUi() {

#if defined(MULTIIMWRAPPER)
	IMWrapperFactory::getFactory().terminatedEvent += boost::bind(&QtQuteCom::terminatedEventHandler,this);
#endif
	// Init parts of the UI which do not rely on Control layer
	QApplication::setQuitOnLastWindowClosed(false);

	loadStyleSheets();

	//Translation
	_qtLanguage = new QtLanguage(this);

	_qutecomPhoneWindow = new QMainWindow(NULL);

	_ui = new Ui::QuteComWindow();
	_ui->setupUi(_qutecomPhoneWindow);
	_ui->tabWidget->setDocumentMode(true);

#ifndef CUSTOM_ACCOUNT
	_ui->actionShowQuteComAccount->setVisible(false);
#endif

#ifdef DISABLE_FILETRANSFER
 	_ui->actionShowFileTransfer->setVisible(false);
#endif

#ifdef OS_MACOSX
	_ui->actionShowAbout->setMenuRole(QAction::AboutRole);
	_ui->actionShowConfig->setMenuRole(QAction::PreferencesRole);
	_ui->actionQuit->setMenuRole(QAction::QuitRole);
	//fixMacOSXMenus();
#endif
        
       LANGUAGE_CHANGE(_qutecomPhoneWindow);

#ifdef OS_LINUX
	const std::string &applicationIconPath = ConfigManager::getInstance().getCurrentConfig().getApplicationIconPath();
	_qutecomPhoneWindow->setWindowIcon(QIcon(QString::fromUtf8(applicationIconPath.c_str())));
#endif

	//Install the close event filter
	CloseEventFilter * closeEventFilter = new CloseEventFilter(this, SLOT(closeWindow()));
	_qutecomPhoneWindow->installEventFilter(closeEventFilter);

	//QtCallBar
	_ui->callBar->getQtPhoneComboBox()->setQtQuteCom(this);

	//QtToolBar
	_qtToolBar = new QtToolBar(*this, _ui, _qutecomPhoneWindow);

	//phoneComboBox
	SAFE_CONNECT(_ui->callBar, SIGNAL(phoneComboBoxClicked()), SLOT(phoneComboBoxClicked()));

	//Buttons initialization
	initCallButtons();

	//QtStatusBar
	_qtStatusBar = new QtStatusBar(this, _ui->statusBar);

#if defined(CUSTOM_ACCOUNT)
	_qtWebDirectory = new QtWebDirectory(0);
#endif

	installQtBrowserWidget();
	
#if defined(OS_MACOSX)
	_qutecomPhoneWindow->setUnifiedTitleAndToolBarOnMac(true);
#endif
	
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	mainWindowGeometry(config);
}

void QtQuteCom::initThreadSafe() {
	// The code in this method relies on the Control layer
	_qtSystray = new QtSystray(this);

	updatePresentation();
}

void QtQuteCom::loadStyleSheets() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QDir dir(QString::fromUtf8(config.getResourcesDir().c_str()) + CSS_DIR);

	QStringList filters;
	filters << "*.css";
	QStringList cssList;
	Q_FOREACH(QFileInfo fileInfo, dir.entryInfoList(filters)) {
		QString path = fileInfo.absoluteFilePath();
		QFile file(path);
		if (!file.open(QIODevice::ReadOnly)) {
			LOG_WARN("Can't open " + path.toUtf8());
			continue;
		}
		QByteArray content = file.readAll();
		cssList << QString::fromUtf8(content);
	}
	QString styleSheet = cssList.join("\n");
	qApp->setStyleSheet(styleSheet);
}

void QtQuteCom::mainWindowGeometry(Config & config) {
	//default position and size given by Qt
	QPoint defaultPos = _qutecomPhoneWindow->pos();
	QSize defaultSize = _qutecomPhoneWindow->size();

	int profileWidth = config.getProfileWidth();
	if (profileWidth == 0) {
		// Make sure the toolbar is fully visible
		profileWidth = qMax(
			_ui->toolBar->sizeHint().width(),
			_qutecomPhoneWindow->sizeHint().width()
			);
	}

	int profileHeight = config.getProfileHeight();

	//Main window size and position saved
	_qutecomPhoneWindow->resize(profileWidth, profileHeight);
	_qutecomPhoneWindow->move(QPoint(config.getProfilePosX(), config.getProfilePoxY()));

	//tests if the QuteCom is visible, if not sends it back to its default position and size
	QDesktopWidget* desktop = QApplication::desktop();
	if (desktop->screenNumber(_qutecomPhoneWindow) == -1) {
		LOG_DEBUG("Main window is NOT visible !!");
		_qutecomPhoneWindow->resize(defaultSize);
	 	_qutecomPhoneWindow->move(defaultPos);
	}
}

QWidget * QtQuteCom::getWidget() const {
	return _qutecomPhoneWindow;
}

QtProfileBar * QtQuteCom::getQtProfileBar() const {
	return _ui->profileBar;
}

QtBrowserWidget * QtQuteCom::getQtBrowserWidget() const {
	return _qtBrowserWidget;
}

CQuteCom & QtQuteCom::getCQuteCom() const {
	return _cQuteCom;
}

void QtQuteCom::setQtSms(QtSms * qtSms) {
	_qtSms = qtSms;
}

QtSms * QtQuteCom::getQtSms() const {
	return _qtSms;
}

QtWebDirectory * QtQuteCom::getQtWebDirectory() const {
	return _qtWebDirectory;
}

QtToolBar & QtQuteCom::getQtToolBar() const {
	return *_qtToolBar;
}

QtStatusBar & QtQuteCom::getQtStatusBar() const {
	return *_qtStatusBar;
}

QtSystray & QtQuteCom::getQtSystray() const {
	return *_qtSystray;
}

QtCallBar & QtQuteCom::getQtCallBar() const {
	return *_ui->callBar;
}

void QtQuteCom::setChatWindow(QWidget * chatWindow) {
	if (!chatWindow) {
		_chatWindow = NULL;
		_ui->actionOpenChatWindow->setEnabled(false);
	}
	else {
		_chatWindow = chatWindow;
		_ui->actionOpenChatWindow->setEnabled(true);
	}
}

QWidget * QtQuteCom::getChatWindow() const {
	return _chatWindow;
}

void QtQuteCom::closeChatWindow() {
        if (_chatWindow)
        {
            ((QtChatWindow *)_chatWindow)->closeAllTabs ();
        }
}

void QtQuteCom::setQtContactList(QtContactList * qtContactList) {
	_qtContactList = qtContactList;

	if (!_ui->tabContactList->layout()) {
		Widget::createLayout(_ui->tabContactList);
	}

	_ui->tabContactList->layout()->addWidget(_qtContactList->getWidget());

	LOG_DEBUG("QtContactList added");
}

QtContactList * QtQuteCom::getQtContactList() const {
	return _qtContactList;
}

void QtQuteCom::setQtHistoryWidget(QtHistoryWidget * qtHistoryWidget) {
	_qtHistoryWidget = qtHistoryWidget;
	_qtHistoryWidget->setQtToolBar(_qtToolBar);
	_qtHistoryWidget->setQtCallBar(_ui->callBar);

	if (!_ui->tabHistory->layout()) {
		Widget::createLayout(_ui->tabHistory);
	}

	_qtHistoryWidget->getWidget()->setParent(_ui->tabHistory);
	_ui->tabHistory->layout()->addWidget(_qtHistoryWidget->getWidget());

	LOG_DEBUG("QtHistoryWidget added");
}

void QtQuteCom::setQtDialpad(QtDialpad * qtDialpad) {
	Widget::createLayout(_ui->tabDialpad)->addWidget(qtDialpad);
	LOG_DEBUG("QtDialpad added");
}

void QtQuteCom::initCallButtons() {
	//callButton
	SAFE_CONNECT(_ui->callBar, SIGNAL(callButtonClicked()), SLOT(callButtonClicked()));
	enableCallButton();

	//hangUpButton
	SAFE_CONNECT(_ui->callBar, SIGNAL(hangUpButtonClicked()), SLOT(hangUpButtonClicked()));
	_ui->callBar->setEnabledHangUpButton(false);

	//phoneComboBox
	SAFE_CONNECT(_ui->callBar, SIGNAL(phoneComboBoxReturnPressed()), SLOT(callButtonClicked()));
	SAFE_CONNECT(_ui->callBar, SIGNAL(phoneComboBoxEditTextChanged(const QString &)), SLOT(enableCallButton()));
}

void QtQuteCom::enableCallButton() {
	std::string phoneNumber = _ui->callBar->getPhoneComboBoxCurrentText();
	_ui->callBar->setEnabledCallButton(!phoneNumber.empty());
}

void QtQuteCom::hangUpButtonClicked() {
	QtContactCallListWidget * widget =
		dynamic_cast<QtContactCallListWidget *>(_ui->tabWidget->currentWidget());
	if (widget) {
		widget->hangup();
	}
}

void QtQuteCom::callButtonClicked() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		std::string phoneNumber = _ui->callBar->getPhoneComboBoxCurrentText();
		if (!phoneNumber.empty()) {
			CUserProfile * cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
			cUserProfile->makeCallErrorEvent += boost::bind(&QtQuteCom::makeCallErrorEventHandler, this, _2);
			cUserProfile->makeCall(phoneNumber);
		}
		_ui->callBar->clearPhoneComboBoxEditText();
	}
}

void QtQuteCom::setActiveTabBeforeCallCurrent() {
	_ui->tabWidget->setCurrentWidget(_activeTabBeforeCall);
}

void QtQuteCom::makeCallErrorEventHandler(EnumMakeCallError::MakeCallError error) {
	typedef PostEvent1<void (EnumMakeCallError::MakeCallError), EnumMakeCallError::MakeCallError> MyPostEvent;
	MyPostEvent * event = new MyPostEvent(boost::bind(&QtQuteCom::makeCallErrorEventHandlerThreadSafe, this, _1), error);
	postEvent(event);
}

void QtQuteCom::makeCallErrorEventHandlerThreadSafe(EnumMakeCallError::MakeCallError error) {
	QString message;
	switch (error) {
	case EnumMakeCallError::CallNotHeld:
		message = tr("Please hold all the phone calls before placing a new one");
		break;
	case EnumMakeCallError::NotConnected:
		message = tr("Can't place a call, you are not connected");
		break;
	case EnumMakeCallError::EmptyPhoneNumber:
		message = tr("You must enter a phone number");
		break;
	case EnumMakeCallError::SipError:
		message = tr("An error has occured trying to place the call");
		break;
	case EnumMakeCallError::NoError:
		LOG_FATAL("Should not happen");
		break;
	}
	QMessageBox::warning(_qutecomPhoneWindow, tr("@product@ - Call Error"),
		message, QMessageBox::Ok);
}

void QtQuteCom::addPhoneCall(QtPhoneCall * qtPhoneCall) {
	_activeTabBeforeCall = _ui->tabWidget->currentWidget();

	QtContactCallListWidget * qtContactCallListWidget = new QtContactCallListWidget(_cQuteCom, NULL);
	SAFE_CONNECT(qtContactCallListWidget, SIGNAL(aboutToClose()), SLOT(setActiveTabBeforeCallCurrent()) );
	_ui->tabWidget->addTab(qtContactCallListWidget, tr("Call"));
	_ui->tabWidget->setCurrentWidget(qtContactCallListWidget);
	qtContactCallListWidget->addPhoneCall(qtPhoneCall);

	_ui->callBar->setEnabledHangUpButton(true);

	if (qtPhoneCall->isIncoming()) {
		_ui->callBar->setEnabledCallButton(true);
	}
}

void QtQuteCom::addToConference(QString phoneNumber, PhoneCall * targetCall) {
	//FIXME conference has to be 100% rewritten...
	/*bool conferenceAlreadyStarted = false;

	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		int nbtab = _ui->tabWidget->count();

		for (int i = 0; i < nbtab; i++) {
			if (_ui->tabWidget->tabText(i) == QString(tr("Conference"))) {
				return;
			}

			for (int j = 0; j < _ui->tabWidget->count(); j++) {
				QtContactCallListWidget * qtContactCallListWidget = dynamic_cast<QtContactCallListWidget *>(_ui->tabWidget->widget(j));
				if (qtContactCallListWidget) {
					if (qtContactCallListWidget->hasPhoneCall(targetCall)) {
						_ui->tabWidget->setTabText(j, tr("Conference"));
						IPhoneLine * phoneLine = _cQuteCom.getCUserProfileHandler().getCUserProfile()->getUserProfile().getActivePhoneLine();

						if (phoneLine != NULL) {
							if (!conferenceAlreadyStarted) {
								conferenceAlreadyStarted = true;
								ConferenceCall * confCall = new ConferenceCall(*phoneLine);
								confCall->addPhoneCall(*targetCall);
								confCall->addPhoneNumber(phoneNumber.toUtf8());
							}
						} else {
							LOG_DEBUG("phoneLine is NULL");
						}
					}
				}
			}
		}
	}*/

	if (!_cQuteCom.getCUserProfileHandler().getCUserProfile())
		return;

	IPhoneLine * phoneLine = _cQuteCom.getCUserProfileHandler().getCUserProfile()->getUserProfile().getActivePhoneLine();
	if(!phoneLine)
		return;

	for (int i = 0; i < _ui->tabWidget->count(); i++) 
	{
		if (_ui->tabWidget->tabText(i) == QString(tr("Conference"))) 
			return;
	}

	ConferenceCall * confCall = new ConferenceCall(*phoneLine);
	targetCall->setConferenceCall(confCall);

	PhoneCall * phoneCall2 = confCall->getPhoneCall(std::string(phoneNumber.toUtf8()));

	QtContactCallListWidget * qtContactCallListWidgetTarget = 0;
	QtPhoneCall * qtPhoneTarget = 0;
	QtPhoneCall * qtPhoneCall2 = 0;
	int index2 = -1;

	for (int j = 0; j < _ui->tabWidget->count(); j++) 
	{
		QtContactCallListWidget * qtContactCallListWidget = dynamic_cast<QtContactCallListWidget *>(_ui->tabWidget->widget(j));
		if (qtContactCallListWidget) 
		{
			if (!qtContactCallListWidgetTarget && qtContactCallListWidget->hasPhoneCall(targetCall)) 
			{
				_ui->tabWidget->setTabText(j, tr("Conference"));

				qtContactCallListWidgetTarget = qtContactCallListWidget;
				qtPhoneTarget = qtContactCallListWidgetTarget->takeQtPhoneCall(targetCall);
			}
			else if(!qtPhoneCall2 && qtContactCallListWidget->hasPhoneCall(phoneCall2))
			{
				qtPhoneCall2 = qtContactCallListWidget->takeQtPhoneCall(phoneCall2);
				index2 = j;
				
			}
		}
	}

	if( qtContactCallListWidgetTarget)
	{
		if(qtPhoneCall2)
		{
			qtPhoneCall2->hideConfButton();
			qtContactCallListWidgetTarget->addPhoneCall(qtPhoneCall2);
			_ui->tabWidget->removeTab(index2);
		}
		if(qtPhoneTarget)
		{
			qtPhoneTarget->hideConfButton();
			qtContactCallListWidgetTarget->addPhoneCall(qtPhoneTarget);
		}
	}

	QString address = QString::fromUtf8(targetCall->getPeerSipAddress().getSipAddress().c_str());
	
	if(address.contains("@"))
		address = address.section("@",0,0);

	if (address.startsWith("sip:")) {
		address = address.mid(4);
	}

	confCall->addPhoneNumber(std::string(address.toUtf8()));
	confCall->addPhoneNumber(std::string(phoneNumber.toUtf8()));
	
	if(phoneCall2)
		phoneCall2->setConferenceCall(confCall);

	if(qtContactCallListWidgetTarget)
		qtContactCallListWidgetTarget->slotStartedTalking(qtPhoneTarget);
}

void QtQuteCom::addToConference(QtPhoneCall * qtPhoneCall) {

	QtContactCallListWidget * qtContactCallListWidget;

	int nbtab = _ui->tabWidget->count();

	for (int i = 0; i < nbtab; i++) {
		if (_ui->tabWidget->tabText(i) == QString(tr("Conference"))) {
			//i is the index of the conference tab
			qtContactCallListWidget = (QtContactCallListWidget *) _ui->tabWidget->widget(i);
			qtContactCallListWidget->addPhoneCall(qtPhoneCall);
			_ui->tabWidget->setCurrentWidget(qtContactCallListWidget);
			return;
		}
	}

	_activeTabBeforeCall = _ui->tabWidget->currentWidget();

	//conference tab not found, create a new one
	qtContactCallListWidget = new QtContactCallListWidget(_cQuteCom,_qutecomPhoneWindow);
	SAFE_CONNECT(qtContactCallListWidget, SIGNAL(aboutToClose()), SLOT(setActiveTabBeforeCallCurrent()) );
	_ui->tabWidget->addTab(qtContactCallListWidget, tr("Conference"));
	_ui->tabWidget->setCurrentWidget(qtContactCallListWidget);
	qtContactCallListWidget->addPhoneCall(qtPhoneCall);
	_ui->callBar->setEnabledHangUpButton(true);
}

QtFileTransfer * QtQuteCom::getFileTransfer() const {
	return _qtFileTransfer;
}

void QtQuteCom::updatePresentation() {
	typedef PostEvent0<void ()> MyPostEvent;
	MyPostEvent * event = new MyPostEvent(boost::bind(&QtQuteCom::updatePresentationThreadSafe, this));
	postEvent(event);
}

void QtQuteCom::updatePresentationThreadSafe() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		//disabled some actions if no SIP Account is used
		bool hasSipAccount = _cQuteCom.getCUserProfileHandler().getCUserProfile()->getUserProfile().hasSipAccount();

		_ui->actionShowQuteComAccount->setEnabled(_cQuteCom.getCUserProfileHandler().getCUserProfile()->getUserProfile().hasQuteComAccount());
		_ui->actionSendSms->setEnabled(hasSipAccount);
		_ui->actionCreateConferenceCall->setEnabled(hasSipAccount);
		_ui->actionSearchQuteComContact->setEnabled(hasSipAccount);
	}
}

void QtQuteCom::dialpad(const std::string & tone) {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		_ui->callBar->setPhoneComboBoxEditText(_ui->callBar->getPhoneComboBoxCurrentText() + tone);
	}
}

void QtQuteCom::dialpadBackspace() {
  int s;
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
	  s = _ui->callBar->getPhoneComboBoxCurrentText().size();
	  if(s>0)
	  {
	    std::string str = _ui->callBar->getPhoneComboBoxCurrentText();
	    str.resize(s-1);
		  _ui->callBar->setPhoneComboBoxEditText(str);
		}
	}
}

void QtQuteCom::dialpadEnter() {
  callButtonClicked();
}

void QtQuteCom::prepareToExitApplication() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	// check for pending calls
	CUserProfile *cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
	if (cUserProfile) {
		CPhoneLine *cPhoneLine = cUserProfile->getCPhoneLine();
		if (cPhoneLine) {
			if (cPhoneLine->hasPendingCalls()) {

				if (QMessageBox::question(
					getWidget(),
					tr("@product@ - Warning"),
					tr("You have unfinished call(s).") + "\n" +
						tr("Are you sure you want to exit the application?"),
					tr("&Exit"),
					tr("&Cancel")
				) == 1)  {
					return;
				}
			}
		}
	}
	////


	_qtSystray->hide();

	//Save the window size
	QSize winsize = _qutecomPhoneWindow->size();
	config.set(Config::PROFILE_WIDTH_KEY, winsize.width());
	config.set(Config::PROFILE_HEIGHT_KEY, winsize.height());

	//Save the window position
	QPoint winpos = _qutecomPhoneWindow->pos();
	config.set(Config::PROFILE_POSX_KEY, winpos.x());
	config.set(Config::PROFILE_POSY_KEY, winpos.y());

	QApplication::closeAllWindows();
	QCoreApplication::processEvents();

	//destroyed chatWindow so that chats are saved
	OWSAFE_DELETE(_chatWindow);
	////

	_cQuteCom.terminate();
	
	_want_to_exit = true;
}

void QtQuteCom::phoneComboBoxClicked() {

	_ui->callBar->clearPhoneComboBox();
	_ui->callBar->clearPhoneComboBoxEditText();
}

void QtQuteCom::exitEvent() {
#ifndef MULTIIMWRAPPER
	QCoreApplication::exit(EXIT_SUCCESS);
#endif
}

void QtQuteCom::showHistory() {
	_ui->tabWidget->setCurrentWidget(_ui->tabHistory);
}

void QtQuteCom::currentUserProfileWillDieEventHandlerSlot() {
	//Signal for re-initializing graphical components
	//when there is no user profile anymore
	userProfileDeleted();

	// Reset _qtIMAccountMonitor so that it does not emit signals anymore
	_qtIMAccountMonitor.reset(0);

	OWSAFE_DELETE(_qtFileTransfer);
	OWSAFE_DELETE(_qtIdle);
	_ui->profileBar->reset();

	if (_qtContactList) {
		_ui->tabContactList->layout()->removeWidget(_qtContactList->getWidget());
		OWSAFE_DELETE(_qtContactList);
	}

	if (_qtHistoryWidget) {
		_ui->tabHistory->layout()->removeWidget(_qtHistoryWidget->getWidget());
		OWSAFE_DELETE(_qtHistoryWidget);
	}

	uninstallQtBrowserWidget();
}

void QtQuteCom::userProfileInitializedEventHandlerSlot() {
	CUserProfile * cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();

	//Idle detection
	//FIXME: QtIdle must not use UserProfile but CUserProfile
	_qtIdle = new QtIdle(cUserProfile->getUserProfile(), _qutecomPhoneWindow);

	// Create account monitor
	_qtIMAccountMonitor.reset( new QtIMAccountMonitor(0, cUserProfile) );

	//Create the profilebar
	_ui->profileBar->init(&_cQuteCom, cUserProfile, _qtIMAccountMonitor.get());

	// Systray
	SAFE_CONNECT_RECEIVER_TYPE(_qtIMAccountMonitor.get(), SIGNAL(imAccountAdded(QString)),
		_qtSystray, SLOT(updateSystrayIcon()), Qt::QueuedConnection);
	SAFE_CONNECT_RECEIVER_TYPE(_qtIMAccountMonitor.get(), SIGNAL(imAccountUpdated(QString)),
		_qtSystray, SLOT(updateSystrayIcon()), Qt::QueuedConnection);
	SAFE_CONNECT_RECEIVER_TYPE(_qtIMAccountMonitor.get(), SIGNAL(imAccountRemoved(QString)),
		_qtSystray, SLOT(updateSystrayIcon()), Qt::QueuedConnection);

	_qtSystray->setTrayMenu();
	_qtSystray->updateSystrayIcon();

	_qtFileTransfer = new QtFileTransfer(this, _cQuteCom.getQuteCom().getCoIpManager());

	//menu
	_qtToolBar->userProfileIsInitialized();
}

void QtQuteCom::proxyNeedsAuthenticationEventHandler(NetworkProxyDiscovery & sender, NetworkProxy networkProxy) {
	typedef PostEvent1<void (NetworkProxy networkProxy), NetworkProxy> MyPostEvent;
	MyPostEvent * event =
		new MyPostEvent(boost::bind(&QtQuteCom::proxyNeedsAuthenticationEventHandlerThreadSafe, this, _1), networkProxy);
	postEvent(event);
}

void QtQuteCom::wrongProxyAuthenticationEventHandler(NetworkProxyDiscovery & sender, NetworkProxy networkProxy) {
	typedef PostEvent1<void (NetworkProxy networkProxy), NetworkProxy> MyPostEvent;
	MyPostEvent * event =
		new MyPostEvent(boost::bind(&QtQuteCom::proxyNeedsAuthenticationEventHandlerThreadSafe, this, _1), networkProxy);
	postEvent(event);
}

void QtQuteCom::proxyNeedsAuthenticationEventHandlerThreadSafe(NetworkProxy networkProxy) {
	static QtHttpProxyLogin * httpProxy =
		new QtHttpProxyLogin(getWidget(),
			networkProxy.getServer(), networkProxy.getServerPort());

	int ret = httpProxy->show();

	if (ret == QDialog::Accepted) {
		NetworkProxy myNetworkProxy;
		myNetworkProxy.setServer(httpProxy->getProxyAddress());
		myNetworkProxy.setServerPort(httpProxy->getProxyPort());
		myNetworkProxy.setLogin(httpProxy->getLogin());
		myNetworkProxy.setPassword(httpProxy->getPassword());

		NetworkProxyDiscovery::getInstance().setProxySettings(myNetworkProxy);
	}
}

void QtQuteCom::closeWindow() {
	_qutecomPhoneWindow->hide();
}

//#if defined(OS_MACOSX)
//void QtQuteCom::fixMacOSXMenus() {
	// Avoids translation of these menus on Mac OS X. Thus Qt
	// will put these under the Application menu
	//_ui->actionShowConfig->setText("Preferences");
	//_ui->actionShowAbout->setText("About");
	//_ui->actionShowAbout->setText("Close");
//}
//#endif

void QtQuteCom::languageChanged() {
//#if defined(OS_MACOSX)
	//fixMacOSXMenus();
//#endif
	LOG_DEBUG("retranslate main window ui");
	_ui->retranslateUi(_qutecomPhoneWindow);
	_qtToolBar->retranslateUi();

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	if (_qtBrowserWidget) {
		_ui->tabWidget->setTabText(_qtBrowserWidgetTabIndex, tr("Home"));
	}

	if (_qtHistoryWidget) {
		_qtHistoryWidget->retranslateUi();
	}
        
    _ui->menuQuteCom->setTitle(tr("Account"));
}

void QtQuteCom::showAddContact(ContactInfo contactInfo) {

	ensureVisible();

	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {

		//FIXME this method should not be called if no UserProfile has been set
		ContactProfile contactProfile;
		QtProfileDetails qtProfileDetails(*_cQuteCom.getCUserProfileHandler().getCUserProfile(),
			contactProfile, _qutecomPhoneWindow, tr("Add a Contact"));

		//FIXME to remove when wdeal will be able to handle SIP presence
		if (contactInfo.group == "WDeal") {
			qtProfileDetails.setHomePhone(QUrl::fromPercentEncoding(QByteArray(contactInfo.sip.c_str())));
		} else {
			qtProfileDetails.setQuteComName(QUrl::fromPercentEncoding(QByteArray(contactInfo.qutecomName.c_str())));
		}
		///

		if (contactInfo.group == "WDeal") {
			qtProfileDetails.setFirstName(QUrl::fromPercentEncoding(QByteArray(contactInfo.wdealServiceTitle.c_str())));
		} else {
			qtProfileDetails.setFirstName(QUrl::fromPercentEncoding(QByteArray(contactInfo.firstname.c_str())));
		}

		qtProfileDetails.setLastName(QUrl::fromPercentEncoding(QByteArray(contactInfo.lastname.c_str())));
		qtProfileDetails.setCountry(QUrl::fromPercentEncoding(QByteArray(contactInfo.country.c_str())));
		qtProfileDetails.setCity(QUrl::fromPercentEncoding(QByteArray(contactInfo.city.c_str())));
		qtProfileDetails.setState(QUrl::fromPercentEncoding(QByteArray(contactInfo.state.c_str())));
		qtProfileDetails.setGroup(QUrl::fromPercentEncoding(QByteArray(contactInfo.group.c_str())));
		qtProfileDetails.setWebsite(QUrl::fromPercentEncoding(QByteArray(contactInfo.website.c_str())));

		if (qtProfileDetails.show()) {
			_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCContactList().addContact(contactProfile);
		}
	}
}

void QtQuteCom::bringMainWindowToFront() {
	ensureVisible();
}

void QtQuteCom::ensureVisible() {
	_qutecomPhoneWindow->activateWindow();
	_qutecomPhoneWindow->showNormal();
	_qutecomPhoneWindow->raise();
}

void QtQuteCom::installQtBrowserWidget() 
{	
	if(!_qtBrowserWidget)
	{
		_qtBrowserWidget = new QtBrowserWidget(*this);
		_qtBrowserWidgetTabIndex = _ui->tabWidget->insertTab(0,_qtBrowserWidget->getWidget(), tr("Home"));
		_ui->tabWidget->setCurrentIndex(0);
	}
	_qtBrowserWidget->loadProgressURL();
}

void QtQuteCom::uninstallQtBrowserWidget() 
{
	//if (_qtBrowserWidget) {
	//	_ui->tabWidget->widget(_qtBrowserWidgetTabIndex)->layout()->removeWidget(_qtBrowserWidget->getWidget());
	//	_ui->tabWidget->removeTab(_qtBrowserWidgetTabIndex);
	//	OWSAFE_DELETE(_qtBrowserWidget);
	//}
	_qtBrowserWidget->loadDefaultURL();
}

void QtQuteCom::setCurrentUserProfileEventHandlerSlot()
{
	_qtBrowserWidget->loadProgressURL();
	_qtToolBar->tryingToConnect();
}
