/*
  CoreLinux++ 
  Copyright (C) 2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__BANKACCOUNT_HPP)
#include <BankAccount.hpp>
#endif

using namespace corelinux;

const Amount   ZeroAmount(0.0);

//
// Default constructor
//

BankAccount::BankAccount( void )
   :
   theAvailableFunds( ZeroAmount )
{
   ;  // do nothing
}

//
// Starting balance amount
//

BankAccount::BankAccount( AmountCref aBalance )
   throw( InvalidAmountException )
   :
   theAvailableFunds( aBalance )
{
   if( aBalance < ZeroAmount )
   {
      throw InvalidAmountException
               (
                  "Account can not have a negative balance", 
                  LOCATION 
               );
   }
   else
   {
      ;  // do nothing
   }
}

//
// Copy constructor
//

BankAccount::BankAccount( BankAccountCref anAccount )
   :
   theAvailableFunds( anAccount.getBalance() )
{
   ;  // do nothing
}

//
// Destructor - set balance to zero
//

BankAccount::~BankAccount( void )
{
   theAvailableFunds = ZeroAmount;
}

//
// Operator assignment
//

BankAccountRef BankAccount::operator=( BankAccountCref anAccount )
{
   if( (*this == anAccount) == false )
   {
      theAvailableFunds = anAccount.getBalance();
   }
   else
   {
      ;  // do nothing
   }

   return ( *this );
}

//
// Operator equality
//

bool  BankAccount::operator==( BankAccountCref anAccount ) const
{
   return ( this == &anAccount );
}

//
// Retrieve the balance
//

AmountCref  BankAccount::getBalance( void ) const
{
   return theAvailableFunds;
}

//
// Deposit into the account
//

void  BankAccount::depositFunds( Amount anAmount )
                        throw(InsufficientFundsException)
{
   if( ( getBalance() + anAmount ) > ZeroAmount )
   {
      theAvailableFunds += anAmount;
   }
   else
   {
      throw InsufficientFundsException
               (
                  "Deposit leaves insufficient funds",
                  LOCATION
               );
   }
}

//
// Withdraw from the account
//

void  BankAccount::withdrawFunds( Amount anAmount )
                        throw(InsufficientFundsException)
{
   if( ( getBalance() - anAmount ) > ZeroAmount )
   {
      theAvailableFunds -= anAmount;
   }
   else
   {
      throw InsufficientFundsException
               (
                  "Withdraw leaves insufficient funds.",
                  LOCATION
               );
   }
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.1 $
   $Date: 2000/02/08 13:13:17 $
   $Locker:  $
*/


