/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.query.functionscore.lin;

import org.apache.lucene.search.ComplexExplanation;
import org.apache.lucene.search.Explanation;
import org.elasticsearch.index.query.functionscore.DecayFunction;
import org.elasticsearch.index.query.functionscore.DecayFunctionParser;

public class LinearDecayFunctionParser extends DecayFunctionParser {

    public static final String[] NAMES = { "linear" };

    @Override
    public String[] getNames() {
        return NAMES;
    }

    static final DecayFunction decayFunction = new LinearDecayScoreFunction();

    @Override
    public DecayFunction getDecayFunction() {
        return decayFunction;
    }

    final static class LinearDecayScoreFunction implements DecayFunction {

        @Override
        public double evaluate(double value, double scale) { 
            return Math.max(0.0, (scale - value) / scale);
        }

        @Override
        public Explanation explainFunction(String valueExpl, double value, double scale) {
            ComplexExplanation ce = new ComplexExplanation();
            ce.setValue((float) evaluate(value, scale));
            ce.setDescription("max(0.0, ((" + scale + " - " + valueExpl + ")/" + scale + ")");
            return ce;
        }

        @Override
        public double processScale(double scale, double decay) {
            return scale / (1.0 - decay);
        }

    }
}
