/* This is a simple little program that users can use to 
   easily get the version of the distribution installed,
   and if wanted the version of the specific program
   that CAME WITH the distribution this program is 
   distributed with.  This program is updated with each
   release, and simply displays hard coded information.
   Only covers boot and base.
*/

#define DIST_VERSION "FreeDOS Ripcord BETA 8 H01 (\"Nikita\") Distribution"
#define HOT_RELEASE

#define NA 1                /* not currently available                          */
#define INTERNAL 32768      /* command is internal to FreeCom                   */
#define UPDATED 2           /* updated in this ripcord release                  */
#define RIPCORD 4           /* updated in a ripcord, but not the current one    */
#define BETA6 8             /* indicates available in Beta6 or earlier          */
#define BETA7 16            /* indicates available released in Beta7            */
#define BETA8 32            /* indicates available released in Beta8            */
#define ALL (NA | BETA8 | BETA7 | BETA6 | RIPCORD | UPDATED | INTERNAL)
#define BETA (BETA8 | BETA7 | BETA6)

#define NUM_PROGRAMS 70
typedef struct PROGINFO
{
  char * name;              /* The name of the program                    */
  char * version;           /* String representing the version            */
  unsigned int  updated;    /* indicates if updated this release or not   */
} PROGINFO;

PROGINFO programs[NUM_PROGRAMS] = 
{
  { "Kernel",   "Build 2.0.27test !upd to 2027! ", UPDATED },
  { "Install",  "3.76                           ", BETA8 },
  { "TE",       "1.0 Revision 1.4               ", BETA8 },
  { "Assign",   "1.4                            ", BETA6 },
  { "Attrib",   "2.00                           ", UPDATED },
  { "Backup",   "Not Available                  ", NA },
  { "break",    "internal                       ", INTERNAL },
  { "cd",       "internal (aka chdir)           ", INTERNAL },
/*{ "Chcp",     "GNU Chcp 1.0                   ", RIPCORD },  --- UTIL LoadFont */
  { "chcp",     "Not Available                  ", NA },
  { "Chkdsk",   "Not Available                  ", NA },
  { "cls",      "internal                       ", INTERNAL },
  { "Choice",   "3.6c                           ", UPDATED },
  { "FreeCom",  "0.83 beta 32                   ", BETA8 },
  { "Comp",     "0.1pv2                         ", BETA6 },
  { "copy",     "internal                       ", INTERNAL },
  { "Debug",    "0.95c                          ", BETA8 },
  { "Defrag",   "alpha 0.2                      ", BETA6 },
  { "del",      "internal (aka erase)           ", INTERNAL },
  { "Deltree",  "1.02c                          ", BETA6 },
  { "dir",      "internal                       ", INTERNAL },
  { "Diskcomp", "0.72                           ", BETA7 },
  { "Diskcopy", "beta 0.6                       ", BETA7 },
  { "echo",     "internal                       ", INTERNAL },
  { "Edit",     "0.5                            ", BETA7 },
  { "Emm386",   "0.4                            ", BETA8 },
/*{ "Fremm386", "Not Available                  ", NA }, */  /* maybe in the future if anyone has time! :-) */
  { "Exe2bin",  "1.0                            ", BETA6 },
  { "Fc",       "2.0                            ", BETA6 },
  { "FDHelp",   "3.4a                           ", UPDATED },
  { "Fdisk",    "1.01 / 1.10beta                ", UPDATED },
  { "FDXMS",    "0.9.Sand                       ", BETA8 },
  { "FDXMS286", "0.01.Espansiva                 ", BETA8 },
  { "Find",     "2.5b                           ", UPDATED },
  { "Format",   "0.90                           ", UPDATED },
  { "Graphics", "Not Available                  ", NA },
  { "Htmlhelp", "4.2                            ", BETA8 },
  { "Label",    "1.3                            ", BETA6 },
  { "loadhigh", "internal                       ", INTERNAL },
  { "Mem",      "1.3                            ", BETA7 },
  { "md",       "internal (aka mkdir)           ", INTERNAL },
  { "Mirror",   "0.2                            ", BETA6 },
  { "Mkeyb",    "0.27                           ", UPDATED }, /* NA */
  { "Mode",     "1.04                           ", BETA6 },
  { "More",     "3.4e                           ", UPDATED },
  { "Move",     "2.0                            ", UPDATED },
  { "Nansi",    "3.4                            ", BETA6 },
  { "Nlsfunc",  "Not Available                  ", NA },
  { "pause",    "internal                       ", INTERNAL },
  { "Print",    "1.0                            ", BETA6 },
  { "PrintQ",   "05-Jul-1997                    ", BETA6 },
  { "Replace",  "1.2                            ", BETA6 },
  { "ren",      "internal (aka rename)          ", INTERNAL },
  { "rd",       "internal (aka rmdir)           ", INTERNAL },
  { "Restore",  "Not Available                  ", NA },
  { "Scandisk", "0.5                            ", BETA6 },
  { "Share",    "FreeDOS share, from Kernel     ", BETA7 },
  { "Shsucdx",  "1.4b (aka fdcdex)              ", BETA6 },
  { "Sort",     "1.0                            ", BETA6 },
  { "Swsubst",  "3.1                            ", BETA6 },
  { "Sys",      "2.5                            ", UPDATED },
  { "Tdsk",     "2.3                            ", BETA7 },
  { "time",     "internal                       ", INTERNAL },
  { "Tree",     "3.7.2                          ", UPDATED },
  { "type",     "internal                       ", INTERNAL },
  { "Undelete", "ver ?, 2002-06-23              ", UPDATED },  /* NA */
  { "Unformat", "0.8                            ", BETA6 },
  { "ver",      "internal                       ", INTERNAL },
  { "verify",   "internal                       ", INTERNAL },
  { "vol",      "internal                       ", INTERNAL },
  { "Xcopy",    "1.0 (Rene's)                   ", BETA8 },
  { "Xkeyb",    "1.13 (1 point thirteen)        ", UPDATED }
};

#include <stdio.h>
#include <string.h>

void print_usage(void)
{
  printf("ripcord [PROGRAM | ALL | BETA[6|7|8] | "
#ifdef HOT_RELEASE
         "RIPCORD | UPDATED | "
#endif
         "NA | INTERNAL]\n");
  printf("  PROGRAM is the program to display version information for.\n");
  printf("  Do NOT include extension, eg use FORMAT not FORMAT.EXE\n");
  printf("\n");
  printf("  The remaining choices are keywords that will display\n");
  printf("  version information for the corresponding programs included\n");
  printf("  on the boot disk and in the base package:\n");
  printf("    ALL     : all programs on boot disk & in base (including INTERNAL)\n");
  printf("    BETA#   : only those updated in Beta [6/7/8]\n");
#ifdef HOT_RELEASE
  printf("    RIPCORD : all programs updated since Beta 8\n");
  printf("    UPDATED : all programs updated in this Ripcord release\n");
#endif
  printf("    NA      : \"standard\" programs not yet available\n");
  printf("    INTERNAL: programs internal to FreeCOM\n");
  printf("\n");
  printf("This program displays hard coded information and only\n");
  printf("reflects information for the distribution it is included with.\n");
  printf("\n");
  printf("Maintained by Kenneth J Davis <jeremyd@computer.org>\n");
  printf("It is public domain.\n");
}


/* Procedures */

int getIndexFor(char *progName)
{
  register int i;
  for (i = 0; i < NUM_PROGRAMS; i++)
    if (strncmpi(progName, programs[i].name, 8) == 0)
      return i;

  return -1;
}

void display_info(int index)
{
  printf("%8s : %s [%s]\n", programs[index].name, programs[index].version,
         ((programs[index].updated & BETA6)?"Beta6" :
         ((programs[index].updated & BETA7)?"Beta7" :
         ((programs[index].updated & BETA8)?"Beta8" :
         ((programs[index].updated & RIPCORD)?"Updated since Beta8" : 
         ((programs[index].updated & UPDATED)?"Updated this release" :
         ((programs[index].updated & NA)?"NA" :
         ((programs[index].updated & INTERNAL)?"Internal to FreeCom" : "Unknown status" ))))))));
}

void display(int flags)
{
  register int i;
  for (i = 0; i < NUM_PROGRAMS; i++)
    if (programs[i].updated & flags)
      display_info(i);
}

int main(int argc, char *argv[])
{
  register int i;
  register int index;

  printf(DIST_VERSION "\n\n");

  for (i = 1; i < argc; i++)
  {
    if ((argv[i][1] == '?') || (strcmpi(argv[i]+1, "HELP") == 0))
    {
      print_usage();
      return 0;
    }
    else if (strcmpi(argv[i], "ALL") == 0)
    {
      display(ALL);
    }
    else if (strcmpi(argv[i], "NA") == 0)
    {
      display(NA);
    }
    else if (strcmpi(argv[i], "INTERNAL") == 0)
    {
      display(INTERNAL);
    }
    else if (strcmpi(argv[i], "BETA") == 0)
    {
      display(BETA);
    }
    else if (strcmpi(argv[i], "BETA6") == 0)
    {
      display(BETA6);
    }
    else if (strcmpi(argv[i], "BETA7") == 0)
    {
      display(BETA7);
    }
    else if (strcmpi(argv[i], "BETA8") == 0)
    {
      display(BETA8);
    }
#ifdef HOT_RELEASE
    else if (strcmpi(argv[i], "RIPCORD") == 0)
    {
      display(RIPCORD | UPDATED);
    }
    else if (strcmpi(argv[i], "UPDATED") == 0)
    {
      display(UPDATED);
    }
#endif
    else /* probably a specific program */
    {
      index = getIndexFor(argv[i]);
      if ((index >= 0) && (index < NUM_PROGRAMS))
        display_info(index);
      else
        printf("Unknown program [%s]\n", argv[i]);
    }
  }

  return 0;
}
