{
    This file is the source for a series of routines to detect the make and
    model of the CPU the program is running on.  
    Copyright (C) 1998 by Phil Brutsche

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA  02111-1307  USA.
}

unit cpu;

interface

function is_8088 : boolean;
function is_80286 : boolean;
function is_80386 : boolean;
function is_nexgen : boolean;
function is_80486 : boolean;
procedure do_cpuid (level : longint; var eax, ebx, ecx, edx : longint);
function iscyrix : boolean;

implementation

function is_8088 : boolean;
label done;
var
  rv : byte;
begin
  asm
    { The 8086 thanks to Robert Collins at rcollins@x86.org. }
    {mov ax,00FFh                    { clear CPU type return register         }
    push sp			                    { save SP on stack to look at            }
    pop bx                          { get SP saved on stack                  }
    cmp bx,sp                       { if 8086/8088, these values will differ }
    mov rv, 0
    jz done                         { nope, must be other CPU type           }
    mov rv, 1
done:
  end;
  is_8088 := rv = 1;
end;

function is_80286 : boolean;
{  we know its at least a second generation cpu.  if bits 12-16 are
   toggleable, then it's 80286.  microcode-wise indistinguishable from a
   80186, but the 186 doesn't support protected mode.
}
label done;
var
  rv : byte;
begin
  asm
    or cx, $F000
    push cx
    popf
    pushf
    pop ax
    and ax, $F000
    mov rv, 0
    jnz done
    mov rv, 1
done:
  end;
  is_80286 := rv = 1;
end;

function is_80386 : boolean;
{  we know that this cpu supports 32-bit instructions, so its at least
   a 80386.  if bit 18 of eflags is toggleable, then we know that its
   a 80386 or clone and we stop. }
label done;
var
  rv : byte;
begin
  asm
    mov bx, sp
    and sp, not 3
    db $66; pushf
    db $66; pop ax
    db $66; mov cx, ax
    db $66; db $35; dd $40000          { xor eax, $40000 }
    db $66; push ax
    db $66; popf
    db $66; pushf
    db $66; pop ax
    db $66; xor ax, cx
    mov sp, bx
    { jz to end if we don't have a 486 here }
    { jnz to '486 checking if we do }
    mov rv, 0
    jnz done
    mov rv, 1
done:
  end;
  is_80386 := rv = 1;
end;

function is_nexgen : boolean;
label done, not_Nx586, is_Nx586;
var
  rv : byte;
begin
  asm
    and sp, not 3
    db $66; push cx
    db $66; popf
    mov sp, bx
    mov ax, $5555
    xor dx, dx
    mov cx, $2
    div cx
    jnz not_Nx586
    jz is_Nx586
  end;
not_Nx586:
  is_nexgen := false;
  goto done;
is_Nx586:
  is_nexgen := true;
done:
end;

function is_80486 : boolean;
  {  we know that its at least a 80486 or clone.  now we check for the
     cpuid instruction.  intel included it in 1992 on all processors, with
     the advent of the pentium.  (i mean, when intel started making the
     pentium, they introduced this instruction.  when they did that, they
     said:  what the hell, lets put it in everything we make from now on.
     and so we have it.)
     we try to toggle eflags register 21.  if its toggleable, then the cpuid
     instruction is not supported.
     if its not, then we know that it's a 80486.
  }
label done;
var
  rv : byte;
begin
  asm
    db $66; mov ax, cx
    db $66; db $35; dd $200000
    db $66; push ax
    db $66; popf
    db $66; pushf
    db $66; pop ax
    db $66; xor ax, cx
    {je end_getcpuid;}
    mov rv, 1
    je done
    mov rv, 0
  end;
done:
  is_80486 := rv = 1;
end;

procedure do_cpuid (level : longint; var eax, ebx, ecx, edx : longint);
var
  _eax, _ebx, _ecx, _edx : longint;
begin
  asm
    db $66; xor ax, ax
    db $66; mov ax, word ptr level
    db $0F, $A2                        { CPUID }
    db $66; mov word ptr _eax, ax
    db $66; mov word ptr _ebx, bx
    db $66; mov word ptr _edx, dx
    db $66; mov word ptr _ecx, cx
  end;
  eax := _eax;
  ebx := _ebx;
  ecx := _ecx;
  edx := _edx;
end;

function iscyrix : boolean;
label not_cyrix, done;
begin
  asm
    xor ax, ax               { clear ax }
    sahf                     { clear flags, bit 1 is always 1 in flags }
    mov ax, 5
    mov bx, 2
    div bl                   { do an operation that does not change flags }
    lahf                     { get flags }
    cmp ah, 2                { check for change in flags }
    jne not_cyrix            { flags changed not Cyrix }
    mov ax, 1                { TRUE Cyrix CPU }
    jmp done
not_cyrix:
    mov  ax, 0               { FALSE NON-Cyrix CPU }
done:
  end;
end;

begin
  writeln ('CPU.TPU initializing...Done');
end.
