/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1999, 2000, 2001 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/disk_bsd.h>
#include <parted/disk_dos.h>
#include <parted/disk_loop.h>
#include <parted/disk_mac.h>
#include <parted/disk_pc98.h>
#include <parted/disk_sun.h>
#include <parted/disk_gpt.h>

#include <string.h>
#include <stdlib.h>

#include <locale.h>
#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

/* TODO: make file systems modules.  Actually this will probably require
 * 2 parts, because we always want clobber and probe to be available
 */
extern void ped_file_system_ext2_init ();
extern void ped_file_system_fat_init ();
extern void ped_file_system_hfs_init ();
extern void ped_file_system_jfs_init ();
extern void ped_file_system_linux_swap_init ();
extern void ped_file_system_ntfs_init ();
extern void ped_file_system_reiserfs_init ();
extern void ped_file_system_ufs_init ();
extern void ped_file_system_xfs_init ();
	
extern void ped_file_system_ext2_done ();
extern void ped_file_system_fat_done ();
extern void ped_file_system_hfs_done ();
extern void ped_file_system_jfs_done ();
extern void ped_file_system_linux_swap_done ();
extern void ped_file_system_ntfs_done ();
extern void ped_file_system_reiserfs_done ();
extern void ped_file_system_ufs_done ();
extern void ped_file_system_xfs_done ();

/* ped_malloc() debugging.  Stick the address and size of memory blocks that
 * weren't ped_free()d in here, and an exception will be thrown when it is
 * allocated.  That way, you can find out what, exactly, the allocated thing
 * is, and where it is created.
 */
typedef struct
{
    void*	pointer;
    size_t	size;
} pointer_size_type;

static pointer_size_type dodgey_malloc_list[] = {
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0},
 {0,		0}
};

static int	dodgey_memory_active[100];

/* Note: loop must be added AFTER msdos, because DOS partition tables and 
 * FAT file systems are hard to tell apart :-/
 */
static void
init_disk_types ()
{
	ped_disk_loop_init ();
	ped_disk_msdos_init ();
#ifdef ENABLE_PC98
	ped_disk_pc98_init ();
#endif
	ped_disk_mac_init ();
	ped_disk_bsd_init ();
	ped_disk_gpt_init ();
	ped_disk_sun_init ();
}

static void
init_file_system_types ()
{
	ped_file_system_xfs_init ();
	ped_file_system_ufs_init ();
	ped_file_system_reiserfs_init ();
	ped_file_system_ntfs_init ();
	ped_file_system_linux_swap_init ();
	ped_file_system_jfs_init ();
	ped_file_system_hfs_init ();
	ped_file_system_fat_init ();
	ped_file_system_ext2_init ();
}

int
ped_init ()
{
#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif

	init_disk_types ();
	init_file_system_types ();

	return 1;
}

static void
done_disk_types ()
{
	ped_disk_msdos_done ();
#ifdef ENABLE_PC98
        ped_disk_pc98_done ();
#endif
	ped_disk_loop_done ();
	ped_disk_mac_done ();
	ped_disk_bsd_done ();
	ped_disk_gpt_done ();
	ped_disk_sun_done ();
}

static void
done_file_system_types ()
{
	ped_file_system_ext2_done ();
	ped_file_system_fat_done ();
	ped_file_system_hfs_done ();
	ped_file_system_jfs_done ();
	ped_file_system_linux_swap_done ();
	ped_file_system_ntfs_done ();
	ped_file_system_reiserfs_done ();
	ped_file_system_ufs_done ();
	ped_file_system_xfs_done ();
}

void
ped_done ()
{
	ped_device_free_all ();

	done_disk_types ();
	done_file_system_types ();
}

const char*
ped_get_version ()
{
	return VERSION;
}

static void
_check_dodgey_pointer (const void* ptr, size_t size, int is_malloc)
{
	int		i;

	for (i=0; dodgey_malloc_list[i].pointer; i++) {
		if (dodgey_malloc_list[i].pointer != ptr)
			continue;
		if (is_malloc && dodgey_malloc_list[i].size != size)
			continue;
		if (!is_malloc && !dodgey_memory_active[i])
			continue;


		if (is_malloc) {
			ped_exception_throw (
				PED_EXCEPTION_INFORMATION,
				PED_EXCEPTION_OK,
				"Dodgey malloc(%x) == %p occured (active==%d)",
				size, ptr, dodgey_memory_active[i]);
			dodgey_memory_active[i]++;
		} else {
			ped_exception_throw (
				PED_EXCEPTION_INFORMATION,
				PED_EXCEPTION_OK,
				"Dodgey free(%p) occured (active==%d)",
				ptr, dodgey_memory_active[i]);
			dodgey_memory_active[i]--;
		}

		return;
	}
}

void*
ped_malloc (size_t size)
{
	void*		mem;

	mem = (void*) malloc (size);
	if (!mem) {
		ped_exception_throw (PED_EXCEPTION_FATAL, PED_EXCEPTION_CANCEL,
				     _("Out of memory."));
		return NULL;
	}
	memset (mem, 0xff, size);	/* assist debugging */

	_check_dodgey_pointer (mem, size, 1);

	return mem;
}

int
ped_realloc (void** old, size_t size)
{
	void*		mem;

	mem = (void*) realloc (*old, size);
	if (!mem) {
		ped_exception_throw (PED_EXCEPTION_FATAL, PED_EXCEPTION_CANCEL,
				     _("Out of memory."));
		return 0;
	}
	*old = mem;
	return 1;
}

void
ped_free (void* ptr)
{
	_check_dodgey_pointer (ptr, 0, 0);
	free (ptr);
}
