/*
    libparted - a library for manipulating disk partitions

    disk_gpt.[ch] by Matt Domsch <Matt_Domsch@dell.com>
    Disclaimed into the Public Domain

    EFI GUID Partition Table handling
    Per Intel EFI Specification v1.02
    http://developer.intel.com/technology/efi/efi.htm

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 
   TODO:
   - Make partition labels get/set properly
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/crc32.h>
#include <parted/disk_gpt.h>
#include <parted/natmath.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <uuid/uuid.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif				/* ENABLE_NLS */

#define MIN_FREESPACE		(150 * 2)	/* 150k */


static int gpt_probe(const PedDevice * dev);
static PedDisk *gpt_open(PedDevice * dev);
static PedDisk *gpt_create(PedDevice * dev);
static int gpt_clobber(PedDevice * dev);
static int gpt_close(PedDisk * disk);
static int gpt_read(PedDisk * disk);
static int gpt_write(PedDisk * disk);


static PedPartition *gpt_partition_new(const PedDisk *disk,
			     PedPartitionType part_type,
			     const PedFileSystemType* fs_type,
			     PedSector start,
			     PedSector end);
static void gpt_partition_destroy(PedPartition *part);
static int gpt_partition_set_flag(PedPartition *part,
				  PedPartitionFlag flag,
				  int state);
static int gpt_partition_get_flag(const PedPartition *part,
				  PedPartitionFlag flag);
static int gpt_partition_is_flag_available(const PedPartition * part,
					   PedPartitionFlag flag);
static void gpt_partition_set_name(PedPartition *part,
				   const char *name);
static const char * gpt_partition_get_name (const PedPartition * part);
static int gpt_partition_align(PedPartition * part,
			       const PedConstraint * constraint);
static int gpt_partition_enumerate(PedPartition * part);

static int gpt_alloc_metadata(PedDisk * disk);
static int gpt_get_max_primary_partition_count(const PedDisk *disk);

/* gpt private function */
static PedDisk * gpt_new(PedDisk * disk);

static PedDiskOps gpt_disk_ops = {
	probe                           : gpt_probe,
	open                            : gpt_open,
	create                          : gpt_create,
	clobber                         : gpt_clobber,
	close                           : gpt_close,
	read                            : gpt_read,
	write                           : gpt_write,
	partition_new                   : gpt_partition_new,
	partition_destroy               : gpt_partition_destroy,
	partition_set_flag              : gpt_partition_set_flag,
	partition_get_flag              : gpt_partition_get_flag,
	partition_is_flag_available     : gpt_partition_is_flag_available,
	partition_set_name              : gpt_partition_set_name,
	partition_get_name              : gpt_partition_get_name,
	partition_align                 : gpt_partition_align,
	partition_enumerate             : gpt_partition_enumerate,
	partition_set_extended_system   : NULL,
	alloc_metadata                  : gpt_alloc_metadata,
	get_max_primary_partition_count : gpt_get_max_primary_partition_count

};

static PedDiskType gpt_disk_type = {
	next:NULL,
	name:GPT_NAME,
	ops:&gpt_disk_ops,
	features: PED_DISK_TYPE_PARTITION_NAME
};



static void
gpt_le_guid_to_cpu(efi_guid_t *guid)
{
	guid->time_low            = PED_LE32_TO_CPU(guid->time_low);
	guid->time_mid            = PED_LE16_TO_CPU(guid->time_mid);
	guid->time_hi_and_version = PED_LE16_TO_CPU(guid->time_hi_and_version);
	/* no need to change clock_seq and node[6].
	   They're already arrays of chars */
	return;
}

static void
gpt_cpu_to_le_guid(efi_guid_t *guid)
{
	guid->time_low            = PED_CPU_TO_LE32(guid->time_low);
	guid->time_mid            = PED_CPU_TO_LE16(guid->time_mid);
	guid->time_hi_and_version = PED_CPU_TO_LE16(guid->time_hi_and_version);
	/* no need to change clock_seq and node[6].
	   They're already arrays of chars */
	return;
}

static void
gpt_le_part_attributes_to_cpu(GuidPartitionEntryAttributes_t *a)
{
	uint64_t *b = (uint64_t *)a;
	*b = PED_LE64_TO_CPU(*b);
}

static void
gpt_cpu_to_le_part_attributes(GuidPartitionEntryAttributes_t *a)
{
	uint64_t *b = (uint64_t *)a;
	*b = PED_CPU_TO_LE64(*b);
}


/************************************************************
 * efi_crc32()
 * Requires:
 *  - a buffer of length len
 * Modifies: nothing
 * Returns:
 *  EFI-style CRC32 value for buf
 *  
 * This function uses the crc32 function by Gary S. Brown,
 * but seeds the function with ~0, and xor's with ~0 at the end.
 ************************************************************/

static inline uint32_t
efi_crc32(const void *buf, unsigned long len)
{
	return (__efi_crc32(buf, len, ~0L) ^ ~0L);
}


static inline int
is_pmbr_valid(LegacyMBR_t * mbr)
{
	int i, found = 0, signature = 0;
	PED_ASSERT(mbr != NULL, return 0);
	signature = (mbr->Signature == MSDOS_MBR_SIGNATURE);
	for (i=0; signature && i<4; i++) {
		if (mbr->PartitionRecord[i].OSType ==
		    EFI_PMBR_OSTYPE_EFI_GPT) {
			found = 1;
			break;
		}
	}
	if (signature && found) return 1;
	return 0;
}

static inline int
efi_guidcmp(efi_guid_t left, efi_guid_t right)
{
	return memcmp(&left, &right, sizeof(efi_guid_t));
}


/************************************************************
 * last_lba()
 * Requires:
 *  - dev
 * Modifies: nothing
 * Returns:
 *  Last LBA value on success 
 *  0 on error
 ************************************************************/


static uint64_t
last_lba(const PedDevice * dev)
{
	PED_ASSERT(dev != NULL, return 0);
	return dev->length - 1;
}

/************************************************************
 * gpt_read_part_entries()
 * Requires:
 *  - dev
 *  - lba is the Logical Block Address of the partition table
 *  - gpt is a buffer into which the GPT will be put  
 * Modifies:
 *  - dev
 *  - gpt
 * Returns:
 *   pte on success
 *   NULL on error
 * Notes: remember to free pte when you're done!
 ************************************************************/
static GuidPartitionEntry_t *
gpt_read_part_entries(const PedDevice * dev,
		      GuidPartitionTableHeader_t * gpt)
{
	GuidPartitionEntry_t *ptes;
	unsigned int i, j;

	PED_ASSERT(dev != NULL, return NULL);
	PED_ASSERT(dev->sector_size != 0, return NULL);
	PED_ASSERT(gpt != NULL, return NULL);

	ptes = (GuidPartitionEntry_t *)
		ped_malloc(gpt->NumberOfPartitionEntries *
			   gpt->SizeOfPartitionEntry);

	PED_ASSERT(ptes != NULL, return NULL);

	memset(ptes, 0, gpt->NumberOfPartitionEntries *
	       gpt->SizeOfPartitionEntry);


	if (!ped_device_read(dev, ptes, gpt->PartitionEntryLBA,
		     gpt->NumberOfPartitionEntries *
		     gpt->SizeOfPartitionEntry / dev->sector_size)) {
		ped_free(ptes);
		return NULL;
	}

	/* Fixup endianness */
	for (i=0; i<gpt->NumberOfPartitionEntries; i++) {
		gpt_le_guid_to_cpu(&ptes[i].PartitionTypeGuid);
		gpt_le_guid_to_cpu(&ptes[i].UniquePartitionGuid);
		ptes[i].StartingLBA = PED_LE64_TO_CPU(ptes[i].StartingLBA);
		ptes[i].EndingLBA   = PED_LE64_TO_CPU(ptes[i].EndingLBA);
		gpt_le_part_attributes_to_cpu(&ptes[i].Attributes);
		for (j=0; j<(72/sizeof(efi_char16_t)); j++) {
			ptes[i].PartitionName[j] = (efi_char16_t)(PED_LE16_TO_CPU((uint16_t)(ptes[i].PartitionName[j])));
		}
	}

	return ptes;
}

/************************************************************
 * gpt_write_part_entries()
 * Requires:
 *  - PedDevice *dev
 *  - gpt 
 *  - pte is a buffer that will be written
 * Modifies:
 *  - dev
 *  - gpt
 * Returns:
 *   pte on success
 *   NULL on error
 * Notes: remember to free pte when you're done!
 ************************************************************/
static GuidPartitionEntry_t *
gpt_write_part_entries(PedDevice * dev,
		       GuidPartitionTableHeader_t * gpt,
		       GuidPartitionEntry_t *ptes)
{
	GuidPartitionEntry_t *new_ptes;
	unsigned long ptes_size;
	int rc;
	unsigned int i, j;

	PED_ASSERT(gpt != NULL, return NULL);
	PED_ASSERT(ptes != NULL, return NULL);
	PED_ASSERT(dev != NULL, return NULL);
	PED_ASSERT(dev->sector_size != 0, return NULL);

	ptes_size = gpt->NumberOfPartitionEntries *
		gpt->SizeOfPartitionEntry;

	new_ptes = ped_malloc(ptes_size);
	if (!new_ptes) return NULL;
	memcpy(new_ptes, ptes, ptes_size);

	/* Fixup endianness */
	for (i=0; i<gpt->NumberOfPartitionEntries; i++) {
		gpt_cpu_to_le_guid(&new_ptes->PartitionTypeGuid);
		gpt_cpu_to_le_guid(&new_ptes->UniquePartitionGuid);
		new_ptes[i].StartingLBA = PED_CPU_TO_LE64(new_ptes[i].StartingLBA);
		new_ptes[i].EndingLBA   = PED_CPU_TO_LE64(new_ptes[i].EndingLBA);
		gpt_cpu_to_le_part_attributes(&new_ptes[i].Attributes);

		for (j=0; j<(72/sizeof(efi_char16_t)); j++) {
			new_ptes[i].PartitionName[j] = (efi_char16_t)(PED_CPU_TO_LE16((uint16_t)(new_ptes[i].PartitionName[j])));
		}
	}

	rc = ped_device_write(dev, new_ptes, gpt->PartitionEntryLBA,
			      ptes_size / dev->sector_size);
	ped_free(new_ptes);
	if (!rc) return NULL;
	return ptes;
}


static void
gpt_print_part_entry(GuidPartitionEntry_t * pte, int i)
{
	efi_guid_t unused_guid = UNUSED_ENTRY_GUID;
	char uuid_buffer[40];
	uuid_t uuid;
	PED_ASSERT(pte != NULL, return);
	if (!efi_guidcmp(pte->PartitionTypeGuid, unused_guid)) {
		// printf("UNUSED_ENTRY_GUID\n");
		return;
	}
	printf("GUID Partition Entry %d:\n", i);
	memcpy(uuid, &pte->PartitionTypeGuid, sizeof(uuid_t));
	uuid_unparse(uuid, uuid_buffer);
	printf("\tPartitionTypeGuid : %s\n", uuid_buffer);
	memcpy(uuid, &pte->UniquePartitionGuid, sizeof(uuid_t));
	uuid_unparse(uuid, uuid_buffer);
	printf("\tUniquePartitionGuid : %s\n", uuid_buffer);
	printf("\tStartingLBA : " PRIx64 "\n", pte->StartingLBA);
	printf("\tEndingLBA   : " PRIx64 "\n", pte->EndingLBA);
	printf("\tAttributes  : ");
	printf("\tRequiredToFunction: %x",
	       pte->Attributes.RequiredToFunction);
	printf("\tGuidSpecific: %x\n",
	       pte->Attributes.GuidSpecific);

	//  printf("\tPartitionName : Unicode string.\n");
	return;
}

static void
gpt_print_header(GuidPartitionTableHeader_t * gpt)
{
	char uuid_buffer[40];
	uuid_t uuid;
	printf("GUID Partition Table Header\n");
	PED_ASSERT(gpt != NULL, return);
	printf("Signature      : 0x" PRIx64 "\n", gpt->Signature);
	printf("Revision       : 0x%x\n", gpt->Revision);
	printf("HeaderSize     : 0x%x\n", gpt->HeaderSize);
	printf("HeaderCRC32    : 0x%x\n", gpt->HeaderCRC32);
	printf("MyLBA          : 0x" PRIx64 "\n", gpt->MyLBA);
	printf("AlternateLBA   : 0x" PRIx64 "\n", gpt->AlternateLBA);
	printf("FirstUsableLBA : 0x" PRIx64 "\n", gpt->FirstUsableLBA);
	printf("LastUsableLBA  : 0x" PRIx64 "\n", gpt->LastUsableLBA);
	memcpy(uuid, &gpt->DiskGUID, sizeof(uuid_t));
	uuid_unparse(uuid, uuid_buffer);
	printf("DiskGUID : %s\n", uuid_buffer);
	printf("PartitionEntryLBA : " PRIx64 "\n", gpt->PartitionEntryLBA);
	printf("NumberOfPartitionEntries : %x\n",
	       gpt->NumberOfPartitionEntries);
	printf("SizeOfPartitionEntry : %x\n", gpt->SizeOfPartitionEntry);
	printf("PartitionEntryArrayCRC32 : %x\n",
	       gpt->PartitionEntryArrayCRC32); return;
}


/************************************************************
 * gpt_read_header()
 * Requires:
 *  - dev
 *  - lba is the Logical Block Address of the partition table
 * Modifies:
 *  - dev
 * Returns:
 *   GPTH on success
 *   NULL on error
 ************************************************************/
static GuidPartitionTableHeader_t *
gpt_read_header(const PedDevice * dev,
		uint64_t lba)
{
	GuidPartitionTableHeader_t *gpt;
	PED_ASSERT(dev != NULL, return 0);
	gpt = (GuidPartitionTableHeader_t *)
	    ped_malloc(sizeof(GuidPartitionTableHeader_t));
	if (!gpt) return NULL;
	memset(gpt, 0, sizeof (*gpt));
	if (!ped_device_read(dev, gpt, lba, 1)) {
		ped_free(gpt);
		return NULL;
	}


	/* Fixup endianness */
	gpt->Signature                = PED_LE64_TO_CPU(gpt->Signature);
	gpt->Revision                 = PED_LE32_TO_CPU(gpt->Revision);
	gpt->HeaderSize               = PED_LE32_TO_CPU(gpt->HeaderSize);
	gpt->HeaderCRC32              = PED_LE32_TO_CPU(gpt->HeaderCRC32);
	gpt->Reserved1                = PED_LE32_TO_CPU(gpt->Reserved1);
	gpt->MyLBA                    = PED_LE64_TO_CPU(gpt->MyLBA);
	gpt->AlternateLBA             = PED_LE64_TO_CPU(gpt->AlternateLBA);
	gpt->FirstUsableLBA           = PED_LE64_TO_CPU(gpt->FirstUsableLBA);
	gpt->LastUsableLBA            = PED_LE64_TO_CPU(gpt->LastUsableLBA);
	gpt->PartitionEntryLBA        = PED_LE64_TO_CPU(gpt->PartitionEntryLBA);
	gpt->NumberOfPartitionEntries = PED_LE32_TO_CPU(gpt->NumberOfPartitionEntries);
	gpt->SizeOfPartitionEntry     = PED_LE32_TO_CPU(gpt->SizeOfPartitionEntry);
	gpt->PartitionEntryArrayCRC32 = PED_LE32_TO_CPU(gpt->PartitionEntryArrayCRC32);
	gpt_le_guid_to_cpu(&gpt->DiskGUID);
	/* Ignore the reserved bytes */

	return gpt;
}

/************************************************************
 * gpt_write_header()
 * Requires:
 *  - dev
 *  - lba is the Logical Block Address of the partition table
 *  - gpt is a buffer into which the GPT will be put  
 * Modifies:
 *  - dev
 * Returns:
 *   1 on success
 *   0 on error
 ************************************************************/
static int
gpt_write_header(PedDevice *dev,
		 GuidPartitionTableHeader_t * gpt)
{
	GuidPartitionTableHeader_t *new_gpt;
	int rc;

	PED_ASSERT(gpt != NULL, return 0);

	new_gpt = ped_malloc(sizeof(*gpt));
	if (!new_gpt) return 0;
	memcpy(new_gpt, gpt, sizeof(*gpt));

	/* Fixup endianness */
	new_gpt->Signature                = PED_CPU_TO_LE64(gpt->Signature);
	new_gpt->Revision                 = PED_CPU_TO_LE32(gpt->Revision);
	new_gpt->HeaderSize               = PED_CPU_TO_LE32(gpt->HeaderSize);
	new_gpt->HeaderCRC32              = PED_CPU_TO_LE32(gpt->HeaderCRC32);
	new_gpt->Reserved1                = PED_CPU_TO_LE32(gpt->Reserved1);
	new_gpt->MyLBA                    = PED_CPU_TO_LE64(gpt->MyLBA);
	new_gpt->AlternateLBA             = PED_CPU_TO_LE64(gpt->AlternateLBA);
	new_gpt->FirstUsableLBA           = PED_CPU_TO_LE64(gpt->FirstUsableLBA);
	new_gpt->LastUsableLBA            = PED_CPU_TO_LE64(gpt->LastUsableLBA);
	new_gpt->PartitionEntryLBA        = PED_CPU_TO_LE64(gpt->PartitionEntryLBA);
	new_gpt->NumberOfPartitionEntries = PED_CPU_TO_LE32(gpt->NumberOfPartitionEntries);
	new_gpt->SizeOfPartitionEntry = PED_CPU_TO_LE32(gpt->SizeOfPartitionEntry);
	new_gpt->PartitionEntryArrayCRC32 = PED_CPU_TO_LE32(gpt->PartitionEntryArrayCRC32);
	gpt_cpu_to_le_guid(&new_gpt->DiskGUID);
	/* Ignore the reserved bytes */

	rc = ped_device_write(dev, new_gpt, gpt->MyLBA, GPT_HEADER_SECTORS);
	ped_free(new_gpt);
	return rc;
}


/************************************************************
 * gpt_is_valid()
 * Requires:
 *  - dev
 *  - lba is the Logical Block Address of the partition table
 * Modifies:
 *  - dev
 *  - gpt  - reads data into gpt
 *  - ptes - reads data into ptes
 * Returns:
 *   1 if valid
 *   0 on error
 ************************************************************/


static int
gpt_is_valid(const PedDevice * dev, uint64_t lba,
	     GuidPartitionTableHeader_t ** gpt,
	     GuidPartitionEntry_t ** ptes)
{
	uint32_t crc, origcrc;
	PED_ASSERT(gpt != NULL, return 0);
	PED_ASSERT(ptes != NULL, return 0);

	if (!(*gpt = gpt_read_header(dev, lba)))
		return 0;
	/* Check the GUID Partition Table Signature */
	if ((*gpt)->Signature != GPT_HEADER_SIGNATURE) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("GUID Partition Table Header Signature is wrong: "
			  "" PRIx64 " should be " PRIx64 ""),
			(*gpt)->Signature, GPT_HEADER_SIGNATURE);
		goto error_free_gpt;
	}

	/* Check the GUID Partition Table Header CRC */
	origcrc = (*gpt)->HeaderCRC32;
	(*gpt)->HeaderCRC32 = 0;
	crc = efi_crc32(*gpt, (*gpt)->HeaderSize);
	if (crc != origcrc) {
		ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_CANCEL,
			_("GPT Header CRC check failed, %x should be %x."),
			origcrc, crc);
		(*gpt)->HeaderCRC32 = origcrc;
		goto error_free_gpt;
	}
	(*gpt)->HeaderCRC32 = origcrc;

	/* Check that the MyLBA entry points to the LBA
	   that contains the GPT we read */
	if ((*gpt)->MyLBA != lba) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			"MyLBA " PRIx64 " != lba " PRIx64 ".\n",
			(*gpt)->MyLBA, lba);
		goto error_free_gpt;
	}

	if (!(*ptes = gpt_read_part_entries(dev, *gpt)))
		goto error_free_gpt;

	/* Check the GUID Partition Entry Array CRC */
	crc = efi_crc32(*ptes, (*gpt)->NumberOfPartitionEntries *
			   (*gpt)->SizeOfPartitionEntry);
	if (crc != (*gpt)->PartitionEntryArrayCRC32) {
		ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_CANCEL,
			_("GPT Partition Entry Array CRC check failed, "
			  "%x should be %x."),
			(*gpt)->PartitionEntryArrayCRC32, crc);
		goto error_free_ptes;
	}

	/* We're done, all's well */
	return 1;

error_free_ptes:
	ped_free (*ptes);
	*ptes = NULL;
error_free_gpt:
	ped_free (*gpt);
	*gpt = NULL;
error:
	return 0;
}

/************************************************************
 * gpt_write_pmbr()
 * Requires:
 *  - dev
 * Modifies:
 *  - dev
 * Returns:
 *   1 on success
 *   0 on error
 ************************************************************/


static int
gpt_write_pmbr(PedDevice * dev)
{
	LegacyMBR_t pmbr;
	memset(&pmbr, 0, sizeof(pmbr));
	pmbr.Signature = PED_CPU_TO_LE16(MSDOS_MBR_SIGNATURE);
	pmbr.PartitionRecord[0].OSType      = EFI_PMBR_OSTYPE_EFI_GPT;
	pmbr.PartitionRecord[0].EndHead     = 0xFE;
	pmbr.PartitionRecord[0].EndSector   = 0xFF;
	pmbr.PartitionRecord[0].EndTrack    = 0xFF;
	pmbr.PartitionRecord[0].StartingLBA = PED_CPU_TO_LE32(1);
	pmbr.PartitionRecord[0].SizeInLBA   = PED_CPU_TO_LE32(last_lba(dev));

	return ped_device_write(dev, &pmbr, GPT_PMBR_LBA, GPT_PMBR_SECTORS);
}



/************************************************************n
 * gpt_update_headers()
 * Updates the CRC fields for both primary and alternate GPTs
 *
 */
static int
gpt_update_headers(GuidPartitionTableHeader_t *pgpt,
		   GuidPartitionTableHeader_t *agpt,
		   GuidPartitionEntry_t *ptes)
{
	// printf("in UpdateGuidPartitionTableHeaders()\n");
	PED_ASSERT(pgpt != NULL, return 0);
	PED_ASSERT(agpt != NULL, return 0);
	PED_ASSERT(ptes != NULL, return 0);

	pgpt->PartitionEntryArrayCRC32 =
		agpt->PartitionEntryArrayCRC32 =
		efi_crc32(ptes, pgpt->NumberOfPartitionEntries *
			  pgpt->SizeOfPartitionEntry);

	pgpt->HeaderCRC32 = 0;
	pgpt->HeaderCRC32 = efi_crc32(pgpt, pgpt->HeaderSize);
	agpt->HeaderCRC32 = 0;
	agpt->HeaderCRC32 = efi_crc32(agpt, agpt->HeaderSize);
	return 1;
}



static int
gpt_write_new(PedDevice *dev)
{
	uint64_t size;
	uuid_t uuid;
	GuidPartitionTableHeader_t gpt, agpt;
	GuidPartitionEntry_t ptes[GPT_DEFAULT_RESERVED_PARTITION_ENTRIES];
	int count;

	// printf("in CreateNewGuidPartitionTableHeader()\n");
	memset(&gpt, 0, sizeof(gpt));
	gpt.Signature = GPT_HEADER_SIGNATURE;
	gpt.Revision = GPT_HEADER_REVISION_V1_02;
	gpt.HeaderSize = 92; /* per 1.02 spec */
	gpt.MyLBA = 1;
	gpt.AlternateLBA = last_lba(dev);
	gpt.FirstUsableLBA = (GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
			      dev->sector_size) + 2;
	gpt.LastUsableLBA =
		gpt.AlternateLBA -
		(GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
		 dev->sector_size) - 1;
	uuid_generate(uuid);
	memcpy(&(gpt.DiskGUID), uuid, sizeof(uuid));
	gpt.PartitionEntryLBA = 2;
	gpt.NumberOfPartitionEntries = GPT_DEFAULT_RESERVED_PARTITION_ENTRIES;
	gpt.SizeOfPartitionEntry = sizeof(GuidPartitionEntry_t);

	memset(ptes, 0,
	       gpt.NumberOfPartitionEntries * gpt.SizeOfPartitionEntry);

	/* Fix up Alternate GPT */
	memcpy(&agpt, &gpt, sizeof(gpt));
	agpt.MyLBA        = gpt.AlternateLBA;
	agpt.AlternateLBA = gpt.MyLBA;
	agpt.PartitionEntryLBA = agpt.MyLBA -
		(GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
		 dev->sector_size);

	if (!gpt_update_headers(&gpt, &agpt, ptes))
		return 0;

	/* Before the writes */
	/* Write PTH and PTEs */
	if (!gpt_write_header(dev, &gpt))
		return 0;
	if (!gpt_write_part_entries(dev, &gpt, ptes))
		return 0;


	/* Write Alternate PTH & PTEs */
	if (!gpt_write_part_entries(dev, &agpt, ptes))
		return 0;
	if (!gpt_write_header(dev, &agpt))
		return 0;

	return 1;
}



static int
gpt_write_new_disk(PedDevice * dev)
{
	if (!gpt_write_pmbr(dev))
		return 0;
	if (!gpt_write_new(dev))
		return 0;
	return 1;
}


/************************************************************
 * gpt_fix_broken_header()
 * Uses values from alternate GPT and puts them in primary GPT.
 * Requires:
 * pgpt, agpt, ptes.
 *  
 * Modifies:
 * pgpt
 * 
 * Returns:
 *   1 if valid
 *   0 on error
 ************************************************************/

static int
gpt_fix_broken_header(PedDevice *dev,
		      GuidPartitionTableHeader_t **badgpt,
		      GuidPartitionTableHeader_t *goodgpt,
		      GuidPartitionEntry_t     *ptes)
{

	PED_ASSERT(badgpt != NULL, return 0);
	PED_ASSERT(goodgpt != NULL, return 0);
	PED_ASSERT(ptes != NULL, return 0);

	// printf("GPT: Fixing broken header.\n");

	*badgpt = (GuidPartitionTableHeader_t *)
		ped_malloc(sizeof(GuidPartitionTableHeader_t));
	if (!*badgpt) return 0;
	memset(*badgpt, 0, sizeof(**badgpt));

	memcpy(*badgpt, goodgpt, sizeof(*goodgpt));

	/* Change badgpt values */
	(*badgpt)->MyLBA        = goodgpt->AlternateLBA;
	(*badgpt)->AlternateLBA = goodgpt->MyLBA;


	if ((*badgpt)->MyLBA == 1) 
		(*badgpt)->PartitionEntryLBA = (*badgpt)->MyLBA + 1;
	else
		(*badgpt)->PartitionEntryLBA = (*badgpt)->MyLBA - 
			(GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
			 dev->sector_size);

	return gpt_update_headers(*badgpt, goodgpt, ptes);
}




static void
gpt_print_legacy_part(PartitionRecord_t * pr, int i)
{
	if (!pr)
		return;
	if (!(pr->OSType))
		return;
	printf("Legacy Partition Record %d\n", i);
	printf("\tBootInd %02x", pr->BootIndicator);
	printf("\tSHead %02x", pr->StartHead);
	printf("\tSSector %02x", pr->StartSector);
	printf("\tSTrack %02x\n", pr->StartTrack);
	printf("\tOSType %02x", pr->OSType);
	printf("\tEHead %02x", pr->EndHead);
	printf("\tESector %02x", pr->EndSector);
	printf("\tETrack %02x\n", pr->EndTrack);
	printf("\tStartingLBA : %x", pr->StartingLBA);
	printf("\tSizeInLBA : %x\n", pr->SizeInLBA);
	return;
}

static void
gpt_print_mbr(LegacyMBR_t * mbr)
{
	int i;
	if (!mbr)
		return;

	printf("UniqueMBRSignature: %x\n", mbr->UniqueMBRSignature);
	for (i = 0; i < 4; i++)
		gpt_print_legacy_part(&(mbr->PartitionRecord[i]), i);
	printf("Signature: %x\n", mbr->Signature);
	return;
}

static void
erase_gpt(PedDevice * dev, GuidPartitionTableHeader_t * gpt,
	  GuidPartitionEntry_t * ptes)
{
	uint64_t header_lba;
	PED_ASSERT(dev  != NULL, return);
	if (gpt == NULL || ptes == NULL) return;

	header_lba = gpt->MyLBA;
	memset(ptes, 0,
	       gpt->NumberOfPartitionEntries *
	       gpt->SizeOfPartitionEntry);
	gpt_write_part_entries(dev, gpt, ptes);
	memset(gpt, 0, sizeof(*gpt));
	ped_device_write(dev, gpt, header_lba, GPT_HEADER_SECTORS);
}

/**************************************************
 * bad_pmbr_handler()
 *
 * Requires: dev, good_pgpt, good_agpt
 * Modifies: dev, pgpt, agpt, ptes
 * Returns:  1 if disk is considered GPT, 0 otherwise
 */


static int
bad_pmbr_handler(PedDevice * dev, int good_pgpt, int good_agpt,
		 GuidPartitionTableHeader_t ** pgpt,
		 GuidPartitionTableHeader_t ** agpt,
		 GuidPartitionEntry_t * ptes)
{
	PedExceptionOption ex_status=0;

	ped_exception_leave_all();

	if (good_pgpt && good_agpt) {
		ex_status = ped_exception_throw(PED_EXCEPTION_ERROR,
						PED_EXCEPTION_YES +
						PED_EXCEPTION_NO  +
						PED_EXCEPTION_IGNORE,
_("This disk contains a valid Primary and Alternate GUID Partition Table "
  "but the Protective MBR is invalid.  This generally means that the disk "
  "had GPT partitions on it, but then a legacy partition editing tool "
  "was used to change the partition table stored in the MBR.\n"
  "Which data is valid,  GPT or MBR?\n"
  "Yes will assume that the GPT information is correct, and rewrite the PMBR.\n"
  "No will assume that the MBR is correct, and erase the GPT information.\n"
  "Ignore will assume that the MBR is correct, but not change the disk."
	  ));
	}  else if (!good_pgpt && good_agpt) {
		ex_status = ped_exception_throw(PED_EXCEPTION_ERROR,
						PED_EXCEPTION_YES +
						PED_EXCEPTION_NO  +
						PED_EXCEPTION_IGNORE,
_("This disk contains a valid Alternate GUID Partition Table "
  "but the Primary GPT and Protective MBR are invalid.  "
  "This generally means that the disk had GPT partitions on it, "
  "but then a legacy partition editing tool was used to change "
  "the partition table stored in the MBR.\n"
  "Which data is valid,  GPT or MBR?\n"
  "Yes will assume that the GPT information is correct, and "
  "will rewrite the PMBR and Primary GPT.\n"
  "No will assume that the MBR is correct, and erase the GPT information.\n"
  "Ignore will assume that the MBR is correct, but not change the disk."
	));
	} else if (good_pgpt && !good_agpt) {
		ex_status = ped_exception_throw(PED_EXCEPTION_ERROR,
						PED_EXCEPTION_YES +
						PED_EXCEPTION_NO  +
						PED_EXCEPTION_IGNORE,
_("This disk contains a valid Primary GUID Partition Table "
  "but the Alternate GPT and Protective MBR are invalid.  "
  "This generally means that the disk had GPT partitions on it, "
  "but then a legacy partition editing tool was used to change "
  "the partition table stored in the MBR.\n"
  "Which data is valid,  GPT or MBR?\n"
  "Yes will assume that the GPT information is correct, and "
  "will rewrite the PMBR and Alternate GPT.\n"
  "No will assume that the MBR is correct, and erase the GPT information.\n"
  "Ignore will assume that the MBR is correct, but not change the disk.\n"
	));
	} else {
		ped_exception_fetch_all();
		return 1;
	}

	switch (ex_status) {
	case PED_EXCEPTION_YES:
		gpt_write_pmbr(dev);
		if (good_pgpt && !good_agpt) {
			gpt_fix_broken_header(dev, agpt, *pgpt, ptes);
			gpt_write_header(dev, *agpt);
			gpt_write_part_entries(dev, *agpt, ptes);
		}
		else if (!good_pgpt && good_agpt) {
			gpt_fix_broken_header(dev, pgpt, *agpt, ptes);
			gpt_write_header(dev, *pgpt);
			gpt_write_part_entries(dev, *pgpt, ptes);
		}
		ped_exception_fetch_all();
		return 1;
		break;
	case PED_EXCEPTION_NO:
		if (good_pgpt) {
			erase_gpt(dev, *pgpt, ptes);
		}
		if (good_agpt) {
			erase_gpt(dev, *agpt, ptes);
		}
		break;
	case PED_EXCEPTION_IGNORE:
	default:
		break;
	}
	ped_exception_fetch_all();
	return 0;
}


/************************************************************
 * gpt_find_valid()
 * Requires:
 *  - dev 
 *  - pgpt is a GPTH if it's valid
 *  - agpt is a GPTH if it's valid
 *  - ptes is a PTE
 * Modifies:
 *  - gpt & ptes
 * Returns:
 *   1 if valid
 *   0 on error
 * Comments:
 *  Intel is changing the EFI Spec. (after v1.02) to say that a
 *  disk is considered to have a GPT label only if the GPT
 *  structures are correct, and the MBR is actually a Protective
 *  MBR (has one 0xEE type partition).
 *  Problem occurs when a GPT-partitioned disk is then
 *  edited with a legacy (non-GPT-aware) application, such as
 *  fdisk (which doesn't generally erase the PGPT or AGPT).
 *  How should such a disk get handled?  As a GPT disk (throwing
 *  away the fdisk changes), or as an MSDOS disk (throwing away
 *  the GPT information).  Previously, I've taken the GPT-is-right,
 *  MBR is wrong, approach, to stay consistent with the EFI Spec.
 *  Intel disagrees, saying the disk should then be treated
 *  as having a msdos label, not a GPT label.  If this is true,
 *  then what's the point of having an AGPT, since if the PGPT
 *  is screwed up, likely the PMBR is too, and the PMBR becomes
 *  a single point of failure.
 *  So, in the Linux kernel, I'm going to test for PMBR, and
 *  warn if it's not there, and treat the disk as MSDOS, with a note
 *  for users to use Parted to "fix up" their disk if they
 *  really want it to be considered GPT.
 *  Hence, gpt_find_valid() needs to ask the user how to fix
 *  this (which one to consider right).
 ************************************************************/
static int
gpt_find_valid(const PedDevice * dev,
	       GuidPartitionTableHeader_t ** pgpt,
	       GuidPartitionTableHeader_t ** agpt,
	       GuidPartitionEntry_t       ** ptes)
{
	int good_pgpt=0, good_agpt=0, good_pmbr=0;
	GuidPartitionEntry_t *pptes = NULL, *aptes = NULL;
	LegacyMBR_t *legacyMbr = NULL;
	uint64_t header_lba, ptes_lba;

	PED_ASSERT(dev  != NULL, return 0);
	PED_ASSERT(pgpt != NULL, return 0);
	PED_ASSERT(agpt != NULL, return 0);
	PED_ASSERT(ptes != NULL, return 0);

	/* Check the Primary GPT */
	good_pgpt = gpt_is_valid(dev, 1, pgpt, &pptes);
	if (good_pgpt) {
		/* Primary GPT is OK, check the alternate and warn if bad */
		good_agpt = gpt_is_valid(dev, (*pgpt)->AlternateLBA,
				  agpt, &aptes);

		if (!good_agpt) {
			*agpt = NULL;
			printf("Alternate GPT is invalid, using primary GPT.\n");
		}
		if (aptes) ped_free(aptes);
		*ptes = pptes;
	} /* if primary is valid */
	else {
		/* Primary GPT is bad, check the Alternate GPT */
		*pgpt = NULL;
		good_agpt = gpt_is_valid(dev, last_lba(dev),
				  agpt, &aptes);
		if (good_agpt) {
			/* Primary is bad, alternate is good.
			   Return values from the alternate and warn.
			 */
			printf("Primary GPT is invalid, using alternate GPT.\n");
			*ptes = aptes;
		}
	}
	
	/* Now test for valid PMBR */
	/* This will be added to the EFI Spec. per Intel after v1.02. */
	if (good_pgpt || good_agpt) {
		legacyMbr = malloc(sizeof(*legacyMbr));
		if (legacyMbr) {
			memset(legacyMbr, 0, sizeof(*legacyMbr));
			ped_device_read(dev, legacyMbr, 0, 1);
			good_pmbr = is_pmbr_valid(legacyMbr);
			ped_free(legacyMbr);
		}
		if (good_pmbr) return 1;
		else {
			if (bad_pmbr_handler((PedDevice *) dev,
					     good_pgpt, good_agpt,
					     pgpt, agpt, *ptes))
			return 1;
		}
	}
	
/*
 * Both primary and alternate GPTs are bad.
	 * This isn't our disk, return 0.
	 */
	if (*pgpt) {ped_free(*pgpt); *pgpt = NULL;}
	if (*agpt) {ped_free(*agpt); *agpt = NULL;}
	if (*ptes) {ped_free(*ptes); *ptes = NULL;}
	return 0;
}

#ifdef GPT_DEBUG
static void
print_disk_info(PedDevice *dev)
{
	unsigned int i;
	LegacyMBR_t mbr;
	GuidPartitionTableHeader_t *pgpt = NULL, *agpt = NULL, *gpt = NULL;
	GuidPartitionEntry_t *pte = NULL, zeropte;

	memset(&zeropte, 0, sizeof(zeropte));
	if (!ped_device_read(dev, &mbr, GPT_PMBR_LBA, GPT_PMBR_SECTORS)) {
		printf("print_disk_info error: ped_device_read(mbr) error.\n");
		return;
	}

	if (mbr.PartitionRecord[0].OSType == EFI_PMBR_OSTYPE_EFI_GPT) {
		/* This is an EFI GPT disk */
		if (!ped_device_read(dev, &pgpt,
				     GPT_PRIMARY_HEADER_LBA,
				     GPT_HEADER_SECTORS)) {
			printf("print_disk_info error: ped_device_read(gpt) error.\n");
			return;
		}
		printf("This is an EFI GPT disk.\n");
		if (gpt_find_valid(dev, &pgpt, &agpt, &pte)) {
			if (pgpt) gpt = pgpt;
			else if (agpt) gpt = agpt;
		}
		else {
			printf("GUID Partition Table is invalid.\n");
			return;
		}
		for (i = 0; pte && i < gpt->NumberOfPartitionEntries; i++) {
			/* Partition entry is unused if all bytes are 0 */
			if (memcmp(&zeropte, &pte[i], sizeof(zeropte)))
				gpt_print_part_entry(&pte[i], i);
		}
		if (pgpt) ped_free(pgpt);
		if (agpt) ped_free(agpt);
		if (pte)  ped_free(pte);
	} else {
		printf("This is not an EFI GPT disk.  Try using vanilla msdos partition tables.\n");
	}
	return;
}
#endif





void
ped_disk_gpt_init()
{
	PED_ASSERT(sizeof(GuidPartitionTableHeader_t) == 512, return);
	PED_ASSERT(sizeof(GuidPartitionEntryAttributes_t) == 8, return);
	PED_ASSERT(sizeof(GuidPartitionEntry_t) == 128, return);

	ped_register_disk_type(&gpt_disk_type);
}

void
ped_disk_gpt_done()
{
	ped_unregister_disk_type(&gpt_disk_type);
}

static int
gpt_probe(const PedDevice * dev)
{
	GuidPartitionTableHeader_t *pgpt = NULL, *agpt = NULL;
	GuidPartitionEntry_t *ptes = NULL;

	PED_ASSERT(dev != NULL, return 0);

	if (!ped_device_open((PedDevice *) dev))
		return 0;

	if (!(gpt_find_valid(dev, &pgpt, &agpt, &ptes))) {
		ped_device_close((PedDevice *) dev);
		return 0;
	}

	ped_device_close((PedDevice *) dev);

	if (pgpt) ped_free(pgpt);
	if (agpt) ped_free(agpt);
	if (ptes) ped_free(ptes);
	return 1;
}

static PedDisk *
gpt_open(PedDevice * dev)
{
	PedDisk *disk;

	PED_ASSERT(dev != NULL, return 0);

	if (!gpt_probe(dev))
		goto error;


	ped_device_open((PedDevice *) dev);

	disk = ped_disk_alloc(dev, &gpt_disk_type);

	if (!disk)
		goto error;

	if (!gpt_read(disk))
	  goto error_free_disk_specific;

	return disk;

      error_free_disk_specific:
	ped_free(disk->disk_specific);
      error_free_disk:
	ped_free(disk);
      error:
	return NULL;
}

static PedDisk *
gpt_create(PedDevice * dev)
{
	PedDisk *newdisk;
	PED_ASSERT(dev != NULL, return 0);

	if (!ped_device_open(dev))
		goto error;


	gpt_write_new_disk(dev);

	if (!ped_device_sync(dev))
		goto error_close_dev;

	ped_device_close(dev);
	newdisk = gpt_open(dev);
	return newdisk;

      error_close_dev:
	ped_device_close(dev);
      error:
	return 0;
}

static int
gpt_close(PedDisk * disk)
{
	PED_ASSERT(disk != NULL, return 0);

	ped_device_close(disk->dev);
	ped_disk_delete_all(disk);
	if (disk->disk_specific) ped_free(disk->disk_specific);
	ped_free(disk);
	return 1;
}






static int
gpt_read(PedDisk * disk)
{
	PedPartition *part;
	unsigned int i;
	GPTDiskData *gpt_disk_data;
	GPTPartitionData *gpt_part_data;
	PedConstraint*		constraint_exact;
	efi_guid_t unused = UNUSED_ENTRY_GUID;

	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);


	ped_disk_delete_all(disk);

	if (!disk->disk_specific) gpt_new(disk);
	PED_ASSERT(disk->disk_specific != NULL, return 0);
	gpt_disk_data = disk->disk_specific;


	if (!gpt_find_valid(disk->dev, &(gpt_disk_data->pgpt), &(gpt_disk_data->agpt), &(gpt_disk_data->ptes)))
		return 0;

	/* If one of the gpts are broken, fix it. */
	if (!gpt_disk_data->pgpt) {
		gpt_fix_broken_header(disk->dev,
				      &gpt_disk_data->pgpt,
				      gpt_disk_data->agpt,
				      gpt_disk_data->ptes);
	}
	else if (!gpt_disk_data->agpt) {
		gpt_fix_broken_header(disk->dev,
				      &gpt_disk_data->agpt,
				      gpt_disk_data->pgpt,
				      gpt_disk_data->ptes);
	}

	for (i = 0; i < gpt_disk_data->pgpt->NumberOfPartitionEntries; i++) {
		
		if (!efi_guidcmp(gpt_disk_data->ptes[i].PartitionTypeGuid,
				 unused)) continue;

		part = ped_partition_alloc(disk, PED_PARTITION_PRIMARY, NULL,
					   gpt_disk_data->ptes[i].StartingLBA,
					   gpt_disk_data->ptes[i].EndingLBA);
		if (!part)
			return 0;

		part->fs_type = ped_file_system_probe(&part->geom);
		part->num = i+1;

		gpt_part_data = part->disk_specific =
			ped_malloc(sizeof(GPTPartitionData));
		if (!gpt_part_data) {
			ped_free(part);
			return 0;
		}
		memset(gpt_part_data, 0, sizeof(*gpt_part_data));
		
		gpt_part_data->pte  = &(gpt_disk_data->ptes[i]);

		constraint_exact = ped_constraint_exact (&part->geom);
		if (!ped_disk_add_partition(disk, part, constraint_exact)) {
			ped_free(gpt_part_data);
			ped_free(part);
			return 0;
		}
		ped_constraint_destroy (constraint_exact);

	}
	return 1;
}

static int
gpt_fixup_ondisk_info(PedDisk *disk)
{
	GPTDiskData *gpt_disk_data;
	GPTPartitionData *gpt_part_data;
	PedPartition *part = NULL;

	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);
	PED_ASSERT(disk->disk_specific != NULL, return 0);
	PED_ASSERT(disk->part_list != NULL, return 0);
	gpt_disk_data = disk->disk_specific;

	/* For each partition, make sure the pte
	   has the right start/end.
	*/

	for (part = disk->part_list; part; part = part->next) {
		if (part->type != PED_PARTITION_PRIMARY) continue;
		PED_ASSERT(part->disk_specific != NULL, return 0);
		gpt_part_data = part->disk_specific;
		PED_ASSERT(gpt_part_data->pte != NULL, return 0);
		gpt_part_data->pte->StartingLBA = part->geom.start;
		gpt_part_data->pte->EndingLBA   = part->geom.end;
	}

	return 1;
}

static int
gpt_write(PedDisk * disk)
{

	GPTDiskData *gpt_disk_data;
	PedPartition *part = NULL;

	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);
	PED_ASSERT(disk->disk_specific != NULL, return 0);
	gpt_disk_data = disk->disk_specific;

	if (!gpt_write_pmbr(disk->dev)) return 0;

	gpt_fixup_ondisk_info(disk);

	gpt_update_headers(gpt_disk_data->pgpt,
			   gpt_disk_data->agpt,
			   gpt_disk_data->ptes);

	/* Write PTH and PTEs */
	if (!gpt_write_header(disk->dev, gpt_disk_data->pgpt))
		return 0;
	if (!gpt_write_part_entries(disk->dev, gpt_disk_data->pgpt, gpt_disk_data->ptes))
		return 0;


	/* Write Alternate PTH & PTEs */
	if (!gpt_write_part_entries(disk->dev, gpt_disk_data->agpt, gpt_disk_data->ptes))
		return 0;
	if (!gpt_write_header(disk->dev, gpt_disk_data->agpt))
		return 0;

	if (!ped_device_sync(disk->dev))
		return 0;

	return 1;
}



static int
add_metadata_part(PedDisk * disk, PedPartitionType type, PedSector start,
		  PedSector length)
{
	PedPartition *new_part;
	PedConstraint * constraint_exact;
	PED_ASSERT(disk != NULL, return 0);


	new_part =
	    ped_partition_new(disk, type | PED_PARTITION_METADATA, NULL,
			      start, start + length - 1);
	if (!new_part)
		goto error;

	constraint_exact = ped_constraint_exact (&new_part->geom);

	if (!ped_disk_add_partition(disk, new_part, constraint_exact))
		goto error_destroy_new_part;

	return 1;

      error_destroy_new_part:
	ped_partition_destroy(new_part);
      error:
	return 0;
}

static PedPartition *
gpt_partition_new(const PedDisk *disk,
		  PedPartitionType part_type,
		  const PedFileSystemType* fs_type,
		  PedSector start,
		  PedSector end)
{
	unsigned int i;
	uuid_t uuid;
	efi_guid_t unused_entry_guid = UNUSED_ENTRY_GUID;
	GPTDiskData *gpt_disk_data;
	GPTPartitionData *gpt_part_data;
	PedPartition *part;

	PED_ASSERT(disk != NULL, return NULL);


	part = ped_partition_alloc (disk, part_type, fs_type, start, end);
	if (!part) return NULL;

	if (part_type != PED_PARTITION_PRIMARY)
		return part;

#if 0
	if (!ped_disk_check_overlap(disk, part)) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("The new partition overlaps with another "
			  "partition."));
		ped_free(part);
		return NULL;
	}
#endif

	PED_ASSERT(disk->disk_specific != NULL, return NULL);

	gpt_disk_data = disk->disk_specific;


	PED_ASSERT(gpt_disk_data->pgpt != NULL, return NULL);
	PED_ASSERT(gpt_disk_data->agpt != NULL, return NULL);
	PED_ASSERT(gpt_disk_data->ptes != NULL, return NULL);


	for (i = 0; i < gpt_disk_data->pgpt->NumberOfPartitionEntries; i++) {
		if (!efi_guidcmp
		    (gpt_disk_data->ptes[i].PartitionTypeGuid, unused_entry_guid)) {
			break;
		}
	}
	/* No unused entries */
	if (i == gpt_disk_data->pgpt->NumberOfPartitionEntries){
		ped_free(part);
		return NULL;
	}

	part->num = i + 1;
	gpt_disk_data->ptes[i].StartingLBA = part->geom.start;
	gpt_disk_data->ptes[i].EndingLBA = part->geom.end;
	uuid_generate(uuid);
	memcpy(&(gpt_disk_data->ptes[i].UniquePartitionGuid), uuid, sizeof(uuid));

	gpt_disk_data->ptes[i].PartitionTypeGuid = PARTITION_BASIC_DATA_GUID;

	if (fs_type && fs_type->name && !strcmp(fs_type->name, "linux-swap"))
		gpt_disk_data->ptes[i].PartitionTypeGuid = 
			PARTITION_SWAP_GUID;


	gpt_part_data = part->disk_specific =
		ped_malloc(sizeof(GPTPartitionData));
	if (!gpt_part_data) return part;
	memset(gpt_part_data, 0, sizeof(*gpt_part_data));

	gpt_part_data->pte = &(gpt_disk_data->ptes[i]);

	return part;
}

static void
gpt_partition_destroy(PedPartition *part)
{
	GPTPartitionData *gpt_part_data;

	PED_ASSERT(part != NULL, return);

	if (part->type != PED_PARTITION_PRIMARY)
		return;

	PED_ASSERT(part->disk_specific != NULL, return);
	gpt_part_data = part->disk_specific;
	if (gpt_part_data->pte)
		memset(gpt_part_data->pte, 0, sizeof(*(gpt_part_data->pte)));

	ped_free(part->disk_specific);
	part->disk_specific = NULL;
	ped_free(part);
}

/**********************************************************
 * Allocate metadata partitions for the GPTH and PTES.
 *
 */
static int
_alloc_metadata_unknown(PedDisk * disk)
{
	uint64_t pte_reserved_blocks;
	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);

	pte_reserved_blocks = 
		(GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
		 disk->dev->sector_size);


	/* metadata at the start of the disk includes the MBR */
	if (!add_metadata_part(disk, PED_PARTITION_PRIMARY,
			       GPT_PMBR_LBA,
			       GPT_PMBR_SECTORS
			       + GPT_HEADER_SECTORS
			       + pte_reserved_blocks))
		return 0;

	/* metadata at the end of the disk */
	if (!add_metadata_part(disk, PED_PARTITION_PRIMARY,
			       last_lba(disk->dev) -
			       pte_reserved_blocks,
			       GPT_HEADER_SECTORS
			       + pte_reserved_blocks))
		return 0;

	return 1;
}


/**********************************************************
 * Allocate metadata partitions for the GPTH and PTES
 * when we know the actual metadata size
 */
static int
_alloc_metadata_known(PedDisk * disk)
{
	int i;
	PedSector gptlength, pteslength = 0;
	GPTDiskData *gpt_disk_data;

	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);
	PED_ASSERT(disk->disk_specific != NULL, return 0);
	gpt_disk_data = disk->disk_specific;

	/* allocate space for the header */
	gptlength = gpt_disk_data->pgpt->HeaderSize / disk->dev->sector_size;
	if (gpt_disk_data->pgpt->HeaderSize % disk->dev->sector_size)
		gptlength++;
	/* allocate space for the ptes */
	pteslength =
	    (gpt_disk_data->pgpt->NumberOfPartitionEntries *
	     gpt_disk_data->pgpt->SizeOfPartitionEntry) /
		disk->dev->sector_size;
	if ((gpt_disk_data->pgpt->NumberOfPartitionEntries *
	     gpt_disk_data->pgpt->SizeOfPartitionEntry) %
	    disk->dev->sector_size)
		pteslength++;

	/* metadata at the start of the disk includes the MBR */
	if (!add_metadata_part(disk, PED_PARTITION_PRIMARY,
			       GPT_PMBR_LBA,
			       GPT_PMBR_SECTORS + gptlength + pteslength))
		return 0;

	/* metadata at the end of the disk */
	if (!add_metadata_part(disk, PED_PARTITION_PRIMARY,
			       last_lba(disk->dev) - gptlength - pteslength + 1,
			       gptlength + pteslength))
		return 0;

	return 1;
}




/**********************************************************
 * Allocate metadata partitions for the GPTH and PTES.
 *
 */
static int
gpt_alloc_metadata(PedDisk * disk)
{
	GPTDiskData *gpt_disk_data;
	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);

	gpt_disk_data = disk->disk_specific;

	if (!gpt_disk_data ||
	    !gpt_disk_data->pgpt ||
	    !gpt_disk_data->agpt ||
	    !gpt_disk_data->ptes)
		return _alloc_metadata_unknown(disk);

	return _alloc_metadata_known(disk);
}

/************************************************************
 * gpt_partition_enumerate()
 * Requires:
 *  - part 
 * Modifies:
 *  - part->num
 * Returns:
 *   1 if valid
 *   0 on error
 * Actions:
 *   Does nothing, as the read/new/destroy functions maintain
 *   part->num.
 *   
 *   
 ************************************************************/
static int
gpt_partition_enumerate(PedPartition * part)
{
	return 1;
}


/************************************************************
 * gpt_clobber()
 * Requires:
 *  - dev 
 * Modifies:
 *  - dev
 * Returns: (per gpt_msdos.c)
 *   1 if valid
 *   0 on error
 * Actions:
 *   This writes zeros to the PMBR and the primary and
 *   alternate GPTHs and PTEs.
 ************************************************************/
static int
gpt_clobber(PedDevice * dev)
{
	LegacyMBR_t pmbr;
	GuidPartitionTableHeader_t gpt;
	GuidPartitionEntry_t ptes[GPT_DEFAULT_RESERVED_PARTITION_ENTRIES];
	uint64_t lastLBA, pte_sectors;


	PED_ASSERT (dev != NULL, return 0);
	PED_ASSERT (gpt_probe(dev), return 0);

	if (!ped_device_open(dev)) return 0;

	PED_ASSERT(dev->sector_size != 0, return 0);
	pte_sectors = GPT_DEFAULT_RESERVED_PARTITION_ENTRY_ARRAY_SIZE /
		dev->sector_size;

	memset(&pmbr, 0, sizeof(pmbr));
	memset(&gpt, 0, sizeof(gpt));
	memset(ptes, 0, sizeof(ptes));

	lastLBA = last_lba(dev);
	       
	if (!ped_device_write(dev, &pmbr, GPT_PMBR_LBA, GPT_PMBR_SECTORS))
		return 0;
	if (!ped_device_write(dev, &gpt,
			      GPT_PRIMARY_HEADER_LBA,
			      GPT_HEADER_SECTORS))
		return 0;
	if (!ped_device_write(dev, &ptes,
			      GPT_PRIMARY_PART_TABLE_LBA,
			      pte_sectors))
		return 0;

	if (!ped_device_write(dev, &ptes, lastLBA - pte_sectors,
			      pte_sectors)) 
		return 0;
	if (!ped_device_write(dev, &gpt, lastLBA, GPT_HEADER_SECTORS))
		return 0;
	return 1;
}


static int
gpt_partition_set_flag(PedPartition *part,
		       PedPartitionFlag flag,
		       int state)
{
	GPTPartitionData *gpt_part_data;
	PED_ASSERT(part != NULL, return 0);
	PED_ASSERT(part->disk_specific != NULL, return 0);
	gpt_part_data = part->disk_specific;
	PED_ASSERT(gpt_part_data->pte != NULL, return 0);


	switch (flag) {
	case PED_PARTITION_BOOT:
		if (state)
			gpt_part_data->pte->PartitionTypeGuid = 
				PARTITION_SYSTEM_GUID;
		else
			gpt_part_data->pte->PartitionTypeGuid = 
				PARTITION_BASIC_DATA_GUID;

		break;
	case PED_PARTITION_RAID:
		if (state)
			gpt_part_data->pte->PartitionTypeGuid =
				PARTITION_RAID_GUID;
		else
			gpt_part_data->pte->PartitionTypeGuid =
				PARTITION_BASIC_DATA_GUID;
		break;
	case PED_PARTITION_LVM:
		if (state)
			gpt_part_data->pte->PartitionTypeGuid =
				PARTITION_LVM_GUID;
		else
			gpt_part_data->pte->PartitionTypeGuid =
				PARTITION_BASIC_DATA_GUID;

		break;
	case PED_PARTITION_LBA:
		if (!state) return 0;
		break;
	case PED_PARTITION_ROOT:
	case PED_PARTITION_SWAP:
	case PED_PARTITION_HIDDEN:
	default:
		return 0;
	}

	if (part->fs_type && part->fs_type->name &&
	    !strcmp(part->fs_type->name, "linux-swap"))
			gpt_part_data->pte->PartitionTypeGuid = 
				PARTITION_SWAP_GUID;
		

	return 1;
}

static int
gpt_partition_get_flag(const PedPartition *part,
		       PedPartitionFlag flag)
{
	GPTPartitionData *gpt_part_data = NULL;
	PED_ASSERT(part->disk_specific != NULL, return 0);
	gpt_part_data = part->disk_specific;

	switch (flag) {
	case PED_PARTITION_RAID:
		return (!efi_guidcmp(gpt_part_data->pte->PartitionTypeGuid, 
				     PARTITION_RAID_GUID));
	case PED_PARTITION_LVM:
		return (!efi_guidcmp(gpt_part_data->pte->PartitionTypeGuid, 
				     PARTITION_LVM_GUID));
	case PED_PARTITION_BOOT:
		return (!efi_guidcmp(gpt_part_data->pte->PartitionTypeGuid,
				     PARTITION_SYSTEM_GUID));
	case PED_PARTITION_LBA:
		return 1;
	case PED_PARTITION_ROOT:
	case PED_PARTITION_SWAP:
	case PED_PARTITION_HIDDEN:
	default:
		return 0;
	}
	return 0;
}

static int
gpt_partition_is_flag_available(const PedPartition * part,
				PedPartitionFlag flag)
{
	switch (flag) {
	case PED_PARTITION_RAID:
	case PED_PARTITION_LVM:
	case PED_PARTITION_LBA:
	case PED_PARTITION_BOOT:
		return 1;
	case PED_PARTITION_ROOT:
	case PED_PARTITION_SWAP:
	case PED_PARTITION_HIDDEN:
	default:
		return 0;
	}
	return 0;
}

static void
gpt_partition_set_name(PedPartition *part,
		       const char *name)
{
	unsigned int i;
	GPTPartitionData *gpt_part_data = NULL;
	PED_ASSERT(part->disk_specific != NULL, return);
	gpt_part_data = part->disk_specific;

	if (!gpt_part_data->pte) return;

	memset(gpt_part_data->pte->PartitionName, 0, sizeof(gpt_part_data->pte->PartitionName));

	for (i=0; i < (72 / sizeof(efi_char16_t)) && i < strlen(name); i++) {
		gpt_part_data->pte->PartitionName[i] = name[i];
	}
	return;
}
static const char *
gpt_partition_get_name (const PedPartition * part)
{
	/* The name is stored in Unicode in the GPT entry */
	char *name;
	unsigned int i, namelen = (72 / sizeof(efi_char16_t));
	GPTPartitionData *gpt_part_data = NULL;
	PED_ASSERT(part->disk_specific != NULL, return 0);
	gpt_part_data = part->disk_specific;

	if (!gpt_part_data->pte) return NULL;

	name = ped_malloc(namelen);
	if (!name) return NULL;

	memset(name, 0, namelen);

	for (i=0; i < namelen ; i++) {
		name[i] = gpt_part_data->pte->PartitionName[i];
	}

	return name;
}


static int
gpt_get_max_primary_partition_count(const PedDisk *disk)
{
	int rc = GPT_DEFAULT_RESERVED_PARTITION_ENTRIES; /* 128 */
	GPTDiskData *gpt_disk_data;
	PED_ASSERT(disk != NULL, return 0);
	gpt_disk_data = disk->disk_specific;

	if (gpt_disk_data  && gpt_disk_data->pgpt)
		rc = gpt_disk_data->pgpt->NumberOfPartitionEntries;
	return rc;
}


static PedConstraint*
_non_metadata_constraint (PedDisk* disk)
{
	PedGeometry	max_geom;
	GPTDiskData    *gpt_disk_data;
	GuidPartitionTableHeader_t *gpt;

	PED_ASSERT(disk != NULL, return 0);
	PED_ASSERT(disk->dev != NULL, return 0);
	PED_ASSERT(disk->disk_specific != NULL, return 0);
	gpt_disk_data = disk->disk_specific;
	PED_ASSERT(gpt_disk_data->pgpt != NULL, return 0);
	gpt = gpt_disk_data->pgpt;

	/* Alignments can be on any sector */
	/* Geometry must reside within the usable LBAs */
	if (!ped_geometry_init (&max_geom, disk,
				gpt->FirstUsableLBA,
				gpt->LastUsableLBA - gpt->FirstUsableLBA + 1))
		return NULL;

	return ped_constraint_new (ped_alignment_any, ped_alignment_any,
				   &max_geom,
				   &max_geom, 1);
}

/*  _try_constraint() copied verbatim from disk_mac.c
 *  per Clausen's recommendation
 */

static int
_try_constraint (PedPartition* part, const PedConstraint* external,
		 PedConstraint* internal)
{
	PedConstraint*		intersection;
	PedGeometry*		solution;

	intersection = ped_constraint_intersect (external, internal);
	ped_constraint_destroy (internal);
	if (!intersection)
		goto fail;

	solution = ped_constraint_solve_nearest (intersection, &part->geom);
	if (!solution)
		goto fail_free_intersection;
	ped_geometry_set (&part->geom, solution->start, solution->length);
	ped_geometry_destroy (solution);
	ped_constraint_destroy (intersection);
	return 1;

fail_free_intersection:
	ped_constraint_destroy (intersection);
fail:
	return 0;
}


/* YUCK! FIXME (err, the whole new-then-add thing sucks) */
static PedPartition*
_find_left_part (PedPartition* part)
{
	PedPartition*	walk;

	for (walk = part->geom.disk->part_list; walk && walk->next;
	     walk = walk->next) {
		if (walk->geom.end < part->geom.start
		    && walk->next->geom.end >= part->geom.start)
			break;
	}
	if (walk && walk->geom.end < part->geom.start)
		return walk;
	else
		return NULL;
}

static PedPartition*
_find_right_part (PedPartition* part, PedPartition* left)
{
	PedPartition*	walk = left;

	for (walk = left; walk; walk = walk->next) {
		if (walk->geom.start > part->geom.end)
			break;
	}
	return walk;
}

/* if the gap between the new partition and the partition-to-be-aligned is
 * less than MIN_FREESPACE, then gobble up the gap!  (Only does this if it
 * still satisfies the constraint)
 */
static int
_grow_over_small_freespace (PedPartition* part, const PedConstraint* constraint)
{
	PedPartition*	left;
	PedPartition*	right;

	/* there should always be a partition on the left - if nothing else -
	 * the partition map.  Unless "part" is the partition map ;-)
	 */
	left = _find_left_part (part);
	if (!left)
		return 1;
	right = _find_right_part (part, left);
	PED_ASSERT (left->geom.end < part->geom.start, return 0);
	if (right)
		PED_ASSERT (part->geom.end < right->geom.start, return 0);

	if (part->geom.start - left->geom.end < MIN_FREESPACE
	    && ped_alignment_is_aligned (constraint->start_align, NULL,
					 left->geom.end + 1)
	    && ped_geometry_test_sector_inside (constraint->start_range,
		    				left->geom.end + 1)) {
		if (!ped_geometry_set_start (&part->geom, left->geom.end + 1))
			return 0;
	}
	if (right
	    && right->geom.start - part->geom.end < MIN_FREESPACE
	    && ped_alignment_is_aligned (constraint->end_align, NULL,
					 right->geom.start - 1)
	    && ped_geometry_test_sector_inside (constraint->end_range,
		    				right->geom.start - 1)) {
		if (!ped_geometry_set_end (&part->geom, right->geom.start - 1))
			return 0;
	}

	return 1;
}

static int
gpt_partition_align(PedPartition * part,
		    const PedConstraint * constraint)
{
	PED_ASSERT (part != NULL, return 0);

	if (!_try_constraint (part, constraint,
			      _non_metadata_constraint (part->geom.disk))) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Unable to align partition."));
		return 0;
	}

	if (!_grow_over_small_freespace (part, constraint))
		return 0;


	return 1;
}

static PedDisk *
gpt_new(PedDisk * disk)
{
	PED_ASSERT(disk != NULL, return NULL);
	if (disk->disk_specific) return disk;

	disk->disk_specific = ped_malloc(sizeof(GPTDiskData));
	PED_ASSERT(disk->disk_specific != NULL, return NULL);
	memset(disk->disk_specific, 0, sizeof(GPTDiskData));
	return disk;
}  


/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 4 
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -4
 * c-argdecl-indent: 4
 * c-label-offset: -4
 * c-continued-statement-offset: 4
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
