# -*- coding: utf-8 -*- #
# Copyright 2025 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Implements commands for the GOAUTH environment variable."""

from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.artifacts import endpoint_util
from googlecloudsdk.command_lib.artifacts import go_util
from googlecloudsdk.core import log


@base.DefaultUniverseOnly
@base.Hidden
@base.ReleaseTracks(base.ReleaseTrack.GA)
class Auth(base.Command):
  """Print authentication commands for the GOAUTH environment variable.

  This command implements the GOAUTH credential provider command introduced in
  Go 1.24. For more details about the GOAUTH environment variable, see
  https://pkg.go.dev/cmd/go#hdr-GOAUTH_environment_variable.  When you configure
  the GOAUTH environment variable for repositories, Artifact Registry looks for
  credentials in the following order:

  * Application Default Credentials (ADC)
  * Credentials provided by the Google Cloud CLI, including user credentials
      from the command `gcloud auth application-default login`.

  """

  detailed_help = {
      'DESCRIPTION':
          '{description}',
      'EXAMPLES':
          """\
    To configure the GOAUTH environment variable for repositories in `us-central1` and use your credentials:

        $ export GOAUTH="{command} --location=us-central1"

    To configure the GOAUTH environment variable for repositories in `us-central1` and use the credentials from a service account:

        $ export GOAUTH="{command} --location=us-central1 --json-key=path/to/key.json"
    """,
  }

  @staticmethod
  def Args(parser):
    """Set up arguements for this command.

    Args:
      parser: An argparse.ArgumentPaser.
    """
    # Go may or may not pass the URL to authenticate against, so we need
    # to support an optional positional argument. This argument is currently
    # unused.
    parser.add_argument(
        'url',
        nargs='*',
        help='A URL generated by Go to set up authentication.',
    )
    parser.add_argument(
        '--location',
        metavar='LOCATION',
        required=True,
        help='The location of the repository to print commands for.',
    )
    parser.add_argument(
        '--json-key',
        metavar='JSON_KEY',
        help=(
            'The path to the JSON key file to use for authentication. If not'
            ' specified, the authentication commands printed will use the token'
            ' from the logged in user.'
        ),
    )

  def Run(self, args):
    """Run the go goauth command."""
    # TODO(b/399155579): Add support for regional endpoints.
    url_line = endpoint_util.ArtifactRegistryDomainEndpoint(
        args.location, 'go', rep=False, protocol='https'
    )
    header_line = go_util.AuthorizationHeader(args.json_key)
    log.out.Print(url_line)
    log.out.Print('')
    log.out.Print(header_line)
    log.out.Print('')
