import events from 'node:events';
import { TsType as AdapterTsType } from '../adapter';
import * as Events from './events';
import { Device } from './model';
import Group from './model/group';
import { DeviceType } from './tstype';
interface Options {
    network: AdapterTsType.NetworkOptions;
    serialPort: AdapterTsType.SerialPortOptions;
    databasePath: string;
    databaseBackupPath: string;
    backupPath: string;
    adapter: AdapterTsType.AdapterOptions;
    /**
     * This lambda can be used by an application to explictly reject or accept an incoming device.
     * When false is returned zigbee-herdsman will not start the interview process and immidiately
     * try to remove the device from the network.
     */
    acceptJoiningDeviceHandler: (ieeeAddr: string) => Promise<boolean>;
}
export interface ControllerEventMap {
    message: [data: Events.MessagePayload];
    adapterDisconnected: [];
    deviceJoined: [data: Events.DeviceJoinedPayload];
    deviceInterview: [data: Events.DeviceInterviewPayload];
    deviceAnnounce: [data: Events.DeviceAnnouncePayload];
    deviceNetworkAddressChanged: [data: Events.DeviceNetworkAddressChangedPayload];
    deviceLeave: [data: Events.DeviceLeavePayload];
    permitJoinChanged: [data: Events.PermitJoinChangedPayload];
    lastSeenChanged: [data: Events.LastSeenChangedPayload];
}
/**
 * @noInheritDoc
 */
export declare class Controller extends events.EventEmitter<ControllerEventMap> {
    private options;
    private database;
    private adapter;
    private greenPower;
    private touchlink;
    private permitJoinTimer;
    private permitJoinEnd?;
    private backupTimer;
    private databaseSaveTimer;
    private stopping;
    private adapterDisconnected;
    private networkParametersCached;
    /** List of unknown devices detected during a single runtime session. Serves as de-dupe and anti-spam. */
    private unknownDevices;
    /**
     * Create a controller
     *
     * To auto detect the port provide `null` for `options.serialPort.path`
     */
    constructor(options: Options);
    /**
     * Start the Herdsman controller
     */
    start(): Promise<AdapterTsType.StartResult>;
    touchlinkIdentify(ieeeAddr: string, channel: number): Promise<void>;
    touchlinkScan(): Promise<{
        ieeeAddr: string;
        channel: number;
    }[]>;
    touchlinkFactoryReset(ieeeAddr: string, channel: number): Promise<boolean>;
    touchlinkFactoryResetFirst(): Promise<boolean>;
    addInstallCode(installCode: string): Promise<void>;
    permitJoin(time: number, device?: Device): Promise<void>;
    getPermitJoin(): boolean;
    getPermitJoinEnd(): number | undefined;
    isStopping(): boolean;
    isAdapterDisconnected(): boolean;
    stop(): Promise<void>;
    private databaseSave;
    backup(): Promise<void>;
    coordinatorCheck(): Promise<{
        missingRouters: Device[];
    }>;
    reset(type: 'soft' | 'hard'): Promise<void>;
    getCoordinatorVersion(): Promise<AdapterTsType.CoordinatorVersion>;
    getNetworkParameters(): Promise<AdapterTsType.NetworkParameters>;
    /**
     * Get all devices
     * @deprecated use getDevicesIterator()
     */
    getDevices(): Device[];
    /**
     * Get iterator for all devices
     */
    getDevicesIterator(predicate?: (value: Device) => boolean): Generator<Device>;
    /**
     * Get all devices with a specific type
     */
    getDevicesByType(type: DeviceType): Device[];
    /**
     * Get device by ieeeAddr
     */
    getDeviceByIeeeAddr(ieeeAddr: string): Device | undefined;
    /**
     * Get device by networkAddress
     */
    getDeviceByNetworkAddress(networkAddress: number): Device | undefined;
    /**
     * Get IEEE address for all devices
     */
    getDeviceIeeeAddresses(): string[];
    /**
     * Get group by ID
     */
    getGroupByID(groupID: number): Group | undefined;
    /**
     * Get all groups
     * @deprecated use getGroupsIterator()
     */
    getGroups(): Group[];
    /**
     * Get iterator for all groups
     */
    getGroupsIterator(predicate?: (value: Group) => boolean): Generator<Group>;
    /**
     * Create a Group
     */
    createGroup(groupID: number): Group;
    /**
     * Broadcast a network-wide channel change.
     */
    private changeChannel;
    identifyUnknownDevice(nwkAddress: number): Promise<Device | undefined>;
    private checkDeviceNetworkAddress;
    private onNetworkAddress;
    private onIEEEAddress;
    private onDeviceAnnounce;
    private onDeviceLeave;
    private onAdapterDisconnected;
    private onDeviceJoinedGreenPower;
    private selfAndDeviceEmit;
    private onDeviceJoined;
    private onZdoResponse;
    private onZclPayload;
}
export default Controller;
//# sourceMappingURL=controller.d.ts.map