.\"
.\" Copyright (c) 2023 Andrea Biscuola <a@abiscuola.com>
.\"
.\" Permission to use, copy, modify, and distribute this software for any
.\" purpose with or without fee is hereby granted, provided that the above
.\" copyright notice and this permission notice appear in all copies.
.\"
.\" THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
.\" WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
.\" MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
.\" ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
.\" WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
.\" ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
.\" OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
.\"
.Dd August 18, 2023
.Dt IRCTK-IRC n
.Os
.Sh NAME
.Nm irctk-irc
.Nd the irctk irc namespace library
.Sh SYNOPSIS
.Cm irc cap Ar chan tags capability
.Pp
.Cm irc dial Ar cmd callback
.Pp
.Cm irc join Ar chan tags channels Op Ar keys
.Pp
.Cm irc nick Ar chan tags nick
.Pp
.Cm irc notice Ar chan tags target Op Ar args
.Pp
.Cm irc part Ar chan tags channels Op Ar args
.Pp
.Cm irc ping Ar chan tags text
.Pp
.Cm irc pong Ar chan tags text
.Pp
.Cm irc privmsg Ar chan tags target Op Ar args
.Pp
.Cm irc quit Ar chan tags Op Ar text
.Pp
.Cm irc receive Ar chan
.Pp
.Cm irc register Ar chan tags user nick pass mode name
.Pp
.Cm irc send Ar chan tags cmd Op Ar args
.Sh DESCRIPTION
The irc namespace implements the IRC client specification as defined in
RFC2812 and a subset of the modern IRCv3 specification.
.Pp
The
.Cm irc dial
command must be used to establish a connection to an irc server.
.Ar cmd
is a shell script that will be launched as a subprocess
and connected to
.Xr irctk 1
through a bidirectional pipe, exchanging IRC commands through the
subprocess standard input and standard output channels.
In effect, the subprocess is responsible for the whole transport layer.
The
.Ar callback
argument will be called whenever new data are ready to on the read side of
the established communication channel and it will be called with the
form:
.Pp
.Dl callback chan
.Pp
where
.Ar chan
is the communication channel just created.
The returned channel must then be passed to all the other commands
when a specific IRC operation must be performed.
.Pp
The
.Cm irc receive
command can be called to receive a new IRC message from the specified
.Ar chan
message in a structured form.
The command returns a dictionary containing the following fields:
.Bl -tag -width command
.It tags
Dictionary containing all the processed IRCv3 tags stored as key-value
pairs.
.It server
The host the message is coming from
.It user
The user that sent the message
.It nick
The nickname of the user that sent the message, or the server that
forwarded it if it was generated by one of the IRC network servers.
.It command
The IRC command.
.It args
List of command arguments
.It date
Timestamp in UNIX epoch format.
.El
.Pp
The
.Cm irc send
command can be used to send an IRC message.
The message will be composed by zero or more
.Ar tags
passed as a dictionary containing a series of key-value pairs, an IRC
command passed as
.Ar cmd
and zero or more command arguments in
.Ar args .
.Ss High level commands
The following high level commands should be used to send commands
to an IRC server.
If a command accept an arguments called
.Ar tags ,
it means, as mentioned, a dictionary of key-value pairs identifying IRCv3
tags unless differently specified.
.Pp
.Cm irc cap
is used to manage the IRCv3 capabilities subsystem of the server
connected to
.Ar chan .
By default, the
.Cm irc cap
command will send a
.Dq CAP REQ Ar capability
command to the server, asking it to enable
.Ar capability
for the client.
The only exceptions for the
.Ar capability
arguments are:
.Bl -tag -width Ds
.It ls
Send a
.Dq CAP LS
command, asking the server for the list of capabilities it supports.
.It end
Send a
.Dq CAP END
command, terminating the capabilities negotiation.
.El
.Pp
.Cm irc register
can be used to handle the client registration and login process to
the irc server.
If the
.Ar pass
argument is an empty string no PASS command will be issued.
.Pp
.Cm irc join
can be used to join one or more IRC channels.
.Ar channels
and
.Ar keys
are comma-separated lists of IRC channels and related passwords,
in case joining a channel requires one.
.Pp
.Cm irc part
can be used to leave one or more channels.
As for
.Cm irc join ,
.Ar channels
is a comma-separated list of channels to leave.
.Pp
.Cm irc nick
can be used to change the current user nickname.
.Pp
.Cm irc notice
and
.Cm irc privmsg
sends, respectively, a NOTICE or a PRIVMSG command to the IRC server
directed at
.Ar target ,
be it a channel or another user nickame.
.Ar args
is a variable-length list of strings that will compose the message to send.
.Pp
.Cm irc quit
will QUIT the connection with
.Ar text
as a reason.
To note that the command does not close
.Ar chan .
It's responsibility of the caller to do so, either immediately, or after
receiving the related ERROR response from the irc server.
.Pp
.Cm irc ping
and
.Cm irc pong
are used for the periodic check with the server if the connection is still
alive.
.Xr irctk 1
may initiate e PING, waiting for the server response, or it may answer
with a PONG, if the PING is initiated by the irc server itself.
.Ar text
is the message passed to identify the PING or PONG transactions.
.Sh RETURN VALUES
.Cm irc dial
returns the opened bi-directional channel on success, or nothing if
an error occured.
.Pp
.Cm irc receive
returns a dictionary with the irc command field values, or -1 in case
an error occured during the read from the channel.
.Pp
.Cm irc send
and all the other commands don't return any value at all.
.Pp
In general, errors on the channel are expected to be handled in the
tcl event-loop, as a
.Dq readable
event is fired whenever a channel is closed, or if any error happened
on it.
.Sh EXAMPLES
Dial to libera.chat, using
.Xr nc 1
with TLS and use the
.Dq newmsg
Command as a callback:
.Bd -literal
proc newmsg {ch} {
	set msg [irc receive $ch]

	if {$rmsg == -1} {
		if {[eof $chan]} {
			puts stderr "error on $chan"
			close $chan
		}

		return
	}

	puts $msg
}

irc dial "nc -c irc.libera.chat 6697" newmsg
.Ed
.Sh SEE ALSO
.Xr irctk 1 ,
.Xr irctk-exts n ,
.Xr irctk-gui n ,
.Xr irctk-irctk n ,
.Xr irctk-log n
.Sh AUTHORS
.An Andrea Biscuola Aq Mt a@abiscuola.com
.An Omar Polo Aq Mt op@omarpolo.com
