// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_LOGIN_LOGIN_MANAGER_TEST_H_
#define CHROME_BROWSER_CHROMEOS_LOGIN_LOGIN_MANAGER_TEST_H_

#include <string>

#include "base/macros.h"
#include "chrome/browser/chromeos/login/test/embedded_test_server_mixin.h"
#include "chrome/test/base/mixin_based_in_process_browser_test.h"

class AccountId;

namespace chromeos {

class UserContext;

// Base class for Chrome OS Login tests. Should be used if you need to start at
// the Chrome OS Login screen (especially with existing users). For the tests
// that are focused more on OOBE - prefer OobeBaseTest. Use LoginManagerMixin to
// configure users for tests.
class LoginManagerTest : public MixinBasedInProcessBrowserTest {
 public:
  LoginManagerTest();
  ~LoginManagerTest() override;

  void SetUpCommandLine(base::CommandLine* command_line) override;
  void SetUpOnMainThread() override;

  // Could be used to registers the user with the given `account_id` on the
  // device. This method should be called in PRE_* test. Use only if necessary,
  // prefer LoginManagerMixin instead.
  void RegisterUser(const AccountId& account_id);

  static const char kPassword[];
  UserContext CreateUserContext(const AccountId& account_id,
                                const std::string& password);

  // Set expected credentials for next login attempt.
  void SetExpectedCredentials(const UserContext& user_context);

  // Tries to login with the credentials in `user_context`. The return value
  // indicates whether the login attempt succeeded.
  bool TryToLogin(const UserContext& user_context);

  // Tries to add the user identified and authenticated by `user_context` to the
  // session. The return value indicates whether the attempt succeeded. This
  // method does the same as TryToLogin() but doesn't verify that the new user
  // has become the active user.
  bool AddUserToSession(const UserContext& user_context);

  // Log in user with `user_id`. User should be registered using RegisterUser().
  void LoginUser(const AccountId& account_id);

  // Add user with `user_id` to session.
  void AddUser(const AccountId& user_id);

  void set_should_launch_browser(bool launch) {
    should_launch_browser_ = launch;
  }

 private:
  bool should_launch_browser_ = false;
  EmbeddedTestServerSetupMixin embedded_test_server_{&mixin_host_,
                                                     embedded_test_server()};
  DISALLOW_COPY_AND_ASSIGN(LoginManagerTest);
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_LOGIN_LOGIN_MANAGER_TEST_H_
