/* commands.c: -*- C -*-  Implementation of the commands for MDB. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Sat Sep 30 13:27:28 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include "language.h"
#include "mdb.h"
#include "commands.h"
#include "breakpoints.h"

/* readline/readline.c */
extern char *readline (char *prompt);

/* An array of MDB_File pointers. */
static MDB_File **mdb_files = (MDB_File **)NULL;
static int mdb_files_slots = 0;
static int mdb_files_index = 0;
static int mdb_files_offset = 0;

/* Information about the recent file and line number. */
static char *mdb_recent_file = (char *)NULL;
static int mdb_recent_line_number = -1;

/* The "current" breakpoint. */
static MDB_Breakpoint *mdb_current_bp = (MDB_Breakpoint *)NULL;

/* When non-zero, it is time to quit. */
int MDB_QuitFlag = 0;

/* When non-zero, it is time to continue. */
int MDB_ContFlag = 0;

static MFunction *MDB_LastCommand = (MFunction *)NULL;
static int mdb_allow_redo = 1;
#define CANNOT_REDO mdb_allow_redo = 0

static void initialize_page_handlers (void);
static void parse_file_and_line (char *string, char **file,
				 int *line, int *defaulted_p);

static char *mdb_quit (MDBArgs);
static char *mdb_eval (MDBArgs);
static char *mdb_help (MDBArgs);
static char *mdb_exec (MDBArgs);
static char *mdb_load (MDBArgs);
static char *mdb_list (MDBArgs);
static char *mdb_info (MDBArgs);
static char *mdb_breakpoint (MDBArgs);
static char *mdb_delete (MDBArgs);
static char *mdb_cont (MDBArgs);
static char *mdb_next (MDBArgs);

MDBCommand mdb_command_table[] = {
  { "quit",	(char *)NULL,	mdb_quit,	"quit",
      "Quit using the debugger" },

  { "eval",	(char *)NULL,	mdb_eval,	"eval EXPR",
      "Evaluate EXPR in the current context" },

  { "help",	"?",		mdb_help,	"help [COMMAND]",
      "Provide help on using MDB" },

  { "exec",	"r",		mdb_exec,	"exec FILENAME",
      "Load FILENAME, and evaluate the contents" },

  { "break",	"b",		mdb_breakpoint,	"break [FILENAME:]LINE",
      "Set a breakpoint for the RUN command" },

  { "delete",	"d",		mdb_delete,	"delete BREAKPOINT",
      "Delete an existing breakpoint" },
  
  { "cont",	"c",		mdb_cont,	"cont [COUNT]",
      "Continue from a breakpoint, perhaps skipping COUNT more" },

  { "next",	"n",		mdb_next,	"next [COUNT]",
      "Executes COUNT more statements" },

  { "load",	(char *)NULL,	mdb_load,	"load FILENAME",
      "Load FILENAME, making it the current file" },

  { "list",	"l",		mdb_list,	"list [FILENAME:]LINE",
      "List the page around LINE-NUMBER" },

  { "info",	"i",		mdb_info,	"info [files|breakpoints]",
      "Get information about files or breakpoints" },

  { (char *)NULL, (char *)NULL, (MFunction *)NULL, (char *)NULL, (char *)NULL }
};

static MDBCommand *
find_command (char *name)
{
  register int i;
  int possibles = 0;
  MDBCommand *close_match = (MDBCommand *)NULL;
  int namelen = strlen (name);
  
  for (i = 0; mdb_command_table[i].name; i++)
    {
      if ((strcasecmp (mdb_command_table[i].name, name) == 0) ||
	  (mdb_command_table[i].alias &&
	   (strcasecmp (mdb_command_table[i].alias, name) == 0)))
	return (&mdb_command_table[i]);
      else if (strncasecmp (mdb_command_table[i].name, name, namelen) == 0)
	{
	  possibles++;
	  close_match = &mdb_command_table[i];
	}
    }      

  /* Okay, that failed.  If the command is uniquely expressed by what was
     typed, then use that command. */
  if (possibles == 1)
    return (close_match);
  else
    return ((MDBCommand *)NULL);
}

static char *
make_error (char *format, ...)
{
  BPRINTF_BUFFER *buffer = bprintf_create_buffer ();
  char *result;
  va_list args;

  va_start (args, format);
  bprintf (buffer, "Error: ");
  vbprintf (buffer, format, args);

  if (buffer->buffer && buffer->bindex &&
      buffer->buffer[buffer->bindex - 1] != '\n')
    bprintf (buffer, "\n");

  result = buffer->buffer;
  free (buffer);

  return (result);
}

static char *
make_message (char *format, ...)
{
  BPRINTF_BUFFER *buffer = bprintf_create_buffer ();
  char *result;
  va_list args;

  va_start (args, format);
  vbprintf (buffer, format, args);

  if (buffer->buffer && buffer->bindex &&
      buffer->buffer[buffer->bindex - 1] != '\n')
    bprintf (buffer, "\n");

  result = buffer->buffer;
  free (buffer);

  return (result);
}

static MDB_File *
mdb_add_file (char *filename, PAGE *page)
{
  MDB_File *file = (MDB_File *)xmalloc (sizeof (MDB_File));

  file->filename = strdup (filename);

  file->nameonly = strrchr (file->filename, '/');
  if (file->nameonly)
    file->nameonly += 1;
  else
    file->nameonly = file->filename;

  file->contents = page;
  file->line_number = 0;
  stat (filename, &(file->finfo));

  if (mdb_files_index + 2 > mdb_files_slots)
    mdb_files = (MDB_File **)xrealloc
      (mdb_files, ((mdb_files_slots += 10) * sizeof (MDB_File *)));

  mdb_files[mdb_files_index++] = file;
  mdb_files[mdb_files_index] = (MDB_File *)NULL;

  return (file);
}

static MDB_File *
mdb_find_file (char *filename)
{
  register int i;
  MDB_File *file = (MDB_File *)NULL;
  char *incpref = pagefunc_get_variable ("mhtml::include-prefix");
  char *relpref = pagefunc_get_variable ("mhtml::relative-prefix");
  char *canon_name;

  if ((filename && incpref) &&
      (strncmp (filename, incpref, strlen (incpref)) == 0))
    canon_name = strdup (filename);
  else
    canon_name = mhtml_canonicalize_file_name (filename, incpref, relpref);

  if (canon_name != (char *)NULL)
    {
      for (i = 0; i < mdb_files_index; i++)
	if ((strcmp (canon_name, mdb_files[i]->filename) == 0) ||
	    (strcmp (filename, mdb_files[i]->filename) == 0) ||
	    (strcmp (filename, mdb_files[i]->nameonly) == 0))
	  {
	    file = mdb_files[i];
	    break;
	  }

      if (!file)
	{
	  PAGE *page = page_read_template (canon_name);

	  if (page)
	    file = mdb_add_file (canon_name, page);
	}
      else
	{
	  struct stat finfo;

	  if (stat (canon_name, &finfo) > -1)
	    {
	      if (finfo.st_mtime > file->finfo.st_mtime)
		{
		  page_free_page (file->contents);
		  file->contents = page_read_template (canon_name);
		  memmove (&(file->finfo), &finfo, sizeof (finfo));
		  fprintf (stderr, "Reloaded changed `%s'.\n", file->nameonly);
		}
	    }
	}
      free (canon_name);
    }

  return (file);
}

static void
mdb_set_current_file (char *filename)
{
  register int i;

  for (i = 0; i < mdb_files_index; i++)
    if (strcmp (filename, mdb_files[i]->filename) == 0)
      {
	mdb_files_offset = i;
	if (mdb_recent_file)
	  free (mdb_recent_file);
	mdb_recent_file = strdup (mdb_files[i]->filename);
	break;
      }
}

char *
mdb_command (char *line)
{
  register int start, i;
  char *name = (char *)NULL;
  char *result = (char *)NULL;
  MDBCommand *command = (MDBCommand *)NULL;
  static int initialized = 0;

  if (!initialized)
    {
      initialize_page_handlers ();
      initialized = 1;
    }

  /* Kill leading whitespace. */
  for (start = 0; whitespace (line[start]); start++);

  mdb_allow_redo = 1;
  if (line[start] == '<')
    {
      name = strdup ("eval");
      i = start;
    }
  else
    {
      for (i = start; line[i] && !whitespace (line[i]); i++);

      if (i == start)
	return ((char *)NULL);

      name = (char *)xmalloc (1 + (i - start));
      strncpy (name, line + start, i - start);
      name[i - start] = '\0';

      while (whitespace (line[i])) i++;
    }

  command = find_command (name);

  /* Trim trailing whitespace from LINE. */
  start = i;
  i = strlen (line);
  if (i)
    {
      i--;
      while ((i > start) && (whitespace (line[i]))) i--;

      if (!whitespace (line[i]))
	line[i + 1] = '\0';
      else
	line[i] = '\0';
    }

  if (!command)
    result =  make_error ("%s: Undefined MDB command.  Try `help'.", name);
  else
    {
      result = (*(command->handler)) (name, line + start);
      if (mdb_allow_redo)
	MDB_LastCommand = command->handler;
      else
	MDB_LastCommand = (MFunction *)NULL;
    }

  free (name);
  return (result);
}

char *
mdb_redo (void)
{
  char *result = (char *)NULL;

  if (MDB_LastCommand)
    result = (*MDB_LastCommand) ("redo", "");

  return (result);
}

static char *
mdb_help (MDBArgs)
{
  register int i;
  BPRINTF_BUFFER *buffer = bprintf_create_buffer ();
  char *result;

  for (i = 0; mdb_command_table[i].name; i++)
    {
      bprintf (buffer, "%25s    %s\n", mdb_command_table[i].invocation,
	       mdb_command_table[i].description);
    }

  result = buffer->buffer;
  free (buffer);

  return (result);
}


static char *
mdb_eval (MDBArgs)
{
  PAGE *page = page_create_page ();
  char *result = (char *)NULL;
  int syntax_complete = 0;

  CANNOT_REDO;

  page_debug_clear ();
  page_syserr_clear ();
  bprintf (page, "%s\n", line);
  syntax_complete = page_check_syntax (page);

  while (!syntax_complete)
    {
      line = readline ("   ");

      if (line == (char *)NULL)
	{
	  syntax_complete = 1;
	  page_free_page (page);
	  page = (PAGE *)NULL;
	}
      else
	{
	  bprintf (page, "%s\n", line);
	  free (line);
	  syntax_complete = page_check_syntax (page);
	}
    }

  if ((page != (PAGE *)NULL) && (page->buffer != (char *)NULL))
    {
      page_debug_clear ();
      page_syserr_clear ();
      page_process_page (page);

      if (page)
	{
	  result = page->buffer;
	  free (page);
	}
    }

  if (page_debug_buffer ())
    {
      fprintf (stderr, "[DEBUGGING-OUTPUT]: %s\n", page_debug_buffer ());
      page_debug_clear ();
    }

  if (page_syserr_buffer ())
    {
      fprintf (stderr, "[SYSTEM-ERROR-OUTPUT]: %s\n", page_syserr_buffer ());
      page_syserr_clear ();
    }

  return (result);
}

static char *
mdb_quit (MDBArgs)
{
  char *answer = "y";

  CANNOT_REDO;

  if (mdb_loop_level > 1)
    {
      int done = 0;

      while (!done)
	{
	  answer =
	    readline ("The program is running.  Quit anyway? (y or n) ");

	  if (answer &&
	      (strcasecmp (answer, "yes") != 0) &&
	      (strcasecmp (answer, "no") != 0) &&
	      (strcasecmp (answer, "y") != 0) &&
	      (strcasecmp (answer, "n") != 0))
	    printf ("\rPlease answer \"yes\" or \"no\".\n");
	  else
	    done = 1;
	}
    }

  if (!answer || (strncasecmp (answer, "y", 1) == 0))
    {
      MDB_QuitFlag = 1;
      mdb_throw_to_top_level ();
    }

  return ((char *)NULL);
}

static char *
mdb_exec (MDBArgs)
{
  char *filename;
  int line_number, defaulted;
  MDB_File *file;
  char *result = (char *)NULL;
  PAGE *page = (PAGE *)NULL;
  static char *last_executed_filename = (char *)NULL;

  CANNOT_REDO;

  parse_file_and_line (line, &filename, &line_number, &defaulted);

  if (defaulted)
    {
      if (!last_executed_filename)
	last_executed_filename = strdup (filename);
      else
	{
	  free (filename);
	  filename = strdup (last_executed_filename);
	}
    }
  else
    {
      if (last_executed_filename)
	free (last_executed_filename);
      last_executed_filename = strdup (filename);
    }

  file = mdb_find_file (filename);

  if (!file)
    result = make_error ("Couldn't find `%s' for `%s'", filename, name);
  else
    {
      printf ("\rRunning %s...\n", file->filename);
      mdb_set_current_file (file->filename);
      mdb_recent_line_number = 1;

      if (mdb_recent_file)
	free (mdb_recent_file);

      mdb_recent_file = strdup (file->filename);

      page = page_copy_page (file->contents);

      mdb_insert_breakpoints (file, page, mdb_breakpoint_list ());
      page_process_page (page);

      if (page)
	{
	  result = page->buffer;
	  free (page);
	}
    }

  return (result);
}

static char *
mdb_load (MDBArgs)
{
  char *filename;
  int line_number, defaulted;
  MDB_File *file;
  char *result = (char *)NULL;

  CANNOT_REDO;

  parse_file_and_line (line, &filename, &line_number, &defaulted);
  file = mdb_find_file (filename);
  free (filename);

  if (!file)
    result = make_error ("Couldn't find `%s' for `%s'", line, name);
  else
    {
      mdb_set_current_file (file->filename);
      mdb_recent_line_number = 1;
      result = make_message ("Loaded `%s'", file->filename);
    }

  return (result);
}

static void
parse_file_and_line (char *string, char **file, int *line, int *defaulted_p)
{
  register int i;
  char *colon = strchr (string, ':');
  char *parseloc = string;
  int digits_only = 1;
  int defaulted = 1;

  *file = (char *)NULL;
  *line = -1;

  if (string && *string)
    {
      if (colon)
	parseloc = colon + 1;

      /* Check to see if there are only digits here. */
      for (i = 0; parseloc[i]; i++)
	{
	  if (!isdigit (parseloc[i]))
	    {
	      digits_only = 0;
	      break;
	    }
	}

      if (digits_only)
	*line = atoi (parseloc);

      if (colon || !digits_only)
	{
	  if (!colon)
	    colon = string + strlen (string);

	  *file = (char *)xmalloc (1 + (colon - string));
	  strncpy (*file, string, (colon - string));
	  (*file)[colon - string] = '\0';
	}
    }

  /* If no file supplied, then use the old one. */
  if (!*file)
    *file = mdb_recent_file ? strdup (mdb_recent_file) : (char *)NULL;
  else
    defaulted = 0;

  /* If no line number, then use the old one. */
  if (*line < 0)
    *line = mdb_recent_line_number;
  else if (defaulted)
    defaulted = 0;

  *defaulted_p = defaulted;
}

int
mdb_count_lines (char *string)
{
  register int i;
  int lines = 0;

  if (string && *string)
    {
      lines++;

      for (i = 0; string[i]; i++)
	if (string[i] == '\n')
	  lines++;
    }
  return (lines);
}

static char *
make_listing (MDB_File *file, int line_number, int arrow_line)
{
  register int i;
  int counter = 1, limit;
  char *content = file->contents->buffer;
  BPRINTF_BUFFER *listing = bprintf_create_buffer ();
  char *result;

  limit = mdb_count_lines (content);

  if (line_number > limit)
    line_number = limit - 10;
  else
    line_number -= 10;

  if (line_number < 1)
    line_number = 1;

  /* Okay, count newlines until we are at the right number. */
  for (i = 0; i < file->contents->bindex; i++)
    {
      if (counter == line_number)
	break;

      if (content[i] == '\n')
	counter++;
    }

  /* Okay, ready to produce the listing. */
  limit = counter + 20;

  for (; counter < limit; counter++)
    {
      /* If this is the arrow line, print it now. */
      bprintf (listing, "%s%4d     ",
	       counter == arrow_line ? "->" : "  ", counter);

      /* Print one line. */
      for (; i < file->contents->bindex && content[i] != '\n'; i++)
	bprintf (listing, "%c", content[i]);

      i++;
      bprintf (listing, "\n");
      if (i >= file->contents->bindex)
	break;
    }

  result = listing->buffer;
  free (listing);

  return (result);
}

static char *
mdb_list (MDBArgs)
{
  int line_number, defaulted;
  int arrow_line = -1;
  char *filename;
  MDB_File *file;

  parse_file_and_line (line, &filename, &line_number, &defaulted);

  /* If there was no recent file, and we defaulted everything,
     there is nothing to list. */
  if (!mdb_recent_file && defaulted)
    return (make_error ("No source file specified.  Use `load'"));

  /* If the file and line were defaulted, and the last command was
     a LIST, then increase the line number now. */
  if (defaulted && (MDB_LastCommand == mdb_list))
    {
      line_number += 20;
      mdb_recent_line_number = line_number;
    }

  /* Find the file that the user specified. */
  file = mdb_find_file (filename);

  if (!file)
    return (make_error ("Cannot find `%s' for `%s'", filename, name));

  if (mdb_recent_file != filename)
    {
      mdb_recent_line_number = line_number;
      mdb_set_current_file (file->filename);
    }

  if ((mdb_loop_level > 1) && (mdb_current_bp))
    arrow_line = mdb_current_bp->line_number;

  return (make_listing (file, line_number, arrow_line));
}


static char *
mdb_file_info (void)
{
  BPRINTF_BUFFER *listing = bprintf_create_buffer ();
  char *result;

  if (mdb_files_index == 0)
    {
      bprintf (listing, "There are no loaded files.");
    }
  else
    {
      register int i;

      bprintf (listing, " Lines Bpt's  File\n");

      for (i = 0; i < mdb_files_index; i++)
	{
	  MDB_File *file = mdb_files[i];
	  bprintf (listing, "%5d  %3d    %s\n",
		   mdb_count_lines (file->contents->buffer),
		   mdb_count_breakpoints (file),
		   file->filename);
	}
    }

  result = listing->buffer;
  free (listing);

  return (result);
}

static char *
mdb_info (MDBArgs)
{
  char *result;

  if ((strcasecmp (line, "files") == 0) ||
      (!*line))
    result = mdb_file_info ();
  else if (strncasecmp (line, "bre", 3) == 0)
    result = mdb_breakpoint_info ();
  else
    result = make_error ("Unrecognized argument to %s, `%s'.", name, line);

  return (result);
}

/* The honest to goodness workhorse function declaration. */
static void pf_break_handler (PFunArgs);
static void pf_include_handler (PFunArgs);
static PFunDesc MDB_BreakPFunDesc = { "*MDB*::BREAK", 0, 0, pf_break_handler };
static PFunDesc MDB_IncludePFunDesc = { "INCLUDE", 0, 0, pf_include_handler };
extern char *get_positional_arg (Package *package, int position);

/* The BREAK function looks like "<*mdb*::break which existing-statements>". */
static void
pf_break_handler (PFunArgs)
{
  int which = atoi (get_positional_arg (vars, 0));
  MDB_Breakpoint *bp = mdb_this_breakpoint (which);
  MDB_Breakpoint *previous = mdb_current_bp;

  mdb_current_bp = bp;

  /* Before we do anything else, put back the remainder of the expression to
     be evaluated.  */
  bp->position = start;
  bp->code = page;

  if (body && body->buffer)
    {
      register int i;

      /* Skip leading whitespace. */
      for (i = 0; whitespace (body->buffer[i]); i++);

      /* Skip the digits which identify this breakpoint. */
      if (body->buffer[i] == '-') i++;

      while (isdigit (body->buffer[i])) i++;

      /* Skip the single trailing space. */
      i++;

      /* Insert the remainder. */
      bprintf_insert (page, start, "%s", body->buffer + i);
    }

  if (MDB_ContFlag)
    {
      mdb_current_bp = previous;
      MDB_ContFlag--;
      return;
    }

  if (bp->type != break_DELETED)
    {
      char *listing;

      mdb_set_current_file (bp->file->filename);

      if (bp->type != break_INTERNAL)
	{
	  printf ("\n*Bpt %d in %s at line %d\n", which + 1,
		  bp->file->filename, bp->line_number);

	  listing =
	    make_listing (mdb_find_file (bp->file->filename),
			  bp->line_number, bp->line_number);
	  printf ("%s", listing);
	  free (listing);
	}
      else
	{
	  /* Print the remainder of the breakpoint here. */
	  PAGE *source = bp->file->contents;
	  int pos = mdb_position_of_line (source->buffer, bp->line_number);

	  printf ("\r%5d  ", bp->line_number);

	  while ((pos > 0) && (pos < source->bindex))
	    {
	      if (source->buffer[pos] == '\n')
		break;
	      printf ("%c", source->buffer[pos++]);
	    }
	  printf ("\n");
	}

      mdb_loop ();

      mdb_current_bp = previous;

      /* If the user has quit, under NO CIRCUMSTANCES should we
	 continue to execute the code. */
      if (MDB_QuitFlag)
	mdb_throw_to_top_level ();
    }
}

/* Continue execution from a breakpoint. */
static char *
mdb_cont (MDBArgs)
{
  int count = atoi (line);
  char *result = (char *)NULL;

  if (mdb_loop_level > 1)
    {
      MDB_ContFlag += count ? count : 1;
      result = strdup ("Continuing...");
    }

  return (result);
}

/* Do the next instruction. */
static char *
mdb_next (MDBArgs)
{
  int count = atoi (line);

  if (mdb_loop_level > 1)
    {
      mdb_set_next_breakpoint (mdb_current_bp);
      MDB_ContFlag += count ? count : 1;
    }
  return (char *)NULL;
}

/* Remove a breakpoint. */
static char *
mdb_delete (MDBArgs)
{
  register int i = 0;
  int which = atoi (line) - 1;
  MDB_Breakpoint **bps = mdb_breakpoint_list ();
  char *result = (char *)NULL;

  if (bps)
    for (i = 0; bps[i]; i++);

  if ((which < 0) || (which >= i) || (bps[which]->type == break_DELETED))
    result = make_error ("There is no breakpoint `%s'", line);
  else
    {
      bps[which]->type = break_DELETED;
      result = make_message ("Deleted breakpoint %d", which + 1);
    }

  return (result);
}

static void
initialize_page_handlers (void)
{
  Symbol *sym;
  PAGE *page = page_create_page ();

  bprintf (page, "<defsubst *mdb-prog*>%%body</defsubst>\n");
  page_process_page (page);
  page_free_page (page);

  sym = symbol_intern_in_package 
    (mhtml_function_package, MDB_BreakPFunDesc.tag);
  sym->values = (char **)&MDB_BreakPFunDesc;

  sym = symbol_intern_in_package (mhtml_function_package, "INCLUDE");
  sym->values = (char **)&MDB_IncludePFunDesc;
}

static char *
mdb_breakpoint (MDBArgs)
{
  int line_number, defaulted;
  char *filename;
  char *result = (char *)NULL;
  MDB_File *file;

  CANNOT_REDO;

  parse_file_and_line (line, &filename, &line_number, &defaulted);

  if (!*line || line_number < 1)
    return (make_error ("`%s' requires a line number", name));

  file = mdb_find_file (filename);

  if (!file)
    result = make_error ("Cannot load `%s' for `%s'", name, filename);
  else
    {
      mdb_add_breakpoint (file, line_number, break_USER);
      result = make_message ("Break at line %05d, in file %s.",
			     line_number, file->nameonly);
    }
  return (result);
}

#if defined (MDB_EDITOR_COMMAND)
static char *
mdb_edit (MDBArgs)
{
  int line_number, defaulted;
  char *filename;
  char *result = (char *)NULL;
}
#endif

static int include_recursive_calls = 0;

static void
pf_include_handler (PFunArgs)
{
  int verbatim_p = var_present_p (vars, "VERBATIM");

  include_recursive_calls++;
  if (include_recursive_calls < MHTML_INCLUDE_RECURSION_LIMIT)
    {
      char *arg = mhtml_evaluate_string (get_positional_arg (vars, 0));
      char *alt = mhtml_evaluate_string
	(get_one_of (vars, "ALT", "ALTERNATE", (char *)0));
      char *canonicalized_name = (char *)NULL;
      char *incpref = pagefunc_get_variable ("%%::incpref");
      char *relpref = pagefunc_get_variable ("%%::relpref");

      if (!incpref) incpref = pagefunc_get_variable ("mhtml::include-prefix");
      if (!relpref) relpref = pagefunc_get_variable ("mhtml::relative-prefix");

      if (arg != (char *)NULL)
	canonicalized_name =
	  mhtml_canonicalize_file_name (arg, incpref, relpref);

      if (canonicalized_name != (char *)NULL)
	{
	  PAGE *file_contents = (PAGE *)NULL;
	  MDB_File *file;

	  /* Instead of simply reading the page, we load it into
	     MDB if it isn't already loaded.  That way, breakpoints
	     already set in that file are not deleted. */
	  file = mdb_find_file (canonicalized_name);

	  /* Did the user specify some alternate HTML if the file
	     couldn't be found? */
	  if (!file)
	    {
	      if (alt != (char *)NULL)
		{
		  verbatim_p = 0;
		  file_contents = page_create_page ();
		  bprintf (file_contents, "%s", alt);
		}
	    }
	  else
	    {
	      file_contents = page_copy_page (file->contents);
	      mdb_insert_breakpoints
		(file, file_contents, mdb_breakpoint_list ());
	    }

	  if (file_contents)
	    {
	      /* Manually insert the file instead of letting bprintf
		 do it for us.  This is because the file could contain
		 binary data, and then file->bindex wouldn't necessarily
		 reflect the length of what was inserted. */
	      if ((file_contents->bindex + page->bindex) >= page->bsize)
		page->buffer = (char *)xrealloc
		(page->buffer, (page->bsize += (file_contents->bindex + 100)));

	      memmove (page->buffer + start + file_contents->bindex,
		       page->buffer + start,
		       (page->bindex + 1) - start);

	      memcpy (page->buffer + start, file_contents->buffer,
		      file_contents->bindex);
	      page->bindex += file_contents->bindex;

	      if (verbatim_p)
		*newstart += file_contents->bindex;
#if defined (MHTML_INCLUDE_IS_RELATIVE)
	      else if (!empty_string_p (relpref))
		{
		  char *temp = strstr (canonicalized_name, incpref);

		  if (temp != (char *)NULL)
		    {
		      char *last;

		      temp += strlen (incpref);
		      last = strrchr (temp, '/');

		      if (last != (char *)NULL)
			{
			  *last = '\0';

			  bprintf_insert
			    (page, start + file_contents->bindex,
			     "<set-var %%::relpref=%s>", relpref);

			  pagefunc_set_variable ("%%::relpref", temp);

			}
		    }
		}
#endif /* MHTML_INCLUDE_IS_RELATIVE */

	      page_free_page (file_contents);
	    }
	  free (canonicalized_name);
	}
    }
  include_recursive_calls--;
}

