/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: simals.h
 * Header file for asynchronous logic simulator
 * From algorithms by: Brent Serbin and Peter J. Gallant
 * Last maintained by: Steven M. Rubin
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

typedef enum {DELAY_MIN=0, DELAY_TYP, DELAY_MAX} DELAY_TYPES;

/* The trace buffer in ALS can currently hold 10000 events.
 * More events can be accommodated in the trace buffer by
 * increasing its size.
 */
#define DEFAULT_TRACE_SIZE	10000

typedef struct model_ds
{
	INTBIG              num;
	char               *name;
	char                type;
	char               *ptr;	/* may be CONPTR or ROWPTR or FUNCPTR */
	struct export_ds   *exptr;
	struct io_ds       *setptr;
	struct load_ds     *loadptr;
	char                fanout;
	INTSML              priority;
	struct model_ds    *next;
	char               *level;  /* hierarchical level */
} MODEL;
typedef MODEL *MODPTR;

typedef struct row_ds
{
	struct io_ds   *inptr;
	struct io_ds   *outptr;
	float           delta;
	float           linear;
	float           exp;
	float           random;
	float           abs;    /* BA delay - SDF absolute port delay */
	struct row_ds  *next;
	char           *delay;  /* delay transition name (01, 10, etc) */
} ROW;
typedef ROW *ROWPTR;

typedef struct io_ds
{
	struct node_ds *nodeptr;
	unsigned char   operatr;
	char           *operand;
	INTSML          strength;
	struct io_ds   *next;
} IO;
typedef IO *IOPTR;

typedef struct connect_ds
{
	char               *inst_name;
	char               *model_name;
	struct export_ds   *exptr;
	struct connect_ds  *parent;
	struct connect_ds  *child;
	struct connect_ds  *next;
	struct channel_ds  *display_page;  /* pointer to the display page */
	INTBIG              num_chn;       /* number of exported channels in this level */
} CONNECT;
typedef CONNECT *CONPTR;

typedef struct export_ds
{
	char              *node_name;
	struct node_ds    *nodeptr;
	struct export_ds  *next;
	INTBIG            td[12];  /* transition delays */
} EXPORT;
typedef EXPORT *EXPTR;

typedef struct load_ds
{
	char            *ptr;
	float            load;
	struct load_ds  *next;
} LOAD;
typedef LOAD *LOADPTR;

typedef struct func_ds
{
	void             (*procptr)(MODPTR);
	struct export_ds  *inptr;
	float              delta;
	float              linear;
	float              exp;
	float              abs;    /* absolute delay for back annotation */
	float              random;
	char              *userptr;
	INTBIG             userint;
	float              userfloat;
} FUNC;
typedef FUNC *FUNCPTR;

typedef struct node_ds
{
	struct connect_ds  *cellptr;
	INTBIG              num;
	INTBIG              sum_state;
	INTSML              sum_strength;
	INTBIG              new_state;
	INTSML              new_strength;
	BOOLEAN             tracenode;
	INTBIG              plot_node;
	struct stat_ds     *statptr;
	struct load_ds     *pinptr;
	float               load;
	INTBIG              visit;
	INTBIG              maxsize;
	INTBIG              arrive;
	INTBIG              depart;
	float               tk_sec;
	double              t_last;
	struct node_ds     *next;
} NODE;
typedef NODE *NODEPTR;

typedef struct stat_ds
{
	struct model_ds  *primptr;
	struct node_ds   *nodeptr;
	INTBIG            new_state;
	INTSML            new_strength;
	unsigned char     sched_op;
	INTBIG            sched_state;
	INTSML            sched_strength;
	struct stat_ds   *next;
} STAT;
typedef STAT *STATPTR;

typedef struct link_ds
{
	struct link_ds  *left;
	struct link_ds  *right;
	struct link_ds  *up;
	struct link_ds  *down;
	char            *ptr;
	char             type;
	unsigned char    operatr;
	INTBIG           state;
	INTSML           strength;
	INTSML           priority;
	double           time;
	MODPTR           primhead;
} LINK;
typedef LINK *LINKPTR;

typedef struct trak_ds
{
	struct node_ds  *ptr;
	INTBIG           state;
	INTSML           strength;
	double           time;
} TRAK;
typedef TRAK *TRAKPTR;

typedef struct channel_ds
{
	char            *name;
	struct node_ds  *nodeptr;
	INTBIG           displayptr;
} CHANNEL;
typedef CHANNEL *CHNPTR;

/*
 * Now come all the global variables declared extern for reference by other files
 */
extern MODPTR      simals_modroot, simals_primroot;
extern ROWPTR      simals_rowptr2;
extern IOPTR       simals_ioptr2;
extern CONPTR      simals_levelptr, simals_cellroot;
extern EXPTR       simals_exptr2;
extern NODEPTR     simals_noderoot, simals_drive_node;
extern LINKPTR     simals_linkfront, simals_linkback, simals_setroot;
extern TRAKPTR     simals_trakroot;
extern LOADPTR     simals_chekroot;
extern char      **simals_rowptr1, **simals_ioptr1;
extern char       *simals_instbuf;
extern INTBIG      simals_pseq, simals_nseq, *simals_instptr,
				   simals_trakfull, simals_trakptr;
extern INTBIG      simals_no_update_key;		/* variable key for "SIM_als_no_update" */
extern BOOLEAN     simals_seed_flag, simals_trace_all_nodes;
extern double      simals_time_abs;
extern NODEPROTO  *simals_mainproto;
extern INTBIG      simals_trace_size;

/* prototypes for intratool interface */
LINKPTR   simals_alloc_link_mem(void);
#ifdef DEBUGMEMORY
  char   *_simals_alloc_mem(INTBIG, char*, INTBIG);
# define  simals_alloc_mem(a) _simals_alloc_mem((a), (char *)__FILE__, (INTBIG)__LINE__)
#else
  char   *simals_alloc_mem(INTBIG);
#endif
void      simals_annotate_command(INTBIG, char*[]);
INTBIG    simals_atoi(char*);
void      simals_build_actel_command(INTBIG, char*[]);
void      simals_clearallvectors(BOOLEAN);
void      simals_clock_command(INTBIG, char*[]);
void      simals_com_comp(INTBIG, char*[10]);
void      simals_compute_node_name(NODEPTR, char*);
char     *simals_compute_path_name(CONPTR);
void      simals_convert_to_upper(char*);
void      simals_erase_model(void);
void      simals_fill_display_arrays(void);
CONPTR    simals_find_level(char*);
NODEPTR   simals_find_node(char*);
BOOLEAN   simals_flatten_network(void);
BOOLEAN   simals_fragment_command(char*);
void      simals_free_link_mem(LINKPTR);
void      simals_freeflatmemory(void);
void      simals_freesimmemory(void);
INTBIG   *simals_get_function_address(char*);
void      simals_go_command(INTBIG, char*[]);
void      simals_help_command(void);
void      simals_init(void);
void      simals_init_display(void);
double    simals_initialize_simulator(BOOLEAN);
void      simals_insert_link_list(LINKPTR);
char    **simals_insert_set_list(LINKPTR);
void      simals_level_up_command(void);
void      simals_level_set_command(char *instname);
char     *simals_nextinstance(void);
void      simals_order_command(INTBIG, char*[]);
void      simals_print_command(INTBIG, char*[]);
BOOLEAN   simals_read_net_desc(NODEPROTO*);
void      simals_seed_command(INTBIG, char*[]);
void      simals_set_command(INTBIG, char*[]);
BOOLEAN   simals_set_current_level(void);
void      simals_term(void);
BOOLEAN   simals_topofinstances(char**);
void      simals_trace_command(INTBIG, char*[]);
void      simals_trans_number_to_state(INTBIG, char*);
INTBIG    simals_trans_state_to_number(char*);
void      simals_vector_command(INTBIG, char*[]);

BOOLEAN   simals_startsimulation(NODEPROTO *np);
BOOLEAN   simals_charhandlerschem(WINDOWPART*, INTSML, INTBIG);
BOOLEAN   simals_charhandlerwave(WINDOWPART*, INTSML, INTBIG);
