/*
 * Electric(tm) VLSI Design System
 *
 * File: projecttool.c
 * Project management tool
 * Written by: Steven M. Rubin
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if PROJECTAID

#include "global.h"
#include "projecttool.h"
#include "edialogs.h"
#include "usr.h"
#include "usrdiacom.h"
#include "tecgen.h"

/***** command parsing *****/

static COMCOMP projcop = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be checked-out", "check-out current facet"};
static COMCOMP projcip = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be checked-in", "check-in current facet"};
static COMCOMP projoldp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet from which old version should be retrieved",
		"use current facet"};
static COMCOMP projap = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be added", "add current facet"};
static COMCOMP projdp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet to be deleted", "delete current facet"};
static KEYWORD projopt[] =
{
	{"build-project",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"check-out",            1,{&projcop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"check-in",             1,{&projcip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"get-old-version",      1,{&projoldp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"add-facet",            1,{&projap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"delete-facet",         1,{&projdp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"update",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set-user",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"list-facets",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP proj_projp = {projopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Project management tool action", ""};

/***** dialogs *****/

/* Project User */
DIALOGITEM proj_usersdialogitems[] =
{
 /*  1 */ {0, {28,184,52,312}, BUTTON, "Select User"},
 /*  2 */ {0, {188,212,212,276}, BUTTON, "Cancel"},
 /*  3 */ {0, {28,4,220,172}, SCROLL, ""},
 /*  4 */ {0, {8,60,24,112}, MESSAGE, "Users:"},
 /*  5 */ {0, {136,184,160,312}, BUTTON, "Delete User"},
 /*  6 */ {0, {64,184,88,312}, BUTTON, "Change Password"},
 /*  7 */ {0, {100,184,124,312}, BUTTON, "New User"}
};
DIALOG proj_usersdialog = {{50,75,279,398}, "User Name", 7, proj_usersdialogitems};

/* Project Password */
DIALOGITEM proj_passworddialogitems[] =
{
 /*  1 */ {0, {72,12,96,76}, BUTTON, "OK"},
 /*  2 */ {0, {72,132,96,196}, BUTTON, "Cancel"},
 /*  3 */ {0, {12,8,28,205}, MESSAGE, "Password for"},
 /*  4 */ {0, {40,48,56,165}, EDITTEXT, ""}
};
DIALOG proj_passworddialog = {{298,75,404,292}, "User / Password", 4, proj_passworddialogitems};

/* Project List */
DIALOGITEM proj_listdialogitems[] =
{
 /*  1 */ {0, {204,312,228,376}, BUTTON, "Done"},
 /*  2 */ {0, {4,4,196,376}, SCROLL, ""},
 /*  3 */ {0, {260,4,324,376}, MESSAGE, ""},
 /*  4 */ {0, {240,4,256,86}, MESSAGE, "Comments:"},
 /*  5 */ {0, {204,120,228,220}, BUTTON, "Check It Out"},
 /*  6 */ {0, {236,4,237,376}, DIVIDELINE, ""},
 /*  7 */ {0, {204,8,228,108}, BUTTON, "Check It In"},
 /*  8 */ {0, {204,232,228,296}, BUTTON, "Update"}
};
DIALOG proj_listdialog = {{50,75,383,462}, "Project Management", 8, proj_listdialogitems};

/* Project Comments */
DIALOGITEM proj_commentsdialogitems[] =
{
 /*  1 */ {0, {104,244,128,308}, BUTTON, "OK"},
 /*  2 */ {0, {104,48,128,112}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,8,20,363}, MESSAGE, "Reason for checking out facet"},
 /*  4 */ {0, {28,12,92,358}, EDITTEXT, ""}
};
DIALOG proj_commentsdialog = {{108,75,248,447}, "Project Comments", 4, proj_commentsdialogitems};

/* Project Old Version */
DIALOGITEM proj_oldversdialogitems[] =
{
 /*  1 */ {0, {176,224,200,288}, BUTTON, "OK"},
 /*  2 */ {0, {176,16,200,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,8,20,308}, MESSAGE, "Old version of facet"},
 /*  4 */ {0, {28,12,166,312}, SCROLL, ""}
};
DIALOG proj_oldversdialog = {{108,75,318,396}, "Get Old Version of Facet", 4, proj_oldversdialogitems};

/***** facet checking queue *****/

#define	NOFCHECK ((FCHECK *)-1)

typedef struct Ifcheck
{
	NODEPROTO      *entry;
	INTBIG          batchnumber;
	struct Ifcheck *nextfcheck;
} FCHECK;

static FCHECK *proj_firstfcheck = NOFCHECK;
static FCHECK *proj_fcheckfree = NOFCHECK;

/***** user database information *****/

#define	NOPUSER ((PUSER *)-1)
#define PUSERFILE "projectusers"

typedef struct Ipuser
{
	char *username;
	char *userpassword;
	struct Ipuser *nextpuser;
} PUSER;

PUSER *proj_users = NOPUSER;

/***** project file information *****/

#define	NOPROJECTFACET ((PROJECTFACET *)-1)

typedef struct Iprojectfacet
{
	char  *libname;						/* name of the library */
	char  *cellname;					/* name of the cell */
	VIEW  *cellview;					/* cell view */
	INTBIG cellversion;					/* cell version */
	char  *owner;						/* current owner of this facet (if checked out) */
	char  *lastowner;					/* previous owner of this facet (if checked in) */
	char  *comment;						/* comments for this facet */
	struct Iprojectfacet *nextprojectfacet;
} PROJECTFACET;

PROJECTFACET *proj_firstprojectfacet = NOPROJECTFACET;
PROJECTFACET *proj_projectfacetfree = NOPROJECTFACET;

/***** miscellaneous *****/

AIDENTRY     *proj_aid;					/* this tool */
char          proj_username[256];		/* current user's name */
INTBIG        proj_lockedkey;			/* key for "PROJ_locked" */
INTBIG        proj_pathkey;				/* key for "PROJ_path" */
INTBIG        proj_userkey;				/* key for "PROJ_user" */
INTSML        proj_active;				/* nonzero if the system is active */
INTSML        proj_ignorechanges;		/* nonzero to ignore broadcast changes */
AIDENTRY     *proj_source;				/* source of changes */
INTBIG        proj_batchnumber;			/* ID number of this batch of changes */
PUSER        *proj_userpos;				/* current user for dialog display */
FILE         *proj_io;					/* channel to project file */
INTBIG       *proj_saveaidstate = 0;	/* saved aid state information */

/* prototypes for local routines */
void          proj_addfacet(char *facetname);
FCHECK       *proj_allocfcheck(void);
PROJECTFACET *proj_allocprojectfacet(void);
void          proj_buildproject(LIBRARY *lib);
void          proj_checkin(char *facetname);
void          proj_checkinmany(LIBRARY *lib);
void          proj_checkout(char *facetname, INTSML showfacet);
void          proj_deletefacet(char *facetname);
void          proj_endwritingprojectfile(void);
PROJECTFACET *proj_findfacet(NODEPROTO *np);
void          proj_freefcheck(FCHECK *f);
void          proj_freeprojectfacet(PROJECTFACET *pf);
INTSML        proj_getcomments(PROJECTFACET *pf, char *direction);
NODEPROTO    *proj_getfacet(PROJECTFACET *pf, LIBRARY *lib);
void          proj_getoldversion(char *facetname);
INTSML        proj_getprojinfo(LIBRARY *lib, char *path, char *projfile);
INTSML        proj_getusername(INTSML newone, LIBRARY *lib);
INTSML        proj_initusers(char **c);
INTSML        proj_loadlist(LIBRARY *lib);
INTSML        proj_lockprojfile(char *projectpath, char *projectfile);
void          proj_marklocked(NODEPROTO *np, INTSML locked);
char         *proj_nextuser(void);
void          proj_queuecheck(NODEPROTO *facet);
INTSML        proj_readprojectfile(char *pathname, char *filename);
void          proj_restoretoolstate(void);
void          proj_showlistdialog(LIBRARY *lib);
INTSML        proj_startwritingprojectfile(char *pathname, char *filename);
char         *proj_templibraryname(void);
INTSML        proj_turnofftools(void);
void          proj_unlockprojfile(char *projectpath, char *projectfile);
void          proj_update(LIBRARY *lib);
INTSML        proj_usenewestversion(NODEPROTO *oldnp, NODEPROTO *newnp);
void          proj_validatelocks(LIBRARY *lib);
char         *proj_wantpassword(INTSML mode, char *username);
INTSML        proj_writefacet(NODEPROTO *np);

/* prototypes for local routines that should be in the system */
NODEPROTO    *copyrecursively(NODEPROTO *fromnp, LIBRARY *tolib);
NODEPROTO    *copyskeleton(NODEPROTO *fromnp, LIBRARY *tolib);

/************************ CONTROL ***********************/

void proj_init(INTBIG *argc, char *argv[], AIDENTRY *thisaid)
{
	/* ignore pass 3 initialization */
	if (thisaid == 0) return;

	/* miscellaneous initialization during pass 2 */
	if (thisaid == NOAID)
	{
		proj_lockedkey = makekey("PROJ_locked");
		proj_pathkey = makekey("PROJ_path");
		proj_userkey = makekey("PROJ_user");
		proj_username[0] = 0;
		proj_active = 0;
		proj_ignorechanges = 0;
		return;
	}

	/* copy aid pointer during pass 1 */
	proj_aid = thisaid;
}

void proj_done(void)
{
	REGISTER FCHECK *f;

	while (proj_firstfcheck != NOFCHECK)
	{
		f = proj_firstfcheck;
		proj_firstfcheck = proj_firstfcheck->nextfcheck;
		proj_freefcheck(f);
	}
	while (proj_fcheckfree != NOFCHECK)
	{
		f = proj_fcheckfree;
		proj_fcheckfree = proj_fcheckfree->nextfcheck;
		efree((char *)f);
	}
	if (proj_saveaidstate != 0)
		efree((char *)proj_saveaidstate);
}

void proj_slice(void)
{
	REGISTER FCHECK *f, *nextf;
	REGISTER NODEPROTO *np;
	AIDENTRY *aid;
	REGISTER INTSML undoit;
	REGISTER INTBIG lowbatch, highbatch, retval;

	if (proj_active == 0) return;
	if (proj_firstfcheck == NOFCHECK) return;

	undoit = 0;
	for(f = proj_firstfcheck; f != NOFCHECK; f = nextf)
	{
		nextf = f->nextfcheck;
		np = f->entry;

		/* make sure facet np is checked-out */
		if (getvalkey((INTBIG)np, VNODEPROTO, VINTEGER, proj_lockedkey) != NOVARIABLE)
		{
			if (undoit == 0)
			{
				(void)initinfstr();
				lowbatch = highbatch = f->batchnumber;
			} else
			{
				if (f->batchnumber < lowbatch) lowbatch = f->batchnumber;
				if (f->batchnumber > highbatch) highbatch = f->batchnumber;
				(void)addstringtoinfstr(", ");
			}
			(void)addstringtoinfstr(describenodeproto(np));
			undoit++;
		}
		proj_freefcheck(f);
	}
	proj_firstfcheck = NOFCHECK;
	if (undoit != 0)
	{
		ttyputerr("Cannot change unchecked-out %s: %s", makeplural("facet", undoit),
			returninfstr());
		for(;;)
		{
			retval = undoabatch(&aid);
			if (retval == 0) break;
			if (retval < 0) retval = -retval;
			if (retval <= lowbatch) break;
		}
		noredoallowed();
	}
}

INTSML proj_set(INTSML count, char *par[])
{
	REGISTER INTSML l;
	REGISTER char *pp, *facetname;
	REGISTER NODEPROTO *np;

	if (count <= 0)
	{
		ttyputerr("Missing command to display tool");
		return(1);
	}
	l = strlen(pp = par[0]);
	if (namesamen(pp, "add-facet", l) == 0)
	{
		if (count >= 2) facetname = par[1]; else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet to add");
				return(1);
			}
			facetname = describenodeproto(np);
		}
		proj_addfacet(facetname);
		proj_active++;
		return(1);
	}
	if (namesamen(pp, "build-project", l) == 0)
	{
		if (el_curlib == NOLIBRARY)
		{
			ttyputerr("No current library to enter");
			return(1);
		}
		proj_buildproject(el_curlib);
		proj_active++;
		return(0);
	}
	if (namesamen(pp, "check-in", l) == 0 && l >= 7)
	{
		if (count >= 2) facetname = par[1]; else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet to check in");
				return(1);
			}
			facetname = describenodeproto(np);
		}
		proj_checkin(facetname);
		proj_active++;
		return(0);
	}
	if (namesamen(pp, "check-out", l) == 0 && l >= 7)
	{
		if (count >= 2) facetname = par[1]; else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet to check out");
				return(1);
			}
			facetname = describenodeproto(np);
		}
		proj_checkout(facetname, 1);
		proj_active++;
		return(0);
	}
	if (namesamen(pp, "delete-facet", l) == 0)
	{
		if (count >= 2) facetname = par[1]; else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet to delete");
				return(1);
			}
			facetname = describenodeproto(np);
		}
		proj_deletefacet(facetname);
		proj_active++;
		return(1);
	}
	if (namesamen(pp, "get-old-version", l) == 0)
	{
		if (count >= 2) facetname = par[1]; else
		{
			np = getcurfacet();
			if (np == NONODEPROTO)
			{
				ttyputerr("No current facet to retrieve old versions");
				return(1);
			}
			facetname = describenodeproto(np);
		}
		proj_getoldversion(facetname);
		proj_active++;
		return(0);
	}
	if (namesamen(pp, "list-facets", l) == 0)
	{
		proj_showlistdialog(el_curlib);
		return(0);
	}
	if (namesamen(pp, "set-user", l) == 0)
	{
		(void)proj_getusername(1, el_curlib);
		return(0);
	}
	if (namesamen(pp, "update", l) == 0)
	{
		if (el_curlib == NOLIBRARY)
		{
			ttyputerr("No current library to update");
			return(1);
		}
		proj_update(el_curlib);
		proj_active++;
		return(0);
	}
	return(1);
}

void proj_startbatch(AIDENTRY *source, INTSML undoredo)
{
	if (proj_ignorechanges != 0) proj_source = proj_aid; else
		proj_source = source;
	proj_batchnumber = getcurrentbatchnumber();
}

void proj_modifynodeinst(NODEINST *ni, INTBIG olx, INTBIG oly, INTBIG ohx, INTBIG ohy,
	INTSML orot, INTSML otran)
{
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	proj_queuecheck(ni->parent);
}

void proj_modifyarcinst(ARCINST *ai, INTBIG oxA, INTBIG oyA, INTBIG oxB, INTBIG oyB,
	INTBIG owid, INTBIG olen)
{
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	proj_queuecheck(ai->parent);
}

void proj_modifyportproto(PORTPROTO *pp, NODEINST *oni, PORTPROTO *opp)
{
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	proj_queuecheck(pp->parent);
}

void proj_newobject(INTBIG addr, INTBIG type)
{
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	switch (type&VTYPE)
	{
		case VNODEINST:  proj_queuecheck(((NODEINST *)addr)->parent);   break;
		case VARCINST:   proj_queuecheck(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: proj_queuecheck(((PORTPROTO *)addr)->parent);  break;
	}
}

void proj_killobject(INTBIG addr, INTBIG type)
{
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	switch (type&VTYPE)
	{
		case VNODEINST:  proj_queuecheck(((NODEINST *)addr)->parent);   break;
		case VARCINST:   proj_queuecheck(((ARCINST *)addr)->parent);    break;
		case VPORTPROTO: proj_queuecheck(((PORTPROTO *)addr)->parent);  break;
	}
}

void proj_readlibrary(LIBRARY *lib)
{
	REGISTER NODEPROTO *np;

	/* scan the library to see if any facets are locked */
	if (proj_ignorechanges != 0 || proj_source == proj_aid) return;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (getvalkey((INTBIG)np, VNODEPROTO, VINTEGER, proj_lockedkey) != NOVARIABLE)
			proj_active++;
	}
}

/************************ PROJECT MANAGEMENT ***********************/

void proj_checkin(char *facetname)
{
	REGISTER NODEPROTO *np, *onp;
	REGISTER NODEINST *ni;
	REGISTER INTSML total, propagated;
	REGISTER LIBRARY *lib;
	REGISTER PROJECTFACET *pf;

	/* find out which facet is being checked in */
	np = getnodeproto(facetname);
	if (np == NONODEPROTO)
	{
		ttyputerr("Cannot identify facet '%s'", facetname);
		return;
	}

	/* mark the facet to be checked-in */
	lib = np->cell->lib;
	for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		onp->temp1 = 0;
	np->temp1 = 1;

	/* look for facets above this one that must also be checked in */
	for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto) onp->temp2 = 0;
	np->temp2 = 1;
	propagated = 1;
	while (propagated != 0)
	{
		propagated = 0;
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		{
			if (onp->temp2 == 1)
			{
				propagated = 1;
				onp->temp2 = 2;
				for(ni = onp->firstinst; ni != NONODEINST; ni = ni->nextinst)
				{
					if (ni->parent->temp2 == 0) ni->parent->temp2 = 1;
				}
			}
		}
	}
	np->temp2 = 0;
	total = 0;
	for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
	{
		if (onp->temp2 == 0) continue;
		pf = proj_findfacet(onp);
		if (pf == NOPROJECTFACET) continue;
		if (namesame(pf->owner, proj_username) == 0)
		{
			onp->temp1 = 1;
			total++;
		}
	}

	/* look for facets below this one that must also be checked in */
	for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto) onp->temp2 = 0;
	np->temp2 = 1;
	propagated = 1;
	while (propagated != 0)
	{
		propagated = 0;
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		{
			if (onp->temp2 == 1)
			{
				propagated = 1;
				onp->temp2 = 2;
				for(ni = onp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				{
					if (ni->proto->primindex != 0) continue;
					if (ni->proto->temp2 == 0) ni->proto->temp2 = 1;
				}
			}
		}
	}
	np->temp2 = 0;
	for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
	{
		if (onp->temp2 == 0) continue;
		pf = proj_findfacet(onp);
		if (pf == NOPROJECTFACET) continue;
		if (namesame(pf->owner, proj_username) == 0)
		{
			onp->temp1 = 1;
			total++;
		}
	}

	/* advise of additional facets that must be checked-in */
	if (total > 0)
	{
		total = 0;
		(void)initinfstr();
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		{
			if (onp == np || onp->temp1 == 0) continue;
			if (total > 0) (void)addstringtoinfstr(", ");
			(void)addstringtoinfstr(describenodeproto(onp));
			total++;
		}
		ttyputmsg("Also checking in related facet(s): %s", returninfstr());
	}

	/* check it in */
	proj_checkinmany(lib);
}

void proj_checkout(char *facetname, INTSML showfacet)
{
	REGISTER NODEPROTO *np, *newvers, *oldvers;
	REGISTER NODEINST *ni;
	REGISTER INTSML worked, propagated, total;
	REGISTER LIBRARY *lib;
	char projectpath[256], projectfile[256], *argv[3];
	PROJECTFACET *pf;

	/* find out which facet is being checked out */
	oldvers = getnodeproto(facetname);
	if (oldvers == NONODEPROTO)
	{
		ttyputerr("Cannot identify facet '%s'", facetname);
		return;
	}
	lib = oldvers->cell->lib;

	/* make sure there is a valid user name */
	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return;
	}

	/* read the project file */
	worked = 0;
	if (proj_readprojectfile(projectpath, projectfile) != 0)
		ttyputerr("Cannot read project file"); else
	{
		/* find this in the project file */
		pf = proj_findfacet(oldvers);
		if (pf == NOPROJECTFACET) ttyputerr("This facet is not in the project"); else
		{
			/* see if it is available */
			if (*pf->owner != 0)
			{
				if (namesame(pf->owner, proj_username) == 0)
				{
					ttyputerr("This facet is already checked out to you");
					proj_marklocked(oldvers, 0);
				} else
				{
					ttyputerr("This facet is already checked out to '%s'", pf->owner);
				}
			} else
			{
				/* make sure we have the latest version */
				if (pf->cellversion > oldvers->version)
				{
					ttyputerr("Cannot check out %s because you don't have the latest version (yours is %d, project has %d)",
						describenodeproto(oldvers), oldvers->version, pf->cellversion);
					ttyputmsg("Do an 'update' first");
				} else
				{
					if (proj_getcomments(pf, "out") == 0)
					{
						if (proj_startwritingprojectfile(projectpath, projectfile) != 0)
							ttyputerr("Cannot write project file"); else
						{
							/* prevent tools (including this one) from seeing the change */
							(void)proj_turnofftools();

							/* make new version */
							newvers = copynodeproto(oldvers, lib, oldvers->cell->cellname);

							/* restore tool state */
							proj_restoretoolstate();

							if (newvers == NONODEPROTO)
								ttyputerr("Error making new version of facet"); else
							{
								(*el_curconstraint->solve)(newvers);

								/* replace former usage with new version */
								if (proj_usenewestversion(oldvers, newvers) != 0)
									ttyputerr("Error replacing instances of new %s",
										describenodeproto(oldvers)); else
								{
									/* update record for the facet */
									(void)reallocstring(&pf->owner, proj_username, proj_aid->cluster);
									(void)reallocstring(&pf->lastowner, "", proj_aid->cluster);
									proj_marklocked(newvers, 0);
									worked = 1;
								}
							}
						}
						proj_endwritingprojectfile();
					}
				}
			}
		}
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);

	/* if it worked, print dependencies and display */
	if (worked != 0)
	{
		ttyputmsg("Facet %s checked out for your use", describenodeproto(newvers));

		/* advise of possible problems with other checkouts higher up in the hierarchy */
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto) np->temp1 = 0;
		newvers->temp1 = 1;
		propagated = 1;
		while(propagated != 0)
		{
			propagated = 0;
			for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (np->temp1 == 1)
				{
					propagated = 1;
					np->temp1 = 2;
					for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
						if (ni->parent->temp1 == 0) ni->parent->temp1 = 1;
				}
			}
		}
		newvers->temp1 = 0;
		total = 0;
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if (np->temp1 == 0) continue;
			pf = proj_findfacet(np);
			if (pf == NOPROJECTFACET) continue;
			if (*pf->owner != 0 && namesame(pf->owner, proj_username) != 0)
			{
				np->temp1 = 3;
				np->temp2 = (INTBIG)pf;
				total++;
			}
		}
		if (total != 0)
		{
			ttyputmsg("*** Warning: the following facets are above this in the hierarchy");
			ttyputmsg("*** and are checked out to others.  This may cause problems");
			for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (np->temp1 != 3) continue;
				pf = (PROJECTFACET *)np->temp2;
				ttyputmsg("    %s is checked out to %s", describenodeproto(np), pf->owner);
			}
		}

		/* advise of possible problems with other checkouts lower down in the hierarchy */
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto) np->temp1 = 0;
		newvers->temp1 = 1;
		propagated = 1;
		while(propagated != 0)
		{
			propagated = 0;
			for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (np->temp1 == 1)
				{
					propagated = 1;
					np->temp1 = 2;
					for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
					{
						if (ni->proto->primindex != 0) continue;
						if (ni->proto->temp1 == 0) ni->proto->temp1 = 1;
					}
				}
			}
		}
		newvers->temp1 = 0;
		total = 0;
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if (np->temp1 == 0) continue;
			pf = proj_findfacet(np);
			if (pf == NOPROJECTFACET) continue;
			if (*pf->owner != 0 && namesame(pf->owner, proj_username) != 0)
			{
				np->temp1 = 3;
				np->temp2 = (INTBIG)pf;
				total++;
			}
		}
		if (total != 0)
		{
			ttyputmsg("*** Warning: the following facets are below this in the hierarchy");
			ttyputmsg("*** and are checked out to others.  This may cause problems");
			for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				if (np->temp1 != 3) continue;
				pf = (PROJECTFACET *)np->temp2;
				ttyputmsg("    %s is checked out to %s", describenodeproto(np), pf->owner);
			}
		}

		/* display the checked-out facet */
		if (showfacet != 0)
		{
			argv[0] = describenodeproto(newvers);
			us_editfacet(1, argv);
			us_endchanges(NOWINDOWPART);
		}
	}
}

void proj_getoldversion(char *facetname)
{
	PROJECTFACET *pf, statpf;
	REGISTER INTBIG version;
	REGISTER UINTBIG date;
	REGISTER INTBIG itemHit, count, i, len;
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;
	char line[256], projectpath[256], projectfile[256], **filelist, sep[2], *pt;

	/* find out which facet is being checked out */
	np = getnodeproto(facetname);
	if (np == NONODEPROTO)
	{
		ttyputerr("Cannot identify facet '%s'", facetname);
		return;
	}
	lib = np->cell->lib;

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	if (proj_readprojectfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot read project file");
		return;
	}

	pf = proj_findfacet(np);
	if (pf == NOPROJECTFACET)
	{
		ttyputerr("Facet %s is not in the project", describenodeproto(np));
		return;
	}

	/* find all files in the directory for this cell */
	projectfile[strlen(projectfile)-5] = 0;
	strcat(projectpath, projectfile);
	sep[0] = DIRSEP;   sep[1] = 0;
	strcat(projectpath, sep);
	strcat(projectpath, pf->cellname);
	count = filesindirectory(projectpath, &filelist);

	if (DiaInitDialog(&proj_oldversdialog) != 0) return;
	DiaInitTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE);
	for(i=0; i<count; i++)
	{
		strcpy(line, filelist[i]);
		len = strlen(line);
		if (strcmp(&line[len-5], ".elib") != 0) continue;
		line[len-5] = 0;
		version = atoi(line);
		if (version <= 0) continue;
		if (version >= pf->cellversion) continue;
		for(pt = line; *pt != 0; pt++) if (*pt == '-') break;
		if (*pt != '-') continue;
		if (strcmp(&pt[1], pf->cellview->viewname) != 0) continue;

		/* file is good, display it */
		strcpy(projectfile, projectpath);
		strcat(projectfile, sep);
		strcat(projectfile, filelist[i]);
		date = filedate(projectfile);
		sprintf(line, "Version %ld, %s", version, timetostring(date));
		DiaStuffLine(4, line);
	}
	DiaSelectLine(4, -1);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}
	version = -1;
	i = DiaGetCurLine(4);
	if (i >= 0)
	{
		pt = DiaGetScrollLine(4, i);
		if (strlen(pt) > 8) version = atoi(&pt[8]);
	}
	DiaDoneDialog();
	if (itemHit == CANCEL) return;
	if (version < 0) return;

	/* build a fake record to describe this facet */
	statpf = *pf;
	statpf.cellversion = version;

	np = proj_getfacet(&statpf, lib);
	if (np == NONODEPROTO)
		ttyputerr("Error retrieving old version of facet");
	proj_marklocked(np, 0);
	(*el_curconstraint->solve)(np);
}

void proj_update(LIBRARY *lib)
{
	char projectpath[256], projectfile[256];
	PROJECTFACET *pf;
	REGISTER INTSML total;
	REGISTER CELL *cell;
	REGISTER NODEPROTO *oldnp, *newnp;

	/* make sure there is a valid user name */
	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return;
	}

	/* read the project file */
	us_clearhighlightcount();
	if (proj_readprojectfile(projectpath, projectfile) != 0)
		ttyputerr("Cannot read project file"); else
	{
		/* check to see which facets are changed/added */
		total = 0;
		for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
		{
			cell = getcell(pf->cellname);
			oldnp = NONODEPROTO;
			if (cell != NOCELL)
			{
				for(oldnp = cell->firstincell; oldnp != NONODEPROTO; oldnp = oldnp->nextincell)
					if (oldnp->cellview == pf->cellview) break;
				if (oldnp != NONODEPROTO && oldnp->version >= pf->cellversion) continue;
			}

			/* this is a new one */
			newnp = proj_getfacet(pf, lib);
			if (newnp == NONODEPROTO)
			{
				if (pf->cellview == el_unknownview)
				{
					ttyputerr("Error bringing in %s;%d", pf->cellname, pf->cellversion);
				} else
				{
					ttyputerr("Error bringing in %s{%s};%d", pf->cellname, pf->cellview->sviewname,
						pf->cellversion);
				}
			} else
			{
				if (oldnp != NONODEPROTO && proj_usenewestversion(oldnp, newnp) != 0)
					ttyputerr("Error replacing instances of new %s", describenodeproto(oldnp)); else
				{
					if (pf->cellview == el_unknownview)
					{
						ttyputmsg("Brought in facet %s;%d", pf->cellname, pf->cellversion);
					} else
					{
						ttyputmsg("Brought in facet %s{%s};%d", pf->cellname, pf->cellview->sviewname,
							pf->cellversion);
					}
					total++;
				}
			}
		}
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);

	/* make sure all facet locks are correct */
	proj_validatelocks(lib);

	/* summarize */
	if (total == 0) ttyputmsg("Project is up-to-date"); else
		ttyputmsg("Updated %d %s", total, makeplural("facet", total));
}

void proj_buildproject(LIBRARY *lib)
{
	char libraryname[256], librarypath[256], newname[256], sep[2], projfile[256], *pars[2];
	REGISTER INTBIG i, extpos;
	REGISTER NODEPROTO *np;
	FILE *io;

	/* verify that a project is to be built */
	us_noyesdlog("Are you sure you want to create a multi-user project from this library?", pars);
	if (namesame(pars[0], "yes") != 0) return;

	/* get path prefix for facet libraries */
	strcpy(librarypath, lib->libfile);
	extpos = -1;
	for(i = strlen(librarypath)-1; i > 0; i--)
	{
		if (librarypath[i] == DIRSEP) break;
		if (librarypath[i] == '.')
		{
			if (extpos < 0) extpos = i;
		}
	}
	if (extpos < 0) strcat(librarypath, "ELIB"); else
		strcpy(&librarypath[extpos], "ELIB");
	if (librarypath[i] == DIRSEP)
	{
		strcpy(libraryname, &librarypath[i+1]);
		librarypath[i] = 0;
	} else
	{
		strcpy(libraryname, librarypath);
		librarypath[0] = 0;
	}

	/* create the top-level directory for this library */
	sep[0] = DIRSEP;   sep[1] = 0;
	strcpy(newname, librarypath);
	strcat(newname, sep);
	strcat(newname, libraryname);
	if (fileexistence(newname) != 0)
	{
		ttyputerr("Project directory '%s' already exists", newname);
		return;
	}
	if (createdirectory(newname) != 0)
	{
		ttyputerr("Could not create project directory '%s'", newname);
		return;
	}
	ttyputmsg("Making project directory '%s'...", newname);

	/* create the project file */
	strcpy(projfile, librarypath);
	strcat(projfile, sep);
	strcat(projfile, libraryname);
	strcat(projfile, ".proj");
	io = xcreate(projfile, FILETYPEPROJ, 0, 0);
	if (io == NULL)
	{
		ttyputerr("Could not create project file '%s'", projfile);
		return;
	}

	/* turn off all tools */
	if (proj_turnofftools() != 0)
	{
		ttyputerr("Could not save tool state");
		return;
	}

	/* make libraries for every facet */
	setvalkey((INTBIG)lib, VLIBRARY, proj_pathkey, (INTBIG)projfile, VSTRING);
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		/* ignore old unused facet versions */
		if (np->newestversion != np)
		{
			if (np->firstinst == NONODEINST) continue;
			ttyputmsg("Warning: including old version of facet %s", describenodeproto(np));
		}

		/* write the facet to disk in its own library */
		ttyputmsg("Entering facet %s", describenodeproto(np));
		if (proj_writefacet(np) != 0) break;

		/* make an entry in the project file */
		xprintf(io, ":%s:%s:%d-%s.elib:::Initial checkin\n", libraryname, np->cell->cellname,
			np->version, np->cellview->viewname);

		/* mark this facet "checked in" and locked */
		proj_marklocked(np, 1);
	}

	/* restore tool state */
	proj_restoretoolstate();

	/* close project file */
	xclose(io);

	/* make sure library variables are proper */
	if (getvalkey((INTBIG)lib, VLIBRARY, VSTRING, proj_userkey) != NOVARIABLE)
		delvalkey((INTBIG)lib, VLIBRARY, proj_userkey);

	/* advise the user of this library */
	ttyputmsg("The current library should be saved and used by new users");
}

void proj_addfacet(char *facetname)
{
	REGISTER NODEPROTO *np;
	REGISTER LIBRARY *lib;
	char projectpath[256], projectfile[256], libname[256];
	PROJECTFACET *pf, *lastpf;

	/* find out which facet is being added */
	np = getnodeproto(facetname);
	if (np == NONODEPROTO)
	{
		ttyputerr("Cannot identify facet '%s'", facetname);
		return;
	}
	lib = np->cell->lib;
	if (np->newestversion != np)
	{
		ttyputerr("Cannot add an old version of the facet");
		return;
	}

	/* make sure there is a valid user name */
	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return;
	}

	/* read the project file */
	if (proj_readprojectfile(projectpath, projectfile) != 0)
		ttyputerr("Cannot read project file"); else
	{
		/* find this in the project file */
		lastpf = NOPROJECTFACET;
		for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
		{
			if (strcmp(pf->cellname, np->cell->cellname) == 0 &&
				pf->cellview == np->cellview) break;
			lastpf = pf;
		}
		if (pf != NOPROJECTFACET) ttyputerr("This facet is already in the project"); else
		{
			if (proj_startwritingprojectfile(projectpath, projectfile) != 0)
				ttyputerr("Cannot write project file"); else
			{
				if (proj_writefacet(np) != 0)
					ttyputerr("Error writing facet file"); else
				{
					/* create new entry for this facet */
					pf = proj_allocprojectfacet();
					if (pf == 0)
						ttyputerr("Cannot add project record"); else
					{
						strcpy(libname, projectfile);
						libname[strlen(libname)-5] = 0;
						(void)allocstring(&pf->libname, libname, proj_aid->cluster);
						(void)allocstring(&pf->cellname, np->cell->cellname, proj_aid->cluster);
						pf->cellview = np->cellview;
						pf->cellversion = np->version;
						(void)allocstring(&pf->owner, "", proj_aid->cluster);
						(void)allocstring(&pf->lastowner, proj_username, proj_aid->cluster);
						(void)allocstring(&pf->comment, "Initial checkin", proj_aid->cluster);

						/* link it in */
						pf->nextprojectfacet = NOPROJECTFACET;
						if (lastpf == NOPROJECTFACET) proj_firstprojectfacet = pf; else
							lastpf->nextprojectfacet = pf;

						/* mark this facet "checked in" and locked */
						proj_marklocked(np, 1);

						ttyputmsg("Facet %s added to the project", describenodeproto(np));
					}
				}

				/* save new project file */
				proj_endwritingprojectfile();
			}
		}
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);
}

void proj_deletefacet(char *facetname)
{
	REGISTER NODEPROTO *np, *onp;
	REGISTER NODEINST *ni;
	REGISTER LIBRARY *lib;
	char projectpath[256], projectfile[256], *pt;
	PROJECTFACET *pf, *lastpf;

	/* find out which facet is being deleted */
	np = getnodeproto(facetname);
	if (np == NONODEPROTO)
	{
		ttyputerr("Cannot identify facet '%s'", facetname);
		return;
	}
	lib = np->cell->lib;

	/* make sure the facet is not being used */
	if (np->firstinst != NONODEINST)
	{
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
			onp->temp1 = 0;
		for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
			ni->parent->temp1++;
		ttyputerr("Cannot delete facet %s because it is still being used by:", facetname);
		for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
			if (onp->temp1 != 0)
				ttyputmsg("   Facet %s has %ld instance(s)", describenodeproto(onp), onp->temp1);
		return;
	}

	/* make sure there is a valid user name */
	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return;
	}

	/* read the project file */
	if (proj_readprojectfile(projectpath, projectfile) != 0)
		ttyputerr("Cannot read project file"); else
	{
		/* make sure the user has no cells checked-out */
		for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
			if (namesame(pf->owner, proj_username) == 0) break;
		if (pf != NOPROJECTFACET)
		{
			ttyputerr("Before deleting a facet from the project, you must check-in all of your work.");
			ttyputerr("This is because the deletion may be dependent upon changes recently made.");
			(void)initinfstr();
			for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
			{
				if (namesame(pf->owner, proj_username) != 0) continue;
				(void)addstringtoinfstr(pf->cellname);
				if (pf->cellview != el_unknownview)
				{
					(void)addstringtoinfstr("{");
					(void)addstringtoinfstr(pf->cellview->sviewname);
					(void)addstringtoinfstr("}");
				}
				(void)addstringtoinfstr(", ");
			}
			pt = returninfstr();
			pt[strlen(pt)-2] = 0;
			ttyputerr("These facets are checked out to you: %s", pt);
		} else
		{
			/* find this in the project file */
			lastpf = NOPROJECTFACET;
			for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
			{
				if (strcmp(pf->cellname, np->cell->cellname) == 0 &&
					pf->cellview == np->cellview) break;
				lastpf = pf;
			}
			if (pf == NOPROJECTFACET) ttyputerr("This facet is not in the project"); else
			{
				if (proj_startwritingprojectfile(projectpath, projectfile) != 0)
					ttyputerr("Cannot write project file"); else
				{
					/* unlink it */
					if (lastpf == NOPROJECTFACET)
						proj_firstprojectfacet = pf->nextprojectfacet; else
							lastpf->nextprojectfacet = pf->nextprojectfacet;

					/* delete the entry */
					efree(pf->libname);
					efree(pf->cellname);
					efree(pf->owner);
					efree(pf->lastowner);
					efree(pf->comment);
					proj_freeprojectfacet(pf);

					/* mark this facet unlocked */
					proj_marklocked(np, 0);

					/* save new project file */
					proj_endwritingprojectfile();

					ttyputmsg("Facet %s deleted from the project", describenodeproto(np));
				}
			}
		}
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);
}

/************************ USER DATABASE ***********************/

/*
 * Routine to obtain the user's name.  If "newone" is nonzero, force input of a name,
 * otherwise, only ask if unknown.  Returns nonzero if user name is invalid.
 */
INTSML proj_getusername(INTSML newone, LIBRARY *lib)
{
	REGISTER char *pt, *pwd;
	REGISTER INTBIG itemHit, writeback, i;
	REGISTER PUSER *pu, *lastpu;
	REGISTER VARIABLE *var;
	char *truename, userline[256], pwdline[256], projectpath[256], projectfile[256];
	FILE *io;

	/* if name exists and not forcing a new user name, stop now */
	if (newone == 0 && proj_username[0] != 0) return(0);

	/* read user database */
	proj_users = NOPUSER;
	(void)initinfstr();
	(void)addstringtoinfstr(el_libdir);
	(void)addstringtoinfstr(PUSERFILE);
	io = xopen(returninfstr(), FILETYPEPROJUSERS, "", &truename);
	if (io == NULL) ttyputmsg("Creating new user database"); else
	{
		/* read the users file */
		lastpu = NOPUSER;
		for(;;)
		{
			if (xfgets(userline, 256, io) != 0) break;
			for(pt = userline; *pt != 0; pt++) if (*pt == ':') break;
			if (*pt != ':')
			{
				ttyputerr("Missing ':' in user file: %s", userline);
				break;
			}
			*pt++ = 0;
			pu = (PUSER *)emalloc(sizeof (PUSER), proj_aid->cluster);
			if (pu == 0) break;
			(void)allocstring(&pu->username, userline, proj_aid->cluster);
			(void)allocstring(&pu->userpassword, pt, proj_aid->cluster);
			pu->nextpuser = NOPUSER;
			if (proj_users == NOPUSER) proj_users = lastpu = pu; else
				lastpu->nextpuser = pu;
			lastpu = pu;
		}
		xclose(io);
	}

	/* if not forcing a new user name, see if it is on the library */
	if (newone == 0)
	{
		/* if already on the library, get it */
		var = getvalkey((INTBIG)lib, VLIBRARY, VSTRING, proj_userkey);
		if (var != NOVARIABLE)
		{
			strcpy(proj_username, (char *)var->addr);
			for(pu = proj_users; pu != NOPUSER; pu = pu->nextpuser)
				if (strcmp(proj_username, pu->username) == 0) break;
			if (pu != NOPUSER)
			{
				pwd = proj_wantpassword(1, proj_username);
				if (strcmp(pwd, pu->userpassword) == 0) return(0);
				ttyputmsg("Incorrect password");
				proj_username[0] = 0;
				return(1);
			}
			proj_username[0] = 0;
			(void)delvalkey((INTBIG)lib, VLIBRARY, proj_userkey);
		}
	}


	/* show the users dialog */
	writeback = 0;
	if (DiaInitDialog(&proj_usersdialog) != 0) return(1);
	DiaInitTextDialog(3, proj_initusers, proj_nextuser, DiaNullDlogDone, 0,
		SCSELMOUSE | SCSELKEY | SCDOUBLEQUIT);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK)
		{
			/* validate the user name */
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			pt = DiaGetScrollLine(3, i);
			for(pu = proj_users; pu != NOPUSER; pu = pu->nextpuser)
				if (strcmp(pt, pu->username) == 0) break;
			if (pu == NOPUSER) continue;
			pwd = proj_wantpassword(1, pt);
			if (strcmp(pwd, pu->userpassword) == 0)
			{
				strcpy(proj_username, pu->username);
				(void)setvalkey((INTBIG)lib, VLIBRARY, proj_userkey, (INTBIG)proj_username, VSTRING);
				break;
			}
			ttyputmsg("Incorrect password");
			continue;
		}
		if (itemHit == 5)
		{
			/* delete user */
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			pt = DiaGetScrollLine(3, i);
			lastpu = NOPUSER;
			for(pu = proj_users; pu != NOPUSER; pu = pu->nextpuser)
			{
				if (strcmp(pt, pu->username) == 0) break;
				lastpu = pu;
			}
			if (pu == NOPUSER) continue;
			pwd = proj_wantpassword(1, pt);
			if (strcmp(pwd, pu->userpassword) != 0)
			{
				ttyputerr("Incorrect password");
				continue;
			}
			if (lastpu == NOPUSER) proj_users = pu->nextpuser; else
				lastpu->nextpuser = pu->nextpuser;
			efree(pu->username);
			efree(pu->userpassword);
			efree((char *)pu);
			DiaLoadTextDialog(3, proj_initusers, proj_nextuser, DiaNullDlogDone, 0);
			writeback = 1;
			continue;
		}
		if (itemHit == 6)
		{
			/* change password */
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			pt = DiaGetScrollLine(3, i);
			for(pu = proj_users; pu != NOPUSER; pu = pu->nextpuser)
				if (strcmp(pt, pu->username) == 0) break;
			if (pu == NOPUSER) continue;
			pwd = proj_wantpassword(1, pt);
			if (strcmp(pwd, pu->userpassword) != 0)
			{
				ttyputerr("Incorrect password");
				continue;
			}
			pwd = proj_wantpassword(2, pt);
			strcpy(pwdline, pwd);
			pwd = proj_wantpassword(3, pt);
			if (strcmp(pwdline, pwd) != 0)
			{
				ttyputerr("Failed to type password the same way twice");
				continue;
			}
			reallocstring(&pu->userpassword, pwdline, proj_aid->cluster);
			writeback = 1;
			continue;
		}
		if (itemHit == 7)
		{
			/* new user */
			pt = proj_wantpassword(0, "");
			strcpy(userline, pt);
			pwd = proj_wantpassword(1, userline);
			strcpy(pwdline, pwd);
			pwd = proj_wantpassword(3, userline);
			if (strcmp(pwdline, pwd) != 0)
			{
				ttyputerr("Failed to type password the same way twice");
				continue;
			}

			pu = (PUSER *)emalloc(sizeof (PUSER), proj_aid->cluster);
			if (pu == 0) continue;
			(void)allocstring(&pu->username, userline, proj_aid->cluster);
			(void)allocstring(&pu->userpassword, pwdline, proj_aid->cluster);
			pu->nextpuser = proj_users;
			proj_users = pu;
			DiaLoadTextDialog(3, proj_initusers, proj_nextuser, DiaNullDlogDone, 0);
			writeback = 1;
			continue;
		}
	}
	DiaDoneDialog();

	if (itemHit == CANCEL) return(1);

	/* write the file back if necessary */
	if (writeback != 0)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(el_libdir);
		(void)addstringtoinfstr(PUSERFILE);
		io = xopen(returninfstr(), FILETYPEPROJUSERS | FILETYPEWRITE, "", &truename);
		if (io == NULL)
		{
			ttyputmsg("Cannot save users database");
			return(1);
		}

		/* write the users file */
		for(pu = proj_users; pu != NOPUSER; pu = pu->nextpuser)
			xprintf(io, "%s:%s\n", pu->username, pu->userpassword);
		xclose(io);

		/* if switching users, update all locks in the library */
		if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
		{
			ttyputerr("Cannot find project file");
			return(0);
		}
		if (proj_readprojectfile(projectpath, projectfile) != 0)
		{
			ttyputerr("Cannot read project file");
			return(0);
		}
		proj_validatelocks(lib);
	}
	return(0);
}

/*
 * Routine to prompt for a user name/password, according to the "mode" and return
 * the string.  "mode" is:
 *   0 to prompt for a user name
 *   1 to prompt for an existing password
 *   2 to prompt for a new password
 *   3 to prompt for a password verification
 */
char *proj_wantpassword(INTSML mode, char *username)
{
	REGISTER INTBIG itemHit;
	static char typedstuff[256];

	if (DiaInitDialog(&proj_passworddialog) != 0) return("");
	switch (mode)
	{
		case 0:
			DiaSetText(3, "New User Name");
			break;
		case 1:
			sprintf(typedstuff, "Password for %s", username);
			DiaSetText(3, typedstuff);
			break;
		case 2:
			sprintf(typedstuff, "New Password for %s", username);
			DiaSetText(3, typedstuff);
			break;
		case 3:
			sprintf(typedstuff, "Verify Password for %s", username);
			DiaSetText(3, typedstuff);
			break;
	}
	if (mode != 0) DiaOpaqueEdit(4);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
	}
	strcpy(typedstuff, DiaGetText(4));
	DiaDoneDialog();
	if (mode != 0)
		myencrypt(typedstuff, "BicIsSchediwy");
	return(typedstuff);
}

INTSML proj_initusers(char **c)
{
	proj_userpos = proj_users;
	return(1);
}

char *proj_nextuser(void)
{
	REGISTER char *ret;

	if (proj_userpos == NOPUSER) return(0);
	ret = proj_userpos->username;
	proj_userpos = proj_userpos->nextpuser;
	return(ret);
}

/************************ PROJECT DATABASE ***********************/

void proj_checkinmany(LIBRARY *lib)
{
	REGISTER NODEPROTO *np;
	char projectpath[256], projectfile[256];
	PROJECTFACET *pf;

	/* make sure there is a valid user name */
	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return;
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return;
	}

	/* read the project file */
	if (proj_readprojectfile(projectpath, projectfile) != 0)
		ttyputerr("Cannot read project file"); else
	{
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			if (np->temp1 == 0) continue;
			if (stopping("Checkin") != 0) break;

			/* find this in the project file */
			pf = proj_findfacet(np);
			if (pf == NOPROJECTFACET)
				ttyputerr("Facet %s is not in the project", describenodeproto(np)); else
			{
				/* see if it is available */
				if (strcmp(pf->owner, proj_username) != 0)
					ttyputerr("Facet %s is not checked out to you", describenodeproto(np)); else
				{
					if (proj_getcomments(pf, "in") == 0)
					{
						/* prepare to write it back */
						if (proj_startwritingprojectfile(projectpath, projectfile) != 0)
							ttyputerr("Cannot write project file"); else
						{
							/* write the facet out there */
							if (proj_writefacet(np) != 0)
								ttyputerr("Error writing facet %s", describenodeproto(np)); else
							{
								(void)reallocstring(&pf->owner, "", proj_aid->cluster);
								(void)reallocstring(&pf->lastowner, proj_username, proj_aid->cluster);
								pf->cellversion = np->version;
								proj_marklocked(np, 1);
								ttyputmsg("Facet %s checked in", describenodeproto(np));
							}
							proj_endwritingprojectfile();
						}
					}
				}
			}
		}
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);
}

/*
 * Routine to obtain information about the project associated with library "lib".
 * The path to the project is placed in "path" and the name of the project file
 * in that directory is placed in "projfile".  Returns nonzero on error.
 */
INTSML proj_getprojinfo(LIBRARY *lib, char *path, char *projfile)
{
	REGISTER VARIABLE *var;
	char *params[2], *truename;
	FILE *io;
	REGISTER INTSML i;
	extern COMCOMP us_colorreadp;

	/* see if there is a variable in the current library with the project path */
	var = getvalkey((INTBIG)lib, VLIBRARY, VSTRING, proj_pathkey);
	if (var != NOVARIABLE)
	{
		strcpy(path, (char *)var->addr);
		io = xopen(path, FILETYPEPROJ, "", &truename);
		if (io != 0) xclose(io); else
			var = NOVARIABLE;
	}
	if (var == NOVARIABLE)
	{
		i = ttygetparam("proj/Project File", &us_colorreadp, 1, params);
		if (i == 0) return(1);
		strcpy(path, params[0]);
		setvalkey((INTBIG)lib, VLIBRARY, proj_pathkey, (INTBIG)path, VSTRING);
	}

	for(i = strlen(path)-1; i>0; i--) if (path[i] == DIRSEP) break;
	if (path[i] == DIRSEP)
	{
		strcpy(projfile, &path[i+1]);
		path[i+1] = 0;
	} else
	{
		strcpy(projfile, path);
		path[0] = 0;
	}
	return(0);
}

void proj_showlistdialog(LIBRARY *lib)
{
	REGISTER INTBIG itemHit, i, j;
	REGISTER PROJECTFACET *pf;

	if (proj_getusername(0, lib) != 0)
	{
		ttyputerr("No valid user");
		return;
	}

	/* show the dialog */
	if (DiaInitDialog(&proj_listdialog) != 0) return;
	DiaInitTextDialog(2, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE | SCREPORT);

	/* load project information into the scroll area */
	if (proj_loadlist(lib) != 0)
	{
		DiaDoneDialog();
		return;
	}

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK) break;
		if (itemHit == 8)
		{
			/* update project */
			proj_update(lib);
			(void)proj_loadlist(lib);
			continue;
		}
		if (itemHit == 5 || itemHit == 7 || itemHit == 2)
		{
			/* figure out which facet is selected */
			i = DiaGetCurLine(2);
			if (i < 0) continue;
			for(j=0, pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet, j++)
				if (i == j) break;
			if (pf == NOPROJECTFACET) continue;

			if (itemHit == 5)
			{
				/* check it out */
				(void)initinfstr();
				(void)addstringtoinfstr(pf->cellname);
				if (pf->cellview != el_unknownview)
				{
					(void)addtoinfstr('{');
					(void)addstringtoinfstr(pf->cellview->sviewname);
					(void)addtoinfstr('}');
				}
				proj_checkout(returninfstr(), 0);
				(void)proj_loadlist(lib);
				continue;
			}
			if (itemHit == 7)
			{
				/* check it in */
				(void)initinfstr();
				(void)addstringtoinfstr(pf->cellname);
				if (pf->cellview != el_unknownview)
				{
					(void)addtoinfstr('{');
					(void)addstringtoinfstr(pf->cellview->sviewname);
					(void)addtoinfstr('}');
				}
				proj_checkin(returninfstr());
				(void)proj_loadlist(lib);
				continue;
			}
			if (itemHit == 2)
			{
				if (*pf->comment != 0) DiaSetText(3, pf->comment);
				if (*pf->owner == 0) DiaUnDimItem(5); else DiaDimItem(5);
				if (strcmp(pf->owner, proj_username) == 0) DiaUnDimItem(7); else DiaDimItem(7);
				continue;
			}
		}
	}
	DiaDoneDialog();
}


/*
 * Routine to display the current project information in the list dialog.
 * Returns nonzero on error.
 */
INTSML proj_loadlist(LIBRARY *lib)
{
	REGISTER INTSML failed, whichline, thisline, uptodate;
	REGISTER PROJECTFACET *pf, *curpf;
	char line[256], projectpath[256], projectfile[256];
	REGISTER NODEPROTO *np, *curfacet;
	REGISTER CELL *cell;

	/* get location of project file */
	if (proj_getprojinfo(lib, projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot find project file");
		return(1);
	}

	/* lock the project file */
	if (proj_lockprojfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Couldn't lock project file");
		return(1);
	}

	/* read the project file */
	failed = 0;
	if (proj_readprojectfile(projectpath, projectfile) != 0)
	{
		ttyputerr("Cannot read project file");
		failed = 1;
	}

	/* relase project file lock */
	proj_unlockprojfile(projectpath, projectfile);
	if (failed != 0) return(1);

	/* find current facet */
	curfacet = getcurfacet();

	/* show what is in the project file */
	DiaLoadTextDialog(2, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
	whichline = -1;
	thisline = 0;
	uptodate = 1;
	for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
	{
		/* see if project is up-to-date */
		cell = getcell(pf->cellname);
		if (cell == NOCELL) uptodate = 0; else
		{
			for(np = cell->firstincell; np != NONODEPROTO; np = np->nextincell)
				if (np->cellview == pf->cellview) break;
			if (np == NONODEPROTO || np->version < pf->cellversion) uptodate = 0;
		}

		/* remember this line if it describes the current facet */
		if (curfacet != NONODEPROTO && namesame(curfacet->cell->cellname, pf->cellname) == 0 &&
			curfacet->cellview == pf->cellview)
		{
			whichline = thisline;
			curpf = pf;
		}

		/* describe this project facet */
		if (pf->cellview == el_unknownview)
		{
			sprintf(line, "%s;%d", pf->cellname, pf->cellversion);
		} else
		{
			sprintf(line, "%s{%s};%d", pf->cellname, pf->cellview->sviewname,
				pf->cellversion);
		}
		if (*pf->owner == 0)
		{
			strcat(line, " AVAILABLE");
			if (*pf->lastowner != 0)
			{
				strcat(line, ", last mod by ");
				strcat(line, pf->lastowner);
			}
		} else
		{
			if (strcmp(pf->owner, proj_username) == 0)
			{
				strcat(line, " EDITABLE, checked out to you");
			} else
			{
				strcat(line, " UNAVAILABLE, checked out to ");
				strcat(line, pf->owner);
			}
		}
		DiaStuffLine(2, line);
		thisline++;
	}
	DiaSelectLine(2, whichline);

	DiaDimItem(5);
	DiaDimItem(7);
	if (whichline >= 0)
	{
		if (*curpf->comment != 0) DiaSetText(3, curpf->comment);
		if (*curpf->owner == 0) DiaUnDimItem(5); else DiaDimItem(5);
		if (strcmp(curpf->owner, proj_username) == 0) DiaUnDimItem(7); else DiaDimItem(7);
	}

	if (uptodate != 0) 	DiaDimItem(8); else
	{
		ttyputmsg("Your library does not contain the most recent additions to the project.");
		ttyputmsg("You should do an 'Update' to make it current.");
		DiaUnDimItem(8);
	}
	return(0);
}

/*
 * Routine to obtain comments about a checkin/checkout for facet "pf".  "direction" is
 * either "in" or "out".  Returns nonzero if the dialog is cancelled.
 */
INTSML proj_getcomments(PROJECTFACET *pf, char *direction)
{
	REGISTER INTBIG itemHit;
	static char line[256];

	if (DiaInitDialog(&proj_commentsdialog) != 0) return(1);
	sprintf(line, "Reason for checking-%s facet %s", direction, pf->cellname);
	DiaSetText(3, line);
	DiaSetText(-4, pf->comment);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
	}
	if (itemHit == OK)
		(void)reallocstring(&pf->comment, DiaGetText(4), proj_aid->cluster);
	DiaDoneDialog();
	if (itemHit == CANCEL) return(1);
	return(0);
}

/************************ PROJECT FILE ***********************/

INTSML proj_readprojectfile(char *pathname, char *filename)
{
	char *truename, projline[256], *pt, *start, fullname[256], origprojline[256];
	FILE *io;
	REGISTER INTSML err;
	PROJECTFACET *pf, *pftail, *nextpf, *opf;

	/* delete previous project database */
	for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = nextpf)
	{
		nextpf = pf->nextprojectfacet;
		efree(pf->libname);
		efree(pf->cellname);
		efree(pf->owner);
		efree(pf->lastowner);
		efree(pf->comment);
		proj_freeprojectfacet(pf);
	}

	/* read the project file */
	proj_firstprojectfacet = pftail = NOPROJECTFACET;
	strcpy(fullname, pathname);
	strcat(fullname, filename);
	io = xopen(fullname, FILETYPEPROJ, "", &truename);
	if (io == 0)
	{
		ttyputerr("Couldn't read project file '%s'", fullname);
		return(1);
	}

	err = 0;
	for(;;)
	{
		if (xfgets(projline, 256, io) != 0) break;
		strcpy(origprojline, projline);
		pf = proj_allocprojectfacet();
		if (pf == 0) break;

		pt = projline;
		if (*pt++ != ':')
		{
			ttyputerr("Missing initial ':' in project file: '%s'", origprojline);
			err = 1;
			break;
		}

		/* get library name */
		for(start = pt; *pt != 0; pt++) if (*pt == ':') break;
		if (*pt != ':')
		{
			ttyputerr("Missing ':' after library name in project file: '%s'", origprojline);
			err = 1;
			break;
		}
		*pt++ = 0;
		(void)allocstring(&pf->libname, start, proj_aid->cluster);

		/* get cell name */
		for(start = pt; *pt != 0; pt++) if (*pt == ':') break;
		if (*pt != ':')
		{
			ttyputerr("Missing ':' after cell name in project file: '%s'", origprojline);
			err = 1;
			break;
		}
		*pt++ = 0;
		(void)allocstring(&pf->cellname, start, proj_aid->cluster);

		/* get version */
		pf->cellversion = atoi(pt);
		for( ; *pt != 0; pt++) if (*pt == '-') break;
		if (*pt++ != '-')
		{
			ttyputerr("Missing '-' after version number in project file: '%s'", origprojline);
			err = 1;
			break;
		}

		/* get view */
		for(start = pt; *pt != 0; pt++)
			if (pt[0] == '.' && pt[1] == 'e' && pt[2] == 'l' && pt[3] == 'i' && pt[4] == 'b' &&
				pt[5] == ':') break;
		if (*pt == 0)
		{
			ttyputerr("Missing '.elib' after view name in project file: '%s'", origprojline);
			err = 1;
			break;
		}
		*pt = 0;
		pt += 6;
		pf->cellview = getview(start);

		/* get owner */
		for(start = pt; *pt != 0; pt++) if (*pt == ':') break;
		if (*pt != ':')
		{
			ttyputerr("Missing ':' after owner name in project file: '%s'", origprojline);
			err = 1;
			break;
		}
		*pt++ = 0;
		(void)allocstring(&pf->owner, start, proj_aid->cluster);

		/* get last owner */
		for(start = pt; *pt != 0; pt++) if (*pt == ':') break;
		if (*pt != ':')
		{
			ttyputerr("Missing ':' after last owner name in project file: '%s'", origprojline);
			err = 1;
			break;
		}
		*pt++ = 0;
		(void)allocstring(&pf->lastowner, start, proj_aid->cluster);

		/* get comments */
		(void)allocstring(&pf->comment, pt, proj_aid->cluster);

		/* check for duplication */
		for(opf = proj_firstprojectfacet; opf != NOPROJECTFACET; opf = opf->nextprojectfacet)
		{
			if (namesame(opf->cellname, pf->cellname) != 0) continue;
			if (opf->cellview != pf->cellview) continue;
			ttyputerr("Error in project file: view '%s' of cell '%s' exists twice (versions %d and %d)",
				pf->cellview->viewname, pf->cellname, pf->cellversion, opf->cellversion);
		}

		/* link it in */
		pf->nextprojectfacet = NOPROJECTFACET;
		if (proj_firstprojectfacet == NOPROJECTFACET) proj_firstprojectfacet = pftail = pf; else
		{
			pftail->nextprojectfacet = pf;
			pftail = pf;
		}
	}
	xclose(io);
	return(err);
}

INTSML proj_startwritingprojectfile(char *pathname, char *filename)
{
	char *truename, fullname[256];

	/* read the project file */
	strcpy(fullname, pathname);
	strcat(fullname, filename);
	proj_io = xopen(fullname, FILETYPEPROJ | FILETYPEWRITE, "", &truename);
	if (proj_io == 0)
	{
		ttyputerr("Couldn't write project file '%s'", fullname);
		return(1);
	}
	return(0);
}

void proj_endwritingprojectfile(void)
{
	PROJECTFACET *pf;

	for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
		xprintf(proj_io, ":%s:%s:%d-%s.elib:%s:%s:%s\n", pf->libname, pf->cellname,
			pf->cellversion, pf->cellview->viewname, pf->owner, pf->lastowner, pf->comment);
	xclose(proj_io);
}

PROJECTFACET *proj_allocprojectfacet(void)
{
	PROJECTFACET *pf;

	if (proj_projectfacetfree != NOPROJECTFACET)
	{
		pf = proj_projectfacetfree;
		proj_projectfacetfree = proj_projectfacetfree->nextprojectfacet;
	} else
	{
		pf = (PROJECTFACET *)emalloc(sizeof (PROJECTFACET), proj_aid->cluster);
		if (pf == 0) return(0);
	}
	return(pf);
}

void proj_freeprojectfacet(PROJECTFACET *pf)
{
	pf->nextprojectfacet = proj_projectfacetfree;
	proj_projectfacetfree = pf;
}

PROJECTFACET *proj_findfacet(NODEPROTO *np)
{
	PROJECTFACET *pf;

	for(pf = proj_firstprojectfacet; pf != NOPROJECTFACET; pf = pf->nextprojectfacet)
		if (strcmp(pf->cellname, np->cell->cellname) == 0 && pf->cellview == np->cellview)
			return(pf);
	return(NOPROJECTFACET);
}

/************************ LOCKING ***********************/

#define MAXTRIES 10
#define NAPTIME  5

INTSML proj_lockprojfile(char *projectpath, char *projectfile)
{
	char lockfilename[256];
	REGISTER INTSML i, j;

	sprintf(lockfilename, "%s%sLOCK", projectpath, projectfile);
	for(i=0; i<MAXTRIES; i++)
	{
		if (lockfile(lockfilename) != 0) return(0);
		if (i == 0) ttyputmsg("Project file locked.  Waiting..."); else
			ttyputmsg("Still waiting (will try %d more times)...", MAXTRIES-i);
		for(j=0; j<NAPTIME; j++)
		{
			gotosleep(60);
			if (stopping("Lock wait") != 0) return(1);
		}
	}
	return(1);
}

void proj_unlockprojfile(char *projectpath, char *projectfile)
{
	char lockfilename[256];

	sprintf(lockfilename, "%s%sLOCK", projectpath, projectfile);
	unlockfile(lockfilename);
}

void proj_validatelocks(LIBRARY *lib)
{
	REGISTER NODEPROTO *np;
	PROJECTFACET *pf;

	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		pf = proj_findfacet(np);
		if (pf == NOPROJECTFACET)
		{
			/* facet not in the project: writable */
			proj_marklocked(np, 0);
		} else
		{
			if (np->version < pf->cellversion)
			{
				/* facet is an old version: writable */
				proj_marklocked(np, 0);
			} else
			{
				if (namesame(pf->owner, proj_username) == 0)
				{
					/* facet checked out to current user: writable */
					proj_marklocked(np, 0);
				} else
				{
					/* facet checked out to someone else: not writable */
					proj_marklocked(np, 1);
				}
			}
		}
	}
}

void proj_marklocked(NODEPROTO *np, INTSML locked)
{
	REGISTER NODEPROTO *onp;

	if (locked == 0)
	{
		for(onp = np->cell->lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		{
			if (onp->cell != np->cell || onp->cellview != np->cellview) continue;
			if (getvalkey((INTBIG)onp, VNODEPROTO, VINTEGER, proj_lockedkey) != NOVARIABLE)
				delvalkey((INTBIG)onp, VNODEPROTO, proj_lockedkey);
		}
	} else
	{
		for(onp = np->cell->lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
		{
			if (onp->cell != np->cell || onp->cellview != np->cellview) continue;
			if (onp->newestversion == onp)
			{
				if (getvalkey((INTBIG)onp, VNODEPROTO, VINTEGER, proj_lockedkey) == NOVARIABLE)
					setvalkey((INTBIG)onp, VNODEPROTO, proj_lockedkey, 1, VINTEGER);
			} else
			{
				if (getvalkey((INTBIG)onp, VNODEPROTO, VINTEGER, proj_lockedkey) != NOVARIABLE)
					delvalkey((INTBIG)onp, VNODEPROTO, proj_lockedkey);
			}
		}
	}
}

/************************ CHANGE TRACKING ***********************/

void proj_queuecheck(NODEPROTO *facet)
{
	REGISTER FCHECK *f;

	/* see if the facet is already queued */
	for(f = proj_firstfcheck; f != NOFCHECK; f = f->nextfcheck)
		if (f->entry == facet && f->batchnumber == proj_batchnumber) return;

	f = proj_allocfcheck();
	if (f == NOFCHECK)
	{
		ttyputerr("PROJ: out of space");
		return;
	}
	f->entry = facet;
	f->batchnumber = proj_batchnumber;
	f->nextfcheck = proj_firstfcheck;
	proj_firstfcheck = f;
}

FCHECK *proj_allocfcheck(void)
{
	REGISTER FCHECK *f;

	if (proj_fcheckfree == NOFCHECK)
	{
		f = (FCHECK *)emalloc((sizeof (FCHECK)), proj_aid->cluster);
		if (f == 0) return(NOFCHECK);
	} else
	{
		f = proj_fcheckfree;
		proj_fcheckfree = f->nextfcheck;
	}

	return(f);
}

void proj_freefcheck(FCHECK *f)
{
	f->nextfcheck = proj_fcheckfree;
	proj_fcheckfree = f;
}

/************************ COPYING FACETS IN AND OUT OF DATABASE ***********************/

/*
 * Routine to get the latest version of the facet described by "pf" and return
 * the newly created facet.  Returns NONODEPROTO on error.
 */
NODEPROTO *proj_getfacet(PROJECTFACET *pf, LIBRARY *lib)
{
	char facetlibname[256], facetlibpath[256], facetprojfile[256], facetname[256],
		sep[2], *templibname;
	REGISTER LIBRARY *flib;
	REGISTER NODEPROTO *newnp;
	REGISTER INTBIG oldverbose, ret;
	extern AIDENTRY *io_aid, *net_aid;

	/* create the library */
	if (proj_getprojinfo(lib, facetlibpath, facetprojfile) != 0)
	{
		ttyputerr("Cannot find project info on library %s", lib->libname);
		return(NONODEPROTO);
	}
	sprintf(facetlibname, "%d-%s.elib", pf->cellversion, pf->cellview->viewname);
	facetprojfile[strlen(facetprojfile)-5] = 0;
	strcat(facetlibpath, facetprojfile);
	sep[0] = DIRSEP;   sep[1] = 0;
	strcat(facetlibpath, sep);
	strcat(facetlibpath, pf->cellname);
	strcat(facetlibpath, sep);
	strcat(facetlibpath, facetlibname);

	/* prevent tools (including this one) from seeing the change */
	(void)proj_turnofftools();

	templibname = proj_templibraryname();
	flib = newlibrary(templibname, facetlibpath);
	if (flib == NOLIBRARY)
	{
		ttyputerr("Cannot create library %s", facetlibpath);
		proj_restoretoolstate();
		return(NONODEPROTO);
	}
	oldverbose = askaid(io_aid, "verbose", (INTSML)0);
	ret = askaid(io_aid, "read", (INTBIG)flib, (INTBIG)"binary");
	(void)askaid(io_aid, "verbose", oldverbose);
	if (ret != 0)
	{
		ttyputerr("Cannot read library %s", facetlibpath);
		killlibrary(flib);
		proj_restoretoolstate();
		return(NONODEPROTO);
	}
	if (flib->curnodeproto == NONODEPROTO)
	{
		ttyputerr("Cannot find facet in library %s", facetlibpath);
		killlibrary(flib);
		proj_restoretoolstate();
		return(NONODEPROTO);
	}
	sprintf(facetname, "%s;%d", flib->curnodeproto->cell->cellname, flib->curnodeproto->version);
	if (flib->curnodeproto->cellview != el_unknownview)
	{
		strcat(facetname, "{");
		strcat(facetname, flib->curnodeproto->cellview->sviewname);
		strcat(facetname, "}");
	}
	newnp = copynodeproto(flib->curnodeproto, lib, facetname);
	if (newnp == NONODEPROTO)
	{
		ttyputerr("Cannot copy facet %s from new library", describenodeproto(flib->curnodeproto));
		killlibrary(flib);
		proj_restoretoolstate();
		return(NONODEPROTO);
	}
	(*el_curconstraint->solve)(newnp);

	/* must do this explicitly because the library kill flushes change batches */
	/* (void)askaid(net_aid, "re-number", (INTBIG)newnp); */

	/* kill the library */
	killlibrary(flib);

	/* restore tool state */
	proj_restoretoolstate();

	/* return the new facet */
	return(newnp);
}

INTSML proj_usenewestversion(NODEPROTO *oldnp, NODEPROTO *newnp)
{
	INTBIG lx, hx, ly, hy;
	REGISTER WINDOWPART *w;
	REGISTER NODEINST *ni, *newni, *nextni;

	/* prevent tools (including this one) from seeing the change */
	(void)proj_turnofftools();

	/* replace them all */
	for(ni = oldnp->firstinst; ni != NONODEINST; ni = nextni)
	{
		nextni = ni->nextinst;
		newni = replacenodeinst(ni, newnp);
		if (newni == NONODEINST)
		{
			ttyputerr("Failed to update instance of %s in %s", describenodeproto(newnp),
				describenodeproto(ni->parent));
			proj_restoretoolstate();
			return(1);
		}
	}

	/* redraw windows that updated */
	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
	{
		if (w->curnodeproto == NONODEPROTO) continue;
		if (w->curnodeproto->cell != newnp->cell ||
			w->curnodeproto->cellview != newnp->cellview) continue;
		w->curnodeproto = newnp;

		/* redisplay the window with the new facet */
		us_fullview(newnp, &lx, &hx, &ly, &hy);
		us_squarescreen(w, NOWINDOWPART, 0, &lx, &hx, &ly, &hy);
		startobjectchange((INTBIG)w, VWINDOWPART);
		(void)setval((INTBIG)w, VWINDOWPART, "screenlx", lx, VINTEGER);
		(void)setval((INTBIG)w, VWINDOWPART, "screenhx", hx, VINTEGER);
		(void)setval((INTBIG)w, VWINDOWPART, "screenly", ly, VINTEGER);
		(void)setval((INTBIG)w, VWINDOWPART, "screenhy", hy, VINTEGER);
		us_gridset(w, w->state);
		endobjectchange((INTBIG)w, VWINDOWPART);
	}

	/* update status display if necessary */
	if (us_curnodeproto == oldnp) us_setnodeproto(newnp);

	if (killnodeproto(oldnp))
		ttyputerr("Could not delete old version");

	/* restore tool state */
	proj_restoretoolstate();
	return(0);
}

INTSML proj_writefacet(NODEPROTO *np)
{
	REGISTER LIBRARY *flib;
	REGISTER INTBIG retval;
	char libname[256], libfile[256], projname[256], sep[2], *templibname;
	REGISTER NODEPROTO *npcopy;
	INTSML filetype;
	extern AIDENTRY *io_aid;

	if (proj_getprojinfo(np->cell->lib, libfile, projname) != 0)
	{
		ttyputerr("Cannot find project info on library %s", np->cell->lib->libname);
		return(1);
	}
	projname[strlen(projname)-5] = 0;
	strcat(libfile, projname);
	sep[0] = DIRSEP;   sep[1] = 0;
	strcat(libfile, sep);
	strcat(libfile, np->cell->cellname);

	/* make the directory if necessary */
	filetype = fileexistence(libfile);
	if (filetype == 1)
	{
		ttyputerr("Could not create cell directory '%s'", libfile);
		return(1);
	}
	if (filetype == 0)
	{
		if (createdirectory(libfile) != 0)
		{
			ttyputerr("Could not create cell directory '%s'", libfile);
			return(1);
		}
	}

	strcat(libfile, sep);
	sprintf(libname, "%d-%s.elib", np->version, np->cellview->viewname);
	strcat(libfile, libname);

	/* prevent tools (including this one) from seeing the change */
	(void)proj_turnofftools();

	templibname = proj_templibraryname();
	flib = newlibrary(templibname, libfile);
	if (flib == NOLIBRARY)
	{
		ttyputerr("Cannot create library %s", libfile);
		proj_restoretoolstate();
		return(1);
	}
	npcopy = copyrecursively(np, flib);
	if (npcopy == NONODEPROTO)
	{
		ttyputerr("Could not place %s in a library", describenodeproto(np));
		killlibrary(flib);
		proj_restoretoolstate();
		return(1);
	}

	flib->curnodeproto = npcopy;
	flib->userbits |= READFROMDISK;
	makeoptionstemporary(flib);
	retval = askaid(io_aid, "write", (INTBIG)flib, (INTBIG)"binary");
	restoreoptionstate();
	if (retval != 0)
	{
		ttyputerr("Could not save library with %s in it", describenodeproto(np));
		killlibrary(flib);
		proj_restoretoolstate();
		return(1);
	}
	killlibrary(flib);

	/* restore tool state */
	proj_restoretoolstate();

	return(0);
}

char *proj_templibraryname(void)
{
	static char libname[256];
	REGISTER LIBRARY *lib;
	REGISTER INTBIG i;

	for(i=1; ; i++)
	{
		sprintf(libname, "projecttemp%ld", i);
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			if (namesame(libname, lib->libname) == 0) break;
		if (lib == NOLIBRARY) break;
	}
	return(libname);
}

/*
 * Routine to save the state of all tools and turn them off.
 */
INTSML proj_turnofftools(void)
{
	REGISTER INTSML i;
	REGISTER AIDENTRY *aid;
	extern AIDENTRY *net_aid;

	/* turn off all tools for this operation */
	if (proj_saveaidstate == 0)
	{
		proj_saveaidstate = (INTBIG *)emalloc(el_maxaid * SIZEOFINTBIG, el_tempcluster);
		if (proj_saveaidstate == 0) return(1);
	}
	for(i=0; i<el_maxaid; i++)
	{
		aid = &el_aids[i];
		proj_saveaidstate[i] = aid->aidstate;
		if (aid == us_aid || aid == proj_aid || aid == net_aid) continue;
		aid->aidstate &= ~AIDON;
	}
	proj_ignorechanges = 1;
	return(0);
}

/*
 * Routine to restore the state of all tools that were reset by "proj_turnofftools()".
 */
void proj_restoretoolstate(void)
{
	REGISTER INTSML i;

	if (proj_saveaidstate == 0) return;
	for(i=0; i<el_maxaid; i++)
		el_aids[i].aidstate = proj_saveaidstate[i];
	proj_ignorechanges = 0;
}

/************************ DATABASE OPERATIONS ***********************/

NODEPROTO *copyrecursively(NODEPROTO *fromnp, LIBRARY *tolib)
{
	REGISTER NODEPROTO *np, *onp, *newfromnp;
	REGISTER NODEINST *ni;
	REGISTER char *newname;
	char versnum[20];

	/* must copy subfacets */
	for(ni = fromnp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		np = ni->proto;
		if (np->primindex != 0) continue;

		/* see if there is already a facet with this name and view */
		for(onp = tolib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
			if (namesame(onp->cell->cellname, np->cell->cellname) == 0 &&
				onp->cellview == np->cellview) break;
		if (onp != NONODEPROTO) continue;

		onp = copyskeleton(np, tolib);
		if (onp == NONODEPROTO)
		{
			ttyputerr("Copy of subfacet %s failed", describenodeproto(np));
			return(NONODEPROTO);
		}
	}

	/* copy the facet if it is not already done */
	for(newfromnp = tolib->firstnodeproto; newfromnp != NONODEPROTO; newfromnp = newfromnp->nextnodeproto)
		if (namesame(newfromnp->cell->cellname, fromnp->cell->cellname) == 0 &&
			newfromnp->cellview == fromnp->cellview && newfromnp->version == fromnp->version) break;
	if (newfromnp == NONODEPROTO)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(fromnp->cell->cellname);
		(void)addtoinfstr(';');
		sprintf(versnum, "%d", fromnp->version);
		(void)addstringtoinfstr(versnum);
		if (fromnp->cellview != el_unknownview)
		{
			(void)addtoinfstr('{');
			(void)addstringtoinfstr(fromnp->cellview->sviewname);
			(void)addtoinfstr('}');
		}
		newname = returninfstr();
		newfromnp = copynodeproto(fromnp, tolib, newname);
		if (newfromnp == NONODEPROTO) return(NONODEPROTO);

		/* ensure that the copied facet is the right size */
		(*el_curconstraint->solve)(newfromnp);
	}

	return(newfromnp);
}

NODEPROTO *copyskeleton(NODEPROTO *fromnp, LIBRARY *tolib)
{
	char *newname;
	REGISTER INTSML newang, newtran;
	REGISTER INTBIG i, xc, yc;
	INTBIG newx, newy;
	XARRAY trans, localtrans, ntrans;
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp, *rpp;
	REGISTER NODEINST *ni, *newni;

	/* cannot skeletonize text-only views */
	if ((fromnp->cellview->viewstate&TEXTVIEW) != 0) return(NONODEPROTO);

	(void)initinfstr();
	(void)addstringtoinfstr(fromnp->cell->cellname);
	if (fromnp->cellview != el_unknownview)
	{
		(void)addtoinfstr('{');
		(void)addstringtoinfstr(fromnp->cellview->sviewname);
		(void)addtoinfstr('}');
	}
	newname = returninfstr();
	np = newnodeproto(newname, tolib);
	if (np == NONODEPROTO) return(NONODEPROTO);

	/* place all exported ports in the new facet */
	for(pp = fromnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		/* make a transformation matrix for the node that is exported */
		ni = pp->subnodeinst;
		rpp = pp->subportproto;
		newang = ni->rotation;
		newtran = ni->transpose;
		makerot(ni, trans);
		while (ni->proto->primindex == 0)
		{
			maketrans(ni, localtrans);
			transmult(localtrans, trans, ntrans);
			ni = rpp->subnodeinst;
			rpp = rpp->subportproto;
			if (ni->transpose == 0) newang = ni->rotation + newang; else
				newang = ni->rotation + 3600 - newang;
			newtran = (newtran + ni->transpose) & 1;
			makerot(ni, localtrans);
			transmult(localtrans, ntrans, trans);
		}

		/* create this node */
		xc = (ni->lowx + ni->highx) / 2;   yc = (ni->lowy + ni->highy) / 2;
		xform(xc, yc, &newx, &newy, trans);
		newx -= (ni->highx - ni->lowx) / 2;
		newy -= (ni->highy - ni->lowy) / 2;
		newang = newang % 3600;   if (newang < 0) newang += 3600;
		newni = newnodeinst(ni->proto, newx, newx+ni->highx-ni->lowx,
			newy, newy+ni->highy-ni->lowy, newtran, newang, np);
		if (newni == NONODEINST) return(NONODEPROTO);
		endobjectchange((INTBIG)newni, VNODEINST);

		/* export the port from the node */
		(void)newportproto(np, newni, rpp, pp->protoname);
	}

	/* make sure facet is the same size */
	i = (fromnp->highy+fromnp->lowy)/2 - (gen_invispinprim->highy-gen_invispinprim->lowy)/2;
	(void)newnodeinst(gen_invispinprim, fromnp->lowx, fromnp->lowx+gen_invispinprim->highx-gen_invispinprim->lowx,
		i, i+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);

	i = (fromnp->highy+fromnp->lowy)/2 - (gen_invispinprim->highy-gen_invispinprim->lowy)/2;
	(void)newnodeinst(gen_invispinprim, fromnp->highx-(gen_invispinprim->highx-gen_invispinprim->lowx), fromnp->highx,
		i, i+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);

	i = (fromnp->highx+fromnp->lowx)/2 - (gen_invispinprim->highx-gen_invispinprim->lowx)/2;
	(void)newnodeinst(gen_invispinprim, i, i+gen_invispinprim->highx-gen_invispinprim->lowx,
		fromnp->lowy, fromnp->lowy+gen_invispinprim->highy-gen_invispinprim->lowy, 0, 0, np);

	i = (fromnp->highx+fromnp->lowx)/2 - (gen_invispinprim->highx-gen_invispinprim->lowx)/2;
	(void)newnodeinst(gen_invispinprim, i, i+gen_invispinprim->highx-gen_invispinprim->lowx,
		fromnp->highy-(gen_invispinprim->highy-gen_invispinprim->lowy), fromnp->highy, 0, 0,np);

	(*el_curconstraint->solve)(np);
	return(np);
}

#endif  /* PROJECTAID - at top */
