import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
var _excluded = ["droppableId", "getIndex"],
  _excluded2 = ["contextId", "droppableId"];
// eslint-disable-next-line import/no-extraneous-dependencies

import { extractClosestEdge } from '@atlaskit/pragmatic-drag-and-drop-hitbox/closest-edge';
import { isDraggableData } from '../draggable/data';
import { isDroppableData } from '../droppable/data';
import { customAttributes, getAttribute } from '../utils/attributes';
import { findAllDraggables } from '../utils/find-all-draggables';

/**
 * Derives the `DraggableLocation` of a `<Draggable>`.
 *
 * Accounts for which edge is being hovered over.
 */
function getDraggableLocationFromDraggableData(_ref) {
  var droppableId = _ref.droppableId,
    getIndex = _ref.getIndex,
    data = _objectWithoutProperties(_ref, _excluded);
  /**
   * The index that the draggable is currently occupying.
   */
  var index = getIndex();
  var closestEdge = extractClosestEdge(
  /**
   * TypeScript doesn't like this without casting.
   *
   * The IDE doesn't have an issue, but if you try to build it then
   * there will be an error.
   */
  data);
  /**
   * Whether the user is hovering over the second half of the draggable.
   *
   * For a vertical list it is the bottom half,
   * while for a horizontal list it is the right half.
   */
  var isForwardEdge = closestEdge === 'bottom' || closestEdge === 'right';
  if (isForwardEdge) {
    /**
     * If hovering over the 'forward' half of the draggable,
     * then the user is targeting the index after the draggable.
     */
    index += 1;
  }
  return {
    droppableId: droppableId,
    index: index
  };
}

/**
 * Derives the `DraggableLocation` of a `<Droppable>`.
 *
 * This corresponds to the first or last index of the list,
 * depending on where the user is hovering.
 */
function getDraggableLocationFromDroppableData(_ref2) {
  var contextId = _ref2.contextId,
    droppableId = _ref2.droppableId,
    data = _objectWithoutProperties(_ref2, _excluded2);
  var draggables = findAllDraggables({
    contextId: contextId,
    droppableId: droppableId
  });

  /**
   * If there are no draggables, then the index should be 0
   */
  if (draggables.length === 0) {
    return {
      droppableId: droppableId,
      index: 0
    };
  }
  var closestEdge = extractClosestEdge(data);
  /**
   * Whether the user is closer to the start of the droppable.
   *
   * For a vertical list it is the top half,
   * while for a horizontal list it is the left half.
   */
  var isCloserToStart = closestEdge === 'top' || closestEdge === 'left';
  if (isCloserToStart) {
    /**
     * If the user is closer to the start of the list, we will target the
     * first (0th) index.
     */
    return {
      droppableId: droppableId,
      index: 0
    };
  }

  /**
   * We don't just take the index of the last draggable,
   * because portal-ing can lead to the DOM order not matching indexes.
   */
  var biggestIndex = draggables.reduce(function (max, draggable) {
    var draggableIndex = parseInt(getAttribute(draggable, customAttributes.draggable.index), 10);
    return Math.max(max, draggableIndex);
  }, 0);
  return {
    droppableId: droppableId,
    index: biggestIndex + 1
  };
}

/**
 * Derives a `DraggableLocation` (`react-beautiful-dnd`)
 * from a `DragLocation` (`@atlaskit/pragmatic-drag-and-drop`).
 */
export function getDraggableLocation(location) {
  var dropTargets = location.dropTargets;

  // If there are no drop targets then there is no destination.
  if (dropTargets.length === 0) {
    return null;
  }

  // Obtains the innermost drop target.
  var target = dropTargets[0];

  // If the target is a draggable we can extract its index.
  if (isDraggableData(target.data)) {
    return getDraggableLocationFromDraggableData(target.data);
  }

  // If the target is a droppable, there is no index to extract.
  // We default to the end of the droppable.
  if (isDroppableData(target.data)) {
    return getDraggableLocationFromDroppableData(target.data);
  }

  // The target is not from the migration layer.
  return null;
}

/**
 * Checks if two `DraggableLocation` values are equivalent.
 */
export function isSameLocation(a, b) {
  if ((a === null || a === void 0 ? void 0 : a.droppableId) !== (b === null || b === void 0 ? void 0 : b.droppableId)) {
    return false;
  }
  if ((a === null || a === void 0 ? void 0 : a.index) !== (b === null || b === void 0 ? void 0 : b.index)) {
    return false;
  }
  return true;
}