/*
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-Studio-CLA-applies
 *
 * MuseScore Studio
 * Music Composition & Notation
 *
 * Copyright (C) 2021 MuseScore Limited
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "styledef.h"

#include "draw/types/geometry.h"

#include "types/constants.h"

#include "dom/articulation.h"
#include "dom/mscore.h"
#include "dom/realizedharmony.h"
#include "dom/stafftype.h"
#include "dom/textbase.h"
#include "dom/tuplet.h"
#include "dom/types.h"

using namespace mu;
using namespace muse::draw;
using namespace mu::engraving;

//! Keep in sync with Sid in styledef.h
// Help keeping Sid names and XML tag texts in sync
#define styleDef(sidAndXmlTag, property) { Sid::sidAndXmlTag, #sidAndXmlTag, property }
const std::array<StyleDef::StyleValue, size_t(Sid::STYLES)> StyleDef::styleValues { {
    styleDef(pageWidth,                                  210.0 / INCH),
    styleDef(pageHeight,                                 297.0 / INCH),   // A4
    styleDef(pagePrintableWidth,                         180.0 / INCH),
    styleDef(pageEvenLeftMargin,                         15.0 / INCH),
    styleDef(pageOddLeftMargin,                          15.0 / INCH),
    styleDef(pageEvenTopMargin,                          15.0 / INCH),
    styleDef(pageEvenBottomMargin,                       15.0 / INCH),
    styleDef(pageOddTopMargin,                           15.0 / INCH),
    styleDef(pageOddBottomMargin,                        15.0 / INCH),
    styleDef(pageTwosided,                               true),

    styleDef(staffUpperBorder,                           Spatium(7.0)),
    styleDef(staffLowerBorder,                           Spatium(7.0)),
    styleDef(staffHeaderFooterPadding,                   Spatium(1.0)),
    styleDef(staffDistance,                              Spatium(6.5)),
    styleDef(instrumentNameOffset,                       Spatium(1.0)),
    styleDef(akkoladeDistance,                           Spatium(6.5)),
    styleDef(minSystemDistance,                          Spatium(8.5)),
    styleDef(maxSystemDistance,                          Spatium(15.0)),
    styleDef(alignSystemToMargin,                        true),

    styleDef(enableVerticalSpread,                       true),
    styleDef(spreadSystem,                               2.5),
    styleDef(spreadSquareBracket,                        1.2),
    styleDef(spreadCurlyBracket,                         1.1),
    styleDef(minSystemSpread,                            Spatium(8.5)),
    styleDef(maxSystemSpread,                            Spatium(32.0)),
    styleDef(minStaffSpread,                             Spatium(3.5)),
    styleDef(maxStaffSpread,                             Spatium(20.0)),
    styleDef(maxAkkoladeDistance,                        Spatium(6.5)),
    styleDef(maxPageFillSpread,                          Spatium(6.0)),

    styleDef(lyricsPlacement,                            PlacementV::BELOW),
    styleDef(lyricsPosAbove,                             PointF(0.0, -2.0)),
    styleDef(lyricsPosBelow,                             PointF(.0, 3.0)),
    styleDef(lyricsMinTopDistance,                       Spatium(1.0)),
    styleDef(lyricsMinBottomDistance,                    Spatium(1.5)),
    styleDef(lyricsMinDistance,                          Spatium(0.25)),
    styleDef(lyricsLineHeight,                           1.0),
    styleDef(lyricsDashMinLength,                        Spatium(0.4)),
    styleDef(lyricsDashMaxLength,                        Spatium(0.6)),
    styleDef(lyricsDashMaxDistance,                      Spatium(16.0)),
    styleDef(lyricsDashForce,                            true),
    styleDef(lyricsDashFirstAndLastGapAreHalf,           true),
    styleDef(lyricsAlignVerseNumber,                     true),
    styleDef(lyricsLineThickness,                        Spatium(0.1)),
    styleDef(lyricsMelismaAlign,                         Align(AlignH::LEFT, AlignV::BASELINE)),

    styleDef(lyricsMelismaPad,                           Spatium(0.20)),  // the empty space before a melisma line
    styleDef(lyricsDashPad,                              Spatium(0.05)),  // the min. empty space before and after a dash
    styleDef(lyricsDashLineThickness,                    Spatium(0.1)),   // in sp. units
    styleDef(lyricsDashYposRatio,                        0.50),           // the fraction of lyrics font x-height to raise the dashes above text base line

    styleDef(lyricsShowDashIfSyllableOnFirstNote,        true),
    styleDef(lyricsMelismaForce,                         false),
    styleDef(lyricsMelismaMinLength,                     Spatium(1.0)),
    styleDef(lyricsDashPosAtStartOfSystem,               int(LyricsDashSystemStart::STANDARD)),
    styleDef(lyricsAvoidBarlines,                        true),

    styleDef(lyricsOddFontFace,                          "Edwin"),
    styleDef(lyricsOddFontSize,                          10.0),
    styleDef(lyricsOddLineSpacing,                       1.0),
    styleDef(lyricsOddFontSpatiumDependent,              true),
    styleDef(lyricsOddFontStyle,                         int(FontStyle::Normal)),
    styleDef(lyricsOddColor,                             Color::BLACK),
    styleDef(lyricsOddAlign,                             Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(lyricsOddFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(lyricsOddFramePadding,                      0.2),
    styleDef(lyricsOddFrameWidth,                        0.1),
    styleDef(lyricsOddFrameRound,                        0),
    styleDef(lyricsOddFrameFgColor,                      Color::BLACK),
    styleDef(lyricsOddFrameBgColor,                      Color::transparent),

    styleDef(lyricsEvenFontFace,                         "Edwin"),
    styleDef(lyricsEvenFontSize,                         10.0),
    styleDef(lyricsEvenLineSpacing,                      1.0),
    styleDef(lyricsEvenFontSpatiumDependent,             true),
    styleDef(lyricsEvenFontStyle,                        int(FontStyle::Normal)),
    styleDef(lyricsEvenColor,                            Color::BLACK),
    styleDef(lyricsEvenAlign,                            Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(lyricsEvenFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(lyricsEvenFramePadding,                     0.2),
    styleDef(lyricsEvenFrameWidth,                       0.1),
    styleDef(lyricsEvenFrameRound,                       0),
    styleDef(lyricsEvenFrameFgColor,                     Color::BLACK),
    styleDef(lyricsEvenFrameBgColor,                     Color::transparent),

    styleDef(figuredBassFontFamily,                      String(u"MScoreBC")),

    //styleDef(figuredBassFontSize,                        PropertyValue(8.0)),
    styleDef(figuredBassYOffset,                         6.0),
    styleDef(figuredBassLineHeight,                      1.0),
    styleDef(figuredBassAlignment,                       0),
    styleDef(figuredBassStyle,                           0),
    styleDef(systemFrameDistance,                        Spatium(7.0)),
    styleDef(frameSystemDistance,                        Spatium(7.0)),
    styleDef(minMeasureWidth,                            Spatium(8.0)),
    styleDef(barWidth,                                   Spatium(0.18)),
    styleDef(doubleBarWidth,                             Spatium(0.18)),

    styleDef(endBarWidth,                                Spatium(0.55)),
    styleDef(doubleBarDistance,                          Spatium(0.37)),
    styleDef(endBarDistance,                             Spatium(0.37)),
    styleDef(repeatBarlineDotSeparation,                 Spatium(0.37)),
    styleDef(repeatBarTips,                              false),
    styleDef(startBarlineSingle,                         false),
    styleDef(startBarlineMultiple,                       true),
    styleDef(maskBarlinesForText,                        true),

    styleDef(bracketWidth,                               Spatium(0.45)),
    styleDef(bracketDistance,                            Spatium(0.45)),
    styleDef(akkoladeWidth,                              Spatium(1.5)),
    styleDef(akkoladeBarDistance,                        Spatium(.35)),

    styleDef(dividerLeft,                                false),
    styleDef(dividerLeftSym,                             String(u"systemDivider")),
    styleDef(dividerLeftX,                               0.0),
    styleDef(dividerLeftY,                               0.0),
    styleDef(dividerRight,                               false),
    styleDef(dividerRightSym,                            String(u"systemDivider")),
    styleDef(dividerRightX,                              0.0),
    styleDef(dividerRightY,                              0.0),

    styleDef(clefLeftMargin,                             Spatium(0.75)),
    styleDef(keysigLeftMargin,                           Spatium(0.5)),
    styleDef(ambitusMargin,                              Spatium(0.5)),

    styleDef(timesigLeftMargin,                          Spatium(0.63)),
    styleDef(midClefKeyRightMargin,                      Spatium(1.0)),
    styleDef(clefKeyRightMargin,                         Spatium(0.8)),
    styleDef(clefKeyDistance,                            Spatium(0.75)),
    styleDef(clefTimesigDistance,                        Spatium(1.0)),
    styleDef(keyTimesigDistance,                         Spatium(1.0)),
    styleDef(keyBarlineDistance,                         Spatium(1.0)),
    styleDef(systemHeaderDistance,                       Spatium(2.5)),
    styleDef(systemHeaderTimeSigDistance,                Spatium(2.0)),
    styleDef(systemTrailerRightMargin,                   Spatium(0.5)),

    styleDef(clefBarlineDistance,                        Spatium(0.5)),
    styleDef(timesigBarlineDistance,                     Spatium(0.5)),

    styleDef(timeSigPlacement,                           TimeSigPlacement::NORMAL),

    styleDef(timeSigCenterOnBarline,                     true),
    styleDef(timeSigVSMarginCentered,                    TimeSigVSMargin::HANG_INTO_MARGIN),
    styleDef(timeSigVSMarginNonCentered,                 TimeSigVSMargin::RIGHT_ALIGN_TO_BARLINE),
    styleDef(timeSigCenterAcrossStaveGroup,              false),

    styleDef(timeSigNormalStyle,                         TimeSigStyle::NORMAL),
    styleDef(timeSigNormalScale,                         ScaleF(1.0, 1.0)),
    styleDef(timeSigNormalScaleLock,                     true),
    styleDef(timeSigNormalNumDist,                       Spatium(0.0)),
    styleDef(timeSigNormalY,                             Spatium(0.0)),
    styleDef(timeSigAboveStyle,                          TimeSigStyle::NORMAL),
    styleDef(timeSigAboveScale,                          ScaleF(2.0, 2.0)),
    styleDef(timeSigAboveScaleLock,                      true),
    styleDef(timeSigAboveNumDist,                        Spatium(0.1)),
    styleDef(timeSigAboveY,                              Spatium(-1.0)),
    styleDef(timeSigAcrossStyle,                         TimeSigStyle::NARROW),
    styleDef(timeSigAcrossScale,                         ScaleF(4.0, 4.0)),
    styleDef(timeSigAcrossScaleLock,                     true),
    styleDef(timeSigAcrossNumDist,                       Spatium(0.1)),
    styleDef(timeSigAcrossY,                             Spatium(0.0)),

    styleDef(useStraightNoteFlags,                       false),
    styleDef(stemWidth,                                  Spatium(0.10)),
    styleDef(shortenStem,                                true),
    styleDef(stemLength,                                 PropertyValue(3.5)),
    styleDef(stemLengthSmall,                            PropertyValue(2.25)),
    styleDef(shortStemStartLocation,                     1),
    styleDef(shortestStem,                               PropertyValue(2.5)),
    styleDef(combineVoice,                               true),
    styleDef(beginRepeatLeftMargin,                      Spatium(1.0)),
    styleDef(minNoteDistance,                            Spatium(0.5)),
    styleDef(barNoteDistance,                            Spatium(1.25)),   // was 1.2

    styleDef(barAccidentalDistance,                      Spatium(0.65)),
    styleDef(noteBarDistance,                            Spatium(1.5)),
    styleDef(measureSpacing,                             1.5),
    styleDef(measureRepeatNumberPos,                     Spatium(-0.5)),
    styleDef(mrNumberSeries,                             false),
    styleDef(mrNumberEveryXMeasures,                     4),
    styleDef(mrNumberSeriesWithParentheses,              true),
    styleDef(oneMeasureRepeatShow1,                      false),
    styleDef(fourMeasureRepeatShowExtenders,             false),
    styleDef(staffLineWidth,                             Spatium(0.11)),
    styleDef(ledgerLineWidth,                            Spatium(0.16)),  // 0.1875
    styleDef(ledgerLineLength,                           Spatium(0.33)),  // notehead width * this value
    styleDef(stemSlashPosition,                          Spatium(2.0)),
    styleDef(stemSlashAngle,                             40.0),
    styleDef(stemSlashThickness,                         Spatium(0.125)),
    styleDef(accidentalDistance,                         Spatium(0.25)),
    styleDef(accidentalNoteDistance,                     Spatium(0.25)),
    styleDef(bracketedAccidentalPadding,                 Spatium(0.175)), //padding inside parentheses for bracketed accidentals
    styleDef(alignAccidentalsLeft,                       false),          // OBSOLETE

    styleDef(accidentalOrderFollowsNoteDisplacement,     false),
    styleDef(alignAccidentalOctavesAcrossSubChords,      false),
    styleDef(keepAccidentalSecondsTogether,              false),
    styleDef(alignOffsetOctaveAccidentals,               false),

    styleDef(keysigAccidentalDistance,                   Spatium(0.3)),
    styleDef(keysigNaturalDistance,                      Spatium(0.4)),

    styleDef(beamWidth,                                  Spatium(0.5)),
    styleDef(useWideBeams,                               false),
    styleDef(beamMinLen,                                 Spatium(1.1)),
    styleDef(beamNoSlope,                                false),
    styleDef(snapCustomBeamsToGrid,                      true),
    styleDef(frenchStyleBeams,                           false),

    styleDef(dotMag,                                     PropertyValue(1.0)),
    styleDef(dotNoteDistance,                            Spatium(0.5)),
    styleDef(dotRestDistance,                            Spatium(0.25)),
    styleDef(dotDotDistance,                             Spatium(0.65)),
    styleDef(propertyDistanceHead,                       Spatium(0.4)),
    styleDef(propertyDistanceStem,                       Spatium(0.4)),
    styleDef(propertyDistance,                           Spatium(0.4)),

    styleDef(articulationMag,                            PropertyValue(1.0)),
    styleDef(articulationPosAbove,                       PointF(0.0, 0.0)),
    styleDef(articulationAnchorDefault,                  int(ArticulationAnchor::AUTO)),
    styleDef(articulationAnchorLuteFingering,            int(ArticulationAnchor::BOTTOM)),
    styleDef(articulationAnchorOther,                    int(ArticulationAnchor::TOP)),
    styleDef(articulationStemHAlign,                     int(ArticulationStemSideAlign::AVERAGE)),
    styleDef(articulationKeepTogether,                   true),
    styleDef(trillAlwaysShowCueNote,                  false),
    styleDef(lastSystemFillLimit,                        PropertyValue(0.3)),

    styleDef(hairpinPlacement,                           PlacementV::BELOW),
    styleDef(hairpinPosAbove,                            PointF(0.0, -1.75)),
    styleDef(hairpinPosBelow,                            PointF(.0, 1.75)),
    styleDef(hairpinLinePosAbove,                        PointF(0.0, -1.0)),
    styleDef(hairpinLinePosBelow,                        PointF(.0, 2.0)),
    styleDef(hairpinHeight,                              Spatium(1.15)),
    styleDef(hairpinContHeight,                          Spatium(0.5)),
    styleDef(hairpinLineWidth,                           Spatium(0.12)),
    styleDef(hairpinFontFace,                            "Edwin"),
    styleDef(hairpinFontSize,                            10.0),
    styleDef(hairpinLineSpacing,                         1.0),
    styleDef(hairpinFontSpatiumDependent,                true),
    styleDef(hairpinFontStyle,                           int(FontStyle::Italic)),
    styleDef(hairpinColor,                               Color::BLACK),
    styleDef(hairpinTextAlign,                           Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(hairpinFrameType,                           int(FrameType::NO_FRAME)),
    styleDef(hairpinFramePadding,                        0.2),
    styleDef(hairpinFrameWidth,                          0.1),
    styleDef(hairpinFrameRound,                          0),
    styleDef(hairpinFrameFgColor,                        Color::BLACK),
    styleDef(hairpinFrameBgColor,                        Color::transparent),
    styleDef(hairpinText,                                String()),
    styleDef(hairpinCrescText,                           String(u"cresc.")),
    styleDef(hairpinDecrescText,                         String(u"dim.")),
    styleDef(hairpinCrescContText,                       String(u"(cresc.)")),
    styleDef(hairpinDecrescContText,                     String(u"(dim.)")),
    styleDef(hairpinLineStyle,                           PropertyValue(LineType::SOLID)),
    styleDef(hairpinDashLineLen,                         3.0),
    styleDef(hairpinDashGapLen,                          3.0),
    styleDef(hairpinLineLineStyle,                       PropertyValue(LineType::DASHED)),
    styleDef(hairpinLineDashLineLen,                     6.0),
    styleDef(hairpinLineDashGapLen,                      9.0),

    styleDef(pedalPlacement,                             PlacementV::BELOW),
    styleDef(pedalPosAbove,                              PointF(.0, -1)),
    styleDef(pedalPosBelow,                              PointF(.0, 2.5)),
    styleDef(pedalLineWidth,                             Spatium(0.11)),
    styleDef(pedalLineStyle,                             PropertyValue(LineType::SOLID)),
    styleDef(pedalDashLineLen,                           4.0),
    styleDef(pedalDashGapLen,                            4.0),
    styleDef(pedalHookHeight,                            Spatium(-1.2)),
    styleDef(pedalFontFace,                              "Edwin"),
    styleDef(pedalFontSize,                              10.0),
    styleDef(pedalLineSpacing,                           1.0),
    styleDef(pedalFontSpatiumDependent,                  true),
    styleDef(pedalMusicalSymbolsScale,                   1.0), // percentage of the standard size
    styleDef(pedalFontStyle,                             int(FontStyle::Normal)),
    styleDef(pedalColor,                                 Color::BLACK),
    styleDef(pedalTextAlign,                             Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(pedalFrameType,                             int(FrameType::NO_FRAME)),
    styleDef(pedalFramePadding,                          0.2),
    styleDef(pedalFrameWidth,                            0.1),
    styleDef(pedalFrameRound,                            0),
    styleDef(pedalFrameFgColor,                          Color::BLACK),
    styleDef(pedalFrameBgColor,                          Color::transparent),
    styleDef(pedalText,                                  String(u"<sym>keyboardPedalPed</sym>")),
    styleDef(pedalHookText,                              String()),
    styleDef(pedalContinueText,
             String(u"<sym>keyboardPedalParensLeft</sym><sym>keyboardPedalPed</sym><sym>keyboardPedalParensRight</sym>")),
    styleDef(pedalContinueHookText,                      String()),
    styleDef(pedalEndText,                               String()),
    styleDef(pedalRosetteEndText,                        String(u"<sym>keyboardPedalUp</sym>")),

    styleDef(trillPlacement,                             PlacementV::ABOVE),
    styleDef(trillPosAbove,                              PointF(.0, -0.5)),
    styleDef(trillPosBelow,                              PointF(.0, 2)),

    styleDef(vibratoPlacement,                           PlacementV::ABOVE),
    styleDef(vibratoPosAbove,                            PointF(.0, -1)),
    styleDef(vibratoPosBelow,                            PointF(.0, 1)),

    styleDef(harmonyFretDist,                            Spatium(1.0)),
    styleDef(minHarmonyDistance,                         Spatium(0.5)),
    styleDef(maxHarmonyBarDistance,                      Spatium(3.0)),
    styleDef(maxChordShiftAbove,                         Spatium(0.0)),
    styleDef(maxChordShiftBelow,                         Spatium(0.0)),

    styleDef(harmonyPlacement,                           PlacementV::ABOVE),
    styleDef(romanNumeralPlacement,                      PlacementV::BELOW),
    styleDef(nashvilleNumberPlacement,                   PlacementV::ABOVE),
    styleDef(harmonyVoiceLiteral,                        true),
    styleDef(harmonyVoicing,                             int(Voicing::AUTO)),
    styleDef(harmonyDuration,                            int(HDuration::UNTIL_NEXT_CHORD_SYMBOL)),

    styleDef(chordSymbolAPosAbove,                       PointF(.0, -2.5)),
    styleDef(chordSymbolAPosBelow,                       PointF(.0, 3.5)),

    styleDef(chordSymbolBPosAbove,                       PointF(.0, -5.0)),
    styleDef(chordSymbolBPosBelow,                       PointF(.0, 3.5)),

    styleDef(romanNumeralPosAbove,                       PointF(.0, -2.5)),
    styleDef(romanNumeralPosBelow,                       PointF(.0, 3.5)),

    styleDef(nashvilleNumberPosAbove,                    PointF(.0, -2.5)),
    styleDef(nashvilleNumberPosBelow,                    PointF(.0, 3.5)),

    styleDef(chordSymbolAFontFace,                       "Edwin"),
    styleDef(chordSymbolAFontSize,                       10.0),
    styleDef(chordSymbolALineSpacing,                    1.0),
    styleDef(chordSymbolAFontSpatiumDependent,           true),
    styleDef(chordSymbolAFontStyle,                      int(FontStyle::Normal)),
    styleDef(chordSymbolAColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(chordSymbolAAlign,                          Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(chordSymbolAFrameType,                      int(FrameType::NO_FRAME)),
    styleDef(chordSymbolAFramePadding,                   0.2),
    styleDef(chordSymbolAFrameWidth,                     0.1),
    styleDef(chordSymbolAFrameRound,                     0),
    styleDef(chordSymbolAFrameFgColor,                   PropertyValue::fromValue(Color::BLACK)),
    styleDef(chordSymbolAFrameBgColor,                   PropertyValue::fromValue(Color::transparent)),

    styleDef(chordSymbolBFontFace,                       "Edwin"),
    styleDef(chordSymbolBFontSize,                       10.0),
    styleDef(chordSymbolBLineSpacing,                    1.0),
    styleDef(chordSymbolBFontSpatiumDependent,           true),
    styleDef(chordSymbolBFontStyle,                      int(FontStyle::Italic)),
    styleDef(chordSymbolBColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(chordSymbolBAlign,                          Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(chordSymbolBFrameType,                      int(FrameType::NO_FRAME)),
    styleDef(chordSymbolBFramePadding,                   0.2),
    styleDef(chordSymbolBFrameWidth,                     0.1),
    styleDef(chordSymbolBFrameRound,                     0),
    styleDef(chordSymbolBFrameFgColor,                   PropertyValue::fromValue(Color::BLACK)),
    styleDef(chordSymbolBFrameBgColor,                   PropertyValue::fromValue(Color::transparent)),

    styleDef(romanNumeralFontFace,                       "Campania"),
    styleDef(romanNumeralFontSize,                       12.0),
    styleDef(romanNumeralLineSpacing,                    1.0),
    styleDef(romanNumeralFontSpatiumDependent,           true),
    styleDef(romanNumeralFontStyle,                      int(FontStyle::Normal)),
    styleDef(romanNumeralColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(romanNumeralAlign,                          Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(romanNumeralFrameType,                      int(FrameType::NO_FRAME)),
    styleDef(romanNumeralFramePadding,                   0.2),
    styleDef(romanNumeralFrameWidth,                     0.1),
    styleDef(romanNumeralFrameRound,                     0),
    styleDef(romanNumeralFrameFgColor,                   PropertyValue::fromValue(Color::BLACK)),
    styleDef(romanNumeralFrameBgColor,                   PropertyValue::fromValue(Color::transparent)),

    styleDef(nashvilleNumberFontFace,                    "Edwin"),
    styleDef(nashvilleNumberFontSize,                    12.0),
    styleDef(nashvilleNumberLineSpacing,                 1.0),
    styleDef(nashvilleNumberFontSpatiumDependent,        true),
    styleDef(nashvilleNumberFontStyle,                   int(FontStyle::Normal)),
    styleDef(nashvilleNumberColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(nashvilleNumberAlign,                       Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(nashvilleNumberFrameType,                   int(FrameType::NO_FRAME)),
    styleDef(nashvilleNumberFramePadding,                0.2),
    styleDef(nashvilleNumberFrameWidth,                  0.1),
    styleDef(nashvilleNumberFrameRound,                  0),
    styleDef(nashvilleNumberFrameFgColor,                PropertyValue::fromValue(Color::BLACK)),
    styleDef(nashvilleNumberFrameBgColor,                PropertyValue::fromValue(Color::transparent)),

    styleDef(capoPosition,                               PropertyValue(0)),
    styleDef(fretNumMag,                                 PropertyValue(2.0)), // DEPRECATED
    styleDef(fretNumPos,                                 PropertyValue(0)),
    styleDef(fretY,                                      Spatium(1.0)),
    styleDef(fretMinDistance,                            Spatium(0.5)),
    styleDef(fretMag,                                    PropertyValue(1.0)),
    styleDef(fretPlacement,                              PlacementV::ABOVE),
    styleDef(fretStrings,                                6),
    styleDef(fretFrets,                                  5),
    styleDef(fretNut,                                    true),
    styleDef(fretDotSize,                                PropertyValue(1.0)), // DEPRECATED
    styleDef(fretDotSpatiumSize,                         Spatium(0.5)),
    styleDef(fretStringSpacing,                          Spatium(0.7)),
    styleDef(fretFretSpacing,                            Spatium(0.7)),
    styleDef(fretOrientation,                            int(Orientation::VERTICAL)),
    styleDef(maxFretShiftAbove,                          Spatium(0.0)),
    styleDef(maxFretShiftBelow,                          Spatium(0.0)),
    styleDef(fretNutThickness,                           Spatium(0.3)),
    styleDef(fretUseCustomSuffix,                        false),
    styleDef(fretCustomSuffix,                           "fr"),
    styleDef(barreAppearanceSlur,                        false),
    styleDef(barreLineWidth,                             PropertyValue(0.85)),
    styleDef(fretShowFingerings,                         false),
    styleDef(fretStyleExtended,                          false),

    styleDef(showPageNumber,                             true),
    styleDef(showPageNumberOne,                          false),
    styleDef(pageNumberOddEven,                          true),
    styleDef(showMeasureNumber,                          true),
    styleDef(showMeasureNumberOne,                       false),
    styleDef(measureNumberInterval,                      PropertyValue(5)),
    styleDef(measureNumberSystem,                        true),
    styleDef(measureNumberAllStaves,                     false),
    styleDef(smallNoteMag,                               PropertyValue(.7)),
    styleDef(scaleRythmicSpacingForSmallNotes,           true),
    styleDef(graceNoteMag,                               PropertyValue(0.7)),
    styleDef(graceToMainNoteDist,                        Spatium(0.45)),
    styleDef(graceToGraceNoteDist,                       Spatium(0.3)),
    styleDef(smallStaffMag,                              PropertyValue(0.7)),
    styleDef(smallClefMag,                               PropertyValue(0.8)),

    styleDef(genClef,                                    true),
    styleDef(hideTabClefAfterFirst,                      true),
    styleDef(genKeysig,                                  true),
    styleDef(genCourtesyTimesig,                         true),
    styleDef(genCourtesyKeysig,                          true),
    styleDef(genCourtesyClef,                            true),
    styleDef(keySigCourtesyBarlineMode,                  PropertyValue(int(CourtesyBarlineMode::DOUBLE_BEFORE_COURTESY))),
    styleDef(timeSigCourtesyBarlineMode,                 PropertyValue(int(CourtesyBarlineMode::ALWAYS_SINGLE))),
    styleDef(swingRatio,                                 PropertyValue(60)),
    styleDef(swingUnit,                                  PropertyValue(String())),
    styleDef(useStandardNoteNames,                       true),
    styleDef(useGermanNoteNames,                         false),
    styleDef(useFullGermanNoteNames,                     false),

    styleDef(useSolfeggioNoteNames,                      false),
    styleDef(useFrenchNoteNames,                         false),
    styleDef(automaticCapitalization,                    true),
    styleDef(lowerCaseMinorChords,                       false),
    styleDef(lowerCaseBassNotes,                         false),
    styleDef(allCapsNoteNames,                           false),
    styleDef(chordStyle,                                 PropertyValue(String(u"std"))),
    styleDef(chordsXmlFile,                              false),
    styleDef(chordDescriptionFile,                       PropertyValue(String(u"chords_std.xml"))),
    styleDef(chordExtensionMag,                          PropertyValue(1.0)),
    styleDef(chordExtensionAdjust,                       PropertyValue(0.0)),
    styleDef(chordModifierMag,                           PropertyValue(1.0)),
    styleDef(chordModifierAdjust,                        PropertyValue(0.0)),
    styleDef(concertPitch,                               false),

    styleDef(multiVoiceRestTwoSpaceOffset,               false),
    styleDef(mergeMatchingRests,                         false),
    styleDef(createMultiMeasureRests,                    false),
    styleDef(minEmptyMeasures,                           PropertyValue(1)),
    styleDef(singleMeasureMMRestUseNormalRest,           true),
    styleDef(singleMeasureMMRestShowNumber,              true),
    styleDef(minMMRestWidth,                             Spatium(6)),
    styleDef(mmRestConstantWidth,                        false),
    styleDef(mmRestReferenceWidth,                       PropertyValue(2)),
    styleDef(mmRestMaxWidthIncrease,                     PropertyValue(8)),
    styleDef(mmRestNumberPos,                            Spatium(-0.5)),
    styleDef(mmRestBetweenStaves,                        true),
    styleDef(mmRestNumberMaskHBar,                       true),
    styleDef(multiMeasureRestMargin,                     Spatium(1.25)),
    styleDef(mmRestHBarThickness,                        Spatium(0.7)),
    styleDef(mmRestHBarVStrokeThickness,                 Spatium(0.2)),
    styleDef(mmRestHBarVStrokeHeight,                    Spatium(2.0)),
    styleDef(oldStyleMultiMeasureRests,                  false),
    styleDef(mmRestOldStyleMaxMeasures,                  PropertyValue(9)),
    styleDef(mmRestOldStyleSpacing,                      Spatium(0.85)),
    styleDef(hideEmptyStaves,                            false),
    styleDef(dontHideStavesInFirstSystem,                true),
    styleDef(enableIndentationOnFirstSystem,             true),
    styleDef(firstSystemIndentationValue,                Spatium(5.0)),
    styleDef(alwaysShowBracketsWhenEmptyStavesAreHidden, false),
    styleDef(alwaysShowSquareBracketsWhenEmptyStavesAreHidden, false),
    styleDef(hideInstrumentNameIfOneInstrument,          true),
    styleDef(firstSystemInstNameVisibility,              PropertyValue(int(InstrumentLabelVisibility::LONG))),
    styleDef(subsSystemInstNameVisibility,               PropertyValue(int(InstrumentLabelVisibility::SHORT))),
    styleDef(gateTime,                                   PropertyValue(100)),
    styleDef(tenutoGateTime,                             PropertyValue(100)),
    styleDef(staccatoGateTime,                           PropertyValue(50)),
    styleDef(slurGateTime,                               PropertyValue(100)),

    styleDef(arpeggioNoteDistance,                       Spatium(.4)),
    styleDef(arpeggioAccidentalDistance,                 Spatium(.3)),
    styleDef(arpeggioAccidentalDistanceMin,              Spatium(0.33)),
    styleDef(arpeggioLineWidth,                          Spatium(.18)),
    styleDef(arpeggioHookLen,                            Spatium(.8)),
    styleDef(arpeggioHiddenInStdIfTab,                   false),
    styleDef(slurEndWidth,                               Spatium(.05)),
    styleDef(slurMidWidth,                               Spatium(.21)),
    styleDef(slurDottedWidth,                            Spatium(.10)),
    styleDef(tieEndWidth,                                Spatium(.05)),
    styleDef(tieMidWidth,                                Spatium(.21)),
    styleDef(tieDottedWidth,                             Spatium(.10)),
    styleDef(minTieLength,                               Spatium(1.0)),
    styleDef(minHangingTieLength,                        Spatium(1.5)),
    styleDef(minStraightGlissandoLength,                 Spatium(1.2)),
    styleDef(minWigglyGlissandoLength,                   Spatium(2.0)),
    styleDef(slurMinDistance,                            Spatium(0.5)),
    styleDef(tieMinDistance,                             Spatium(0.5)),
    styleDef(laissezVibMinDistance,                      Spatium(0.5)),
    styleDef(headerToLineStartDistance,                  Spatium(1.0)),
    styleDef(lineEndToBarlineDistance,                   Spatium(0.25)),

    styleDef(tiePlacementSingleNote,                     TiePlacement::OUTSIDE),
    styleDef(tiePlacementChord,                          TiePlacement::OUTSIDE),
    styleDef(tieDotsPlacement,                           TieDotsPlacement::AUTO),
    styleDef(tieMinShoulderHeight,                       Spatium(0.3)),
    styleDef(tieMaxShoulderHeight,                       Spatium(2.0)),

    styleDef(minLaissezVibLength,                        Spatium(2.0)),
    styleDef(laissezVibUseSmuflSym,                      false),

    styleDef(sectionPause,                               PropertyValue(double(3.0))),
    styleDef(musicalSymbolFont,                          PropertyValue(String(u"Leland"))),
    styleDef(musicalTextFont,                            PropertyValue(String(u"Leland Text"))),

    styleDef(showHeader,                                 true),
    styleDef(headerFirstPage,                            false),
    styleDef(headerOddEven,                              true),
    styleDef(evenHeaderL,                                PropertyValue(String(u"$p"))),
    styleDef(evenHeaderC,                                PropertyValue(String())),
    styleDef(evenHeaderR,                                PropertyValue(String())),
    styleDef(oddHeaderL,                                 PropertyValue(String())),
    styleDef(oddHeaderC,                                 PropertyValue(String())),
    styleDef(oddHeaderR,                                 PropertyValue(String(u"$p"))),
    styleDef(showFooter,                                 true),

    styleDef(footerFirstPage,                            true),
    styleDef(footerOddEven,                              true),
    styleDef(evenFooterL,                                PropertyValue(String())),
    styleDef(evenFooterC,                                PropertyValue(String(u"$C"))),
    styleDef(evenFooterR,                                PropertyValue(String())),
    styleDef(oddFooterL,                                 PropertyValue(String())),
    styleDef(oddFooterC,                                 PropertyValue(String(u"$C"))),
    styleDef(oddFooterR,                                 PropertyValue(String())),

    styleDef(voltaPosAbove,                              PointF(0.0, -3.0)),
    styleDef(voltaHook,                                  Spatium(2.2)),
    styleDef(voltaLineWidth,                             Spatium(0.11)),
    styleDef(voltaLineStyle,                             PropertyValue(LineType::SOLID)),
    styleDef(voltaDashLineLen,                           5.0),
    styleDef(voltaDashGapLen,                            5.0),
    styleDef(voltaFontFace,                              "Edwin"),
    styleDef(voltaFontSize,                              11.0),
    styleDef(voltaLineSpacing,                           1.0),
    styleDef(voltaFontSpatiumDependent,                  true),
    styleDef(voltaFontStyle,                             int(FontStyle::Bold)),
    styleDef(voltaColor,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(voltaAlign,                                 Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(voltaOffset,                                PointF(0.6, 2.2)),
    styleDef(voltaFrameType,                             int(FrameType::NO_FRAME)),
    styleDef(voltaFramePadding,                          0.2),
    styleDef(voltaFrameWidth,                            0.1),
    styleDef(voltaFrameRound,                            0),
    styleDef(voltaFrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(voltaFrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(ottava8VAPlacement,                         PlacementV::ABOVE),
    styleDef(ottava8VBPlacement,                         PlacementV::BELOW),
    styleDef(ottava15MAPlacement,                        PlacementV::ABOVE),
    styleDef(ottava15MBPlacement,                        PlacementV::BELOW),
    styleDef(ottava22MAPlacement,                        PlacementV::ABOVE),
    styleDef(ottava22MBPlacement,                        PlacementV::BELOW),

    styleDef(ottava8VAText,                              String(u"<sym>ottavaAlta</sym>")),
    styleDef(ottava8VAContinueText,                      String(u"<sym>ottavaAlta</sym>")),
    styleDef(ottava8VBText,                              String(u"<sym>ottavaBassa</sym>")),
    styleDef(ottava8VBContinueText,                      String(u"<sym>ottavaBassa</sym>")),
    styleDef(ottava15MAText,                             String(u"<sym>quindicesimaAlta</sym>")),
    styleDef(ottava15MAContinueText,                     String(u"<sym>quindicesimaAlta</sym>")),
    styleDef(ottava15MBText,                             String(u"<sym>quindicesimaBassa</sym>")),
    styleDef(ottava15MBContinueText,                     String(u"<sym>quindicesimaBassa</sym>")),
    styleDef(ottava22MAText,                             String(u"<sym>ventiduesimaAlta</sym>")),
    styleDef(ottava22MAContinueText,                     String(u"<sym>ventiduesimaAlta</sym>")),
    styleDef(ottava22MBText,                             String(u"<sym>ventiduesimaBassa</sym>")),
    styleDef(ottava22MBContinueText,                     String(u"<sym>ventiduesimaBassa</sym>")),

    styleDef(ottava8VAnoText,                            String(u"<sym>ottava</sym>")),
    styleDef(ottava8VAnoContinueText,                    String(u"<sym>ottava</sym>")),
    styleDef(ottava8VBnoText,                            String(u"<sym>ottava</sym>")),
    styleDef(ottava8VBnoContinueText,                    String(u"<sym>ottava</sym>")),
    styleDef(ottava15MAnoText,                           String(u"<sym>quindicesima</sym>")),
    styleDef(ottava15MAnoContinueText,                   String(u"<sym>quindicesima</sym>")),
    styleDef(ottava15MBnoText,                           String(u"<sym>quindicesima</sym>")),
    styleDef(ottava15MBnoContinueText,                   String(u"<sym>quindicesima</sym>")),
    styleDef(ottava22MAnoText,                           String(u"<sym>ventiduesima</sym>")),
    styleDef(ottava22MAnoContinueText,                   String(u"<sym>ventiduesima</sym>")),
    styleDef(ottava22MBnoText,                           String(u"<sym>ventiduesima</sym>")),
    styleDef(ottava22MBnoContinueText,                   String(u"<sym>ventiduesima</sym>")),

    styleDef(ottavaPosAbove,                             PointF(.0, -2.0)),
    styleDef(ottavaPosBelow,                             PointF(.0, 2.0)),
    styleDef(ottavaHookAbove,                            Spatium(1)),
    styleDef(ottavaHookBelow,                            Spatium(-1)),
    styleDef(ottavaLineWidth,                            Spatium(0.11)),
    styleDef(ottavaLineStyle,                            PropertyValue(LineType::DASHED)),
    styleDef(ottavaDashLineLen,                          6.0),
    styleDef(ottavaDashGapLen,                           6.0),
    styleDef(ottavaNumbersOnly,                          true),
    styleDef(ottavaFontFace,                             "Edwin"),
    styleDef(ottavaFontSize,                             10.0),
    styleDef(ottavaLineSpacing,                          1.0),
    styleDef(ottavaFontSpatiumDependent,                 true),
    styleDef(ottavaMusicalSymbolsScale,                  1.0), // percentage of the standard size
    styleDef(ottavaFontStyle,                            int(FontStyle::Normal)),
    styleDef(ottavaColor,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(ottavaTextAlignAbove,                       Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(ottavaTextAlignBelow,                       Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(ottavaFrameType,                            int(FrameType::NO_FRAME)),
    styleDef(ottavaFramePadding,                         0.2),
    styleDef(ottavaFrameWidth,                           0.1),
    styleDef(ottavaFrameRound,                           0),
    styleDef(ottavaFrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(ottavaFrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(tabClef,                                    PropertyValue(int(ClefType::TAB))),

    styleDef(tremoloWidth,                               Spatium(1.2)), // tremolo stroke width: notehead width
    styleDef(tremoloBoxHeight,                           Spatium(0.65)),
    styleDef(tremoloLineWidth,                           Spatium(0.5)), // was 0.35
    styleDef(tremoloDistance,                            Spatium(0.8)),
    styleDef(tremoloStyle,                               int(TremoloStyle::DEFAULT)),
    styleDef(tremoloStrokeLengthMultiplier,              0.62),
    styleDef(tremoloNoteSidePadding,                     Spatium(1.25)),
    styleDef(tremoloOutSidePadding,                      Spatium(0.5)),
    //styleDef(tremoloMaxBeamLength,                       PropertyValue(double(1.5))), // TODO

    styleDef(linearStretch,                              PropertyValue(double(1.5))),
    styleDef(crossMeasureValues,                         false),
    styleDef(keySigNaturals,                             PropertyValue(int(KeySigNatural::NONE))),

    styleDef(tupletMaxSlope,                             PropertyValue(double(0.5))),
    styleDef(tupletOutOfStaff,                           true),
    styleDef(tupletVHeadDistance,                        Spatium(.5)),
    styleDef(tupletVStemDistance,                        Spatium(.5)),
    styleDef(tupletStemLeftDistance,                     Spatium(.5)),
    styleDef(tupletStemRightDistance,                    Spatium(.5)),
    styleDef(tupletNoteLeftDistance,                     Spatium(0.0)),
    styleDef(tupletNoteRightDistance,                    Spatium(0.0)),
    styleDef(tupletBracketWidth,                         Spatium(0.1)),
    styleDef(tupletDirection,                            DirectionV::AUTO),
    styleDef(tupletNumberType,                           int(TupletNumberType::SHOW_NUMBER)),
    styleDef(tupletBracketType,                          int(TupletBracketType::AUTO_BRACKET)),
    styleDef(tupletFontFace,                             "Edwin"),
    styleDef(tupletFontSize,                             9.0),
    styleDef(tupletLineSpacing,                          1.0),
    styleDef(tupletFontSpatiumDependent,                 true),
    styleDef(tupletMusicalSymbolsScale,                  1.0), // percentage of the standard size
    styleDef(tupletFontStyle,                            int(FontStyle::Italic)),
    styleDef(tupletColor,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(tupletAlign,                                Align(AlignH::HCENTER, AlignV::VCENTER)),
    styleDef(tupletUseSymbols,                           false),
    styleDef(tupletBracketHookHeight,                    Spatium(0.75)),
    styleDef(tupletOffset,                               PointF()),
    styleDef(tupletFrameType,                            int(FrameType::NO_FRAME)),
    styleDef(tupletFramePadding,                         0.2),
    styleDef(tupletFrameWidth,                           0.1),
    styleDef(tupletFrameRound,                           0),
    styleDef(tupletFrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(tupletFrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(scaleBarlines,                              false),
    styleDef(barGraceDistance,                           Spatium(1.0)),
    styleDef(minVerticalDistance,                        Spatium(0.5)),
    styleDef(skylineMinHorizontalClearance,              Spatium(0.25)),
    styleDef(ornamentStyle,                              int(OrnamentStyle::DEFAULT)),
    styleDef(spatium,                                    24.8),

    styleDef(autoplaceHairpinDynamicsDistance,           Spatium(0.5)),

    styleDef(dynamicsHairpinVoiceBasedPlacement,         DirectionV::AUTO),
    styleDef(dynamicsHairpinsAutoCenterOnGrandStaff,     true),
    styleDef(dynamicsHairpinsAboveForVocalStaves,        true),

    styleDef(dynamicsOverrideFont,                       false),
    styleDef(dynamicsFont,                               PropertyValue(String(u"Leland"))),
    styleDef(dynamicsSize,                               1.0), // percentage of the standard size
    styleDef(dynamicsPlacement,                          PlacementV::BELOW),
    styleDef(dynamicsPosAbove,                           PointF(.0, -1.0)),
    styleDef(dynamicsPosBelow,                           PointF(.0, 2.0)),
    styleDef(avoidBarLines,                              true),
    styleDef(snapToDynamics,                             true),
    styleDef(centerOnNotehead,                           true),
    styleDef(dynamicsMinDistance,                        Spatium(0.5)),
    styleDef(autoplaceVerticalAlignRange,                int(VerticalAlignRange::SYSTEM)),

    styleDef(textLinePlacement,                          PlacementV::ABOVE),
    styleDef(textLinePosAbove,                           PointF(.0, -1.0)),
    styleDef(textLinePosBelow,                           PointF(.0, 1.0)),
    styleDef(textLineFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(textLineFramePadding,                       0.2),
    styleDef(textLineFrameWidth,                         0.1),
    styleDef(textLineFrameRound,                         0),
    styleDef(textLineFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(textLineFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(systemTextLinePlacement,                    PlacementV::ABOVE),
    styleDef(systemTextLinePosAbove,                     PointF(.0, -1.0)),
    styleDef(systemTextLinePosBelow,                     PointF(.0, 1.0)),
    styleDef(systemTextLineFrameType,                    int(FrameType::NO_FRAME)),
    styleDef(systemTextLineFramePadding,                 0.2),
    styleDef(systemTextLineFrameWidth,                   0.1),
    styleDef(systemTextLineFrameRound,                   0),
    styleDef(systemTextLineFrameFgColor,                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(systemTextLineFrameBgColor,                 PropertyValue::fromValue(Color::transparent)),

    styleDef(tremoloBarLineWidth,                        Spatium(0.12)),
    styleDef(jumpPosAbove,                               PointF(.0, -2.0)),
    styleDef(markerPosAbove,                             PointF(.0, -2.0)),

    styleDef(defaultFontFace,                            "Edwin"),
    styleDef(defaultFontSize,                            10.0),
    styleDef(defaultLineSpacing,                         1.0),
    styleDef(defaultFontSpatiumDependent,                true),
    styleDef(defaultFontStyle,                           int(FontStyle::Normal)),
    styleDef(defaultColor,                               PropertyValue::fromValue(Color::BLACK)),
    styleDef(defaultAlign,                               Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(defaultFrameType,                           int(FrameType::NO_FRAME)),
    styleDef(defaultFramePadding,                        Spatium(0.2)),
    styleDef(defaultFrameWidth,                          Spatium(0.1)),
    styleDef(defaultFrameRound,                          0),
    styleDef(defaultFrameFgColor,                        PropertyValue::fromValue(Color::BLACK)),
    styleDef(defaultFrameBgColor,                        PropertyValue::fromValue(Color::transparent)),
    styleDef(defaultOffset,                              PointF()),
    styleDef(defaultOffsetType,                          int(OffsetType::SPATIUM)),
    styleDef(defaultSystemFlag,                          false),
    styleDef(defaultText,                                String()),

    styleDef(titleFontFace,                              "Edwin"),
    styleDef(titleFontSize,                              22.0),
    styleDef(titleLineSpacing,                           1.0),
    styleDef(titleFontSpatiumDependent,                  false),
    styleDef(titleFontStyle,                             int(FontStyle::Normal)),
    styleDef(titleColor,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(titleAlign,                                 Align(AlignH::HCENTER, AlignV::TOP)),
    styleDef(titleOffset,                                PointF()),
    styleDef(titleOffsetType,                            int(OffsetType::ABS)),
    styleDef(titleFrameType,                             int(FrameType::NO_FRAME)),
    styleDef(titleFramePadding,                          0.2),
    styleDef(titleFrameWidth,                            0.1),
    styleDef(titleFrameRound,                            0),
    styleDef(titleFrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(titleFrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(subTitleFontFace,                           "Edwin"),
    styleDef(subTitleFontSize,                           14.0),
    styleDef(subTitleLineSpacing,                        1.0),
    styleDef(subTitleFontSpatiumDependent,               false),
    styleDef(subTitleFontStyle,                          int(FontStyle::Normal)),
    styleDef(subTitleColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(subTitleAlign,                              Align(AlignH::HCENTER, AlignV::TOP)),
    styleDef(subTitleOffset,                             PointF(0.0, 10.0)),
    styleDef(subTitleOffsetType,                         int(OffsetType::ABS)),
    styleDef(subTitleFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(subTitleFramePadding,                       0.2),
    styleDef(subTitleFrameWidth,                         0.1),
    styleDef(subTitleFrameRound,                         0),
    styleDef(subTitleFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(subTitleFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(composerFontFace,                           "Edwin"),
    styleDef(composerFontSize,                           10.0),
    styleDef(composerLineSpacing,                        1.0),
    styleDef(composerFontSpatiumDependent,               false),
    styleDef(composerFontStyle,                          int(FontStyle::Normal)),
    styleDef(composerColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(composerAlign,                              Align(AlignH::RIGHT, AlignV::BOTTOM)),
    styleDef(composerOffset,                             PointF()),
    styleDef(composerOffsetType,                         int(OffsetType::ABS)),
    styleDef(composerFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(composerFramePadding,                       0.2),
    styleDef(composerFrameWidth,                         0.1),
    styleDef(composerFrameRound,                         0),
    styleDef(composerFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(composerFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(lyricistFontFace,                           "Edwin"),
    styleDef(lyricistFontSize,                           10.0),
    styleDef(lyricistLineSpacing,                        1.0),
    styleDef(lyricistFontSpatiumDependent,               false),
    styleDef(lyricistFontStyle,                          int(FontStyle::Normal)),
    styleDef(lyricistColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(lyricistAlign,                              Align(AlignH::LEFT, AlignV::BOTTOM)),
    styleDef(lyricistOffset,                             PointF()),
    styleDef(lyricistOffsetType,                         int(OffsetType::ABS)),
    styleDef(lyricistFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(lyricistFramePadding,                       0.2),
    styleDef(lyricistFrameWidth,                         0.1),
    styleDef(lyricistFrameRound,                         0),
    styleDef(lyricistFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(lyricistFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(fingeringFontFace,                          "Edwin"),
    styleDef(fingeringFontSize,                          8.0),
    styleDef(fingeringLineSpacing,                       1.0),
    styleDef(fingeringFontSpatiumDependent,              true),
    styleDef(fingeringFontStyle,                         int(FontStyle::Normal)),
    styleDef(fingeringColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(fingeringAlign,                             Align(AlignH::HCENTER, AlignV::VCENTER)),
    styleDef(fingeringFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(fingeringFramePadding,                      0.2),
    styleDef(fingeringFrameWidth,                        0.1),
    styleDef(fingeringFrameRound,                        0),
    styleDef(fingeringFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(fingeringFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),
    styleDef(fingeringOffset,                            PointF()),

    styleDef(lhGuitarFingeringFontFace,                  "Edwin"),
    styleDef(lhGuitarFingeringFontSize,                  8.0),
    styleDef(lhGuitarFingeringLineSpacing,               1.0),
    styleDef(lhGuitarFingeringFontSpatiumDependent,      true),
    styleDef(lhGuitarFingeringFontStyle,                 int(FontStyle::Normal)),
    styleDef(lhGuitarFingeringColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(lhGuitarFingeringAlign,                     Align(AlignH::RIGHT, AlignV::VCENTER)),
    styleDef(lhGuitarFingeringFrameType,                 int(FrameType::NO_FRAME)),
    styleDef(lhGuitarFingeringFramePadding,              0.2),
    styleDef(lhGuitarFingeringFrameWidth,                0.1),
    styleDef(lhGuitarFingeringFrameRound,                0),
    styleDef(lhGuitarFingeringFrameFgColor,              PropertyValue::fromValue(Color::BLACK)),
    styleDef(lhGuitarFingeringFrameBgColor,              PropertyValue::fromValue(Color::transparent)),
    styleDef(lhGuitarFingeringOffset,                    PointF(-0.5, 0.0)),

    styleDef(rhGuitarFingeringFontFace,                  "Edwin"),
    styleDef(rhGuitarFingeringFontSize,                  8.0),
    styleDef(rhGuitarFingeringLineSpacing,               1.0),
    styleDef(rhGuitarFingeringFontSpatiumDependent,      true),
    styleDef(rhGuitarFingeringFontStyle,                 int(FontStyle::Italic)),
    styleDef(rhGuitarFingeringColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(rhGuitarFingeringAlign,                     Align(AlignH::HCENTER, AlignV::VCENTER)),
    styleDef(rhGuitarFingeringFrameType,                 int(FrameType::NO_FRAME)),
    styleDef(rhGuitarFingeringFramePadding,              0.2),
    styleDef(rhGuitarFingeringFrameWidth,                0.1),
    styleDef(rhGuitarFingeringFrameRound,                0),
    styleDef(rhGuitarFingeringFrameFgColor,              PropertyValue::fromValue(Color::BLACK)),
    styleDef(rhGuitarFingeringFrameBgColor,              PropertyValue::fromValue(Color::transparent)),
    styleDef(rhGuitarFingeringOffset,                    PointF()),

    styleDef(stringNumberFontFace,                       "Edwin"),
    styleDef(stringNumberFontSize,                       8.0),
    styleDef(stringNumberLineSpacing,                    1.0),
    styleDef(stringNumberFontSpatiumDependent,           true),
    styleDef(stringNumberFontStyle,                      int(FontStyle::Normal)),
    styleDef(stringNumberColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(stringNumberAlign,                          Align(AlignH::HCENTER, AlignV::VCENTER)),
    styleDef(stringNumberFrameType,                      int(FrameType::CIRCLE)),
    styleDef(stringNumberFramePadding,                   0.2),
    styleDef(stringNumberFrameWidth,                     0.1),
    styleDef(stringNumberFrameRound,                     0),
    styleDef(stringNumberFrameFgColor,                   PropertyValue::fromValue(Color::BLACK)),
    styleDef(stringNumberFrameBgColor,                   PropertyValue::fromValue(Color::transparent)),
    styleDef(stringNumberOffset,                         PointF(0.0, 0.0)),
    styleDef(preferSameStringForTranspose,               false),

    styleDef(stringTuningsFontSize,                      9.0),

    styleDef(harpPedalDiagramFontFace,                   "Edwin"),
    styleDef(harpPedalDiagramFontSize,                   10.0),
    styleDef(harpPedalDiagramLineSpacing,                1.0),
    styleDef(harpPedalDiagramFontSpatiumDependent,       true),
    styleDef(harpPedalDiagramMusicalSymbolsScale,        1.0), // percentage of the standard size
    styleDef(harpPedalDiagramFontStyle,                  int(FontStyle::Normal)),
    styleDef(harpPedalDiagramColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(harpPedalDiagramAlign,                      Align(AlignH::HCENTER, AlignV::VCENTER)),
    styleDef(harpPedalDiagramFrameType,                  int(FrameType::NO_FRAME)),
    styleDef(harpPedalDiagramFramePadding,               0.2),
    styleDef(harpPedalDiagramFrameWidth,                 0.1),
    styleDef(harpPedalDiagramFrameRound,                 0),
    styleDef(harpPedalDiagramFrameFgColor,               PropertyValue::fromValue(Color::BLACK)),
    styleDef(harpPedalDiagramFrameBgColor,               PropertyValue::fromValue(Color::transparent)),
    styleDef(harpPedalDiagramOffset,                     PointF()),
    styleDef(harpPedalDiagramPlacement,                  PlacementV::ABOVE),
    styleDef(harpPedalDiagramPosAbove,                   PointF(.0, -1.0)),
    styleDef(harpPedalDiagramPosBelow,                   PointF(.0, 2.5)),
    styleDef(harpPedalDiagramMinDistance,                Spatium(.5)),

    styleDef(harpPedalTextDiagramFontFace,               "Edwin"),
    styleDef(harpPedalTextDiagramFontSize,               8.0),
    styleDef(harpPedalTextDiagramLineSpacing,            1.0),
    styleDef(harpPedalTextDiagramFontSpatiumDependent,   true),
    styleDef(harpPedalTextDiagramFontStyle,              int(FontStyle::Normal)),
    styleDef(harpPedalTextDiagramColor,                  PropertyValue::fromValue(Color::BLACK)),
    styleDef(harpPedalTextDiagramAlign,                  Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(harpPedalTextDiagramFrameType,              int(FrameType::NO_FRAME)),
    styleDef(harpPedalTextDiagramFramePadding,           0.2),
    styleDef(harpPedalTextDiagramFrameWidth,             0.1),
    styleDef(harpPedalTextDiagramFrameRound,             0),
    styleDef(harpPedalTextDiagramFrameFgColor,           PropertyValue::fromValue(Color::BLACK)),
    styleDef(harpPedalTextDiagramFrameBgColor,           PropertyValue::fromValue(Color::transparent)),
    styleDef(harpPedalTextDiagramOffset,                 PointF()),
    styleDef(harpPedalTextDiagramPlacement,              PlacementV::BELOW),
    styleDef(harpPedalTextDiagramPosAbove,               PointF(.0, -1.5)),
    styleDef(harpPedalTextDiagramPosBelow,               PointF(.0, 2.5)),
    styleDef(harpPedalTextDiagramMinDistance,            Spatium(.5)),

    styleDef(longInstrumentFontFace,                     "Edwin"),
    styleDef(longInstrumentFontSize,                     10.0),
    styleDef(longInstrumentLineSpacing,                  1.0),
    styleDef(longInstrumentFontSpatiumDependent,         true),
    styleDef(longInstrumentFontStyle,                    int(FontStyle::Normal)),
    styleDef(longInstrumentColor,                        PropertyValue::fromValue(Color::BLACK)),
    styleDef(longInstrumentAlign,                        Align(AlignH::RIGHT, AlignV::VCENTER)),
    styleDef(longInstrumentOffset,                       PointF(.0, .0)),
    styleDef(longInstrumentFrameType,                    int(FrameType::NO_FRAME)),
    styleDef(longInstrumentFramePadding,                 0.2),
    styleDef(longInstrumentFrameWidth,                   0.1),
    styleDef(longInstrumentFrameRound,                   0),
    styleDef(longInstrumentFrameFgColor,                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(longInstrumentFrameBgColor,                 PropertyValue::fromValue(Color::transparent)),

    styleDef(shortInstrumentFontFace,                    "Edwin"),
    styleDef(shortInstrumentFontSize,                    10.0),
    styleDef(shortInstrumentLineSpacing,                 1.0),
    styleDef(shortInstrumentFontSpatiumDependent,        true),
    styleDef(shortInstrumentFontStyle,                   int(FontStyle::Normal)),
    styleDef(shortInstrumentColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(shortInstrumentAlign,                       Align(AlignH::RIGHT, AlignV::VCENTER)),
    styleDef(shortInstrumentOffset,                      PointF(.0, .0)),
    styleDef(shortInstrumentFrameType,                   int(FrameType::NO_FRAME)),
    styleDef(shortInstrumentFramePadding,                0.2),
    styleDef(shortInstrumentFrameWidth,                  0.1),
    styleDef(shortInstrumentFrameRound,                  0),
    styleDef(shortInstrumentFrameFgColor,                PropertyValue::fromValue(Color::BLACK)),
    styleDef(shortInstrumentFrameBgColor,                PropertyValue::fromValue(Color::transparent)),

    styleDef(partInstrumentFontFace,                     "Edwin"),
    styleDef(partInstrumentFontSize,                     14.0),
    styleDef(partInstrumentLineSpacing,                  1.0),
    styleDef(partInstrumentFontSpatiumDependent,         false),
    styleDef(partInstrumentFontStyle,                    int(FontStyle::Normal)),
    styleDef(partInstrumentColor,                        PropertyValue::fromValue(Color::BLACK)),
    styleDef(partInstrumentAlign,                        Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(partInstrumentOffset,                       PointF()),
    styleDef(partInstrumentFrameType,                    int(FrameType::NO_FRAME)),
    styleDef(partInstrumentFramePadding,                 0.2),
    styleDef(partInstrumentFrameWidth,                   0.1),
    styleDef(partInstrumentFrameRound,                   0),
    styleDef(partInstrumentFrameFgColor,                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(partInstrumentFrameBgColor,                 PropertyValue::fromValue(Color::transparent)),

    // OBSOLETE after version 4.1. Dynamic text now takes its setting from expression.
    styleDef(dynamicsFontFace,                           "Edwin"),
    styleDef(dynamicsFontSize,                           10.0),
    styleDef(dynamicsLineSpacing,                        1.0),
    styleDef(dynamicsFontSpatiumDependent,               true),
    styleDef(dynamicsFontStyle,                          int(FontStyle::Italic)),
    styleDef(dynamicsColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(dynamicsAlign,                              Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(dynamicsFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(dynamicsFramePadding,                       0.2),
    styleDef(dynamicsFrameWidth,                         0.1),
    styleDef(dynamicsFrameRound,                         0),
    styleDef(dynamicsFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(dynamicsFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(expressionFontFace,                         "Edwin"),
    styleDef(expressionFontSize,                         10.0),
    styleDef(expressionLineSpacing,                      1.0),
    styleDef(expressionFontSpatiumDependent,             true),
    styleDef(expressionFontStyle,                        int(FontStyle::Italic)),
    styleDef(expressionColor,                            PropertyValue::fromValue(Color::BLACK)),
    styleDef(expressionAlign,                            Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(expressionPlacement,                        PlacementV::BELOW),
    styleDef(expressionOffset,                           PointF(.0, 2.5)),
    styleDef(expressionPosAbove,                         PointF(.0, -1.5)),
    styleDef(expressionPosBelow,                         PointF(.0, 2.5)),
    styleDef(expressionFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(expressionFramePadding,                     0.2),
    styleDef(expressionFrameWidth,                       0.1),
    styleDef(expressionFrameRound,                       0),
    styleDef(expressionFrameFgColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(expressionFrameBgColor,                     PropertyValue::fromValue(Color::transparent)),
    styleDef(expressionMinDistance,                      Spatium(.5)),

    styleDef(tempoFontFace,                              "Edwin"),
    styleDef(tempoFontSize,                              12.0),
    styleDef(tempoLineSpacing,                           1.0),
    styleDef(tempoFontSpatiumDependent,                  true),
    styleDef(tempoFontStyle,                             int(FontStyle::Bold)),
    styleDef(tempoColor,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(tempoAlign,                                 Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(tempoSystemFlag,                            true),
    styleDef(tempoPlacement,                             PlacementV::ABOVE),
    styleDef(tempoPosAbove,                              PointF(.0, -2.0)),
    styleDef(tempoPosBelow,                              PointF(.0, 3.0)),
    styleDef(tempoMinDistance,                           Spatium(.5)),
    styleDef(tempoFrameType,                             int(FrameType::NO_FRAME)),
    styleDef(tempoFramePadding,                          0.2),
    styleDef(tempoFrameWidth,                            0.1),
    styleDef(tempoFrameRound,                            0),
    styleDef(tempoFrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(tempoFrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(tempoChangeFontFace,                        "Edwin"),
    styleDef(tempoChangeFontSize,                        12.0),
    styleDef(tempoChangeLineSpacing,                     1.0),
    styleDef(tempoChangeFontSpatiumDependent,            true),
    styleDef(tempoChangeFontStyle,                       int(FontStyle::Bold)),
    styleDef(tempoChangeColor,                           PropertyValue::fromValue(Color::BLACK)),
    styleDef(tempoChangeAlign,                           Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(tempoChangeSystemFlag,                      true),
    styleDef(tempoChangePlacement,                       PlacementV::ABOVE),
    styleDef(tempoChangePosAbove,                        PointF(.0, -2.0)),
    styleDef(tempoChangePosBelow,                        PointF(.0, 3.0)),
    styleDef(tempoChangeMinDistance,                     Spatium(.5)),
    styleDef(tempoChangeFrameType,                       int(FrameType::NO_FRAME)),
    styleDef(tempoChangeFramePadding,                    0.2),
    styleDef(tempoChangeFrameWidth,                      0.1),
    styleDef(tempoChangeFrameRound,                      0),
    styleDef(tempoChangeFrameFgColor,                    PropertyValue::fromValue(Color::BLACK)),
    styleDef(tempoChangeFrameBgColor,                    PropertyValue::fromValue(Color::transparent)),
    styleDef(tempoChangeLineWidth,                       Spatium(0.15)),
    styleDef(tempoChangeLineStyle,                       PropertyValue(LineType::DASHED)),
    styleDef(tempoChangeDashLineLen,                     6.0),
    styleDef(tempoChangeDashGapLen,                      9.0),

    styleDef(metronomeFontFace,                          "Edwin"),
    styleDef(metronomeFontSize,                          12.0),
    styleDef(metronomeLineSpacing,                       1.0),
    styleDef(metronomeFontSpatiumDependent,              false),
    styleDef(metronomeFontStyle,                         int(FontStyle::Normal)),
    styleDef(metronomeColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(metronomePlacement,                         PlacementV::ABOVE),
    styleDef(metronomeAlign,                             Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(metronomeOffset,                            PointF()),
    styleDef(metronomeFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(metronomeFramePadding,                      0.2),
    styleDef(metronomeFrameWidth,                        0.1),
    styleDef(metronomeFrameRound,                        0),
    styleDef(metronomeFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(metronomeFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),

    styleDef(measureNumberFontFace,                      "Edwin"),
    styleDef(measureNumberFontSize,                      8.0),
    styleDef(measureNumberLineSpacing,                   1.0),
    styleDef(measureNumberFontSpatiumDependent,          false),
    styleDef(measureNumberFontStyle,                     2),
    styleDef(measureNumberColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(measureNumberPosAbove,                      PointF(0.0, -2.0)),
    styleDef(measureNumberPosBelow,                      PointF(0.0, 2.0)),
    styleDef(measureNumberOffsetType,                    int(OffsetType::SPATIUM)),
    styleDef(measureNumberVPlacement,                    PlacementV::ABOVE),
    styleDef(measureNumberHPlacement,                    PlacementH::LEFT),
    styleDef(measureNumberMinDistance,                   Spatium(0.5)),
    styleDef(measureNumberAlign,                         Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(measureNumberFrameType,                     int(FrameType::NO_FRAME)),
    styleDef(measureNumberFramePadding,                  0.2),
    styleDef(measureNumberFrameWidth,                    0.1),
    styleDef(measureNumberFrameRound,                    0),
    styleDef(measureNumberFrameFgColor,                  PropertyValue::fromValue(Color::BLACK)),
    styleDef(measureNumberFrameBgColor,                  PropertyValue::fromValue(Color::transparent)),

    styleDef(mmRestShowMeasureNumberRange,               false),
    styleDef(mmRestRangeBracketType,                     int(MMRestRangeBracketType::BRACKETS)),

    styleDef(mmRestRangeFontFace,                        "Edwin"),
    styleDef(mmRestRangeFontSize,                        8.0),
    styleDef(mmRestRangeFontSpatiumDependent,            false),
    styleDef(mmRestRangeFontStyle,                       int(FontStyle::Italic)),
    styleDef(mmRestRangeColor,                           PropertyValue::fromValue(Color::BLACK)),
    styleDef(mmRestRangePosAbove,                        PointF(0.0, -3.0)),
    styleDef(mmRestRangePosBelow,                        PointF(0.0, 1.0)),
    styleDef(mmRestRangeOffsetType,                      int(OffsetType::SPATIUM)),
    styleDef(mmRestRangeVPlacement,                      PlacementV::BELOW),
    styleDef(mmRestRangeHPlacement,                      PlacementH::CENTER),
    styleDef(mmRestRangeAlign,                           Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(mmRestRangeFrameType,                       int(FrameType::NO_FRAME)),
    styleDef(mmRestRangeFramePadding,                    0.2),
    styleDef(mmRestRangeFrameWidth,                      0.1),
    styleDef(mmRestRangeFrameRound,                      0),
    styleDef(mmRestRangeFrameFgColor,                    PropertyValue::fromValue(Color::BLACK)),
    styleDef(mmRestRangeFrameBgColor,                    PropertyValue::fromValue(Color::transparent)),
    styleDef(mmRestRangeMinDistance,                     Spatium(0.5)),

    styleDef(translatorFontFace,                         "Edwin"),
    styleDef(translatorFontSize,                         10.0),
    styleDef(translatorLineSpacing,                      1.0),
    styleDef(translatorFontSpatiumDependent,             false),
    styleDef(translatorFontStyle,                        int(FontStyle::Normal)),
    styleDef(translatorColor,                            PropertyValue::fromValue(Color::BLACK)),
    styleDef(translatorAlign,                            Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(translatorOffset,                           PointF()),
    styleDef(translatorFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(translatorFramePadding,                     0.2),
    styleDef(translatorFrameWidth,                       0.1),
    styleDef(translatorFrameRound,                       0),
    styleDef(translatorFrameFgColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(translatorFrameBgColor,                     PropertyValue::fromValue(Color::transparent)),

    styleDef(systemTextFontFace,                         "Edwin"),
    styleDef(systemTextFontSize,                         10.0),
    styleDef(systemTextLineSpacing,                      1.0),
    styleDef(systemTextFontSpatiumDependent,             true),
    styleDef(systemTextFontStyle,                        int(FontStyle::Normal)),
    styleDef(systemTextColor,                            PropertyValue::fromValue(Color::BLACK)),
    styleDef(systemTextAlign,                            Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(systemTextOffsetType,                       int(OffsetType::SPATIUM)),
    styleDef(systemTextPlacement,                        PlacementV::ABOVE),
    styleDef(systemTextPosAbove,                         PointF(.0, -2.0)),
    styleDef(systemTextPosBelow,                         PointF(.0, 3.5)),
    styleDef(systemTextMinDistance,                      Spatium(0.5)),
    styleDef(systemTextFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(systemTextFramePadding,                     0.2),
    styleDef(systemTextFrameWidth,                       0.1),
    styleDef(systemTextFrameRound,                       0),
    styleDef(systemTextFrameFgColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(systemTextFrameBgColor,                     PropertyValue::fromValue(Color::transparent)),

    styleDef(staffTextFontFace,                          "Edwin"),
    styleDef(staffTextFontSize,                          10.0),
    styleDef(staffTextLineSpacing,                       1.0),
    styleDef(staffTextFontSpatiumDependent,              true),
    styleDef(staffTextFontStyle,                         int(FontStyle::Normal)),
    styleDef(staffTextColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(staffTextAlign,                             Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(staffTextOffsetType,                        int(OffsetType::SPATIUM)),
    styleDef(staffTextPlacement,                         PlacementV::ABOVE),
    styleDef(staffTextPosAbove,                          PointF(.0, -1.0)),
    styleDef(staffTextPosBelow,                          PointF(.0, 2.5)),
    styleDef(staffTextMinDistance,                       Spatium(0.5)),
    styleDef(staffTextFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(staffTextFramePadding,                      0.2),
    styleDef(staffTextFrameWidth,                        0.1),
    styleDef(staffTextFrameRound,                        0),
    styleDef(staffTextFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(staffTextFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),

    styleDef(fretDiagramFingeringFontFace,               "FreeSans"),
    styleDef(fretDiagramFingeringFontSize,               6.0),
    styleDef(fretDiagramFingeringLineSpacing,            1.0),
    styleDef(fretDiagramFingeringFontSpatiumDependent,   true),
    styleDef(fretDiagramFingeringFontStyle,              int(FontStyle::Normal)),
    styleDef(fretDiagramFingeringColor,                  PropertyValue::fromValue(Color::BLACK)),
    styleDef(fretDiagramFingeringAlign,                  Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(fretDiagramFingeringPosAbove,               PointF(.0, -1.0)),
    styleDef(fretDiagramFingeringFrameType,              int(FrameType::NO_FRAME)),
    styleDef(fretDiagramFingeringFramePadding,           0.2),
    styleDef(fretDiagramFingeringFrameWidth,             0.1),
    styleDef(fretDiagramFingeringFrameRound,             0),
    styleDef(fretDiagramFingeringFrameFgColor,           PropertyValue::fromValue(Color::BLACK)),
    styleDef(fretDiagramFingeringFrameBgColor,           PropertyValue::fromValue(Color::transparent)),

    styleDef(fretDiagramFretNumberFontFace,              "FreeSans"),
    styleDef(fretDiagramFretNumberFontSize,              6.0),
    styleDef(fretDiagramFretNumberLineSpacing,           1.0),
    styleDef(fretDiagramFretNumberFontSpatiumDependent,  true),
    styleDef(fretDiagramFretNumberFontStyle,             int(FontStyle::Normal)),
    styleDef(fretDiagramFretNumberColor,                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(fretDiagramFretNumberAlign,                 Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(fretDiagramFretNumberPosAbove,              PointF(.0, -1.0)),
    styleDef(fretDiagramFretNumberFrameType,             int(FrameType::NO_FRAME)),
    styleDef(fretDiagramFretNumberFramePadding,          0.2),
    styleDef(fretDiagramFretNumberFrameWidth,            0.1),
    styleDef(fretDiagramFretNumberFrameRound,            0),
    styleDef(fretDiagramFretNumberFrameFgColor,          PropertyValue::fromValue(Color::BLACK)),
    styleDef(fretDiagramFretNumberFrameBgColor,          PropertyValue::fromValue(Color::transparent)),

    styleDef(rehearsalMarkFontFace,                      "Edwin"),
    styleDef(rehearsalMarkFontSize,                      14.0),
    styleDef(rehearsalMarkLineSpacing,                   1.0),
    styleDef(rehearsalMarkFontSpatiumDependent,          true),
    styleDef(rehearsalMarkFontStyle,                     int(FontStyle::Bold)),
    styleDef(rehearsalMarkColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(rehearsalMarkAlign,                         Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(rehearsalMarkFrameType,                     int(FrameType::SQUARE)),
    styleDef(rehearsalMarkFramePadding,                  0.5),
    styleDef(rehearsalMarkFrameWidth,                    0.16),
    styleDef(rehearsalMarkFrameRound,                    0),
    styleDef(rehearsalMarkFrameFgColor,                  PropertyValue::fromValue(Color::BLACK)),
    styleDef(rehearsalMarkFrameBgColor,                  PropertyValue::fromValue(Color::transparent)),
    styleDef(rehearsalMarkPlacement,                     PlacementV::ABOVE),
    styleDef(rehearsalMarkPosAbove,                      PointF(.0, -2.0)),
    styleDef(rehearsalMarkPosBelow,                      PointF(.0, 4.0)),
    styleDef(rehearsalMarkMinDistance,                   Spatium(0.5)),

    styleDef(repeatLeftFontFace,                         "Edwin"),
    styleDef(repeatLeftFontSize,                         18.0),
    styleDef(repeatLeftLineSpacing,                      1.0),
    styleDef(repeatLeftFontSpatiumDependent,             true),
    styleDef(repeatLeftFontStyle,                        int(FontStyle::Normal)),
    styleDef(repeatLeftColor,                            PropertyValue::fromValue(Color::BLACK)),
    styleDef(repeatLeftAlign,                            Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(repeatLeftPlacement,                        PlacementV::ABOVE),
    styleDef(repeatLeftFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(repeatLeftFramePadding,                     0.2),
    styleDef(repeatLeftFrameWidth,                       0.1),
    styleDef(repeatLeftFrameRound,                       0),
    styleDef(repeatLeftFrameFgColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(repeatLeftFrameBgColor,                     PropertyValue::fromValue(Color::transparent)),

    styleDef(repeatRightFontFace,                        "Edwin"),
    styleDef(repeatRightFontSize,                        11.0),
    styleDef(repeatRightLineSpacing,                     1.0),
    styleDef(repeatRightFontSpatiumDependent,            true),
    styleDef(repeatRightFontStyle,                       int(FontStyle::Normal)),
    styleDef(repeatRightColor,                           PropertyValue::fromValue(Color::BLACK)),
    styleDef(repeatRightAlign,                           Align(AlignH::RIGHT, AlignV::BASELINE)),
    styleDef(repeatRightPlacement,                       PlacementV::ABOVE),
    styleDef(repeatRightFrameType,                       int(FrameType::NO_FRAME)),
    styleDef(repeatRightFramePadding,                    0.2),
    styleDef(repeatRightFrameWidth,                      0.1),
    styleDef(repeatRightFrameRound,                      0),
    styleDef(repeatRightFrameFgColor,                    PropertyValue::fromValue(Color::BLACK)),
    styleDef(repeatRightFrameBgColor,                    PropertyValue::fromValue(Color::transparent)),

    styleDef(frameFontFace,                              "Edwin"),
    styleDef(frameFontSize,                              10.0),
    styleDef(frameLineSpacing,                           1.0),
    styleDef(frameFontSpatiumDependent,                  false),
    styleDef(frameFontStyle,                             int(FontStyle::Normal)),
    styleDef(frameColor,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(frameAlign,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(frameOffset,                                PointF()),
    styleDef(frameFrameType,                             int(FrameType::NO_FRAME)),
    styleDef(frameFramePadding,                          0.2),
    styleDef(frameFrameWidth,                            0.1),
    styleDef(frameFrameRound,                            0),
    styleDef(frameFrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(frameFrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(textLineFontFace,                           "Edwin"),
    styleDef(textLineFontSize,                           10.0),
    styleDef(textLineLineSpacing,                        1.0),
    styleDef(textLineFontSpatiumDependent,               true),
    styleDef(textLineFontStyle,                          int(FontStyle::Normal)),
    styleDef(textLineColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(textLineTextAlign,                          Align(AlignH::LEFT, AlignV::VCENTER)),
    styleDef(textLineSystemFlag,                         false),

    styleDef(systemTextLineFontFace,                     "Edwin"),
    styleDef(systemTextLineFontSize,                     10.0),
    styleDef(systemTextLineFontSpatiumDependent,         true),
    styleDef(systemTextLineFontStyle,                    int(FontStyle::Normal)),
    styleDef(systemTextLineColor,                        PropertyValue::fromValue(Color::BLACK)),
    styleDef(systemTextLineTextAlign,                    Align(AlignH::LEFT, AlignV::VCENTER)),
    styleDef(systemTextLineSystemFlag,                   true),

    styleDef(noteLinePlacement,                         PlacementV::ABOVE),
    styleDef(noteLineFontFace,                          "Edwin"),
    styleDef(noteLineFontSize,                          PropertyValue(10.0)),
    styleDef(noteLineLineSpacing,                       1.0),
    styleDef(noteLineFontSpatiumDependent,              true),
    styleDef(noteLineFontStyle,                         int(FontStyle::Normal)),
    styleDef(noteLineColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(noteLineAlign,                             Align(AlignH::LEFT, AlignV::VCENTER)),
    styleDef(noteLineOffset,                            PointF()),
    styleDef(noteLineFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(noteLineFramePadding,                      0.2),
    styleDef(noteLineFrameWidth,                        0.1),
    styleDef(noteLineFrameRound,                        0),
    styleDef(noteLineFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(noteLineFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),
    styleDef(noteLineWidth,                             Spatium(0.15)),
    styleDef(noteLineStyle,                             PropertyValue(LineType::SOLID)),
    styleDef(noteLineDashLineLen,                       5.0),
    styleDef(noteLineDashGapLen,                        5.0),

    styleDef(glissandoFontFace,                          "Edwin"),
    styleDef(glissandoFontSize,                          PropertyValue(8.0)),
    styleDef(glissandoLineSpacing,                       1.0),
    styleDef(glissandoFontSpatiumDependent,              true),
    styleDef(glissandoFontStyle,                         int(FontStyle::Italic)),
    styleDef(glissandoColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(glissandoAlign,                             Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(glissandoOffset,                            PointF()),
    styleDef(glissandoFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(glissandoFramePadding,                      0.2),
    styleDef(glissandoFrameWidth,                        0.1),
    styleDef(glissandoFrameRound,                        0),
    styleDef(glissandoFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(glissandoFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),
    styleDef(glissandoLineWidth,                         Spatium(0.15)),
    styleDef(glissandoText,                              String(u"gliss.")),
    styleDef(glissandoStyle,                             GlissandoStyle::CHROMATIC),
    styleDef(glissandoStyleHarp,                         GlissandoStyle::DIATONIC),
    styleDef(glissandoType,                              GlissandoType::STRAIGHT),
    styleDef(glissandoLineStyle,                         PropertyValue(LineType::SOLID)),
    styleDef(glissandoDashLineLen,                       5.0),
    styleDef(glissandoDashGapLen,                        5.0),
    styleDef(glissandoShowText,                          true),

    styleDef(bendFontFace,                               "Edwin"),
    styleDef(bendFontSize,                               8.0),
    styleDef(bendLineSpacing,                            1.0),
    styleDef(bendFontSpatiumDependent,                   true),
    styleDef(bendFontStyle,                              int(FontStyle::Normal)),
    styleDef(bendColor,                                  PropertyValue::fromValue(Color::BLACK)),
    styleDef(bendAlign,                                  Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(bendOffset,                                 PointF()),
    styleDef(bendFrameType,                              int(FrameType::NO_FRAME)),
    styleDef(bendFramePadding,                           0.2),
    styleDef(bendFrameWidth,                             0.1),
    styleDef(bendFrameRound,                             0),
    styleDef(bendFrameFgColor,                           PropertyValue::fromValue(Color::BLACK)),
    styleDef(bendFrameBgColor,                           PropertyValue::fromValue(Color::transparent)),
    styleDef(bendLineWidth,                              Spatium(0.15)),
    styleDef(bendArrowWidth,                             Spatium(.5)),

    styleDef(guitarBendLineWidth,                        Spatium(0.13)),
    styleDef(guitarBendLineWidthTab,                     Spatium(0.15)),
    styleDef(guitarBendHeightAboveTABStaff,              1.5), // in units of TAB lines dist
    styleDef(guitarBendPartialBendHeight,                Spatium(1.5)),
    styleDef(guitarBendUseFull,                          false),
    styleDef(guitarBendArrowWidth,                       Spatium(1.0)),
    styleDef(guitarBendArrowHeight,                      Spatium(1.0)),
    styleDef(useCueSizeFretForGraceBends,                false),

    styleDef(headerFontFace,                             "Edwin"),
    styleDef(headerFontSize,                             9.0),
    styleDef(headerLineSpacing,                          1.0),
    styleDef(headerFontSpatiumDependent,                 false),
    styleDef(headerFontStyle,                            int(FontStyle::Normal)),
    styleDef(headerColor,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(headerAlign,                                Align(AlignH::HCENTER, AlignV::TOP)),
    styleDef(headerOffset,                               PointF()),
    styleDef(headerFrameType,                            int(FrameType::NO_FRAME)),
    styleDef(headerFramePadding,                         0.2),
    styleDef(headerFrameWidth,                           0.1),
    styleDef(headerFrameRound,                           0),
    styleDef(headerFrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(headerFrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(footerFontFace,                             "Edwin"),
    styleDef(footerFontSize,                             9.0),
    styleDef(footerLineSpacing,                          1.0),
    styleDef(footerFontSpatiumDependent,                 false),
    styleDef(footerFontStyle,                            int(FontStyle::Normal)),
    styleDef(footerColor,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(footerAlign,                                Align(AlignH::HCENTER, AlignV::BOTTOM)),
    styleDef(footerOffset,                               PointF(0.0, 0.0)),
    styleDef(footerFrameType,                            int(FrameType::NO_FRAME)),
    styleDef(footerFramePadding,                         0.2),
    styleDef(footerFrameWidth,                           0.1),
    styleDef(footerFrameRound,                           0),
    styleDef(footerFrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(footerFrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    // New for 4.4 - defaults taken from footer
    styleDef(copyrightFontFace,                          "Edwin"),
    styleDef(copyrightFontSize,                          9.0),
    styleDef(copyrightLineSpacing,                       1.0),
    styleDef(copyrightFontSpatiumDependent,              false),
    styleDef(copyrightFontStyle,                         int(FontStyle::Normal)),
    styleDef(copyrightColor,                             PropertyValue::fromValue(Color::BLACK)),
    styleDef(copyrightAlign,                             Align(AlignH::HCENTER, AlignV::BOTTOM)),
    styleDef(copyrightOffset,                            PointF(0.0, 0.0)),
    styleDef(copyrightFrameType,                         int(FrameType::NO_FRAME)),
    styleDef(copyrightFramePadding,                      0.2),
    styleDef(copyrightFrameWidth,                        0.1),
    styleDef(copyrightFrameRound,                        0),
    styleDef(copyrightFrameFgColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(copyrightFrameBgColor,                      PropertyValue::fromValue(Color::transparent)),

    // New for 4.4 - defaults taken from pre-4.4 header
    styleDef(pageNumberFontFace,                         "Edwin"),
    styleDef(pageNumberFontSize,                         11.0),
    styleDef(pageNumberLineSpacing,                      1.0),
    styleDef(pageNumberFontSpatiumDependent,             false),
    styleDef(pageNumberFontStyle,                        1),
    styleDef(pageNumberColor,                            PropertyValue::fromValue(Color::BLACK)),
    styleDef(pageNumberAlign,                            Align(AlignH::HCENTER, AlignV::TOP)),
    styleDef(pageNumberOffset,                           PointF()),
    styleDef(pageNumberFrameType,                        int(FrameType::NO_FRAME)),
    styleDef(pageNumberFramePadding,                     0.2),
    styleDef(pageNumberFrameWidth,                       0.1),
    styleDef(pageNumberFrameRound,                       0),
    styleDef(pageNumberFrameFgColor,                     PropertyValue::fromValue(Color::BLACK)),
    styleDef(pageNumberFrameBgColor,                     PropertyValue::fromValue(Color::transparent)),

    styleDef(instrumentChangeFontFace,                   "Edwin"),
    styleDef(instrumentChangeFontSize,                   10.0),
    styleDef(instrumentChangeLineSpacing,                1.0),
    styleDef(instrumentChangeFontSpatiumDependent,       true),
    styleDef(instrumentChangeFontStyle,                  int(FontStyle::Bold)),
    styleDef(instrumentChangeColor,                      PropertyValue::fromValue(Color::BLACK)),
    styleDef(instrumentChangeAlign,                      Align(AlignH::LEFT, AlignV::BASELINE)),
    styleDef(instrumentChangeOffset,                     PointF()),
    styleDef(instrumentChangePlacement,                  PlacementV::ABOVE),
    styleDef(instrumentChangePosAbove,                   PointF(.0, -2.0)),
    styleDef(instrumentChangePosBelow,                   PointF(.0, 3.5)),
    styleDef(instrumentChangeMinDistance,                Spatium(0.5)),
    styleDef(instrumentChangeFrameType,                  int(FrameType::NO_FRAME)),
    styleDef(instrumentChangeFramePadding,               0.2),
    styleDef(instrumentChangeFrameWidth,                 0.1),
    styleDef(instrumentChangeFrameRound,                 0),
    styleDef(instrumentChangeFrameFgColor,               PropertyValue::fromValue(Color::BLACK)),
    styleDef(instrumentChangeFrameBgColor,               PropertyValue::fromValue(Color::transparent)),

    styleDef(stickingFontFace,                           "Edwin"),
    styleDef(stickingFontSize,                           10.0),
    styleDef(stickingLineSpacing,                        1.0),
    styleDef(stickingFontSpatiumDependent,               true),
    styleDef(stickingFontStyle,                          int(FontStyle::Normal)),
    styleDef(stickingColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(stickingAlign,                              Align(AlignH::HCENTER, AlignV::BASELINE)),
    styleDef(stickingOffset,                             PointF()),
    styleDef(stickingPlacement,                          PlacementV::BELOW),
    styleDef(stickingPosAbove,                           PointF(.0, -1.0)),
    styleDef(stickingPosBelow,                           PointF(.0, 2.0)),
    styleDef(stickingMinDistance,                        Spatium(0.5)),
    styleDef(stickingFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(stickingFramePadding,                       0.2),
    styleDef(stickingFrameWidth,                         0.1),
    styleDef(stickingFrameRound,                         0),
    styleDef(stickingFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(stickingFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),

    styleDef(figuredBassFontFace,                        "MScoreBC"),
    styleDef(figuredBassFontSize,                        8.0),
    styleDef(figuredBassLineSpacing,                     1.0),
    styleDef(figuredBassFontSpatiumDependent,            true),
    styleDef(figuredBassFontStyle,                       int(FontStyle::Normal)),
    styleDef(figuredBassColor,                           PropertyValue::fromValue(Color::BLACK)),

    styleDef(user1Name,                                  ""),
    styleDef(user1FontFace,                              "Edwin"),
    styleDef(user1FontSize,                              10.0),
    styleDef(user1LineSpacing,                           1.0),
    styleDef(user1FontSpatiumDependent,                  true),
    styleDef(user1FontStyle,                             int(FontStyle::Normal)),
    styleDef(user1Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user1Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user1Offset,                                PointF()),
    styleDef(user1OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user1FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user1FramePadding,                          0.2),
    styleDef(user1FrameWidth,                            0.1),
    styleDef(user1FrameRound,                            0),
    styleDef(user1FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user1FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user2Name,                                  ""),
    styleDef(user2FontFace,                              "Edwin"),
    styleDef(user2FontSize,                              10.0),
    styleDef(user2LineSpacing,                           1.0),
    styleDef(user2FontSpatiumDependent,                  true),
    styleDef(user2FontStyle,                             int(FontStyle::Normal)),
    styleDef(user2Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user2Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user2Offset,                                PointF()),
    styleDef(user2OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user2FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user2FramePadding,                          0.2),
    styleDef(user2FrameWidth,                            0.1),
    styleDef(user2FrameRound,                            0),
    styleDef(user2FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user2FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user3Name,                                  ""),
    styleDef(user3FontFace,                              "Edwin"),
    styleDef(user3FontSize,                              10.0),
    styleDef(user3LineSpacing,                           1.0),
    styleDef(user3FontSpatiumDependent,                  true),
    styleDef(user3FontStyle,                             int(FontStyle::Normal)),
    styleDef(user3Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user3Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user3Offset,                                PointF()),
    styleDef(user3OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user3FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user3FramePadding,                          0.2),
    styleDef(user3FrameWidth,                            0.1),
    styleDef(user3FrameRound,                            0),
    styleDef(user3FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user3FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user4Name,                                  ""),
    styleDef(user4FontFace,                              "Edwin"),
    styleDef(user4FontSize,                              10.0),
    styleDef(user4LineSpacing,                           1.0),
    styleDef(user4FontSpatiumDependent,                  true),
    styleDef(user4FontStyle,                             int(FontStyle::Normal)),
    styleDef(user4Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user4Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user4Offset,                                PointF()),
    styleDef(user4OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user4FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user4FramePadding,                          0.2),
    styleDef(user4FrameWidth,                            0.1),
    styleDef(user4FrameRound,                            0),
    styleDef(user4FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user4FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user5Name,                                  ""),
    styleDef(user5FontFace,                              "Edwin"),
    styleDef(user5FontSize,                              10.0),
    styleDef(user5LineSpacing,                           1.0),
    styleDef(user5FontSpatiumDependent,                  true),
    styleDef(user5FontStyle,                             int(FontStyle::Normal)),
    styleDef(user5Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user5Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user5Offset,                                PointF()),
    styleDef(user5OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user5FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user5FramePadding,                          0.2),
    styleDef(user5FrameWidth,                            0.1),
    styleDef(user5FrameRound,                            0),
    styleDef(user5FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user5FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user6Name,                                  ""),
    styleDef(user6FontFace,                              "Edwin"),
    styleDef(user6FontSize,                              10.0),
    styleDef(user6LineSpacing,                           1.0),
    styleDef(user6FontSpatiumDependent,                  true),
    styleDef(user6FontStyle,                             int(FontStyle::Normal)),
    styleDef(user6Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user6Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user6Offset,                                PointF()),
    styleDef(user6OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user6FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user6FramePadding,                          0.2),
    styleDef(user6FrameWidth,                            0.1),
    styleDef(user6FrameRound,                            0),
    styleDef(user6FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user6FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user7Name,                                  ""),
    styleDef(user7FontFace,                              "Edwin"),
    styleDef(user7FontSize,                              10.0),
    styleDef(user7LineSpacing,                           1.0),
    styleDef(user7FontSpatiumDependent,                  true),
    styleDef(user7FontStyle,                             int(FontStyle::Normal)),
    styleDef(user7Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user7Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user7Offset,                                PointF()),
    styleDef(user7OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user7FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user7FramePadding,                          0.2),
    styleDef(user7FrameWidth,                            0.1),
    styleDef(user7FrameRound,                            0),
    styleDef(user7FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user7FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user8Name,                                  ""),
    styleDef(user8FontFace,                              "Edwin"),
    styleDef(user8FontSize,                              10.0),
    styleDef(user8LineSpacing,                           1.0),
    styleDef(user8FontSpatiumDependent,                  true),
    styleDef(user8FontStyle,                             int(FontStyle::Normal)),
    styleDef(user8Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user8Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user8Offset,                                PointF()),
    styleDef(user8OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user8FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user8FramePadding,                          0.2),
    styleDef(user8FrameWidth,                            0.1),
    styleDef(user8FrameRound,                            0),
    styleDef(user8FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user8FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user9Name,                                  ""),
    styleDef(user9FontFace,                              "Edwin"),
    styleDef(user9FontSize,                              10.0),
    styleDef(user9LineSpacing,                           1.0),
    styleDef(user9FontSpatiumDependent,                  true),
    styleDef(user9FontStyle,                             int(FontStyle::Normal)),
    styleDef(user9Color,                                 PropertyValue::fromValue(Color::BLACK)),
    styleDef(user9Align,                                 Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user9Offset,                                PointF()),
    styleDef(user9OffsetType,                            int(OffsetType::SPATIUM)),
    styleDef(user9FrameType,                             int(FrameType::NO_FRAME)),
    styleDef(user9FramePadding,                          0.2),
    styleDef(user9FrameWidth,                            0.1),
    styleDef(user9FrameRound,                            0),
    styleDef(user9FrameFgColor,                          PropertyValue::fromValue(Color::BLACK)),
    styleDef(user9FrameBgColor,                          PropertyValue::fromValue(Color::transparent)),

    styleDef(user10Name,                                 ""),
    styleDef(user10FontFace,                             "Edwin"),
    styleDef(user10FontSize,                             10.0),
    styleDef(user10LineSpacing,                          1.0),
    styleDef(user10FontSpatiumDependent,                 true),
    styleDef(user10FontStyle,                            int(FontStyle::Normal)),
    styleDef(user10Color,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(user10Align,                                Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user10Offset,                               PointF()),
    styleDef(user10OffsetType,                           int(OffsetType::SPATIUM)),
    styleDef(user10FrameType,                            int(FrameType::NO_FRAME)),
    styleDef(user10FramePadding,                         0.2),
    styleDef(user10FrameWidth,                           0.1),
    styleDef(user10FrameRound,                           0),
    styleDef(user10FrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(user10FrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(user11Name,                                 ""),
    styleDef(user11FontFace,                             "Edwin"),
    styleDef(user11FontSize,                             10.0),
    styleDef(user11LineSpacing,                          1.0),
    styleDef(user11FontSpatiumDependent,                 true),
    styleDef(user11FontStyle,                            int(FontStyle::Normal)),
    styleDef(user11Color,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(user11Align,                                Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user11Offset,                               PointF()),
    styleDef(user11OffsetType,                           int(OffsetType::SPATIUM)),
    styleDef(user11FrameType,                            int(FrameType::NO_FRAME)),
    styleDef(user11FramePadding,                         0.2),
    styleDef(user11FrameWidth,                           0.1),
    styleDef(user11FrameRound,                           0),
    styleDef(user11FrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(user11FrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(user12Name,                                 ""),
    styleDef(user12FontFace,                             "Edwin"),
    styleDef(user12FontSize,                             10.0),
    styleDef(user12LineSpacing,                          1.0),
    styleDef(user12FontSpatiumDependent,                 true),
    styleDef(user12FontStyle,                            int(FontStyle::Normal)),
    styleDef(user12Color,                                PropertyValue::fromValue(Color::BLACK)),
    styleDef(user12Align,                                Align(AlignH::LEFT, AlignV::TOP)),
    styleDef(user12Offset,                               PointF()),
    styleDef(user12OffsetType,                           int(OffsetType::SPATIUM)),
    styleDef(user12FrameType,                            int(FrameType::NO_FRAME)),
    styleDef(user12FramePadding,                         0.2),
    styleDef(user12FrameWidth,                           0.1),
    styleDef(user12FrameRound,                           0),
    styleDef(user12FrameFgColor,                         PropertyValue::fromValue(Color::BLACK)),
    styleDef(user12FrameBgColor,                         PropertyValue::fromValue(Color::transparent)),

    styleDef(letRingFontFace,                            "Edwin"),
    styleDef(letRingFontSize,                            10.0),
    styleDef(letRingLineSpacing,                         1.0),
    styleDef(letRingFontSpatiumDependent,                true),
    styleDef(letRingFontStyle,                           int(FontStyle::Normal)),
    styleDef(letRingColor,                               PropertyValue::fromValue(Color::BLACK)),
    styleDef(letRingTextAlign,                           Align(AlignH::LEFT, AlignV::VCENTER)),
    styleDef(letRingHookHeight,                          Spatium(0.6)),
    styleDef(letRingPlacement,                           PlacementV::BELOW),
    styleDef(letRingPosAbove,                            PointF(.0, 0.0)),
    styleDef(letRingPosBelow,                            PointF(.0, 0.0)),
    styleDef(letRingLineWidth,                           Spatium(0.11)),
    styleDef(letRingLineStyle,                           PropertyValue(LineType::DASHED)),
    styleDef(letRingDashLineLen,                         5.0),
    styleDef(letRingDashGapLen,                          5.0),
    styleDef(letRingText,                                "let ring"),
    styleDef(letRingFrameType,                           int(FrameType::NO_FRAME)),
    styleDef(letRingFramePadding,                        0.2),
    styleDef(letRingFrameWidth,                          0.1),
    styleDef(letRingFrameRound,                          0),
    styleDef(letRingFrameFgColor,                        PropertyValue::fromValue(Color::BLACK)),
    styleDef(letRingFrameBgColor,                        PropertyValue::fromValue(Color::transparent)),
    styleDef(letRingEndHookType,                         HookType::HOOK_90T),

    styleDef(palmMuteFontFace,                           "Edwin"),
    styleDef(palmMuteFontSize,                           10.0),
    styleDef(palmMuteLineSpacing,                        1.0),
    styleDef(palmMuteFontSpatiumDependent,               true),
    styleDef(palmMuteFontStyle,                          int(FontStyle::Normal)),
    styleDef(palmMuteColor,                              PropertyValue::fromValue(Color::BLACK)),
    styleDef(palmMuteTextAlign,                          Align(AlignH::LEFT, AlignV::VCENTER)),
    styleDef(palmMuteHookHeight,                         Spatium(0.6)),
    styleDef(palmMutePlacement,                          PlacementV::BELOW),
    styleDef(palmMutePosAbove,                           PointF(.0, 0.0)),
    styleDef(palmMutePosBelow,                           PointF(.0, 0.0)),
    styleDef(palmMuteLineWidth,                          Spatium(0.11)),
    styleDef(palmMuteLineStyle,                          PropertyValue(LineType::DASHED)),
    styleDef(palmMuteDashLineLen,                        5.0),
    styleDef(palmMuteDashGapLen,                         5.0),
    styleDef(palmMuteText,                               "P.M."),
    styleDef(palmMuteFrameType,                          int(FrameType::NO_FRAME)),
    styleDef(palmMuteFramePadding,                       0.2),
    styleDef(palmMuteFrameWidth,                         0.1),
    styleDef(palmMuteFrameRound,                         0),
    styleDef(palmMuteFrameFgColor,                       PropertyValue::fromValue(Color::BLACK)),
    styleDef(palmMuteFrameBgColor,                       PropertyValue::fromValue(Color::transparent)),
    styleDef(palmMuteEndHookType,                        HookType::HOOK_90T),

    styleDef(fermataPosAbove,                            PointF(.0, -0.5)),
    styleDef(fermataPosBelow,                            PointF(.0, 0.5)),
    styleDef(fermataMinDistance,                         Spatium(0.4)),

    styleDef(fingeringPlacement,                         PlacementV::ABOVE),

    styleDef(articulationMinDistance,                    Spatium(0.4)),
    styleDef(fingeringMinDistance,                       Spatium(0.5)),
    styleDef(hairpinMinDistance,                         Spatium(0.7)),
    styleDef(letRingMinDistance,                         Spatium(0.7)),
    styleDef(ottavaMinDistance,                          Spatium(0.7)),
    styleDef(palmMuteMinDistance,                        Spatium(0.7)),
    styleDef(pedalMinDistance,                           Spatium(0.7)),
    styleDef(repeatMinDistance,                          Spatium(0.5)),
    styleDef(textLineMinDistance,                        Spatium(0.7)),
    styleDef(systemTextLineMinDistance,                  Spatium(0.7)),
    styleDef(trillMinDistance,                           Spatium(0.5)),
    styleDef(vibratoMinDistance,                         Spatium(1.0)),
    styleDef(voltaMinDistance,                           Spatium(1.0)),
    styleDef(figuredBassMinDistance,                     Spatium(0.5)),
    styleDef(tupletMinDistance,                          Spatium(0.5)),

    /// Display options for tab elements (simple and common styles)

    styleDef(slurShowTabSimple,                          true),
    styleDef(slurShowTabCommon,                          true),
    styleDef(fermataShowTabSimple,                       true),
    styleDef(fermataShowTabCommon,                       true),
    styleDef(dynamicsShowTabSimple,                      true),
    styleDef(dynamicsShowTabCommon,                      true),
    styleDef(hairpinShowTabSimple,                       true),
    styleDef(hairpinShowTabCommon,                       true),
    styleDef(accentShowTabSimple,                        true),
    styleDef(accentShowTabCommon,                        true),
    styleDef(staccatoShowTabSimple,                      true),
    styleDef(staccatoShowTabCommon,                      true),
    styleDef(harmonicMarkShowTabSimple,                  true),
    styleDef(harmonicMarkShowTabCommon,                  true),
    styleDef(letRingShowTabSimple,                       true),
    styleDef(letRingShowTabCommon,                       true),
    styleDef(palmMuteShowTabSimple,                      true),
    styleDef(palmMuteShowTabCommon,                      true),
    styleDef(rasgueadoShowTabSimple,                     true),
    styleDef(rasgueadoShowTabCommon,                     true),
    styleDef(mordentShowTabSimple,                       true),
    styleDef(mordentShowTabCommon,                       true),
    styleDef(turnShowTabSimple,                          true),
    styleDef(turnShowTabCommon,                          true),
    styleDef(wahShowTabSimple,                           true),
    styleDef(wahShowTabCommon,                           true),
    styleDef(golpeShowTabSimple,                         true),
    styleDef(golpeShowTabCommon,                         true),

    styleDef(tabShowTiedFret,                            int(ShowTiedFret::TIE_AND_FRET)),
    styleDef(tabParenthesizeTiedFret,                    int(ParenthesizeTiedFret::START_OF_SYSTEM)),
    styleDef(parenthesizeTiedFretIfArticulation,         true),

    styleDef(tabFretPadding,                             Spatium(0.1)),

    styleDef(chordlineThickness,                         Spatium(0.16)),

    styleDef(dummyMusicalSymbolsScale,                   1.0),

    styleDef(autoplaceEnabled,                           true),
    styleDef(defaultsVersion,                            Constants::MSC_VERSION),
    styleDef(changesBeforeBarlineRepeats,                true),
    styleDef(changesBeforeBarlineOtherJumps,             true),
    styleDef(placeClefsBeforeRepeats,                    false),
    styleDef(changesBetweenEndStartRepeat,               true),
    styleDef(showCourtesiesRepeats,                      true),
    styleDef(useParensRepeatCourtesies,                  true),
    styleDef(showCourtesiesOtherJumps,                   true),
    styleDef(useParensOtherJumpCourtesies,               true),
    styleDef(showCourtesiesAfterCancellingRepeats,       true),
    styleDef(useParensRepeatCourtesiesAfterCancelling,   true),
    styleDef(showCourtesiesAfterCancellingOtherJumps,    true),
    styleDef(useParensOtherJumpCourtesiesAfterCancelling, true),
    styleDef(smallParens,                                true),
} };

#undef styleDef
