import { JSONObject } from '@lumino/coreutils';
import { DisposableDelegate } from '@lumino/disposable';
import * as Kernel from './kernel';
import * as KernelMessage from './messages';
/**
 * Comm channel handler.
 */
export declare class CommHandler extends DisposableDelegate implements Kernel.IComm {
    /**
     * Construct a new comm channel.
     */
    constructor(target: string, id: string, kernel: Kernel.IKernelConnection, disposeCb: () => void);
    /**
     * The unique id for the comm channel.
     */
    get commId(): string;
    /**
     * The target name for the comm channel.
     */
    get targetName(): string;
    /**
     * Get the callback for a comm close event.
     *
     * #### Notes
     * This is called when the comm is closed from either the server or client.
     *
     * **See also:** [[ICommClose]], [[close]]
     */
    get onClose(): (msg: KernelMessage.ICommCloseMsg) => void | PromiseLike<void>;
    /**
     * Set the callback for a comm close event.
     *
     * #### Notes
     * This is called when the comm is closed from either the server or client. If
     * the function returns a promise, and the kernel was closed from the server,
     * kernel message processing will pause until the returned promise is
     * fulfilled.
     *
     * **See also:** [[close]]
     */
    set onClose(cb: (msg: KernelMessage.ICommCloseMsg) => void | PromiseLike<void>);
    /**
     * Get the callback for a comm message received event.
     */
    get onMsg(): (msg: KernelMessage.ICommMsgMsg) => void | PromiseLike<void>;
    /**
     * Set the callback for a comm message received event.
     *
     * #### Notes
     * This is called when a comm message is received. If the function returns a
     * promise, kernel message processing will pause until it is fulfilled.
     */
    set onMsg(cb: (msg: KernelMessage.ICommMsgMsg) => void | PromiseLike<void>);
    /**
     * Open a comm with optional data and metadata.
     *
     * #### Notes
     * This sends a `comm_open` message to the server.
     *
     * **See also:** [[ICommOpen]]
     */
    open(data?: JSONObject, metadata?: JSONObject, buffers?: (ArrayBuffer | ArrayBufferView)[]): Kernel.IShellFuture;
    /**
     * Send a `comm_msg` message to the kernel.
     *
     * #### Notes
     * This is a no-op if the comm has been closed.
     *
     * **See also:** [[ICommMsg]]
     */
    send(data: JSONObject, metadata?: JSONObject, buffers?: (ArrayBuffer | ArrayBufferView)[], disposeOnDone?: boolean): Kernel.IShellFuture;
    /**
     * Close the comm.
     *
     * #### Notes
     * This will send a `comm_close` message to the kernel, and call the
     * `onClose` callback if set.
     *
     * This is a no-op if the comm is already closed.
     *
     * **See also:** [[ICommClose]], [[onClose]]
     */
    close(data?: JSONObject, metadata?: JSONObject, buffers?: (ArrayBuffer | ArrayBufferView)[]): Kernel.IShellFuture;
    private _target;
    private _id;
    private _kernel;
    private _onClose;
    private _onMsg;
}
