﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/bedrock-agent-runtime/model/SessionStatus.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace BedrockAgentRuntime {
namespace Model {
namespace SessionStatusMapper {

static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int EXPIRED_HASH = HashingUtils::HashString("EXPIRED");
static const int ENDED_HASH = HashingUtils::HashString("ENDED");

SessionStatus GetSessionStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ACTIVE_HASH) {
    return SessionStatus::ACTIVE;
  } else if (hashCode == EXPIRED_HASH) {
    return SessionStatus::EXPIRED;
  } else if (hashCode == ENDED_HASH) {
    return SessionStatus::ENDED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<SessionStatus>(hashCode);
  }

  return SessionStatus::NOT_SET;
}

Aws::String GetNameForSessionStatus(SessionStatus enumValue) {
  switch (enumValue) {
    case SessionStatus::NOT_SET:
      return {};
    case SessionStatus::ACTIVE:
      return "ACTIVE";
    case SessionStatus::EXPIRED:
      return "EXPIRED";
    case SessionStatus::ENDED:
      return "ENDED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace SessionStatusMapper
}  // namespace Model
}  // namespace BedrockAgentRuntime
}  // namespace Aws
