/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.tools.launchers;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.elasticsearch.tools.java_version_checker.JavaVersion;
import org.elasticsearch.tools.launchers.JvmOption;
import org.elasticsearch.tools.launchers.Launchers;

final class JvmErgonomics {
    private static final Pattern SYSTEM_PROPERTY = Pattern.compile("^-D(?<key>[\\w+].*?)=(?<value>.*)$");

    private JvmErgonomics() {
        throw new AssertionError((Object)"No instances intended");
    }

    static List<String> choose(List<String> userDefinedJvmOptions) throws InterruptedException, IOException {
        ArrayList<String> ergonomicChoices = new ArrayList<String>();
        Map<String, JvmOption> finalJvmOptions = JvmOption.findFinalOptions(userDefinedJvmOptions);
        long heapSize = JvmOption.extractMaxHeapSize(finalJvmOptions);
        long maxDirectMemorySize = JvmOption.extractMaxDirectMemorySize(finalJvmOptions);
        if (System.getProperty("os.name").startsWith("Windows") && JavaVersion.majorVersion((List)JavaVersion.CURRENT) == 8) {
            Launchers.errPrintln("Warning: with JDK 8 on Windows, Elasticsearch may be unable to derive correct");
            Launchers.errPrintln("  ergonomic settings due to a JDK issue (JDK-8074459). Please use a newer");
            Launchers.errPrintln("  version of Java.");
        }
        if (maxDirectMemorySize == 0L && userDefinedJvmOptions.stream().noneMatch(s -> s.startsWith("-XX:MaxDirectMemorySize"))) {
            if (System.getProperty("os.name").startsWith("Windows") && JavaVersion.majorVersion((List)JavaVersion.CURRENT) == 8) {
                Launchers.errPrintln("Warning: MaxDirectMemorySize may have been miscalculated due to JDK-8074459.");
                Launchers.errPrintln("  Please use a newer version of Java or set MaxDirectMemorySize explicitly.");
            }
            ergonomicChoices.add("-XX:MaxDirectMemorySize=" + heapSize / 2L);
        }
        if (JavaVersion.majorVersion((List)JavaVersion.CURRENT) >= 10) {
            boolean tuneG1GCForSmallHeap = JvmErgonomics.tuneG1GCForSmallHeap(heapSize);
            boolean tuneG1GCHeapRegion = JvmErgonomics.tuneG1GCHeapRegion(finalJvmOptions, tuneG1GCForSmallHeap);
            boolean tuneG1GCInitiatingHeapOccupancyPercent = JvmErgonomics.tuneG1GCInitiatingHeapOccupancyPercent(finalJvmOptions);
            int tuneG1GCReservePercent = JvmErgonomics.tuneG1GCReservePercent(finalJvmOptions, tuneG1GCForSmallHeap);
            if (tuneG1GCHeapRegion) {
                ergonomicChoices.add("-XX:G1HeapRegionSize=4m");
            }
            if (tuneG1GCInitiatingHeapOccupancyPercent) {
                ergonomicChoices.add("-XX:InitiatingHeapOccupancyPercent=30");
            }
            if (tuneG1GCReservePercent != 0) {
                ergonomicChoices.add("-XX:G1ReservePercent=" + tuneG1GCReservePercent);
            }
        }
        return ergonomicChoices;
    }

    static boolean tuneG1GCForSmallHeap(long heapSize) {
        return heapSize < 0x200000000L;
    }

    static boolean tuneG1GCHeapRegion(Map<String, JvmOption> finalJvmOptions, boolean tuneG1GCForSmallHeap) {
        JvmOption g1GCHeapRegion = finalJvmOptions.get("G1HeapRegionSize");
        JvmOption g1GC = finalJvmOptions.get("UseG1GC");
        return tuneG1GCForSmallHeap && g1GC.getMandatoryValue().equals("true") && !g1GCHeapRegion.isCommandLineOrigin();
    }

    static int tuneG1GCReservePercent(Map<String, JvmOption> finalJvmOptions, boolean tuneG1GCForSmallHeap) {
        JvmOption g1GC = finalJvmOptions.get("UseG1GC");
        JvmOption g1GCReservePercent = finalJvmOptions.get("G1ReservePercent");
        if (g1GC.getMandatoryValue().equals("true")) {
            if (!g1GCReservePercent.isCommandLineOrigin() && tuneG1GCForSmallHeap) {
                return 15;
            }
            if (!g1GCReservePercent.isCommandLineOrigin() && !tuneG1GCForSmallHeap) {
                return 25;
            }
        }
        return 0;
    }

    static boolean tuneG1GCInitiatingHeapOccupancyPercent(Map<String, JvmOption> finalJvmOptions) {
        JvmOption g1GC = finalJvmOptions.get("UseG1GC");
        JvmOption g1GCInitiatingHeapOccupancyPercent = finalJvmOptions.get("InitiatingHeapOccupancyPercent");
        return !g1GCInitiatingHeapOccupancyPercent.isCommandLineOrigin() && g1GC.getMandatoryValue().equals("true");
    }

    static Map<String, String> extractSystemProperties(List<String> userDefinedJvmOptions) {
        HashMap<String, String> systemProperties = new HashMap<String, String>();
        for (String jvmOption : userDefinedJvmOptions) {
            Matcher matcher = SYSTEM_PROPERTY.matcher(jvmOption);
            if (!matcher.matches()) continue;
            systemProperties.put(matcher.group("key"), matcher.group("value"));
        }
        return systemProperties;
    }
}

