// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0 OR ISC

#![cfg(test)]

#[cfg(feature = "fips")]
mod fips;

use crate::key_wrap::AesKek;

use super::{BlockCipher, BlockCipherId, KeyWrap, KeyWrapPadded, AES_128, AES_256};

macro_rules! block_cipher_test {
    ($name:ident, $alg:expr, $id:expr, $key_len:literal) => {
        #[test]
        fn $name() {
            let x: &dyn BlockCipher = $alg;
            assert_eq!($id, x.id());
            assert_eq!($alg.key_len(), $key_len);
        }
    };
}

block_cipher_test!(aes_128_cipher, &AES_128, BlockCipherId::Aes128, 16);
block_cipher_test!(aes_256_cipher, &AES_256, BlockCipherId::Aes256, 32);

#[test]
fn key_encryption_key_debug_impl() {
    let kek = AesKek::new(&AES_128, &[42u8; 16]).expect("key created");

    assert_eq!(
        "KeyEncryptionKey { cipher: Aes128, .. }",
        format!("{kek:?}")
    );
}

macro_rules! nist_aes_key_wrap_test {
    ($name:ident, $alg:expr, $key:expr, $plaintext:expr, $expect:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const P: &[u8] = $plaintext;
            const C: &[u8] = $expect;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            assert_eq!($alg.id(), kek.block_cipher_id());

            let mut output = vec![0u8; C.len()];

            let wrapped = Vec::from(kek.wrap(P, &mut output).expect("wrap successful"));

            assert_eq!(wrapped, C);

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let unwrapped = kek.unwrap(&wrapped, &mut output).expect("wrap successful");

            assert_eq!(unwrapped, P);
        }
    };
}

macro_rules! nist_aes_key_wrap_with_padding_test {
    ($name:ident, $alg:expr, $key:expr, $plaintext:expr, $expect:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const P: &[u8] = $plaintext;
            const C: &[u8] = $expect;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            assert_eq!($alg.id(), kek.block_cipher_id());

            let mut output = vec![0u8; C.len()];

            let wrapped = Vec::from(
                kek.wrap_with_padding(P, &mut output)
                    .expect("wrap successful"),
            );

            assert_eq!(wrapped, C);

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let unwrapped = kek
                .unwrap_with_padding(&wrapped, &mut output)
                .expect("wrap successful");

            assert_eq!(unwrapped, P);
        }
    };
}

macro_rules! nist_aes_key_unwrap_test {
    ($name:ident, $alg:expr, $key:expr, $ciphertext:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const C: &[u8] = $ciphertext;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            kek.unwrap(C, &mut output).expect_err("unwrap to fail");
        }
    };
    ($name:ident, $alg:expr, $key:expr, $ciphertext:expr, $expect:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const C: &[u8] = $ciphertext;
            const P: &[u8] = $expect;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let unwrapped = Vec::from(kek.unwrap(C, &mut output).expect("unwrap successful"));

            assert_eq!(unwrapped, P);

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let wrapped = kek.wrap(&unwrapped, &mut output).expect("wrap successful");

            assert_eq!(wrapped, C);
        }
    };
}

macro_rules! nist_aes_key_unwrap_with_padding_test {
    ($name:ident, $alg:expr, $key:expr, $ciphertext:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const C: &[u8] = $ciphertext;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            kek.unwrap_with_padding(C, &mut output)
                .expect_err("unwrap to fail");
        }
    };
    ($name:ident, $alg:expr, $key:expr, $ciphertext:expr, $expect:expr) => {
        #[test]
        fn $name() {
            const K: &[u8] = $key;
            const C: &[u8] = $ciphertext;
            const P: &[u8] = $expect;

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let unwrapped = Vec::from(
                kek.unwrap_with_padding(C, &mut output)
                    .expect("unwrap successful"),
            );

            assert_eq!(unwrapped, P);

            let kek = AesKek::new($alg, K).expect("key creation successful");

            let mut output = vec![0u8; C.len()];

            let wrapped = kek
                .wrap_with_padding(&unwrapped, &mut output)
                .expect("wrap successful");

            assert_eq!(wrapped, C);
        }
    };
}

nist_aes_key_wrap_with_padding_test!(
    kwp_ae_aes128_8bit_len,
    &AES_128,
    &[
        0x6d, 0xec, 0xf1, 0x0a, 0x1c, 0xaf, 0x8e, 0x3b, 0x80, 0xc7, 0xa4, 0xbe, 0x8c, 0x9c, 0x84,
        0xe8,
    ],
    &[0x49],
    &[
        0x01, 0xa7, 0xd6, 0x57, 0xfc, 0x4a, 0x5b, 0x21, 0x6f, 0x26, 0x1c, 0xca, 0x4d, 0x05, 0x2c,
        0x2b,
    ]
);

nist_aes_key_wrap_with_padding_test!(
    kwp_ae_aes128_248bit_len,
    &AES_128,
    &[
        0xbe, 0x96, 0xdc, 0x19, 0x5e, 0xc0, 0x34, 0xd6, 0x16, 0x48, 0x6e, 0xd7, 0x0e, 0x97, 0xfe,
        0x83
    ],
    &[
        0x85, 0xb5, 0x43, 0x7b, 0x63, 0x35, 0xeb, 0xba, 0x76, 0x35, 0x90, 0x3a, 0x44, 0x93, 0xd1,
        0x2a, 0x77, 0xd9, 0x35, 0x7a, 0x9e, 0x0d, 0xbc, 0x01, 0x34, 0x56, 0xd8, 0x5f, 0x1d, 0x32,
        0x01
    ],
    &[
        0x97, 0x47, 0x69, 0xb3, 0xa7, 0xb4, 0xd5, 0xd3, 0x29, 0x85, 0xf8, 0x7f, 0xdd, 0xf9, 0x99,
        0x06, 0x31, 0xe5, 0x61, 0x0f, 0xbf, 0xb2, 0x78, 0x38, 0x7b, 0x58, 0xb1, 0xf4, 0x8e, 0x05,
        0xc7, 0x7d, 0x2f, 0xb7, 0x57, 0x5c, 0x51, 0x69, 0xeb, 0x0e
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes128_8bit_len,
    &AES_128,
    &[
        0x49, 0x31, 0x9c, 0x33, 0x12, 0x31, 0xcd, 0x6b, 0xf7, 0x4c, 0x2f, 0x70, 0xb0, 0x7f, 0xcc,
        0x5c
    ],
    &[
        0x9c, 0x21, 0x1f, 0x32, 0xf8, 0xb3, 0x41, 0xf3, 0x2b, 0x05, 0x2f, 0xed, 0x5f, 0x31, 0xa3,
        0x87
    ],
    &[0xe4]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes128_8bit_len_fail,
    &AES_128,
    &[
        0x7a, 0x3f, 0x4d, 0x97, 0x05, 0x01, 0xbf, 0x86, 0x14, 0x7e, 0x91, 0x5f, 0xe1, 0xb9, 0x03,
        0x18
    ],
    &[
        0xad, 0xd7, 0x0b, 0xaf, 0xaf, 0xb1, 0x5e, 0x79, 0xc3, 0xa8, 0x5c, 0xe1, 0xde, 0x55, 0x82,
        0x72
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes128_248bit_len,
    &AES_128,
    &[
        0x28, 0x90, 0x23, 0x37, 0x90, 0x78, 0xb8, 0x21, 0xfc, 0x24, 0xf7, 0x18, 0xbd, 0xc9, 0x43,
        0x31
    ],
    &[
        0xff, 0x51, 0xb7, 0xae, 0x52, 0x46, 0x23, 0x44, 0xfc, 0x45, 0x5f, 0x72, 0xbe, 0x05, 0x9b,
        0x56, 0xa9, 0x8c, 0xc8, 0x33, 0xa1, 0xcf, 0x3b, 0x20, 0xb6, 0x88, 0x71, 0x12, 0xf5, 0xa4,
        0x3f, 0xd4, 0x5e, 0x9c, 0x5f, 0x51, 0xe7, 0xc6, 0x62, 0xf4
    ],
    &[
        0xbe, 0xd5, 0x24, 0xc6, 0x40, 0x2e, 0xeb, 0x77, 0x38, 0x69, 0x6f, 0x31, 0x06, 0x99, 0x9f,
        0xc9, 0x31, 0xbe, 0xd6, 0x76, 0x88, 0x38, 0x34, 0x5d, 0x18, 0xba, 0x44, 0xe1, 0xb0, 0x32,
        0xb8
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes128_248bit_len_fail,
    &AES_128,
    &[
        0x69, 0x29, 0x11, 0x7e, 0x6c, 0xb1, 0x8e, 0xa4, 0xa2, 0x98, 0x58, 0x86, 0xf0, 0x8c, 0x0a,
        0xe1
    ],
    &[
        0x5f, 0xd9, 0xe7, 0x7c, 0x37, 0x04, 0x1c, 0x2e, 0xbd, 0x4c, 0x34, 0x6d, 0x5b, 0x6c, 0x78,
        0xf7, 0xb4, 0x85, 0xca, 0x58, 0x9d, 0x6b, 0x0b, 0x54, 0x16, 0xd0, 0x28, 0x7a, 0x6d, 0xb3,
        0x6b, 0x39, 0xbd, 0xc9, 0x61, 0xb4, 0xdc, 0x2f, 0xec, 0xbc
    ]
);

nist_aes_key_wrap_test!(
    kw_ae_aes128_128bit_len,
    &AES_128,
    &[
        0x75, 0x75, 0xda, 0x3a, 0x93, 0x60, 0x7c, 0xc2, 0xbf, 0xd8, 0xce, 0xc7, 0xaa, 0xdf, 0xd9,
        0xa6
    ],
    &[
        0x42, 0x13, 0x6d, 0x3c, 0x38, 0x4a, 0x3e, 0xea, 0xc9, 0x5a, 0x06, 0x6f, 0xd2, 0x8f, 0xed,
        0x3f
    ],
    &[
        0x03, 0x1f, 0x6b, 0xd7, 0xe6, 0x1e, 0x64, 0x3d, 0xf6, 0x85, 0x94, 0x81, 0x6f, 0x64, 0xca,
        0xa3, 0xf5, 0x6f, 0xab, 0xea, 0x25, 0x48, 0xf5, 0xfb
    ]
);

nist_aes_key_wrap_test!(
    kw_ae_aes128_256bit_len,
    &AES_128,
    &[
        0xe5, 0xd0, 0x58, 0xe7, 0xf1, 0xc2, 0x2c, 0x01, 0x6c, 0x4e, 0x1c, 0xc9, 0xb2, 0x6b, 0x9f,
        0x8f
    ],
    &[
        0x7f, 0x60, 0x4e, 0x9b, 0x8d, 0x39, 0xd3, 0xc9, 0x1e, 0x19, 0x3f, 0xe6, 0xf1, 0x96, 0xc1,
        0xe3, 0xda, 0x62, 0x11, 0xa7, 0xc9, 0xa3, 0x3b, 0x88, 0x73, 0xb6, 0x4b, 0x13, 0x8d, 0x18,
        0x03, 0xe4
    ],
    &[
        0x60, 0xb9, 0xf8, 0xac, 0x79, 0x7c, 0x56, 0xe0, 0x1e, 0x9b, 0x5f, 0x84, 0xd6, 0x58, 0x16,
        0xa9, 0x80, 0x77, 0x78, 0x69, 0xf6, 0x79, 0x91, 0xa0, 0xe6, 0xdc, 0x19, 0xb8, 0xcd, 0x75,
        0xc9, 0xb5, 0x4d, 0xb4, 0xa3, 0x84, 0x56, 0xbb, 0xd6, 0xf3
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes128_128bit_len,
    &AES_128,
    &[
        0x1c, 0xbd, 0x2f, 0x79, 0x07, 0x8b, 0x95, 0x00, 0xfa, 0xe2, 0x36, 0x96, 0x31, 0x19, 0x53,
        0xeb
    ],
    &[
        0xec, 0xbd, 0x7a, 0x17, 0xc5, 0xda, 0x3c, 0xfd, 0xfe, 0x22, 0x25, 0xd2, 0xbf, 0x9a, 0xc7,
        0xab, 0xce, 0x78, 0xc2, 0xb2, 0xae, 0xfa, 0x6e, 0xac
    ],
    &[
        0x9c, 0x4e, 0x67, 0x52, 0x77, 0xa3, 0xbd, 0xc3, 0xa0, 0x71, 0x04, 0x8b, 0x32, 0x7a, 0x01,
        0x1e
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes128_128bit_len_fail,
    &AES_128,
    &[
        0x5e, 0xa3, 0x0c, 0x21, 0xdb, 0x36, 0xc0, 0x57, 0x72, 0x94, 0xcc, 0x70, 0xd3, 0xb8, 0x69,
        0x70
    ],
    &[
        0x37, 0xe4, 0x81, 0x3d, 0x9c, 0x40, 0xc9, 0x16, 0x5b, 0x7f, 0x12, 0x0c, 0xec, 0x34, 0xa8,
        0x5d, 0x3b, 0xf5, 0x6a, 0xe0, 0x7f, 0xad, 0x8f, 0x40
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes128_256bit_len,
    &AES_128,
    &[
        0x83, 0xda, 0x6e, 0x02, 0x40, 0x4d, 0x5a, 0xbf, 0xd4, 0x7d, 0x15, 0xda, 0x59, 0x18, 0x40,
        0xe2
    ],
    &[
        0x3f, 0x4c, 0xbf, 0x3a, 0x98, 0x02, 0x92, 0x43, 0xda, 0x87, 0xa7, 0x56, 0xb3, 0xc5, 0x25,
        0x53, 0xf9, 0x13, 0x66, 0xf4, 0xff, 0x4b, 0x10, 0x3b, 0x2c, 0x73, 0xe6, 0x8a, 0xa8, 0xca,
        0x81, 0xf0, 0x1e, 0xbd, 0xa3, 0x5d, 0x71, 0x87, 0x41, 0xac
    ],
    &[
        0x67, 0xdf, 0xd6, 0x27, 0x34, 0x6e, 0xbd, 0x21, 0x78, 0x49, 0xa5, 0xba, 0x5b, 0xca, 0x6e,
        0x9c, 0xe0, 0x7a, 0x77, 0x47, 0xbe, 0xd1, 0xba, 0x11, 0x9e, 0xc0, 0x15, 0x03, 0x20, 0x2a,
        0x07, 0x5a
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes128_256bit_len_fail,
    &AES_128,
    &[
        0x84, 0xbc, 0x6c, 0xe7, 0xee, 0x4f, 0xd9, 0xdb, 0x51, 0x25, 0x36, 0x66, 0x9d, 0x06, 0x86,
        0xda
    ],
    &[
        0xc3, 0x83, 0xdb, 0x93, 0x0f, 0xfd, 0x02, 0xc0, 0x07, 0x3a, 0xc2, 0xcc, 0x79, 0xec, 0x28,
        0x9e, 0x68, 0x66, 0xbd, 0xcc, 0x6a, 0x13, 0x5a, 0x3b, 0x77, 0x6a, 0xa4, 0x2f, 0x14, 0xee,
        0x04, 0xf9, 0xcc, 0xa0, 0x6e, 0xd6, 0xc0, 0xb2, 0x29, 0x01
    ]
);

nist_aes_key_wrap_with_padding_test!(
    kwp_ae_aes256_8bit_len,
    &AES_256,
    &[
        0x95, 0xda, 0x27, 0x00, 0xca, 0x6f, 0xd9, 0xa5, 0x25, 0x54, 0xee, 0x2a, 0x8d, 0xf1, 0x38,
        0x6f, 0x5b, 0x94, 0xa1, 0xa6, 0x0e, 0xd8, 0xa4, 0xae, 0xf6, 0x0a, 0x8d, 0x61, 0xab, 0x5f,
        0x22, 0x5a
    ],
    &[0xd1],
    &[
        0x06, 0xba, 0x7a, 0xe6, 0xf3, 0x24, 0x8c, 0xfd, 0xcf, 0x26, 0x75, 0x07, 0xfa, 0x00, 0x1b,
        0xc4
    ]
);

nist_aes_key_wrap_with_padding_test!(
    kwp_ae_aes256_248bit_len,
    &AES_256,
    &[
        0xe9, 0xbb, 0x7f, 0x44, 0xc7, 0xba, 0xaf, 0xbf, 0x39, 0x2a, 0xb9, 0x12, 0x58, 0x9a, 0x2f,
        0x8d, 0xb5, 0x32, 0x68, 0x10, 0x6e, 0xaf, 0xb7, 0x46, 0x89, 0xbb, 0x18, 0x33, 0x13, 0x6e,
        0x61, 0x13
    ],
    &[
        0xff, 0xe9, 0x52, 0x60, 0x48, 0x34, 0xbf, 0xf8, 0x99, 0xe6, 0x36, 0x58, 0xf3, 0x42, 0x46,
        0x81, 0x5c, 0x91, 0x59, 0x7e, 0xb4, 0x0a, 0x21, 0x72, 0x9e, 0x0a, 0x8a, 0x95, 0x9b, 0x61,
        0xf2
    ],
    &[
        0x15, 0xb9, 0xf0, 0x6f, 0xbc, 0x76, 0x5e, 0x5e, 0x3d, 0x55, 0xd6, 0xb8, 0x24, 0x61, 0x6f,
        0x21, 0x92, 0x1d, 0x2a, 0x69, 0x18, 0xee, 0x7b, 0xf1, 0x40, 0x6b, 0x52, 0x42, 0x74, 0xe1,
        0x70, 0xb4, 0xa7, 0x83, 0x33, 0xca, 0x5e, 0xe9, 0x2a, 0xf5
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes256_8bit_len,
    &AES_256,
    &[
        0x20, 0xe4, 0xff, 0x6a, 0x88, 0xff, 0xa9, 0xa2, 0x81, 0x8b, 0x81, 0x70, 0x27, 0x93, 0xd8,
        0xa0, 0x16, 0x72, 0x2c, 0x2f, 0xa1, 0xff, 0x44, 0x5f, 0x24, 0xb9, 0xdb, 0x29, 0x3c, 0xb1,
        0x20, 0x69
    ],
    &[
        0x85, 0x01, 0x1d, 0xc9, 0x27, 0xb1, 0x67, 0xf4, 0x11, 0xb0, 0xb8, 0xe2, 0x1b, 0x11, 0xd8,
        0x19
    ],
    &[0xd2]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes256_8bit_len_fail,
    &AES_256,
    &[
        0xc3, 0x2c, 0xb3, 0xe1, 0xe4, 0x1a, 0x4b, 0x9f, 0x4d, 0xe7, 0x99, 0x89, 0x95, 0x78, 0x66,
        0xf5, 0xdd, 0x48, 0xdb, 0xa3, 0x8c, 0x22, 0xa6, 0xeb, 0xb8, 0x0e, 0x14, 0xc8, 0x4b, 0xdd,
        0x95, 0x34
    ],
    &[
        0xc2, 0x9b, 0x05, 0xc2, 0x61, 0x9a, 0x58, 0xec, 0xc1, 0xd2, 0x39, 0xe7, 0xa3, 0x42, 0x73,
        0xcd
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes256_248bit_len,
    &AES_256,
    &[
        0x09, 0xab, 0x42, 0x86, 0xa8, 0x45, 0xc1, 0x8b, 0xb4, 0x81, 0xda, 0x91, 0xc3, 0x9a, 0x58,
        0xfd, 0x52, 0xed, 0x78, 0xd5, 0x49, 0x73, 0xfc, 0x41, 0xf2, 0x51, 0x63, 0xa0, 0xc3, 0x3f,
        0x47, 0x27
    ],
    &[
        0x0a, 0x18, 0x0a, 0x84, 0xb0, 0x1f, 0xc1, 0xe4, 0x4b, 0x9f, 0x93, 0x01, 0xcc, 0x89, 0xaf,
        0x95, 0xde, 0x75, 0x82, 0x19, 0x01, 0x5a, 0xbc, 0x86, 0xc3, 0xe4, 0x8e, 0x76, 0x4e, 0x73,
        0x79, 0x24, 0x6a, 0xe7, 0x20, 0x9a, 0xaa, 0x4f, 0x88, 0x9d
    ],
    &[
        0x4c, 0x1b, 0x6a, 0xcc, 0xb4, 0x92, 0xc8, 0x8b, 0x10, 0xa5, 0x6a, 0x56, 0xeb, 0x9b, 0x6d,
        0x6e, 0xd9, 0x79, 0x70, 0x56, 0xa5, 0x59, 0xfe, 0x3f, 0x0c, 0x7c, 0x04, 0x29, 0xa2, 0x00,
        0xaf
    ]
);

nist_aes_key_unwrap_with_padding_test!(
    kwp_ad_aes256_248bit_len_fail,
    &AES_256,
    &[
        0x8c, 0x35, 0xfb, 0x77, 0x76, 0x6d, 0x04, 0xf4, 0x8d, 0x5b, 0x52, 0x27, 0x5c, 0x5c, 0x5f,
        0x31, 0xf5, 0x68, 0x07, 0x84, 0x19, 0xe5, 0xc2, 0x33, 0x59, 0x18, 0x96, 0x5f, 0xbe, 0x53,
        0xce, 0xdd
    ],
    &[
        0xba, 0xcc, 0xcb, 0x17, 0x14, 0xdb, 0xaa, 0x49, 0x08, 0xc2, 0x65, 0x4a, 0xa8, 0xdb, 0xb1,
        0xdd, 0xbd, 0xdd, 0x8a, 0xb8, 0x19, 0x42, 0x9b, 0x02, 0x66, 0x19, 0xfb, 0x1c, 0x0f, 0xa7,
        0x5a, 0x82, 0x47, 0x37, 0x2b, 0x2f, 0xee, 0xab, 0x1e, 0x1d
    ]
);

nist_aes_key_wrap_test!(
    kw_ae_aes256_128bit_len,
    &AES_256,
    &[
        0xf5, 0x97, 0x82, 0xf1, 0xdc, 0xeb, 0x05, 0x44, 0xa8, 0xda, 0x06, 0xb3, 0x49, 0x69, 0xb9,
        0x21, 0x2b, 0x55, 0xce, 0x6d, 0xcb, 0xdd, 0x09, 0x75, 0xa3, 0x3f, 0x4b, 0x3f, 0x88, 0xb5,
        0x38, 0xda
    ],
    &[
        0x73, 0xd3, 0x30, 0x60, 0xb5, 0xf9, 0xf2, 0xeb, 0x57, 0x85, 0xc0, 0x70, 0x3d, 0xdf, 0xa7,
        0x04
    ],
    &[
        0x2e, 0x63, 0x94, 0x6e, 0xa3, 0xc0, 0x90, 0x90, 0x2f, 0xa1, 0x55, 0x83, 0x75, 0xfd, 0xb2,
        0x90, 0x77, 0x42, 0xac, 0x74, 0xe3, 0x94, 0x03, 0xfc
    ]
);

nist_aes_key_wrap_test!(
    kw_ae_aes256_256bit_len,
    &AES_256,
    &[
        0x8b, 0x54, 0xe6, 0xbc, 0x3d, 0x20, 0xe8, 0x23, 0xd9, 0x63, 0x43, 0xdc, 0x77, 0x6c, 0x0d,
        0xb1, 0x0c, 0x51, 0x70, 0x8c, 0xee, 0xcc, 0x9a, 0x38, 0xa1, 0x4b, 0xeb, 0x4c, 0xa5, 0xb8,
        0xb2, 0x21
    ],
    &[
        0xd6, 0x19, 0x26, 0x35, 0xc6, 0x20, 0xde, 0xe3, 0x05, 0x4e, 0x09, 0x63, 0x39, 0x6b, 0x26,
        0x0a, 0xf5, 0xc6, 0xf0, 0x26, 0x95, 0xa5, 0x20, 0x5f, 0x15, 0x95, 0x41, 0xb4, 0xbc, 0x58,
        0x4b, 0xac
    ],
    &[
        0xb1, 0x3e, 0xeb, 0x76, 0x19, 0xfa, 0xb8, 0x18, 0xf1, 0x51, 0x92, 0x66, 0x51, 0x6c, 0xeb,
        0x82, 0xab, 0xc0, 0xe6, 0x99, 0xa7, 0x15, 0x3c, 0xf2, 0x6e, 0xdc, 0xb8, 0xae, 0xb8, 0x79,
        0xf4, 0xc0, 0x11, 0xda, 0x90, 0x68, 0x41, 0xfc, 0x59, 0x56
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes256_128bit_len,
    &AES_256,
    &[
        0x80, 0xaa, 0x99, 0x73, 0x27, 0xa4, 0x80, 0x6b, 0x6a, 0x7a, 0x41, 0xa5, 0x2b, 0x86, 0xc3,
        0x71, 0x03, 0x86, 0xf9, 0x32, 0x78, 0x6e, 0xf7, 0x96, 0x76, 0xfa, 0xfb, 0x90, 0xb8, 0x26,
        0x3c, 0x5f
    ],
    &[
        0x42, 0x3c, 0x96, 0x0d, 0x8a, 0x2a, 0xc4, 0xc1, 0xd3, 0x3d, 0x3d, 0x97, 0x7b, 0xf0, 0xa9,
        0x15, 0x59, 0xf9, 0x9c, 0x8a, 0xcd, 0x29, 0x3d, 0x43
    ],
    &[
        0x0a, 0x25, 0x6b, 0xa7, 0x5c, 0xfa, 0x03, 0xaa, 0xa0, 0x2b, 0xa9, 0x42, 0x03, 0xf1, 0x5b,
        0xaa
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes256_128bit_len_fail,
    &AES_256,
    &[
        0x08, 0xc9, 0x36, 0xb2, 0x5b, 0x56, 0x7a, 0x0a, 0xa6, 0x79, 0xc2, 0x9f, 0x20, 0x1b, 0xf8,
        0xb1, 0x90, 0x32, 0x7d, 0xf0, 0xc2, 0x56, 0x3e, 0x39, 0xce, 0xe0, 0x61, 0xf1, 0x49, 0xf4,
        0xd9, 0x1b
    ],
    &[
        0xe2, 0x27, 0xeb, 0x8a, 0xe9, 0xd2, 0x39, 0xcc, 0xd8, 0x92, 0x8a, 0xde, 0xc3, 0x9c, 0x28,
        0x81, 0x0c, 0xa9, 0xb3, 0xdc, 0x1f, 0x36, 0x64, 0x44
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes256_256bit_len,
    &AES_256,
    &[
        0x04, 0x9c, 0x7b, 0xcb, 0xa0, 0x3e, 0x04, 0x39, 0x5c, 0x2a, 0x22, 0xe6, 0xa9, 0x21, 0x5c,
        0xda, 0xe0, 0xf7, 0x62, 0xb0, 0x77, 0xb1, 0x24, 0x4b, 0x44, 0x31, 0x47, 0xf5, 0x69, 0x57,
        0x99, 0xfa
    ],
    &[
        0x77, 0x6b, 0x1e, 0x91, 0xe9, 0x35, 0xd1, 0xf8, 0x0a, 0x53, 0x79, 0x02, 0x18, 0x6d, 0x6b,
        0x00, 0xdf, 0xc6, 0xaf, 0xc1, 0x20, 0x00, 0xf1, 0xbd, 0xe9, 0x13, 0xdf, 0x5d, 0x67, 0x40,
        0x70, 0x61, 0xdb, 0x82, 0x27, 0xfc, 0xd0, 0x89, 0x53, 0xd4
    ],
    &[
        0xe6, 0x17, 0x83, 0x1c, 0x7d, 0xb8, 0x03, 0x8f, 0xda, 0x4c, 0x59, 0x40, 0x37, 0x75, 0xc3,
        0xd4, 0x35, 0x13, 0x6a, 0x56, 0x6f, 0x35, 0x09, 0xc2, 0x73, 0xe1, 0xda, 0x1e, 0xf9, 0xf5,
        0x0a, 0xea
    ]
);

nist_aes_key_unwrap_test!(
    kw_ad_aes256_256bit_len_fail,
    &AES_256,
    &[
        0x3c, 0x7c, 0x55, 0x9f, 0xb9, 0x9d, 0x2e, 0x3f, 0x82, 0x80, 0xc9, 0xbe, 0x14, 0xb0, 0xf7,
        0xb6, 0x76, 0xa3, 0x20, 0x53, 0xeb, 0xa8, 0xf7, 0xaf, 0xbb, 0x43, 0x04, 0xc1, 0x17, 0xa6,
        0x50, 0x69
    ],
    &[
        0x86, 0x1b, 0x0a, 0x15, 0xbf, 0x59, 0x07, 0xb5, 0x51, 0xbc, 0x94, 0x82, 0xbc, 0x4d, 0xe3,
        0x61, 0xde, 0x64, 0x5f, 0x18, 0xf9, 0x7f, 0xd8, 0x0f, 0xff, 0xa5, 0xb9, 0x68, 0x79, 0x23,
        0x82, 0x59, 0xc6, 0x67, 0x7e, 0xb5, 0x05, 0x96, 0x20, 0x5b
    ]
);

macro_rules! wrap_input_output_invalid_test {
    ($name:ident, $input_len:expr, $output_len:expr) => {
        #[test]
        fn $name() {
            let kek = AesKek::new(&AES_128, &[16u8; 16]).expect("key creation successful");

            let input_len: usize = $input_len.try_into().unwrap();
            let output_len: usize = $output_len.try_into().unwrap();

            let input = vec![42u8; input_len];
            let mut output = vec![0u8; output_len];

            kek.wrap(input.as_slice(), output.as_mut_slice())
                .expect_err("failure");
        }
    };
}

// Input length < 16
wrap_input_output_invalid_test!(wrap_input_len_less_than_min, 15, 23);

// Input length % 8 != 0
wrap_input_output_invalid_test!(wrap_input_len_not_multiple_of_eight, 17, 25);

// Output length < Input length - 8
wrap_input_output_invalid_test!(wrap_output_len_too_small, 16, 8);

macro_rules! unwrap_input_output_invalid_test {
    ($name:ident, $input_len:expr, $output_len:expr) => {
        #[test]
        fn $name() {
            let kek = AesKek::new(&AES_128, &[16u8; 16]).expect("key creation successful");

            let input_len: usize = $input_len.try_into().unwrap();
            let output_len: usize = $output_len.try_into().unwrap();

            let input = vec![42u8; input_len];
            let mut output = vec![0u8; output_len];

            kek.unwrap(input.as_slice(), output.as_mut_slice())
                .expect_err("failure");
        }
    };
}

// Input length < 24
unwrap_input_output_invalid_test!(unwrap_input_len_smaller_than_min, 16, 16);

// Input length % 8 != 0
unwrap_input_output_invalid_test!(unwrap_input_len_not_multiple_of_eight, 31, 31);

// Output length < Input length - 8
unwrap_input_output_invalid_test!(unwrap_output_len_too_small, 24, 15);

macro_rules! wrap_with_padding_input_output_invalid_test {
    ($name:ident, $input_len:expr, $output_len:expr) => {
        #[test]
        fn $name() {
            let kek = AesKek::new(&AES_128, &[16u8; 16]).expect("key creation successful");

            let input_len: usize = $input_len.try_into().unwrap();
            let output_len: usize = $output_len.try_into().unwrap();

            let input = vec![42u8; input_len];
            let mut output = vec![0u8; output_len];

            kek.wrap_with_padding(input.as_slice(), output.as_mut_slice())
                .expect_err("failure");
        }
    };
}

// Input length == 0
wrap_with_padding_input_output_invalid_test!(wrap_with_padding_input_len_zero, 0, 16);

// Output length is not sufficent for required padding
// In this example an input length of 6 would require 2 additional bytes of padding, plus the additional
// 8 bytes from the wrapping algorithm (So minimum of 16 bytes).
wrap_with_padding_input_output_invalid_test!(wrap_with_padding_output_len_too_small, 6, 15);

macro_rules! unwrap_with_padding_input_output_invalid_test {
    ($name:ident, $input_len:expr, $output_len:expr) => {
        #[test]
        fn $name() {
            let kek = AesKek::new(&AES_128, &[16u8; 16]).expect("key creation successful");

            let input_len: usize = $input_len.try_into().unwrap();
            let output_len: usize = $output_len.try_into().unwrap();

            let input = vec![42u8; input_len];
            let mut output = vec![0u8; output_len];

            kek.unwrap_with_padding(input.as_slice(), output.as_mut_slice())
                .expect_err("failure");
        }
    };
}

// Input length < 16 (AES Block Length)
unwrap_with_padding_input_output_invalid_test!(unwrap_padded_input_len_smaller_than_min, 15, 15);

// Output length < Input length - 8
unwrap_with_padding_input_output_invalid_test!(unwrap_padded_output_len_too_small, 24, 15);
