//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// INCL++ intra-nuclear cascade model
// Alain Boudard, CEA-Saclay, France
// Joseph Cugnon, University of Liege, Belgium
// Jean-Christophe David, CEA-Saclay, France
// Pekka Kaitaniemi, CEA-Saclay, France, and Helsinki Institute of Physics, Finland
// Sylvie Leray, CEA-Saclay, France
// Davide Mancusi, CEA-Saclay, France
//
#define INCLXX_IN_GEANT4_MODE 1

#include "globals.hh"

#include "G4INCLNpiToSKChannel.hh"
#include "G4INCLKinematicsUtils.hh"
#include "G4INCLBinaryCollisionAvatar.hh"
#include "G4INCLCrossSections.hh"
#include "G4INCLRandom.hh"
#include "G4INCLGlobals.hh"
#include "G4INCLLogger.hh"
#include <algorithm>
#include "G4INCLPhaseSpaceGenerator.hh"

namespace G4INCL {
	
	NpiToSKChannel::NpiToSKChannel(Particle *p1, Particle *p2)
		: particle1(p1), particle2(p2)
		{}
	
	NpiToSKChannel::~NpiToSKChannel(){}
	
	void NpiToSKChannel::fillFinalState(FinalState *fs) {
		
		Particle *nucleon;
		Particle *pion;
		
		if(particle1->isNucleon()){
			nucleon = particle1;
			pion = particle2;
		}
		else{
			nucleon = particle2;
			pion = particle1;
		}
		
		const G4int iso = ParticleTable::getIsospin(nucleon->getType()) + ParticleTable::getIsospin(pion->getType());
		
		G4double SmKpCX = CrossSections::p_pimToSmKp(pion,nucleon);
		G4double SzKzCX = CrossSections::p_pimToSzKz(pion,nucleon); // == piz p -> SpKzCX
		G4double SzKpCX = CrossSections::p_pizToSzKp(pion,nucleon);
		G4double rdm = Random::shoot();
		
		ParticleType SigmaType, KaonType;
		
		G4int Channel = 99;
		
		if(iso == 3 || iso == -3){
			SigmaType = ParticleTable::getSigmaType(2*iso/3);
			KaonType = ParticleTable::getKaonType(iso/3);
			Channel = 1; // "3/2" // Measured
		}
		else if(pion->getType() == PiZero){
			if(rdm*(SzKpCX+SzKzCX) < SzKzCX){
				SigmaType = ParticleTable::getSigmaType(2*iso);
				KaonType = ParticleTable::getKaonType(-iso); 
			    Channel = 0; // sqrt(1/2)*"3/2"+sqrt(1/2)*"1/2" // OK
			}
			else{
				SigmaType = SigmaZero;
				KaonType = ParticleTable::getKaonType(iso);
			    Channel = 0; // sqrt(2/3)*"3/2"+sqrt(1/3)*"1/2" // Estimate similar with p pi- -> S0 K0
			}
		}
		else{
			if(rdm*(SmKpCX+SzKzCX) < SzKzCX){
				SigmaType = SigmaZero;
				KaonType = ParticleTable::getKaonType(iso);
			    Channel = 0; // sqrt(1/2)*"3/2"+sqrt(1/2)*"1/2" // Measured
			}
			else{
				SigmaType = ParticleTable::getSigmaType(2*iso);
				KaonType = ParticleTable::getKaonType(-iso);
			    Channel = 0; // sqrt(1/3)*"3/2"+sqrt(2/3)*"1/2" // Estimate similar with p pi- -> S0 K0
			}
		}
		
		
		ThreeVector mom_hyperon = KaonMomentum(pion, nucleon, Channel);
		
		nucleon->setType(SigmaType);
		pion->setType(KaonType);
		
		G4double norm = KinematicsUtils::momentumInCM(pion,nucleon);

		nucleon->setMomentum(mom_hyperon*norm);
		pion->setMomentum(-mom_hyperon*norm);

		nucleon->adjustEnergyFromMomentum();
		pion->adjustEnergyFromMomentum();
		
#ifdef INCLXX_IN_GEANT4_MODE
		// Erase the parent resonance information of the nucleon and pion
		nucleon->setParentResonancePDGCode(0);
		nucleon->setParentResonanceID(0);
		pion->setParentResonancePDGCode(0);
		pion->setParentResonanceID(0);
#endif
		
		fs->addModifiedParticle(nucleon);
		fs->addModifiedParticle(pion);
		
	}
	
	ThreeVector NpiToSKChannel::KaonMomentum(Particle const * const pion, Particle const * const nucleon, G4int WhichChannel){
		
		const G4double pLab = KinematicsUtils::momentumInLab(pion,nucleon);
		
		G4double min = 1040., max = 2375;
		if(WhichChannel == 1){
			min = 1105.;
			max = 2475;
		}
		
		if(pLab < min) return Random::normVector(); // isotropic
		
		G4double cos_theta = 1.;
		G4double sin_theta = 0.;
		const G4double cos_phi = std::cos(Random::shoot()*Math::twoPi);
		const G4double sin_phi = std::sqrt(1-cos_phi*cos_phi);
		
		const G4double x = pion->getMomentum().getX();
		const G4double y = pion->getMomentum().getY();
		const G4double z = pion->getMomentum().getZ();
		
		const G4double r = std::sqrt(x*x+y*y+z*z);
		const G4double rho = std::sqrt(x*x+y*y);
		
		if(pLab >= max){
			const G4double b = 12. * pLab/2375.; // correspond to the forward slope description at 2375 MeV/c in K-p elastic
			cos_theta = std::log(Random::shoot()*(std::exp(b)-std::exp(-b))+std::exp(-b))/b;
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
			
		}
		else{
			const G4double Legendre_coef[2][275][9] = {
				{// pi- p -> Sigma0 K0
				{1040,0.18188,-0.04656,-0.07595,0.01513,0.00374,0.00201,0.,0.},
				{1045,0.16386,-0.02812,-0.07075,0.01418,0.00298,0.00194,0.,0.},
				{1050,0.14587,-0.00965,-0.06554,0.01323,0.00222,0.00187,0.,0.},
				{1055,0.12793,0.00891,-0.0603,0.01228,0.00146,0.00179,0.,0.},
				{1060,0.11008,0.02758,-0.05502,0.01132,7e-04,0.00172,0.,0.},
				{1065,0.09234,0.0464,-0.0497,0.01034,-7e-05,0.00165,0.,0.},
				{1070,0.07475,0.06543,-0.04431,0.00936,-0.00084,0.00157,0.,0.},
				{1075,0.05733,0.08469,-0.03886,0.00836,-0.00161,0.00149,0.,0.},
				{1080,0.04011,0.10423,-0.03331,0.00735,-0.00238,0.00141,0.,0.},
				{1085,0.02312,0.12408,-0.02767,0.00631,-0.00317,0.00133,0.,0.},
				{1090,0.00638,0.14429,-0.02192,0.00525,-0.00395,0.00124,0.,0.},
				{1095,-0.01007,0.1649,-0.01605,0.00417,-0.00474,0.00116,0.,0.},
				{1100,-0.02621,0.18591,-0.01004,0.00306,-0.00554,0.00106,0.,0.},
				{1105,-0.04203,0.20727,-0.00389,0.00193,-0.00635,0.00097,0.,0.},
				{1110,-0.05753,0.22885,0.00243,0.00076,-0.00716,0.00086,0.,0.},
				{1115,-0.0727,0.25057,0.00893,-0.00045,-0.00798,0.00076,0.,0.},
				{1120,-0.08753,0.27232,0.01561,-0.0017,-0.0088,0.00065,0.,0.},
				{1125,-0.10203,0.29401,0.02251,-0.00298,-0.00963,0.00053,0.,0.},
				{1130,-0.11619,0.31553,0.02963,-0.00432,-0.01046,4e-04,0.,0.},
				{1135,-0.13,0.33678,0.03699,-0.0057,-0.0113,0.00027,0.,0.},
				{1140,-0.14346,0.35766,0.0446,-0.00713,-0.01215,0.00013,0.,0.},
				{1145,-0.15657,0.37808,0.05247,-0.00862,-0.013,-1e-05,0.,0.},
				{1150,-0.16932,0.39792,0.06063,-0.01016,-0.01385,-0.00017,0.,0.},
				{1155,-0.1817,0.4171,0.06909,-0.01177,-0.01472,-0.00033,0.,0.},
				{1160,-0.19372,0.43551,0.07785,-0.01344,-0.01558,-0.00051,0.,0.},
				{1165,-0.20536,0.45308,0.08694,-0.01518,-0.01645,-0.00069,0.,0.},
				{1170,-0.2166,0.4698,0.09635,-0.01698,-0.01732,-0.00089,0.,0.},
				{1175,-0.22743,0.48566,0.10609,-0.01885,-0.01819,-0.0011,0.,0.},
				{1180,-0.23783,0.50068,0.11617,-0.02079,-0.01905,-0.00131,0.,0.},
				{1185,-0.24777,0.51485,0.12657,-0.02281,-0.01991,-0.00154,0.,0.},
				{1190,-0.25724,0.52817,0.13732,-0.02489,-0.02076,-0.00178,0.,0.},
				{1195,-0.26622,0.54063,0.1484,-0.02705,-0.0216,-0.00204,0.,0.},
				{1200,-0.27469,0.55225,0.15982,-0.02928,-0.02242,-0.00231,0.,0.},
				{1205,-0.28263,0.56301,0.17159,-0.03158,-0.02323,-0.00259,0.,0.},
				{1210,-0.29002,0.57292,0.1837,-0.03396,-0.02402,-0.00288,0.,0.},
				{1215,-0.29685,0.58199,0.19617,-0.03642,-0.02478,-0.00319,0.,0.},
				{1220,-0.30309,0.5902,0.20898,-0.03895,-0.02552,-0.00352,0.,0.},
				{1225,-0.30872,0.59755,0.22215,-0.04156,-0.02623,-0.00386,0.,0.},
				{1230,-0.31373,0.60409,0.23567,-0.04425,-0.02691,-0.00421,0.,0.},
				{1235,-0.31816,0.60992,0.2495,-0.04702,-0.02756,-0.00459,0.,0.},
				{1240,-0.32203,0.61513,0.26362,-0.04984,-0.02815,-0.00497,0.,0.},
				{1245,-0.32535,0.61986,0.27799,-0.05274,-0.0287,-0.00537,0.,0.},
				{1250,-0.32816,0.62422,0.29257,-0.05568,-0.02919,-0.00579,0.,0.},
				{1255,-0.33049,0.62831,0.30735,-0.05868,-0.02961,-0.00622,0.,0.},
				{1260,-0.33236,0.63225,0.32228,-0.06172,-0.02996,-0.00667,0.,0.},
				{1265,-0.3338,0.63615,0.33733,-0.0648,-0.03024,-0.00713,0.,0.},
				{1270,-0.33482,0.64013,0.35246,-0.06792,-0.03042,-0.0076,0.,0.},
				{1275,-0.33547,0.6443,0.36766,-0.07106,-0.03052,-0.00809,0.,0.},
				{1280,-0.33576,0.64877,0.38287,-0.07422,-0.03051,-0.00859,0.,0.},
				{1285,-0.33573,0.65366,0.39808,-0.0774,-0.0304,-0.00911,0.,0.},
				{1290,-0.33539,0.65907,0.41324,-0.0806,-0.03018,-0.00964,0.,0.},
				{1295,-0.33477,0.6651,0.42834,-0.08379,-0.02983,-0.01018,0.,0.},
				{1300,-0.33388,0.67178,0.44333,-0.08696,-0.02935,-0.01073,0.,0.},
				{1305,-0.33274,0.67915,0.4582,-0.0901,-0.02873,-0.01129,0.,0.},
				{1310,-0.33138,0.68725,0.47292,-0.09319,-0.02795,-0.01186,0.,0.},
				{1315,-0.32979,0.69613,0.48747,-0.0962,-0.027,-0.01244,0.,0.},
				{1320,-0.328,0.70582,0.50181,-0.09912,-0.02587,-0.01302,0.,0.},
				{1325,-0.32602,0.71635,0.51593,-0.10193,-0.02455,-0.0136,0.,0.},
				{1330,-0.32387,0.72778,0.52979,-0.10461,-0.02303,-0.01419,0.,0.},
				{1335,-0.32157,0.74013,0.54338,-0.10715,-0.02129,-0.01477,0.,0.},
				{1340,-0.31911,0.75329,0.5567,-0.10955,-0.01932,-0.01535,0.,0.},
				{1345,-0.31652,0.76713,0.56975,-0.11183,-0.01711,-0.01592,0.,0.},
				{1350,-0.31379,0.78149,0.58256,-0.11401,-0.01463,-0.01648,0.,0.},
				{1355,-0.31094,0.79624,0.59512,-0.11609,-0.01188,-0.01702,0.,0.},
				{1360,-0.30797,0.81121,0.60747,-0.11811,-0.00884,-0.01754,0.,0.},
				{1365,-0.30488,0.82628,0.6196,-0.12007,-0.0055,-0.01804,0.,0.},
				{1370,-0.30169,0.84128,0.63153,-0.12199,-0.00183,-0.01851,0.,0.},
				{1375,-0.29841,0.85607,0.64327,-0.12389,0.00217,-0.01894,0.,0.},
				{1380,-0.29503,0.87051,0.65484,-0.12577,0.00653,-0.01935,0.,0.},
				{1385,-0.29157,0.88445,0.66625,-0.12767,0.01125,-0.01971,0.,0.},
				{1390,-0.28803,0.89774,0.6775,-0.12959,0.01636,-0.02002,0.,0.},
				{1395,-0.28442,0.91023,0.68862,-0.13156,0.02186,-0.02029,0.,0.},
				{1400,-0.28074,0.9218,0.69961,-0.13357,0.02778,-0.02051,0.,0.},
				{1405,-0.27699,0.9324,0.71048,-0.13564,0.03412,-0.02067,0.,0.},
				{1410,-0.27316,0.94199,0.72125,-0.13776,0.0409,-0.02076,0.,0.},
				{1415,-0.26923,0.95056,0.73193,-0.13992,0.04813,-0.02077,0.,0.},
				{1420,-0.26519,0.95808,0.74253,-0.14212,0.05582,-0.02071,0.,0.},
				{1425,-0.26103,0.96451,0.75305,-0.14435,0.06398,-0.02055,0.,0.},
				{1430,-0.25674,0.96982,0.76351,-0.1466,0.07263,-0.02031,0.,0.},
				{1435,-0.25231,0.97399,0.77393,-0.14888,0.08177,-0.01996,0.,0.},
				{1440,-0.24772,0.97699,0.7843,-0.15118,0.09142,-0.0195,0.,0.},
				{1445,-0.24297,0.9788,0.79465,-0.15349,0.10159,-0.01892,0.,0.},
				{1450,-0.23805,0.97937,0.80498,-0.15581,0.11229,-0.01821,0.,0.},
				{1455,-0.23293,0.9787,0.81531,-0.15813,0.12353,-0.01738,0.,0.},
				{1460,-0.22762,0.97675,0.82564,-0.16044,0.13533,-0.01641,0.,0.},
				{1465,-0.22209,0.97364,0.83596,-0.16272,0.14767,-0.01528,0.,0.},
				{1470,-0.21634,0.96945,0.84628,-0.16494,0.16055,-0.014,0.,0.},
				{1475,-0.21034,0.96431,0.85659,-0.16707,0.17396,-0.01255,0.,0.},
				{1480,-0.20408,0.95831,0.86688,-0.16907,0.1879,-0.01092,0.,0.},
				{1485,-0.19755,0.95157,0.87714,-0.17092,0.20235,-0.00911,0.,0.},
				{1490,-0.19073,0.9442,0.88738,-0.17258,0.21731,-0.00709,0.,0.},
				{1495,-0.18361,0.9363,0.89758,-0.17403,0.23277,-0.00486,0.,0.},
				{1500,-0.17616,0.92798,0.90773,-0.17524,0.24872,-0.00242,0.,0.},
				{1505,-0.16839,0.91935,0.91784,-0.17617,0.26515,0.00026,0.,0.},
				{1510,-0.16026,0.91051,0.92789,-0.17679,0.28206,0.00317,0.,0.},
				{1515,-0.15177,0.90158,0.93788,-0.17708,0.29943,0.00633,0.,0.},
				{1520,-0.14291,0.89266,0.9478,-0.177,0.31726,0.00976,0.,0.},
				{1525,-0.13368,0.88383,0.95762,-0.1765,0.33549,0.01345,0.,0.},
				{1530,-0.1241,0.87515,0.9673,-0.17551,0.35406,0.01741,0.,0.},
				{1535,-0.11418,0.86671,0.9768,-0.17398,0.37293,0.02167,0.,0.},
				{1540,-0.10394,0.85857,0.98609,-0.17185,0.39203,0.02622,0.,0.},
				{1545,-0.09338,0.85081,0.99514,-0.16906,0.41131,0.03107,0.,0.},
				{1550,-0.08253,0.84351,1.0039,-0.16555,0.43072,0.03623,0.,0.},
				{1555,-0.0714,0.83672,1.01235,-0.16127,0.45019,0.04172,0.,0.},
				{1560,-0.05999,0.83054,1.02044,-0.15615,0.46967,0.04754,0.,0.},
				{1565,-0.04833,0.82502,1.02813,-0.15013,0.48911,0.0537,0.,0.},
				{1570,-0.03643,0.82025,1.0354,-0.14316,0.50844,0.06021,0.,0.},
				{1575,-0.02429,0.81629,1.04221,-0.13518,0.52762,0.06708,0.,0.},
				{1580,-0.01194,0.8132,1.04852,-0.12614,0.54659,0.07431,0.,0.},
				{1585,6e-04,0.81094,1.05435,-0.11607,0.56533,0.08191,0.,0.},
				{1590,0.01333,0.80944,1.0597,-0.10499,0.58383,0.08987,0.,0.},
				{1595,0.02622,0.80866,1.06461,-0.09294,0.60206,0.0982,0.,0.},
				{1600,0.03926,0.80853,1.06907,-0.07995,0.62002,0.10688,0.,0.},
				{1605,0.05242,0.80899,1.0731,-0.06606,0.63769,0.11592,0.,0.},
				{1610,0.06569,0.80999,1.07673,-0.0513,0.65505,0.12532,0.,0.},
				{1615,0.07905,0.81145,1.07996,-0.03571,0.67209,0.13508,0.,0.},
				{1620,0.09249,0.81333,1.08281,-0.01931,0.6888,0.14519,0.,0.},
				{1625,0.10598,0.81556,1.08529,-0.00214,0.70514,0.15565,0.,0.},
				{1630,0.11951,0.81809,1.08742,0.01576,0.72112,0.16646,0.,0.},
				{1635,0.13306,0.82085,1.08922,0.03437,0.73672,0.17762,0.,0.},
				{1640,0.14662,0.82378,1.09069,0.05364,0.75191,0.18912,0.,0.},
				{1645,0.16016,0.82683,1.09186,0.07355,0.76669,0.20097,0.,0.},
				{1650,0.17367,0.82994,1.09274,0.09406,0.78104,0.21316,0.,0.},
				{1655,0.18715,0.83306,1.09333,0.11511,0.79498,0.22566,0.,0.},
				{1660,0.20061,0.83616,1.09366,0.13665,0.80851,0.23843,0.,0.},
				{1665,0.21404,0.8392,1.09373,0.15861,0.82165,0.25144,0.,0.},
				{1670,0.22745,0.84213,1.09356,0.18094,0.83442,0.26466,0.,0.},
				{1675,0.24084,0.84492,1.09315,0.20358,0.84682,0.27805,0.,0.},
				{1680,0.25423,0.84753,1.09251,0.22648,0.85888,0.29158,0.,0.},
				{1685,0.2676,0.84991,1.09167,0.24957,0.8706,0.30522,0.,0.},
				{1690,0.28096,0.85203,1.09062,0.2728,0.882,0.31892,0.,0.},
				{1695,0.29432,0.85385,1.08938,0.29611,0.8931,0.33266,0.,0.},
				{1700,0.30768,0.85532,1.08797,0.31943,0.9039,0.34641,0.,0.},
				{1705,0.32104,0.85641,1.08639,0.34272,0.91442,0.36013,0.,0.},
				{1710,0.33441,0.85709,1.08465,0.36592,0.92468,0.37378,0.,0.},
				{1715,0.34774,0.8574,1.08275,0.389,0.93468,0.38736,0.,0.},
				{1720,0.36102,0.85739,1.08069,0.41194,0.94443,0.40084,0.,0.},
				{1725,0.3742,0.85709,1.07845,0.43471,0.95393,0.41423,0.,0.},
				{1730,0.38726,0.85657,1.07604,0.45729,0.9632,0.42751,0.,0.},
				{1735,0.40016,0.85586,1.07344,0.47966,0.97223,0.44067,0.,0.},
				{1740,0.41288,0.85501,1.07065,0.5018,0.98104,0.45369,0.,0.},
				{1745,0.42537,0.85408,1.06767,0.52368,0.98963,0.46656,0.,0.},
				{1750,0.43761,0.85312,1.06448,0.54527,0.99801,0.47928,0.,0.},
				{1755,0.44956,0.85216,1.06109,0.56656,1.00619,0.49182,0.,0.},
				{1760,0.4612,0.85126,1.05749,0.58752,1.01417,0.50419,0.,0.},
				{1765,0.47248,0.85047,1.05367,0.60813,1.02196,0.51636,0.,0.},
				{1770,0.48338,0.84982,1.04962,0.62836,1.02956,0.52834,0.,0.},
				{1775,0.49387,0.84939,1.04535,0.6482,1.03699,0.54009,0.,0.},
				{1780,0.50391,0.84919,1.04084,0.66762,1.04424,0.55162,0.,0.},
				{1785,0.51352,0.84924,1.0361,0.68662,1.05131,0.56292,0.,0.},
				{1790,0.52271,0.84953,1.03113,0.70521,1.05818,0.57399,0.,0.},
				{1795,0.53149,0.85007,1.02593,0.72339,1.06484,0.58483,0.,0.},
				{1800,0.53988,0.85085,1.02049,0.74116,1.07128,0.59544,0.,0.},
				{1805,0.54788,0.85186,1.01483,0.75854,1.07748,0.6058,0.,0.},
				{1810,0.55553,0.85311,1.00895,0.77552,1.08342,0.61593,0.,0.},
				{1815,0.56281,0.85459,1.00284,0.79211,1.0891,0.62582,0.,0.},
				{1820,0.56976,0.8563,0.9965,0.80831,1.0945,0.63547,0.,0.},
				{1825,0.57638,0.85823,0.98994,0.82413,1.0996,0.64487,0.,0.},
				{1830,0.58269,0.86039,0.98317,0.83957,1.1044,0.65403,0.,0.},
				{1835,0.58871,0.86277,0.97617,0.85463,1.10887,0.66293,0.,0.},
				{1840,0.59443,0.86538,0.96896,0.86933,1.11301,0.67159,0.,0.},
				{1845,0.59989,0.86819,0.96153,0.88366,1.11681,0.68,0.,0.},
				{1850,0.60508,0.87122,0.95388,0.89763,1.12024,0.68815,0.,0.},
				{1855,0.61003,0.87447,0.94602,0.91124,1.12329,0.69605,0.,0.},
				{1860,0.61475,0.87792,0.93795,0.9245,1.12596,0.70369,0.,0.},
				{1865,0.61926,0.88158,0.92967,0.93741,1.12822,0.71107,0.,0.},
				{1870,0.62356,0.88544,0.92118,0.94998,1.13006,0.71818,0.,0.},
				{1875,0.62767,0.88951,0.91248,0.9622,1.13148,0.72504,0.,0.},
				{1880,0.6316,0.89378,0.90358,0.97409,1.13245,0.73163,0.,0.},
				{1885,0.63537,0.89833,0.89449,0.98565,1.133,0.73795,0.,0.},
				{1890,0.63898,0.90323,0.88523,0.99687,1.13314,0.744,0.,0.},
				{1895,0.64246,0.90855,0.87582,1.00776,1.13289,0.74977,0.,0.},
				{1900,0.64581,0.91436,0.86626,1.01832,1.13227,0.75527,0.,0.},
				{1905,0.64904,0.92075,0.85659,1.02854,1.1313,0.76049,0.,0.},
				{1910,0.65217,0.92778,0.84681,1.03844,1.12999,0.76543,0.,0.},
				{1915,0.65522,0.93553,0.83694,1.048,1.12837,0.77008,0.,0.},
				{1920,0.65818,0.94407,0.82701,1.05723,1.12645,0.77445,0.,0.},
				{1925,0.66108,0.95349,0.81703,1.06614,1.12425,0.77852,0.,0.},
				{1930,0.66392,0.96384,0.80701,1.07471,1.12178,0.7823,0.,0.},
				{1935,0.66673,0.97522,0.79697,1.08296,1.11907,0.78579,0.,0.},
				{1940,0.6695,0.98768,0.78693,1.09088,1.11614,0.78898,0.,0.},
				{1945,0.67226,1.00132,0.77691,1.09848,1.113,0.79186,0.,0.},
				{1950,0.67502,1.01619,0.76692,1.10575,1.10967,0.79444,0.,0.},
				{1955,0.67778,1.03238,0.75698,1.11269,1.10617,0.79672,0.,0.},
				{1960,0.68056,1.04995,0.74712,1.11931,1.10252,0.79868,0.,0.},
				{1965,0.68338,1.06899,0.73733,1.12561,1.09874,0.80033,0.,0.},
				{1970,0.68624,1.08957,0.72765,1.13158,1.09484,0.80167,0.,0.},
				{1975,0.68916,1.11176,0.71809,1.13724,1.09084,0.80269,0.,0.},
				{1980,0.69215,1.13564,0.70866,1.14257,1.08676,0.80339,0.,0.},
				{1985,0.69523,1.16128,0.69939,1.14758,1.08262,0.80376,0.,0.},
				{1990,0.6984,1.18876,0.69029,1.15227,1.07844,0.80381,0.,0.},
				{1995,0.70167,1.21815,0.68138,1.15664,1.07423,0.80353,0.,0.},
				{2000,0.70507,1.24952,0.67267,1.1607,1.07002,0.80292,0.,0.},
				{2005,0.70859,1.2829,0.66418,1.16444,1.06581,0.80198,0.,0.},
				{2010,0.71222,1.31813,0.6559,1.16793,1.06158,0.80076,0.,0.},
				{2015,0.71594,1.35499,0.64783,1.17122,1.05728,0.79932,0.,0.},
				{2020,0.71974,1.39327,0.63995,1.17438,1.05289,0.7977,0.,0.},
				{2025,0.72358,1.43275,0.63226,1.17745,1.04836,0.79597,0.,0.},
				{2030,0.72745,1.47323,0.62475,1.18051,1.04367,0.79418,0.,0.},
				{2035,0.73133,1.51448,0.6174,1.18361,1.03876,0.79238,0.,0.},
				{2040,0.73519,1.5563,0.61021,1.18681,1.03362,0.79063,0.,0.},
				{2045,0.73903,1.59846,0.60318,1.19018,1.02819,0.78898,0.,0.},
				{2050,0.74281,1.64077,0.59628,1.19377,1.02245,0.78748,0.,0.},
				{2055,0.74651,1.683,0.58951,1.19764,1.01636,0.7862,0.,0.},
				{2060,0.75013,1.72495,0.58287,1.20184,1.00989,0.78518,0.,0.},
				{2065,0.75367,1.76647,0.57638,1.20635,1.00308,0.78441,0.,0.},
				{2070,0.75714,1.80742,0.57004,1.21116,0.99597,0.78388,0.,0.},
				{2075,0.76056,1.84767,0.56388,1.21622,0.98861,0.7836,0.,0.},
				{2080,0.76395,1.88707,0.55791,1.22151,0.98104,0.78355,0.,0.},
				{2085,0.76732,1.92548,0.55216,1.227,0.97332,0.78371,0.,0.},
				{2090,0.77069,1.96276,0.54665,1.23267,0.96548,0.78409,0.,0.},
				{2095,0.77407,1.99878,0.54138,1.23849,0.95758,0.78467,0.,0.},
				{2100,0.77748,2.03339,0.53639,1.24443,0.94966,0.78545,0.,0.},
				{2105,0.78093,2.06646,0.53168,1.25046,0.94176,0.7864,0.,0.},
				{2110,0.78443,2.09785,0.52728,1.25655,0.93393,0.78754,0.,0.},
				{2115,0.78801,2.12741,0.5232,1.26267,0.92622,0.78884,0.,0.},
				{2120,0.79168,2.15503,0.51946,1.26881,0.91867,0.79029,0.,0.},
				{2125,0.79544,2.18067,0.51607,1.27496,0.91132,0.79191,0.,0.},
				{2130,0.79929,2.20434,0.51302,1.28111,0.9042,0.79369,0.,0.},
				{2135,0.80323,2.226,0.51031,1.28726,0.89735,0.79563,0.,0.},
				{2140,0.80726,2.24567,0.50795,1.29342,0.89079,0.79774,0.,0.},
				{2145,0.81139,2.26332,0.50593,1.29958,0.88457,0.80002,0.,0.},
				{2150,0.81561,2.27894,0.50426,1.30575,0.87871,0.80247,0.,0.},
				{2155,0.81994,2.29253,0.50293,1.31191,0.87325,0.80509,0.,0.},
				{2160,0.82435,2.30407,0.50195,1.31807,0.86823,0.80789,0.,0.},
				{2165,0.82887,2.31355,0.50131,1.32423,0.86368,0.81087,0.,0.},
				{2170,0.83349,2.32096,0.50102,1.33039,0.85963,0.81403,0.,0.},
				{2175,0.83821,2.3263,0.50107,1.33654,0.85612,0.81737,0.,0.},
				{2180,0.84303,2.32955,0.50147,1.34269,0.85317,0.8209,0.,0.},
				{2185,0.84796,2.33069,0.50221,1.34883,0.85083,0.82462,0.,0.},
				{2190,0.85299,2.32973,0.5033,1.35497,0.84913,0.82852,0.,0.},
				{2195,0.85813,2.32665,0.50473,1.3611,0.8481,0.83262,0.,0.},
				{2200,0.86337,2.32144,0.50652,1.36722,0.84778,0.83692,0.,0.},
				{2205,0.86873,2.31409,0.50865,1.37333,0.8482,0.84141,0.,0.},
				{2210,0.87419,2.30458,0.51112,1.37943,0.84939,0.84611,0.,0.},
				{2215,0.87976,2.29291,0.51395,1.38551,0.85138,0.851,0.,0.},
				{2220,0.88545,2.27907,0.51712,1.39159,0.85422,0.8561,0.,0.},
				{2225,0.89125,2.26305,0.52063,1.39765,0.85794,0.86141,0.,0.},
				{2230,0.89716,2.24483,0.5245,1.40369,0.86256,0.86693,0.,0.},
				{2235,0.90319,2.22441,0.52871,1.40972,0.86813,0.87267,0.,0.},
				{2240,0.90933,2.20179,0.53328,1.41573,0.87466,0.87861,0.,0.},
				{2245,0.91559,2.17703,0.53817,1.42173,0.88214,0.88476,0.,0.},
				{2250,0.92195,2.15022,0.54339,1.42771,0.89052,0.89111,0.,0.},
				{2255,0.92842,2.12143,0.54892,1.43367,0.89978,0.89766,0.,0.},
				{2260,0.93499,2.09074,0.55475,1.43962,0.90987,0.90438,0.,0.},
				{2265,0.94165,2.05824,0.56086,1.44556,0.92077,0.91129,0.,0.},
				{2270,0.94841,2.02399,0.56725,1.45148,0.93243,0.91836,0.,0.},
				{2275,0.95525,1.98809,0.5739,1.45739,0.94483,0.92559,0.,0.},
				{2280,0.96218,1.9506,0.5808,1.46328,0.95792,0.93298,0.,0.},
				{2285,0.96919,1.91161,0.58793,1.46917,0.97168,0.94051,0.,0.},
				{2290,0.97627,1.8712,0.59529,1.47504,0.98606,0.94818,0.,0.},
				{2295,0.98342,1.82945,0.60286,1.48091,1.00104,0.95598,0.,0.},
				{2300,0.99064,1.78643,0.61063,1.48676,1.01657,0.96391,0.,0.},
				{2305,0.99792,1.74223,0.61858,1.49261,1.03263,0.97194,0.,0.},
				{2310,1.00526,1.69692,0.62671,1.49844,1.04917,0.98009,0.,0.},
				{2315,1.01265,1.65058,0.635,1.50427,1.06617,0.98833,0.,0.},
				{2320,1.02009,1.6033,0.64344,1.51009,1.08358,0.99667,0.,0.},
				{2325,1.02758,1.55515,0.65201,1.51591,1.10138,1.00509,0.,0.},
				{2330,1.03511,1.50621,0.66071,1.52172,1.11953,1.01359,0.,0.},
				{2335,1.04268,1.45656,0.66952,1.52752,1.13799,1.02215,0.,0.},
				{2340,1.05028,1.40628,0.67843,1.53332,1.15672,1.03078,0.,0.},
				{2345,1.05791,1.35545,0.68743,1.53911,1.1757,1.03946,0.,0.},
				{2350,1.06556,1.30415,0.6965,1.5449,1.19489,1.04819,0.,0.},
				{2355,1.07323,1.25246,0.70563,1.55069,1.21425,1.05695,0.,0.},
				{2360,1.08093,1.20045,0.71481,1.55647,1.23375,1.06575,0.,0.},
				{2365,1.08863,1.14821,0.72403,1.56225,1.25335,1.07456,0.,0.},
				{2370,1.09634,1.09581,0.73327,1.56804,1.27302,1.0834,0.,0.},
				{2375,1.10406,1.04334,0.74253,1.57382,1.29273,1.09224,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.},
				{0.,0.,0.,0.,0.,0.,0.,0.,0.}
				},
				{// pi+ p -> Sigma+ pi+
				{1105,-0.94614,0.13062,0.15332,-0.38197,0.29933,-0.63182,0.22245,0.2047},
				{1110,-0.92514,0.10676,0.19258,-0.38582,0.29026,-0.59597,0.2148,0.19724},
				{1115,-0.90417,0.08301,0.23175,-0.38957,0.28119,-0.56031,0.20716,0.18979},
				{1120,-0.88327,0.05949,0.27072,-0.39315,0.27209,-0.52506,0.19953,0.18234},
				{1125,-0.86247,0.03632,0.30939,-0.39646,0.26297,-0.49042,0.19192,0.17492},
				{1130,-0.8418,0.01362,0.34765,-0.39941,0.25382,-0.45659,0.18434,0.16751},
				{1135,-0.82129,-0.00851,0.38541,-0.40192,0.24462,-0.42377,0.17679,0.16012},
				{1140,-0.80099,-0.02993,0.42257,-0.40389,0.23536,-0.39217,0.16929,0.15276},
				{1145,-0.78091,-0.05054,0.45902,-0.40522,0.22604,-0.36199,0.16183,0.14544},
				{1150,-0.7611,-0.07022,0.49467,-0.40585,0.21665,-0.33343,0.15443,0.13815},
				{1155,-0.74159,-0.08885,0.52941,-0.40566,0.20717,-0.3067,0.1471,0.1309},
				{1160,-0.7224,-0.10632,0.56314,-0.40458,0.19761,-0.282,0.13983,0.12371},
				{1165,-0.70354,-0.12258,0.59581,-0.40256,0.18797,-0.25936,0.13263,0.11656},
				{1170,-0.68498,-0.13762,0.62737,-0.39958,0.17827,-0.2387,0.12553,0.10948},
				{1175,-0.66669,-0.15144,0.65779,-0.39563,0.16855,-0.21996,0.11851,0.10247},
				{1180,-0.64863,-0.16402,0.68704,-0.39067,0.15883,-0.20306,0.11159,0.09553},
				{1185,-0.63078,-0.17537,0.71507,-0.38471,0.14914,-0.18792,0.10479,0.08867},
				{1190,-0.61311,-0.18548,0.74186,-0.37772,0.1395,-0.17446,0.09809,0.0819},
				{1195,-0.5956,-0.19435,0.76737,-0.36969,0.12994,-0.16261,0.09152,0.07523},
				{1200,-0.5782,-0.20196,0.79156,-0.3606,0.1205,-0.15229,0.08509,0.06867},
				{1205,-0.56089,-0.20832,0.8144,-0.35042,0.11118,-0.14342,0.07879,0.06221},
				{1210,-0.54365,-0.21341,0.83585,-0.33915,0.10203,-0.13593,0.07264,0.05588},
				{1215,-0.52644,-0.21724,0.85588,-0.32677,0.09306,-0.12974,0.06664,0.04967},
				{1220,-0.50923,-0.2198,0.87445,-0.31326,0.08431,-0.12478,0.06081,0.0436},
				{1225,-0.49204,-0.22113,0.89156,-0.29867,0.07581,-0.12094,0.05514,0.03766},
				{1230,-0.4749,-0.22129,0.90727,-0.28311,0.06762,-0.11813,0.04963,0.03185},
				{1235,-0.45787,-0.22038,0.92164,-0.26669,0.0598,-0.11623,0.04427,0.02618},
				{1240,-0.44101,-0.21848,0.93471,-0.24955,0.05241,-0.11512,0.03907,0.02064},
				{1245,-0.42437,-0.21567,0.94654,-0.2318,0.04551,-0.1147,0.03402,0.01523},
				{1250,-0.40799,-0.21204,0.95719,-0.21357,0.03915,-0.11485,0.0291,0.00995},
				{1255,-0.39194,-0.20767,0.96672,-0.19496,0.03339,-0.11546,0.02432,0.00479},
				{1260,-0.37626,-0.20265,0.97517,-0.17612,0.02829,-0.11642,0.01967,-0.00025},
				{1265,-0.36101,-0.19705,0.9826,-0.15715,0.02391,-0.11761,0.01515,-0.00517},
				{1270,-0.34624,-0.19097,0.98907,-0.13818,0.02031,-0.11893,0.01075,-0.00997},
				{1275,-0.33201,-0.18448,0.99464,-0.11933,0.01754,-0.12026,0.00646,-0.01465},
				{1280,-0.31836,-0.17767,0.99935,-0.10071,0.01566,-0.12149,0.00229,-0.01922},
				{1285,-0.30534,-0.17062,1.00326,-0.08245,0.01473,-0.12251,-0.00178,-0.02367},
				{1290,-0.29298,-0.16337,1.00644,-0.06463,0.01474,-0.12326,-0.00574,-0.02802},
				{1295,-0.28127,-0.15592,1.00894,-0.04729,0.01562,-0.12373,-0.0096,-0.03226},
				{1300,-0.27019,-0.14829,1.01082,-0.03048,0.01733,-0.1239,-0.01337,-0.0364},
				{1305,-0.25975,-0.14048,1.01214,-0.01423,0.01979,-0.12375,-0.01704,-0.04044},
				{1310,-0.24992,-0.13249,1.01296,0.0014,0.02296,-0.12327,-0.02062,-0.04439},
				{1315,-0.24071,-0.12434,1.01334,0.01636,0.02677,-0.12244,-0.0241,-0.04825},
				{1320,-0.23211,-0.11602,1.01334,0.03062,0.03117,-0.12125,-0.0275,-0.05202},
				{1325,-0.22411,-0.10755,1.01302,0.04412,0.03608,-0.11967,-0.03082,-0.0557},
				{1330,-0.21669,-0.09894,1.01244,0.05683,0.04146,-0.11769,-0.03405,-0.05931},
				{1335,-0.20985,-0.09018,1.01164,0.06873,0.04726,-0.1153,-0.0372,-0.06284},
				{1340,-0.20356,-0.08129,1.01058,0.07985,0.0534,-0.11252,-0.04027,-0.0663},
				{1345,-0.1978,-0.07227,1.00928,0.0902,0.05986,-0.10934,-0.04327,-0.06969},
				{1350,-0.19254,-0.06312,1.00772,0.09984,0.06658,-0.10577,-0.0462,-0.07301},
				{1355,-0.18776,-0.05386,1.00587,0.10877,0.0735,-0.10181,-0.04906,-0.07626},
				{1360,-0.18344,-0.04449,1.00374,0.11704,0.08057,-0.09747,-0.05185,-0.07945},
				{1365,-0.17954,-0.03502,1.00131,0.12466,0.08776,-0.09276,-0.05457,-0.08257},
				{1370,-0.17605,-0.02545,0.99856,0.13168,0.09499,-0.08768,-0.05724,-0.08564},
				{1375,-0.17294,-0.01579,0.9955,0.13811,0.10224,-0.08224,-0.05984,-0.08866},
				{1380,-0.1702,-0.00605,0.99209,0.14399,0.10943,-0.07644,-0.06239,-0.09161},
				{1385,-0.16779,0.00381,0.98832,0.14938,0.1165,-0.07025,-0.06488,-0.09452},
				{1390,-0.16573,0.0138,0.98412,0.15434,0.12336,-0.06361,-0.06731,-0.09737},
				{1395,-0.164,0.02398,0.97947,0.15895,0.1299,-0.05648,-0.06967,-0.10017},
				{1400,-0.1626,0.03439,0.9743,0.16329,0.13604,-0.04879,-0.07197,-0.10292},
				{1405,-0.16152,0.04506,0.96858,0.16742,0.14169,-0.04049,-0.0742,-0.10562},
				{1410,-0.16076,0.05603,0.96225,0.17143,0.14675,-0.03153,-0.07636,-0.10826},
				{1415,-0.16032,0.06735,0.95526,0.17539,0.15112,-0.02185,-0.07844,-0.11086},
				{1420,-0.16018,0.07904,0.94758,0.17937,0.15471,-0.01139,-0.08045,-0.1134},
				{1425,-0.16032,0.09115,0.93916,0.18348,0.15743,-0.00014,-0.08237,-0.11589},
				{1430,-0.16072,0.10365,0.93,0.18788,0.15919,0.01187,-0.08421,-0.11833},
				{1435,-0.16131,0.11654,0.92008,0.19276,0.15988,0.02457,-0.08596,-0.12072},
				{1440,-0.16205,0.1298,0.90941,0.1983,0.15942,0.03792,-0.08761,-0.12307},
				{1445,-0.16288,0.14342,0.89797,0.20467,0.15771,0.05186,-0.08916,-0.12536},
				{1450,-0.16377,0.15738,0.88576,0.21207,0.15465,0.06634,-0.09061,-0.12762},
				{1455,-0.16466,0.17168,0.87277,0.22066,0.15015,0.08129,-0.09195,-0.12982},
				{1460,-0.16551,0.18629,0.85905,0.23052,0.14422,0.09666,-0.09317,-0.13198},
				{1465,-0.16627,0.20112,0.84472,0.2415,0.13711,0.11238,-0.09429,-0.13409},
				{1470,-0.16693,0.2161,0.82995,0.25341,0.12908,0.12838,-0.09532,-0.13613},
				{1475,-0.16745,0.23114,0.81489,0.26605,0.12041,0.14458,-0.09627,-0.13811},
				{1480,-0.16779,0.24618,0.7997,0.27923,0.11139,0.16091,-0.09713,-0.14},
				{1485,-0.16791,0.26112,0.78452,0.29277,0.10228,0.17731,-0.09793,-0.14181},
				{1490,-0.1678,0.27589,0.76952,0.30647,0.09336,0.19369,-0.09867,-0.14352},
				{1495,-0.1674,0.29043,0.75483,0.3202,0.08487,0.20999,-0.09936,-0.14512},
				{1500,-0.1667,0.30482,0.74056,0.33397,0.07697,0.22618,-0.10001,-0.14661},
				{1505,-0.16566,0.31913,0.7268,0.34786,0.06978,0.24223,-0.10063,-0.14797},
				{1510,-0.16426,0.33345,0.71365,0.36195,0.06342,0.25811,-0.10122,-0.14918},
				{1515,-0.16248,0.34775,0.70117,0.37616,0.05796,0.27376,-0.10179,-0.15025},
				{1520,-0.16029,0.36189,0.68942,0.39033,0.05345,0.28907,-0.10234,-0.15116},
				{1525,-0.15769,0.37579,0.67848,0.40443,0.04992,0.30396,-0.10285,-0.1519},
				{1530,-0.15466,0.38951,0.66844,0.41854,0.04734,0.31837,-0.10332,-0.15246},
				{1535,-0.1512,0.40308,0.65936,0.43276,0.04569,0.33227,-0.10376,-0.15285},
				{1540,-0.1473,0.41655,0.65133,0.44722,0.04496,0.34558,-0.10416,-0.15304},
				{1545,-0.14295,0.42994,0.64444,0.46201,0.04513,0.35827,-0.10452,-0.15303},
				{1550,-0.13812,0.44331,0.63877,0.47723,0.04617,0.37028,-0.10483,-0.15281},
				{1555,-0.13285,0.45667,0.63433,0.49295,0.04807,0.38161,-0.10509,-0.15238},
				{1560,-0.12715,0.46998,0.63098,0.50907,0.0508,0.39235,-0.10532,-0.15175},
				{1565,-0.1211,0.48319,0.62859,0.52549,0.05434,0.4026,-0.1055,-0.15093},
				{1570,-0.11473,0.49626,0.62699,0.5421,0.05865,0.41247,-0.10565,-0.14993},
				{1575,-0.10809,0.50913,0.62604,0.55881,0.06373,0.42208,-0.10577,-0.14877},
				{1580,-0.10123,0.52176,0.62558,0.57551,0.06953,0.43151,-0.10585,-0.14745},
				{1585,-0.09421,0.53411,0.62545,0.5921,0.07604,0.44089,-0.10592,-0.14599},
				{1590,-0.08706,0.54614,0.62557,0.60853,0.0832,0.45027,-0.10596,-0.1444},
				{1595,-0.07979,0.55784,0.62586,0.62474,0.09096,0.45965,-0.10598,-0.14269},
				{1600,-0.07245,0.56918,0.62625,0.6407,0.09925,0.46907,-0.106,-0.14085},
				{1605,-0.06505,0.58016,0.62668,0.65638,0.10801,0.47852,-0.106,-0.13891},
				{1610,-0.05762,0.59074,0.62707,0.67175,0.11718,0.48804,-0.106,-0.13687},
				{1615,-0.05018,0.60091,0.62737,0.68675,0.12668,0.49763,-0.106,-0.13474},
				{1620,-0.04276,0.61068,0.62751,0.70139,0.13648,0.50729,-0.106,-0.13252},
				{1625,-0.03535,0.62007,0.62748,0.71565,0.14651,0.51701,-0.106,-0.13022},
				{1630,-0.02796,0.62911,0.62727,0.72955,0.15674,0.52673,-0.10601,-0.12786},
				{1635,-0.02059,0.63784,0.62686,0.74311,0.16713,0.53644,-0.10603,-0.12544},
				{1640,-0.01324,0.6463,0.62623,0.75633,0.17762,0.54609,-0.10605,-0.12297},
				{1645,-0.00591,0.65451,0.62538,0.76923,0.18819,0.55565,-0.10608,-0.12045},
				{1650,0.00139,0.66253,0.62428,0.78182,0.19879,0.5651,-0.10612,-0.1179},
				{1655,0.00868,0.67037,0.62293,0.7941,0.20937,0.57439,-0.10617,-0.11533},
				{1660,0.01593,0.67807,0.62131,0.8061,0.2199,0.58348,-0.10623,-0.11274},
				{1665,0.02316,0.68568,0.6194,0.81783,0.23032,0.59236,-0.1063,-0.11014},
				{1670,0.03036,0.69322,0.61719,0.82929,0.24061,0.60098,-0.10638,-0.10754},
				{1675,0.03754,0.70073,0.61467,0.84049,0.2507,0.60931,-0.10647,-0.10495},
				{1680,0.04468,0.70824,0.61182,0.85146,0.26058,0.61732,-0.10658,-0.10237},
				{1685,0.05179,0.7158,0.60863,0.8622,0.27018,0.62497,-0.1067,-0.09982},
				{1690,0.05886,0.72342,0.60508,0.87271,0.27946,0.63223,-0.10684,-0.09731},
				{1695,0.06591,0.73112,0.60118,0.88299,0.28841,0.63907,-0.10699,-0.09483},
				{1700,0.07293,0.73887,0.59694,0.893,0.29698,0.64545,-0.10717,-0.09241},
				{1705,0.07993,0.74664,0.59238,0.90269,0.30516,0.65133,-0.10736,-0.09003},
				{1710,0.08691,0.75441,0.58751,0.91203,0.31293,0.65668,-0.10758,-0.08772},
				{1715,0.09388,0.76215,0.58234,0.92098,0.32025,0.66148,-0.10782,-0.08547},
				{1720,0.10084,0.76985,0.5769,0.92954,0.32711,0.66572,-0.10809,-0.08329},
				{1725,0.10779,0.77748,0.57123,0.93769,0.33352,0.66945,-0.10838,-0.08119},
				{1730,0.11473,0.78502,0.56535,0.94547,0.33947,0.67271,-0.10869,-0.07918},
				{1735,0.12166,0.79245,0.5593,0.95286,0.34495,0.67554,-0.10902,-0.07725},
				{1740,0.12857,0.79975,0.55311,0.95988,0.34998,0.67799,-0.10938,-0.07542},
				{1745,0.13547,0.80689,0.54682,0.96655,0.35453,0.68008,-0.10975,-0.07369},
				{1750,0.14236,0.81386,0.54046,0.97285,0.35862,0.68187,-0.11014,-0.07206},
				{1755,0.14923,0.82063,0.53405,0.97881,0.36224,0.6834,-0.11054,-0.07055},
				{1760,0.15609,0.82719,0.52764,0.98443,0.36538,0.6847,-0.11096,-0.06915},
				{1765,0.16293,0.83351,0.52126,0.98972,0.36805,0.68582,-0.1114,-0.06788},
				{1770,0.16976,0.83957,0.51493,0.99469,0.37023,0.68679,-0.11184,-0.06673},
				{1775,0.17657,0.84536,0.5087,0.99933,0.37194,0.68766,-0.1123,-0.06572},
				{1780,0.18336,0.85084,0.50258,1.00367,0.37317,0.68847,-0.11276,-0.06485},
				{1785,0.19009,0.856,0.49662,1.00768,0.37395,0.6892,-0.11323,-0.06413},
				{1790,0.19674,0.86081,0.49082,1.01134,0.37429,0.68985,-0.1137,-0.06355},
				{1795,0.20326,0.86526,0.48522,1.01465,0.37424,0.69042,-0.11416,-0.06312},
				{1800,0.20962,0.86932,0.47982,1.01757,0.37382,0.69088,-0.11461,-0.06286},
				{1805,0.21579,0.87298,0.47467,1.02009,0.37305,0.69124,-0.11505,-0.06275},
				{1810,0.22172,0.8762,0.46977,1.0222,0.37196,0.69148,-0.11547,-0.06281},
				{1815,0.22742,0.87899,0.46514,1.02388,0.37061,0.6916,-0.11586,-0.06304},
				{1820,0.23286,0.88135,0.46074,1.02515,0.36904,0.69159,-0.11622,-0.06343},
				{1825,0.23807,0.88327,0.45655,1.02601,0.36732,0.69146,-0.11655,-0.06399},
				{1830,0.24305,0.88477,0.45254,1.02648,0.36551,0.69121,-0.11683,-0.06471},
				{1835,0.2478,0.88586,0.4487,1.02655,0.36368,0.69084,-0.11707,-0.06558},
				{1840,0.25232,0.88654,0.445,1.02625,0.36189,0.69035,-0.11725,-0.06662},
				{1845,0.25663,0.88681,0.4414,1.02558,0.36021,0.68975,-0.11736,-0.06781},
				{1850,0.26072,0.88668,0.43789,1.02454,0.35868,0.68902,-0.11741,-0.06915},
				{1855,0.26461,0.88616,0.43444,1.02315,0.35739,0.68818,-0.11739,-0.07064},
				{1860,0.2683,0.88526,0.43103,1.02142,0.35638,0.68723,-0.11729,-0.07228},
				{1865,0.27179,0.88397,0.42763,1.01936,0.35573,0.68616,-0.1171,-0.07407},
				{1870,0.27509,0.88231,0.42421,1.01697,0.35549,0.68497,-0.11683,-0.076},
				{1875,0.2782,0.88028,0.42075,1.01426,0.35573,0.68367,-0.11645,-0.07808},
				{1880,0.28114,0.87788,0.41722,1.01124,0.35651,0.68226,-0.11597,-0.08029},
				{1885,0.28391,0.87515,0.41361,1.00795,0.35788,0.68074,-0.11539,-0.08265},
				{1890,0.28657,0.87216,0.40992,1.00444,0.35986,0.67908,-0.11469,-0.08513},
				{1895,0.28917,0.86899,0.40614,1.0008,0.36248,0.67727,-0.11389,-0.08774},
				{1900,0.29176,0.86572,0.40227,0.99711,0.36575,0.67529,-0.11297,-0.09047},
				{1905,0.2944,0.86244,0.39831,0.99344,0.3697,0.67313,-0.11195,-0.09332},
				{1910,0.29714,0.85921,0.39426,0.98985,0.37433,0.67078,-0.11082,-0.09628},
				{1915,0.3,0.85606,0.39014,0.98635,0.37962,0.66822,-0.10957,-0.09935},
				{1920,0.30299,0.85298,0.38597,0.98292,0.38555,0.66549,-0.10821,-0.10252},
				{1925,0.30612,0.84996,0.38177,0.97952,0.39207,0.6626,-0.10674,-0.10578},
				{1930,0.3094,0.84699,0.37755,0.97614,0.39915,0.65955,-0.10515,-0.10914},
				{1935,0.31283,0.84406,0.37334,0.97273,0.40676,0.65637,-0.10345,-0.11259},
				{1940,0.31643,0.84117,0.36916,0.96929,0.41485,0.65306,-0.10163,-0.11612},
				{1945,0.32021,0.83831,0.36503,0.96577,0.42339,0.64965,-0.0997,-0.11973},
				{1950,0.32418,0.83547,0.36097,0.96215,0.43235,0.64615,-0.09765,-0.12341},
				{1955,0.32835,0.83264,0.357,0.95841,0.44169,0.64256,-0.09547,-0.12717},
				{1960,0.33272,0.82981,0.35313,0.95452,0.45138,0.63891,-0.09318,-0.13099},
				{1965,0.3373,0.82698,0.3494,0.95046,0.46137,0.63521,-0.09076,-0.13488},
				{1970,0.34212,0.82414,0.34581,0.94618,0.47163,0.63148,-0.08822,-0.13882},
				{1975,0.34716,0.82128,0.3424,0.94168,0.48214,0.62772,-0.08556,-0.14281},
				{1980,0.35244,0.81839,0.33921,0.93696,0.49286,0.62395,-0.08278,-0.14686},
				{1985,0.35793,0.81546,0.33628,0.93204,0.50378,0.62017,-0.07987,-0.15094},
				{1990,0.36362,0.81249,0.33367,0.92693,0.51489,0.61638,-0.07685,-0.15505},
				{1995,0.3695,0.80947,0.33142,0.92167,0.52616,0.6126,-0.07372,-0.15918},
				{2000,0.37557,0.80638,0.3296,0.91626,0.5376,0.60883,-0.07047,-0.16334},
				{2005,0.38179,0.80326,0.32819,0.91074,0.54915,0.60509,-0.0671,-0.1675},
				{2010,0.38815,0.80015,0.32716,0.90515,0.56081,0.60143,-0.06363,-0.17166},
				{2015,0.39462,0.7971,0.32646,0.89952,0.57253,0.59787,-0.06005,-0.17582},
				{2020,0.40118,0.79416,0.32605,0.8939,0.58428,0.59447,-0.05636,-0.17995},
				{2025,0.40779,0.79138,0.32589,0.88831,0.59603,0.59126,-0.05257,-0.18406},
				{2030,0.41443,0.78881,0.32593,0.8828,0.60774,0.58827,-0.04866,-0.18814},
				{2035,0.42108,0.78649,0.32614,0.87741,0.61939,0.58555,-0.04466,-0.19217},
				{2040,0.42771,0.78448,0.32648,0.87217,0.63095,0.58314,-0.04055,-0.19614},
				{2045,0.4343,0.78283,0.32689,0.86712,0.64237,0.58107,-0.03633,-0.20006},
				{2050,0.44082,0.78157,0.32734,0.8623,0.65363,0.57938,-0.03202,-0.2039},
				{2055,0.44724,0.78077,0.32779,0.85775,0.66469,0.57812,-0.0276,-0.20766},
				{2060,0.45354,0.78047,0.3282,0.8535,0.67553,0.57732,-0.02309,-0.21133},
				{2065,0.4597,0.78072,0.32851,0.84959,0.68611,0.57701,-0.01848,-0.21491},
				{2070,0.46568,0.78157,0.3287,0.84607,0.69639,0.57725,-0.01377,-0.21837},
				{2075,0.47143,0.78303,0.32871,0.84292,0.70636,0.57806,-0.00896,-0.22172},
				{2080,0.47686,0.78509,0.32848,0.84013,0.71598,0.57948,-0.00406,-0.22495},
				{2085,0.48188,0.78773,0.32797,0.83768,0.72524,0.58154,0.00093,-0.22805},
				{2090,0.48641,0.79094,0.32711,0.83553,0.73411,0.58428,0.00601,-0.231},
				{2095,0.49037,0.79472,0.32585,0.83367,0.74256,0.58772,0.01118,-0.23381},
				{2100,0.49366,0.79904,0.32414,0.83207,0.75058,0.5919,0.01644,-0.23647},
				{2105,0.49624,0.80389,0.32195,0.83071,0.75815,0.59681,0.02178,-0.23896},
				{2110,0.49816,0.80918,0.31935,0.82953,0.76531,0.60238,0.02721,-0.24129},
				{2115,0.49948,0.81487,0.31641,0.82851,0.77211,0.60852,0.03272,-0.24344},
				{2120,0.50026,0.82088,0.31321,0.82758,0.77857,0.61513,0.0383,-0.24542},
				{2125,0.50056,0.82715,0.30981,0.82672,0.78474,0.62213,0.04397,-0.24723},
				{2130,0.50044,0.8336,0.3063,0.82587,0.79067,0.62942,0.04971,-0.24885},
				{2135,0.49996,0.84017,0.30274,0.82499,0.79638,0.63691,0.05552,-0.25028},
				{2140,0.49918,0.84679,0.29921,0.82405,0.80192,0.64451,0.0614,-0.25152},
				{2145,0.49816,0.8534,0.29578,0.82298,0.80733,0.65213,0.06735,-0.25257},
				{2150,0.49696,0.85993,0.29252,0.82176,0.81265,0.65968,0.07337,-0.25341},
				{2155,0.49565,0.86631,0.2895,0.82034,0.81792,0.66707,0.07946,-0.25406},
				{2160,0.49428,0.87251,0.28679,0.81869,0.82315,0.67424,0.0856,-0.25449},
				{2165,0.49291,0.87852,0.28443,0.81682,0.82835,0.68115,0.09182,-0.25473},
				{2170,0.49162,0.88433,0.28247,0.81472,0.83352,0.68777,0.0981,-0.25475},
				{2175,0.49046,0.88995,0.28095,0.81239,0.83868,0.69406,0.10444,-0.25457},
				{2180,0.48949,0.89536,0.27992,0.80982,0.84381,0.69997,0.11086,-0.25418},
				{2185,0.48879,0.90056,0.27942,0.807,0.84893,0.70549,0.11735,-0.25358},
				{2190,0.48842,0.90554,0.27951,0.80394,0.85403,0.71055,0.12391,-0.25277},
				{2195,0.48843,0.9103,0.28022,0.80063,0.85913,0.71514,0.13054,-0.25176},
				{2200,0.48889,0.91482,0.28161,0.79707,0.86422,0.71922,0.13725,-0.25053},
				{2205,0.48982,0.91912,0.28367,0.7933,0.86934,0.72283,0.14403,-0.24909},
				{2210,0.4912,0.9232,0.28639,0.7894,0.87452,0.72605,0.15087,-0.24744},
				{2215,0.49301,0.92707,0.28974,0.78543,0.8798,0.72899,0.15778,-0.24558},
				{2220,0.49524,0.93074,0.29369,0.78147,0.88522,0.73175,0.16474,-0.24351},
				{2225,0.49787,0.93422,0.29822,0.7776,0.89083,0.7344,0.17175,-0.24122},
				{2230,0.50089,0.93753,0.30331,0.77388,0.89667,0.73706,0.17881,-0.23871},
				{2235,0.50427,0.94067,0.30894,0.7704,0.90277,0.73982,0.1859,-0.23598},
				{2240,0.508,0.94365,0.31507,0.76723,0.90918,0.74276,0.19303,-0.23304},
				{2245,0.51205,0.94648,0.32167,0.76443,0.91593,0.74598,0.20019,-0.22988},
				{2250,0.51641,0.9492,0.32868,0.76206,0.92297,0.7495,0.20737,-0.2265},
				{2255,0.52101,0.95185,0.336,0.76017,0.93022,0.75331,0.21457,-0.22289},
				{2260,0.52581,0.95446,0.34353,0.75879,0.93762,0.7574,0.22179,-0.21908},
				{2265,0.53077,0.95707,0.35118,0.75797,0.94507,0.76177,0.229,-0.21504},
				{2270,0.53584,0.95972,0.35885,0.75775,0.95251,0.7664,0.23622,-0.21079},
				{2275,0.54099,0.96245,0.36644,0.75818,0.95985,0.77129,0.24343,-0.20631},
				{2280,0.54615,0.96531,0.37387,0.7593,0.96701,0.77643,0.25063,-0.20163},
				{2285,0.5513,0.96832,0.38104,0.76114,0.97393,0.78182,0.25781,-0.19672},
				{2290,0.55637,0.97153,0.38784,0.76377,0.98051,0.78743,0.26496,-0.19161},
				{2295,0.56134,0.97498,0.3942,0.7672,0.9867,0.79328,0.27209,-0.18627},
				{2300,0.56619,0.97866,0.4001,0.77142,0.99246,0.79937,0.27918,-0.18073},
				{2305,0.57092,0.98258,0.40554,0.77638,0.99779,0.80572,0.28624,-0.17498},
				{2310,0.57554,0.98672,0.41053,0.78203,1.0027,0.81236,0.29326,-0.16902},
				{2315,0.58005,0.99108,0.41507,0.78832,1.00719,0.8193,0.30024,-0.16286},
				{2320,0.58445,0.99565,0.41919,0.79521,1.01125,0.82658,0.30717,-0.1565},
				{2325,0.58875,1.00043,0.42288,0.80265,1.01489,0.8342,0.31406,-0.14995},
				{2330,0.59294,1.0054,0.42615,0.81059,1.01809,0.8422,0.3209,-0.14321},
				{2335,0.59702,1.01057,0.42902,0.81897,1.02088,0.85059,0.32769,-0.13629},
				{2340,0.60101,1.01591,0.43149,0.82776,1.02323,0.8594,0.33442,-0.12918},
				{2345,0.60491,1.02144,0.43357,0.83691,1.02516,0.86865,0.3411,-0.12189},
				{2350,0.60869,1.02714,0.43525,0.84638,1.02664,0.87834,0.34772,-0.11443},
				{2355,0.61231,1.03305,0.43655,0.85621,1.02762,0.88845,0.35428,-0.10679},
				{2360,0.61572,1.03919,0.43744,0.86642,1.02803,0.89895,0.36077,-0.09899},
				{2365,0.61886,1.04558,0.43793,0.87705,1.02781,0.90981,0.3672,-0.09103},
				{2370,0.62167,1.05226,0.438,0.88814,1.0269,0.921,0.37355,-0.08291},
				{2375,0.6241,1.05925,0.43764,0.8997,1.02524,0.93248,0.37983,-0.07463},
				{2380,0.62609,1.06658,0.43686,0.91178,1.02275,0.94422,0.38604,-0.06621},
				{2385,0.62761,1.07425,0.43564,0.92439,1.01942,0.95621,0.39216,-0.05765},
				{2390,0.62867,1.08225,0.43401,0.9375,1.01527,0.96843,0.39821,-0.04895},
				{2395,0.62929,1.09054,0.43198,0.95108,1.01035,0.98089,0.40419,-0.04013},
				{2400,0.6295,1.09908,0.4296,0.9651,1.0047,0.99357,0.4101,-0.03119},
				{2405,0.6293,1.10785,0.42686,0.97951,0.99837,1.00648,0.41594,-0.02213},
				{2410,0.62872,1.11682,0.4238,0.9943,0.9914,1.0196,0.42171,-0.01297},
				{2415,0.62778,1.12594,0.42043,1.00942,0.98384,1.03294,0.42742,-0.00372},
				{2420,0.6265,1.1352,0.41678,1.02485,0.97573,1.0465,0.43307,0.00562},
				{2425,0.6249,1.14456,0.41287,1.04055,0.96711,1.06025,0.43866,0.01503},
				{2430,0.62298,1.15399,0.40873,1.05648,0.95803,1.07421,0.4442,0.02452},
				{2435,0.62079,1.16345,0.40436,1.07263,0.94853,1.08837,0.44968,0.03407},
				{2440,0.61833,1.17291,0.3998,1.08894,0.93866,1.10272,0.45511,0.04368},
				{2445,0.61563,1.18237,0.39507,1.1054,0.92847,1.11725,0.4605,0.05333},
				{2450,0.61273,1.19182,0.39019,1.12199,0.918,1.13193,0.46585,0.06302},
				{2455,0.60967,1.20126,0.3852,1.13867,0.9073,1.14673,0.47117,0.07275},
				{2460,0.60647,1.2107,0.38011,1.15544,0.89642,1.16162,0.47647,0.0825},
				{2465,0.60319,1.22014,0.37495,1.17226,0.88542,1.17659,0.48175,0.09227},
				{2470,0.59984,1.22958,0.36976,1.18912,0.87434,1.19159,0.48702,0.10205},
				{2475,0.59648,1.23901,0.36454,1.20599,0.86322,1.20662,0.49228,0.11184}}
				};
					
			const G4int coef_ener = G4int((pLab-Legendre_coef[WhichChannel][0][0])/5);
			const G4double sup_ener = pLab/5. - coef_ener -Legendre_coef[WhichChannel][0][0]/5;
			
// assert(pLab >= Legendre_coef[WhichChannel][coef_ener][0] && pLab < Legendre_coef[WhichChannel][coef_ener+1][0]);
			
			// Legendre coefficient normalized
			const G4double A0 = 1.;
			const G4double A1 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][1] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][1];
			const G4double A2 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][2] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][2];
			const G4double A3 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][3] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][3];
			const G4double A4 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][4] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][4];
			const G4double A5 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][5] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][5];
			const G4double A6 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][6] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][6];
			const G4double A7 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][7] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][7];
			const G4double A8 = (1-sup_ener)*Legendre_coef[WhichChannel][coef_ener][8] + sup_ener*Legendre_coef[WhichChannel][coef_ener+1][8];
			
			// Theoritical max if all Ai > 0 (often the case)
			const G4double A = std::fabs(A0) + std::fabs(A1) + std::fabs(A2) + std::fabs(A3) + std::fabs(A4) + std::fabs(A5) + std::fabs(A6) + std::fabs(A7) + std::fabs(A8);
			
			G4bool success = false;
			G4int maxloop = 0;
			
			while(!success && maxloop < 1000){
				
				cos_theta = Random::shoot()*2-1.; // not optimized
				
				// Legendre Polynomial
				G4double P0 = A0;
				G4double P1 = A1*cos_theta;
				G4double P2 = A2/2.*(3*std::pow(cos_theta,2)-1);
				G4double P3 = A3/2.*(5*std::pow(cos_theta,3)-3*cos_theta);
				G4double P4 = A4/8.*(35*std::pow(cos_theta,4)-30*std::pow(cos_theta,2)+3);
				G4double P5 = A5/8.*(63*std::pow(cos_theta,5)-70*std::pow(cos_theta,3)+15*cos_theta);
				G4double P6 = A6/16.*(231*std::pow(cos_theta,6)-315*std::pow(cos_theta,4)+105*std::pow(cos_theta,2)-5);
				G4double P7 = A7/16.*(429*std::pow(cos_theta,7)-693*std::pow(cos_theta,5)+315*std::pow(cos_theta,3)-35*cos_theta);
				G4double P8 = A8/128.*(6435*std::pow(cos_theta,8)-12012*std::pow(cos_theta,6)+6930*std::pow(cos_theta,4)-1260*std::pow(cos_theta,2)+35);
				
				G4double P = (P0 + P1 + P2 + P3 + P4 + P5 + P6 + P7 + P8)/2; // /2 for the normalisation
				
				if(Random::shoot()*A < P) success = true;
				maxloop +=1 ;
				if(maxloop==1000) cos_theta = std::log(Random::shoot()*(std::exp(10.)-std::exp(-10.))+std::exp(-10.))/10.; // if no success in 1E4 shoot, probably angulard distribution piked very forward 
			}
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
		}
		
		if(rho == 0) return ThreeVector(sin_theta*cos_phi,sin_theta*sin_phi,cos_theta);
		// Rotation in the direction of the incident kaon
		const G4double px = x/r*cos_theta - y/rho*sin_theta*cos_phi + z/r*x/rho*sin_theta*sin_phi;
		const G4double py = y/r*cos_theta + x/rho*sin_theta*cos_phi + z/r*y/rho*sin_theta*sin_phi;
		const G4double pz = z/r*cos_theta - rho/r*sin_theta*sin_phi;
		
		return ThreeVector(px,py,pz);
	}
}
