# spec file for package dtrace
#
# Oracle Linux DTrace.
# Copyright (c) 2011, 2025, Oracle and/or its affiliates. All rights reserved.
# Licensed under the Universal Permissive License v 1.0 as shown at
# http://oss.oracle.com/licenses/upl.

# Redefine 'build_variant' at build time to create a kernel package named
# something like 'kernel-uek-dtrace'.
%define variant %{?build_variant:%{build_variant}}%{!?build_variant:-uek}

# Allow building against libdtrace-ctf even when libctf is available.  Specify
# "--without libctf" to the rpmbuild command to bypass libctf.
%define with_libctf %{?_without_libctf: 0} %{?!_without_libctf: 1}

%if "%{?dist}" == ".el7"
%define with_libctf 0
%endif

# OL10 requires an explicit BPF toolset version.
%if "%{?dist}" == ".el10"
%define bpfv -14
%define bpfc BPFC=bpf-unknown-none-gcc-14
%else
%define bpfv %{nil}
%define bpfc %{nil}
%endif

# By default, do not build with 32-on-64 support.
%define glibc32 %{nil}

# Enable it for non-ARM64 builds excpet for OL10.
# ARM64 does not support 32-on-64 either.
%ifnarch aarch64
%if "%{?dist}" != ".el10"
%define glibc32 glibc-devel(%{__isa_name}-32) libgcc(%{__isa_name}-32)
%endif
%endif

# Build DTrace without LTO.
%global _lto_cflags %{nil}

BuildRequires: rpm
Name:         dtrace
License:      Universal Permissive License (UPL), Version 1.0
Group:        Development/Tools
Requires:     cpp elfutils-libelf zlib libpcap libpfm
BuildRequires: glibc-headers bison flex zlib-devel elfutils-libelf-devel
BuildRequires: systemd systemd-devel glibc-static %{glibc32} wireshark
BuildRequires: libpcap-devel valgrind-devel libpfm-devel libbpf-devel
%if "%{?dist}" == ".el7"
Requires:     fuse
BuildRequires: fuse-devel
%define maybe_use_fuse2 libfuse2=yes
%else
Requires:     fuse3 >= 3.2.0
BuildRequires: fuse3-devel >= 3.2.0
%define maybe_use_fuse2 %{nil}
%endif
%{?systemd_requires}
BuildRequires: gcc-bpf-unknown-none%{bpfv}
BuildRequires: binutils-bpf-unknown-none
%ifnarch aarch64
Requires:     binutils
BuildRequires: binutils-devel
%endif
%if %{with_libctf}
Requires:     binutils >= 2.30-58.0.8
BuildRequires: binutils-devel >= 2.30-58.0.8
%else
Requires:     libdtrace-ctf >= 1.1.0
BuildRequires: libdtrace-ctf-devel >= 1.1.0
%endif
Summary:      DTrace user interface.
Version:      2.0.4
Release:      1%{?dist}
Source:       dtrace-%{version}.tar.bz2
BuildRoot:    %{_tmppath}/%{name}-%{version}-build
ExclusiveArch:    x86_64 aarch64

%description
DTrace user interface and dtrace(1) command.

Maintainers:
-----------
DTrace external development mailing list <dtrace-devel@oss.oracle.com>

%package devel
Summary:      DTrace development headers.
%if %{with_libctf}
Requires:     binutils-devel >= 2.30-58.0.8
%else
Requires:     libdtrace-ctf-devel >= 1.1.0
%endif
Requires:     elfutils-libelf-devel
Requires:     %{name}%{?_isa} = %{version}-%{release}
Provides:     dtrace-headers = 2.0.0
Group:	      Development/System

%description devel
Headers and libraries to develop DTrace applications.

You do not need this package merely to compile providers and probe points into
applications that will be probed by dtrace, but rather when developing
replacements for dtrace(1) itself.

# We turn off dependency generation for the testsuite because it contains
# test shared libraries of its own, which are otherwise picked up as
# (erroneous) deps to nonexistent packages.
%package testsuite
Summary:      DTrace testsuite.
Requires:     make glibc-devel(%{__isa_name}-64) libgcc(%{__isa_name}-64)
Requires:     dtrace-headers >= 2.0.0 module-init-tools
Requires:     %{name}-devel = %{version}-%{release} perl gcc java
Requires:     java-devel perl-IO-Socket-IP xfsprogs
Requires:     exportfs vim-minimal %{name}%{?_isa} = %{version}-%{release}
Requires:     coreutils wireshark %{glibc32}
Requires:     perf time bc nfs-utils
Autoreq:      0
Group:	      Development/System

%description testsuite
The DTrace testsuite.

Installed in /usr/lib64/dtrace/testsuite.

'make check' here is just like 'make check' in the source tree, except that
it always tests the installed DTrace.

%prep
%setup -q

%build
make -j $(getconf _NPROCESSORS_ONLN) %{bpfc} %{maybe_use_fuse2}

# Force off debuginfo splitting.  We have no debuginfo in dtrace proper,
# and the testsuite requires debuginfo for proper operation.
%global debug_package %{nil}
%global __debug_package %{nil}
%global __spec_install_post \
	%{_rpmconfigdir}/brp-compress \
	%{nil}

%install

mkdir -p $RPM_BUILD_ROOT/usr/sbin
make DESTDIR=$RPM_BUILD_ROOT VERSION=%{version} \
     HDRPREFIX="$RPM_BUILD_ROOT/usr/include" \
     install install-test

%if "%{?dist}" == ".el7"
sed -i '/^ProtectSystem=/d; /^ProtectControlGroups=/d; /^RuntimeDirectory/d;' $RPM_BUILD_ROOT/usr/lib/systemd/system/dtprobed.service
%endif

%clean
[ "$RPM_BUILD_ROOT" != "/" ] && rm -rf "$RPM_BUILD_ROOT"
rm -rf $RPM_BUILD_DIR/%{name}-%{version}

%post
/sbin/ldconfig
%udev_rules_update
%systemd_post dtprobed.service
# Force a daemon restart on upgrade even if the previous package did
# not use presets and did not request a restart on uninstallation.
# (When upgrades from 1.13.1 and below are no longer supported, we can
# remove this.)
# systemd_postun_with_restart does the right thing here, though we need
# to wrap it in an extra conditional to make it not run on install.
if [ $1 -ge 2 ] ; then
    # OL8 and below don't reload the unit file properly.
    %if "%{?dist}" == ".el7" || "%{?dist}" == ".el8"
    systemctl daemon-reload || :
    %endif

    %systemd_postun_with_restart dtprobed.service
fi

# Clean up the mess left behind by previous installations' overwriting
# of systemtap's sdt.h: if an sdt-systemtap.h exists and is a regular
# file, move it back to sdt.h.  (RPM itself will identify and delete
# all the other debris.)

SYSINCDIR=/usr/include/sys
if [ -f $SYSINCDIR/sdt-systemtap.h ]; then
    mv -f $SYSINCDIR/sdt-systemtap.h $SYSINCDIR/sdt.h
fi

%preun
%systemd_preun dtprobed.service

%postun
/sbin/ldconfig
%udev_rules_update
%systemd_postun_with_restart dtprobed.service

%posttrans
# Make sure the daemon really is started (OL9 doesn't start the
# daemon properly on fresh installations -- all presets do is enable
# it, not start it, and postun_with_restart only restarts on *upgrade*.)
systemctl start dtprobed || :

%files
%defattr(-,root,root,-)
%{_libdir}/dtrace
%exclude %{_libdir}/dtrace/testsuite
%{_libdir}/libdtrace.so.*
%{_sbindir}/dtrace
%{_sbindir}/dtprobed
%{_datadir}/pkgconfig/dtrace.pc
%{_datadir}/pkgconfig/dtrace_sdt.pc
%{_mandir}/man8/dtrace.8.gz
%{_mandir}/man8/dtprobed.8.gz
%doc %{_docdir}/dtrace-%{version}
%{_unitdir}/dtprobed.service
%{_unitdir}/dtrace-usdt.target
%{_presetdir}/50-dtprobed.preset
%{_udevrulesdir}/60-dtprobed.rules

%files devel
%defattr(-,root,root,-)
%{_libdir}/libdtrace.so
%{_includedir}/dtrace
%{_includedir}/dtrace.h
%{_includedir}/sys/dtrace.h
%{_includedir}/sys/dtrace_types.h

%files testsuite
%defattr(-,root,root,-)
%{_libdir}/dtrace/testsuite

%changelog
* Fri Oct 24 2025 Eugene Loh <eugene.loh@oracle.com> - 2.0.4-1
- TCP, UDP, and stapsdt providers implemented.  (Alan Maguire)
- New learning materials:  the User's Guide in Markdown format,
  example scripts, and a context file for LLMs.
  (Eugene Loh, Bruce McCulloch, Ruud van der Pas, Elena Zannoni).
- Allow [u]stack() to be used as a variable value.  (Kris Van Hees)
  [Orabug: 37950533]
- Comments using // are now supported.  (Kris Van Hees)
- Scalability improvements.  (Kris Van Hees)
- Error injection via return() action.  (Kris Van Hees)
- Improved string handling.  (Kris Van Hees)
- Various bug fixes.  (Eugene Loh, Kris Van Hees)
- Fix dyn vars overwriting one another.  [Orabug: 37994729]
- Fix regression:  list fbt probes by default.  [Orabug: 38249511]
- Various testsuite fixes and improvements.
  (Nick Alcock, Eugene Loh, Alan Maguire, Kris Van Hees)
- Add test for preprocessor definitions.  [Orabug: 28763074]
- Fix some stack tests.  [Orabug: 37459289]

* Tue Jun 10 2025 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.3-1
- This is only released on OL10.
- Redesigned USDT support to work for LTO compilations.  [Orabug: 38011704]
- New builtin variable: execargs.
- Offset probes in pid provider.  (Eugene Loh)
- Various bug fixes.  (Nick Alcock, Eugene Loh, Kris Van Hees)
- Various testsuite fixes and improvements.
  (Nick Alcock, Sam James, Eugene Loh, Kris Van Hees)

* Mon Nov 11 2024 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.2-1
- Translators to support kernels 6.10 and later.
- FBT return probe argument support.
- The print() action is augmented with type information.  (Alan Maguire)
- Support to discover and trace USDT probes after a tracing session has
  started.  (Eugene Loh, Nick Alcock)
- USDT probe argument support (translated types, mapping).  (Nick Alcock)
- Installation locations are now configurable.  (Nick Alcock)
- Valgrind is no longer a required build dependency.  (Nick Alcock)
- Self-grabs have been improved.  (Nick Alcock)
- New provider: rawfbt.  (Kris Van Hees)
- Various bug fixes.  (Nick Alcock, Eugene Loh, Alan Maguire, Kris Van Hees)
- Various testsuite fixes and improvements.
  (Nick Alcock, Sam James, Eugene Loh, Kris Van Hees)
- Various code improvements.  (Nick Alcock, Eugene Loh, Kris Van Hees)
  [Orabug: 37274251]

* Mon May  6 2024 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.1-1
- Implement FBT probes with fentry/fexit tracepoints
- Provide argument info for FBT probes.
- Provide pre-generated translator files sets. [Orabug: 36504847]

* Tue Mar  5 2024 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.14
- Implement provider: io.  (Eugene Loh, Kris Van Hees)
- Implement actions: print().  (Alan Maguire)
- Implement subroutines: link_ntop(), cleanpath().  (Eugene Loh)
- Implement options: -xcpu, -xaggpercpu.  (Eugene Loh)
- Improve providers: pid (offset-based probes) and rawtp (arg info).
- Improve options: -xlockmem (improve default).  (Eugene Loh)
- Ensure USDT probes can survive dtprobed restarts.  (Nick Alcock)
- Improve USDT probe creation/deletion.  (Nick Alcock)
- Improve support for DTrace with upstream kernels.  (Nick Alcock)
- Improve support for compiling DTrace in older environments.  (Kris Van Hees)
- Add support for aggregations of stacks.  (Eugene Loh)
- Improve lexer parsing (top-level wildcard ambiguities and `numerals).
  (Nick Alcock)
- Fix END probe execution with multiple tracers.  (Nick Alcock)
- Preemptive BPF program execution for DTrace probes is not allowed.
- Buffer overrun fix for systems with non-sequential online CPU ids.
  (Kris Van Hees, Nick Alcock) [Orabug: 36356681]
- Various bug fixes.  (Nick Alcock, Eugene Loh, Kris Van Hees)
- Various testsuite fixes and improvements.
  (Nick Alcock, Eugene Loh, Kris Van Hees)
- Various code improvements.  (Nick Alcock, Eugene Loh, Kris Van Hees)
  [Orabug: 36329725]

* Thu Nov  9 2023 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.13.2
- Support using DTrace with upstream kernels.
- Implement provider: ip.
- Implement actions: trunc(), pcap().
- Implement subroutines: inet_ntoa6().
- Implement subroutines: inet_ntop(). (Eugene Loh)
- Support modules.builtin.ranges for builtin module-symbol association.
- Provide a BTF-to-CTF convertor to provide (limited) kernel type information
  when CTF is not available.
- Remove dependency on waitfd(). (Nick Alcock)
- Various testsuite fixes and improvements.
  (Nick Alcock, Eugene Loh, Kris Van Hees)
- Various code improvements. (Nick Alcock, Eugene Loh, Kris Van Hees)

* Wed Jun  7 2023 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.13.1
- Restart dtprobed when upgrading DTrace.
- Report and clean up orphaned tracing events after each test.

* Fri May 26 2023 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.13
- Full support for is-enabled USDT probes. (Nick Alcock)
- Report error on programs that exceed aggsize or dynvarsize.
- Support for drop counters for principal, speculation, and aggregation buffers
  and for dynamic variables.
- Implement probe: proc:::signal-clear.
- Implement provider: sched (partial implementation).
- Implement provider: lockstat (for kernels >= 5.10.0 and UEK6 with fix).
- Support NULL strings. (Eugene Loh)
- Support uregs[] on older kernels. (Eugene Loh)
- New option: lonknommap. (Nick Alcock)
- Support for USDT probes in programs in different fs namespaces. (Nick Alcock)
- Support for USDT probes in non-PIE executables. (Nick Alcock)
- Fix dtprobed to support DOF that exceeds 64KiB. (Nick Alcock)
  [Orabug: 35411920]
- Do not modify input files with dtrace -G if unchanged. (Steven Sistare)
  [Orabug: 35417184]
- Various testsuite fixes and improvements.
  (Nick Alcock, Eugene Loh, Kris Van Hees) [Orabug: 35435195]
- Various code improvements.  (Nick Alcock, Eugene Loh, Kris Van Hees)
  [Orabug: 35435195]

* Mon Feb 27 2023 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.12
- Fix evaluation order of bcopy() arguments and lift non-alloca restriction
  on the source address. (Eugene Loh, Kris Van Hees)
- Implement actions: clear(), tracemem(). (Kris Van Hees, Eugene Loh)
- Implement subroutines: copyout(), copyinstr(). (Kris Van Hees, Eugene Loh)
- Implement options: switchrate, aggrate.
- Implement provider: cpc. (Eugene Loh)
- Implement provider: proc.
- Implement built-in variable: uregs. (Eugene Loh)
- Increase strtab maximum size.
- Support using indirect load instructions for pointers to alloca()'d and
  DTrace managed memory. (Kris Van Hees, Eugene Loh)
- Fix arg0 and arg1 for profile-* and tick-* probes. (Eugene Loh)
- Implement runtime bounds checking for scalar array access. (Eugene Loh)
- Updated manpage and moved to dtrace.8.
- Support arbitrary address pointers for basename(), dirname(), strchr(),
  strrchr(), and inet_ntoa(). (Eugene Loh) [Orabug: 34857846]
- Add runtime bounds checking for scalar array access. (Eugene Loh)
  [Orabug: 35045463]
- Various testsuite fixes and improvements. [Orabug: 34829509]
- Various code improvements. [Orabug: 34829509]

* Tue Nov 08 2022 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.11.1
- Support both libfuse 2 and libfuse 3.

* Fri Oct 28 2022 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.11
- Add initial support for USDT. (Nick Alcock, Kris Van Hees)
- Add support for aggregation keys. (Eugene Loh, Kris Van Hees)
- Add support for copyin(), copyinto(), and copyinstr().
- Add support for built-in variable args[] and sdt probe arg types.
- Fix arg handling for various probes. (Eugene Loh)
- Add basic support for setopt().
- Add -xlockmem, with useful error message. (Eugene Loh)
- Fix -xverbose, -xcpp, and -xctfpath
- Fix handling of multiple args after --. (Nick Alcock)
- Have the pid provider ignore compiler-generated internal function names.
- Fix various bugs with typecasting and internal integer storage. (Eugene Loh)
- Fix access to scalars in kernel space.
- Fix libproc search of rtld_global due to glibc changes. (Nick Alcock)
  [Orabug: 32856318]
- Truly decouple per-CPU BPF agg maps with a "map of maps."
- Unused dual aggregation copies (DT_AGG_NUM_COPIES) have been removed.
  (Eugene Loh)
- Various testsuite fixes and improvements. [Orabug: 34251899]
- Various code improvements. [Orabug: 34251899]

* Tue Apr 26 2022 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.10
- Add support for associative arrays.
- Add support for allcoa() and bcopy(). (Nick Alcock)
- Add support for inet_ntoa(), progenyof(), getmajor(), getminor(),
  mutex_owned(), mutex_owner(), mutex_type_adaptive(), mutex_type_spin(),
  rw_read_held(), rw_write_held(), and rw_iswriter(). (Eugene Loh)
- Improved fault handling. (Nick Alcock, Kris Van Hees)
- Various disassembler improvements, esp. annotations.
- Strings are no longer stored using a length prefix.
- The trace() action now supports arrays, structs, and unions.
- Various testsuite fixes and improvements. [Orabug: 34112342]
- Various code improvements. [Orabug: 34112342]

* Mon Jan 31 2022 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.9.1
- Add support for UEK7. [Orabug: 33806867]

* Wed Dec 08 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.9
- Add support for strtok(). (Eugene Loh)
- Implement TLS (thread-local storage) variables.
- Add support for basename(), dirname(). (Eugene Loh)
- Generic hash table improvements and consistent use of htabs. (Nick Alcock)
- CTF improvements in view of better kernel support. (Nick Alcock)
- Add support for ftruncate(). (Eugene Loh)
- Add support for rand(). (Eugene Loh)
- Fix string constant handling for strings longer than strsize.
- Optimization of substr(), strjoin(), and storing strings in the trace buffer.
- Various string handling fixes and improvements. (Eugene Loh, Kris Van Hees)
  [Orabug: 33651682]
- Various testsuite fixes and improvements. [Orabug: 33651682]
- Various code improvements. [Orabug: 33651682]

* Fri Oct 15 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.8
- Support running dtrace under valgrind. (Nick Alcock) [Orabug: 32760574]
- Implementation of speculative tracing. (Nick Alcock)
- Add support for string comparison. (Eugene Loh)
- Add support for strchr(), strrchr(), index(), rindex(), strstr(), lltostr().
  (Eugene Loh)
- Add support for symbols in compressed kernel modules. (Nick Alcock)
- Add support for htonl, htonll, htons, ntohl, ntohll, ntohs. (Eugene Loh)
- Various testsuite fixes and improvements. [Orabug: 33474154]
- Various code improvements. [Orabug: 33474154]

* Thu Sep  9 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.7
- Implement argument retrieval for SDT probes.
- Introduce 'bpflog' runtime option to request BPF verifier log.
- Implementation improvements for memry copy operations.
- Fix type alignment handling for enums.
- Fix ERROR-in-BEGIN probe handling.
- Transition from variable-length string size to 2-byte string size.
- Fix size of string data in the trace output buffer.
- Fix data size for value copy.
- Add support for strjoin() subroutine.
- Add support for substr() subroutine.

* Fri Jun 18 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.6
- Consolidated development of newly ported features. [Orabug: 33037106]
- Implement built-in variables: probeprov, probemod, probefunc, probename.
- Implement built-in variables: caller, stackdepth, ucaller, ustackdepth,
  errno, walltimestamp.  (Eugene Loh)
- Implement actions: stack(), ustack(). (Eugene Loh)
- Implement subroutine: strlen().
- Implement option: -Z (partial). (Eugene Loh)
- Added support for string constants and string values.
- Added support for strings to trace().
- Fixed storage size determination for global and local variables.
- Improved storage layout for global and local variables.
- Mark aggregation actions as non-data recording actions.  (Eugene Loh)
- Load string constant table into the 'strtab' BPF map.
- Load probe description data into the 'probes' BPF map.
- Consolidate all string hash functions into a single hash function.
- Implement variable length integer support.
- Store the length of each string as a variable length integer inline with its
  character stream.
- Improve memcpy() internal BPF function.
- Testsuite improvements.  (Eugene Loh)

* Mon Apr 12 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.5.1
- Memory corruption fix in pid probe discovery.

* Fri Apr  9 2021 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.5
- Implement the ERROR probe. [Orabug: 32749498]
- Add checks in BPF programs for division by zero and NULL pointer dereference.
  [Orabug: 32749498]
- Implement the pid provider for userspace function boundary tracing.
  [Orabug: 32749504]
- Implement the -w option to explicitly allow destructive actions.
  [Orabug: 31632678]
- Add support in translators for more than two kernel variants.  (Nick Alcock)
  [Orabug: 32748968]
- Switch from condition variable to eventfd for process death notifications.
  [Orabug: 32125018]
- Implement normalize() and denormalize() actions.
- Reworking of kernel tracepoint based providers.  [Orabug: 32125018]
- Reworking of global and local variable storage to allow for values sizes
  beyond 8 bytes. (Eugene Loh) [Orabug: 32125018]
- Implement -xbpflogsize option. (Eugene Loh) [Orabug: 32125018]
- Improvements to -xdisasm support. (Eugene Loh) [Orabug: 32125018]
- Fix bitfield operations. (Eugene Loh) [Orabug: 32125018]
- Sample scripts for building DTrace on Ubuntu. (David McLean)
- Coding style updates. (Eugene Loh)
- Testsuite updates.

* Wed Dec  9 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.4
- Implement aggregation support code for both producer and consumer.
  [Orabug: 32254734]
- Implement aggregation functions: count(), lquantize().
- Implement aggregation functions: avg(), llquantize(), quantize().
  (Eugene Loh)
- Implement aggregation functions: max(), min(), stddev(), sum().
  (David McLean)
- Implement the printa() action.
- Rework BPF code generation to use emit*() macros.
- Fix bitwise negation. (Eugene Loh) [Orabug: 32125018]
- Fix reporting when all quantize() data is in the last bin. (Eugene Loh)
  [Orabug: 32148161]
- Load pre-compiled BPF functions even if they do not have relocations.
  (Eugene Loh)
- Testsuite updates.
- Fix END probe execution when consumer stops the producer.
- Force use of bash as shell.

* Thu Oct  1 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.3
- Implement the freopen() action. (Eugene Loh)
- Implement the system() action. (Eugene Loh)
- Implement the id built-in variable. (David McLean)
- Implement the ppid built-in variable. (Eugene Loh)
- Improve annotations in the disassembler.
- Implement proper BEGIN and END probe semantics.
- Correct exit() action semantics.
- Correct signed divide and modulo operations. (Eugene Loh)
- Correct code generation for post-decrement expressions. (Eugene Loh)
- Various compilation fixes.
- Better support for different distributions and compiler versions.

* Thu Aug  6 2020 Eugene Loh <eugene.loh@oracle.com> - 2.0.0-1.2
- Ensure a minimum locked-memory limit for BPF.
- int8_t is now always signed, even on platforms where char is unsigned.

* Fri Jul 24 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.1
- Implement profile-n and tick-n probes. (Eugene Loh)
- Implement the printf() action.
- Implement the raise() action.
- Support probe specifications with wildcards.
- Support multiple clauses for each probe.
- Support built-in variables: arg0 through arg9 (for probes that provide
  arguments), curcpu curthread, epid, gid, pid, tid, uid, and timestamp.
- Various memory management fixes.
- Register leak fixes.

* Thu Apr 30 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-1.0
- Implement BEGIN and END probes. (Eugene Loh) [Orabug: 31220513]
- Implement a probe cleanup mechanism for providers.
  (Eugene Loh) [Orabug: 31220519]
- Implement the exit() action. [Orabug: 31220520]
- Implement the flowindent option. [Orabug: 31220522]
- Implement the -xdisasm=n option. [Orabug: 31220524]
- Implement linking of precompiled BPF functions. [Orabug: 31220525]
- Fix memory leaks and minor uninitialized-data bugs. [Orabug: 31220517]
- Fix load-before-store BPF verifier errors for local variables.
  [Orabug: 31220527]
- Fix post-increment code generation. [Orabug: 31220528]
- Fix register clobbering and implement register spilling. [Orabug: 31187562]
- Add error and drop tag verification for tests. [Orabug: 31220521]
- Various code cleanup and errata release preparation. [Orabug; 31220516]
- Ensure that bpf_dlib.o is installed.
- Fix the creation of FBT return probes.

* Tue Mar 10 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.8
- Sync with latest development code.
- Implement perf event output buffer management and processing.
- Introduce dt_calloc() to use in all array allocations.
- Provide support for building against libdtrace-ctf and libctf, with
  preference given to libctf (if available).
- Implement association of BPF programs with probe events.

* Sun Jan 26 2020 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.7
- Update spec file for build requirements.
- Sync with latest development code.
- Ensure that pre-release banner does not interfere with test result
  comparison.

* Thu Dec  5 2019 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.6
- Update spec file for build requirements.

* Thu Dec  5 2019 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.5
- Support building on aarch64.

* Wed Dec  4 2019 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.4
- Support building on systems with older kernels and/or glibc.

* Tue Nov 26 2019 Kris Van Hees <kris.van.hees@oracle.com> - 2.0.0-0.3
- Change the kernel build version to 5.4.0-1948.2.

* Wed Aug 14 2019 Kris van Hees <kris.van.hees@oracle.com> - 2.0.0-0.2
- Change the kernel build version to 5.2.8-1933.

* Tue Jul 30 2019 Kris van Hees <kris.van.hees@oracle.com> - 2.0.0-0.1
- DTrace implementation based on BPF (initial portions).

* Fri Aug 10 2018 - <nick.alcock@oracle.com> - 1.1.0-0.1
- Add more DTRACE_PROBE definitions to sdt.h, for SystemTap
  compatibility, and test them (Tomas Jedlicka) [Orabug: 27721525]
- Fix disassembler coredump (Tomas Jedlicka) [Orabug: 28054399]
- Fix process-termination-related crash and deadlock bugs
 (Nick Alcock, Tomas Jedlicka) [Orabug: 27961105, 28133496]
- Fix deadlock on creation of new threads in -c/-p processes
  [Orabug: 28210986]
- Properly handle breakpoints tripped while DTrace is exiting
  [Orabug: 28473826]
- Do not crash on self-grabs and on races with dtrace termination
  [Orabug: 28361373]
- New ctfpath option, allowing explicit specification of ctf archives
  to use for the running kernel (Tomas Jedlicka) [Orabug: 28178265]
- Fix memory leaks and minor uninitialized-data bugs [Orabug: 28247636]
- Add test for SDT argument retrieval from stack, in UEK4QU7
  (Kris Van Hees) [Orabug: 25949088]
- Remove preallocation from the buffering testsuite (Tomas Jedlicka)
  [Orabug: 27998779]
- Speed up aggmod tests (Eugene Loh) [Orabug: 28007146, 28119700]
- Include the smoketests in make check [Orabug: 28128338]
- Test fixes (boost timeouts, kernel 4.17 compatibility,
  unskip a forgotten test).
- Improve testsuite temporary file creation (Eugene Loh)
  [Orabug: 28142056]
- Clean up compiler warnings (Eugene Loh) [Orabug: 27934422, 27998779]

* Thu May 10 2018 - <nick.alcock@oracle.com> - 1.0.2-1
- Testsuite fixes [Orabug: 27995907]

* Fri Apr 27 2018 - <nick.alcock@oracle.com> - 1.0.1-1
- ARM64 USDT and pid probe support (Kris Van Hees) [Orabug: 27847969]
- Fix -c option on ARM64 (Kris Van Hees) [Orabug: 27847946]
- Disable pid provider tests on UEK4.
- Testsuite fixes [Orabug: 27921244, 27919918]

* Thu Mar 29 2018 - <tomas.jedlicka@oracle.com> - 1.0.0-10
- Enabled tst.default.d in smoketests.

* Tue Mar 27 2018 - <tomas.jedlicka@oracle.com> - 1.0.0-9
- Updated translators to match new kernel redesign of per-task and
  per-process data. [Orabug: 27731759]
- Enhanced build system to support multiple kernels [Orabug: 27731756]
- Added pid provider support (Kris Van Hees) [Orabug: 27609459]
- Adjusted the interpretation of /proc/kallmodsyms for aarch64
  (Eugene Loh) [Orabug: 27214992]

* Thu Mar  8 2018 - <nick.alcock@oracle.com> - 1.0.0-8
- Change dtrace_proc_*() public interfaces to take a struct dtrace_proc,
  avoiding lifecycle issues that can cause access to freed memory when
  victims are exec()ing rapidly [Orabug: 27501199]
- Bump library soname to libdtrace.so.1 [Orabug: 27501199]
- Improve handling of symbol name->address and address->name lookup
  (Eugene Loh) [Orabug: 27214992]
- Check array bounds at compile time [Orabug: 27382268]
- Fix dtrace -S not printing actions other than the first (Tomas Jedlicka)
  [Orabug: 27565023]
- Drop expensive, unnecessary bad page state checking from testsuite
  [Orabug: 27577555]
- Handle interruptions in the testsuite runner better [Orabug: 27577593]
- Consider erratically failing tests in some directories to be passes
- Test fixes [Orabug: 27583002, 27651838]

* Fri Feb 16 2018 - <nick.alcock@oracle.com> - 1.0.0-7
- Internal development release.
- Several latent porting bugs fixed (Kris Van Hees)
- Make tests used in smoke testsuite rely on sdt less
  (Kris Van Hees)

* Tue Jan 30 2018 - <nick.alcock@oracle.com> - 1.0.0-6
- Internal development release.
- ARM64 support [Orabug: 27438960, 27438993, 27438977]
- Include correct procfs.h (Tomas Jedlicka) [Orabug: 27266725]

* Thu Jan 18 2018 - <nick.alcock@oracle.com> - 1.0.0-5
- Internal development release.
- Bring back translators for 4.14.

* Fri Jan 12 2018 - <nick.alcock@oracle.com> - 1.0.0-4
- Internal development release.
- Eliminate new symbol at address 0 on KPTI-enabled kernels
  [Orabug: 27364377]
- Boost some test timeouts

* Fri Jan  5 2018 - <nick.alcock@oracle.com> - 1.0.0-3
- Internal development release.
- New tests and test fixes (Kris Van Hees, Nicolas Droux)
  [Orabug: 27194352, 27282243]

* Fri Dec 15 2017 - <nick.alcock@oracle.com> - 1.0.0-2
- Internal development release.
- Fix rare assertion failures at exit [Orabug: 26848964]

* Thu Nov 30 2017 - <nick.alcock@oracle.com> - 1.0.0-1
- Release for smoketesters, not GA
- Fix mod() normalization: add test for it (Tomas Jedlicka, Eugene Loh)
  [Orabug: 26826564]
- Fix faulty logic in dtrace_modsym_update() adding symbols to wrong modules
  (Tomas Jedlicka, Eugene Loh) [Orabug: 26848467]
- ip provider ipv6_tclass and ipv6_flow are not set correctly (Nicolas Droux)
  [Orabug: 27193049]
- Translator changes for kernels 4.12 -- 4.14 (Nicolas Droux,
  Tomas Jedlicka, Nick Alcock)
- Add tagging capability to the testsuite (Vincent Lim, Eugene Loh)
  [Orabug: 26992620]
- Fix kills of hanging dtraces in the testsuite (Eugene Loh, Nick Alcock)
  [Orabug: 27051149]
- New tests for the lockstat provider and fixes to jstack and 3-arg
  tracemem tests (Eugene Loh, Alan Maguire) [Orabug: 26149894, 27015838]

* Mon Oct 16 2017 - <nick.alcock@oracle.com> - 0.6.2-3
- Apply correct version of lockstat patch [Orabug: 26149894]

* Mon Oct 16 2017 - <nick.alcock@oracle.com> - 0.6.2-2
- Add lockstat provider [Orabug: 26149894] (Alan Maguire)
- Add missing obsolete of dtrace-modules-provider-headers (Tomas Jedlicka)
  [Orabug: 26883486]
- Add missing testsuite NFS dependency (Tomas Jedlicka) [Orabug: 26860985]

* Thu Sep 21 2017 - <nick.alcock@oracle.com> - 0.6.2-1
- Internal development release.
- Fix segfault at shutdown time if grabbed processes die at
  precisely the wrong time [Orabug: 26528776]
- New llquantize() aggregation, providing log/linear results
  (Eugene Loh) [Orabug: 26675201]
- New optional third arg for tracemem(): dynamically-variable size
  limit (Eugene Loh) [Orabug: 26675604]
- Fix wrong wrong-number-of-args error messages [Orabug: 26402731]
  (Eugene Loh)
- Fix module address range merging (Eugene Loh) [Orabug: 25767469]
- Allow referencing of structure and union members named with
  the same name as D keywords, e.g. 'self' [Orabug: 26518086]
- Changes for the move of UAPI headers into dtrace-utils-devel.
  (Tomas Jedlicka, Nick Alcock).
- Support CTF in /lib/modules/$(uname -r)/kernel/vmlinux.ctfa
  archive [Orabug: 25815372]
- Testsuite fixes and new tests for inet_*() and lquantize
  (Eugene Loh, Alan Maguire, Nick Alcock)

* Wed Sep 13 2017 - <nick.alcock@oracle.com> - 0.6.1-3
- Relicense all of userspace, including the testsuite, to UPL.
- Merge NEWS from the modules into NEWS for userspace: there is
  only one NEWS now.
- Test fixes (Tomas Jedlicka, Nick Alcock) [Orabug: 26522961]
- make check-quick support, skipping long-running tests

* Fri Jul 14 2017 - <tomas.jedlicka@oracle.com> - 0.6.1-2
- Internal development release
- Fix name of lowest bucket in dtrace_print_lquantize() (Eugene Loh)
  [Orabug: 26261502]
- Fix wrong depth in dtrace_print_ustack() leading to garbage output
  from jstack() (Eugene Loh) [Orabug: 26045010]
- Do not build DTrace with debug info by default [Orabug: 26389302]
- Fix spurious errors and misreading of unaligned data on SPARC,
  breaking ustack() etc with upgraded glibc (Nick Alcock)
  [Orabug: 26378141]
  - Translator changes for the IO provider for NFS (Nicolas Droux)
  [Orabug: 26176938]
- Search for cpp and ld along the PATH, fixing use of dtrace -C with
  devtoolset-6 and other non-default GCCs (Nick Alcock) [Orabug: 26396530]
- Test fixes, test harness extensions (consumer tests) and new tests.
- dtrace-utils-devel now depends on elfutils-libelf-devel.
  (Nick Alcock)
- dtrace-utils-testsuite now depends on dtrace-devel.
  (Nick Alcock) [Orabug: 26415697]

* Thu May 25 2017 - <nick.alcock@oracle.com> - 0.6.1-1
- Add translators and tests for TCP and UDP (Alan Maguire)
  [Orabug: 25815242]
- Add translators for the IO provider (Nicolas Droux) [Orabug: 25816562]
- Add link_ntop() and tests for it (Girish Moodalbail) [Orabug: 25931511]
- sched.d no longer depends on the sched provider [Orabug: 26036143]
- Ship a default /etc/dtrace-modules (Nicolas Droux) [Orabug: 25918164]
- Repeated dtrace -G no longer corrupts the object file (Kris Van Hees)
  [Orabug: 22509887]

* Tue Mar 21 2017 - <nick.alcock@oracle.com> - 0.6.0-3
- Test fixes

* Fri Mar  3 2017 - <nick.alcock@oracle.com> - 0.6.0-2
- Internal development release
- Add translator for ip provider [Orabug: 25557249]
- Test fixes and new tests.

* Tue Jan 17 2017 - <nick.alcock@oracle.com> - 0.6.0-1
- Internal development release
- Allow self-grabs [Orabug: 24829169]
- Use /proc/pid/map_files if available [Orabug: 24843582]
- Fix fd leaks on big-endian systems and during heavy exec() [Orabug: 25040553]
- Add improved multi-argument DTRACE_PROBE macro [Orabug: 24678905]
- Fix infloops in SPARC breakpoint handling [Orabug: 24454127]

* Tue Nov  8 2016 - <nick.alcock@oracle.com> - 0.5.4-1
- Work around elfutils bug causing object file corruption [Orabug: 25059329]

* Wed Jul 20 2016 - <nick.alcock@oracle.com> - 0.5.3-2
- New tests

* Thu Apr 28 2016 - <nick.alcock@oracle.com> - 0.5.3-1
- Prevent intermittent assertion failures crashes and hangs when
  shutdown races with termination of a grabbed process [Orabug: 22824594]
  [Orabug: 23028026]

* Fri Feb  5 2016 - <nick.alcock@oracle.com> - 0.5.2-2
- Fix uregs array on SPARC [Orabug: 22602756]
- Testsuite fixes

* Tue Jan 12 2016 - <nick.alcock@oracle.com> - 0.5.2-1
- Do not crash USDT probe users when shared libraries are in the upper half of
  the address space [Orabug: 22384028]
- Do not waste CPU time busywaiting in a do-nothing ioctl()-heavy loop
  [Orabug: 22370283] [Orabug: 22335130]
- Testsuite triggers are synchronized with dtrace by default [Orabug: 22370283]
- Fix dtrace -c and -p on SPARC and improve error-handling paths
  [Orabug: 22390414]
- Fix smoketests on SPARC [Orabug: 22533468]

* Tue Dec  8 2015 - <nick.alcock@oracle.com> - 0.5.1-4
- Internal development release.
- Prevent testsuite hangs when per-arch options are in use [Orabug: 22030161]

* Thu Nov 19 2015 - <nick.alcock@oracle.com> - 0.5.1-3
- Fix buggy performance improvements to correctly detect out-of-tree modules
  (like dtrace.ko) and speed them up some more [Orabug: 22237449]
  [Orabug: 22238204]

* Wed Nov 18 2015 - <nick.alcock@oracle.com> - 0.5.1-2
- Internal development release.
- Remove typoed non-bug from rpm changelog.

* Wed Nov 18 2015 - <nick.alcock@oracle.com> - 0.5.1-1
- Internal development release.
- Improve startup performance when disk cache is cold [Orabug: 22185787]
  [Orabug: 22185763] [Orabug: 22083846]
- Fix various problems in the testsuite and in DTRACE_DEBUG output
  [Orabug: 21431540] [Orabug: 22170799]

* Wed Nov  4 2015 - <nick.alcock@oracle.com> - 0.5.0-4
- Improve identification of system daemons that must not be ptraced
  unless explicitly specified [Orabug: 21914902]
- Improve symbol resolution in the absence of ptrace() [Orabug: 22106441]
- Fix dtrace -p with an invalid PID [Orabug: 21974221]
- Close any inherited fds before running testsuite [Orabug: 21914934]

* Wed Sep 23 2015 - <nick.alcock@oracle.com> - 0.5.0-3
- Internal development release
- No longer reference UEK3 kernels on SPARC.
- Do not require 32-bit glibc on SPARC.

* Wed Sep  9 2015 - <nick.alcock@oracle.com> - 0.5.0-2
- Internal development release
- No longer Provide: an unversioned dtrace-utils [Orabug: 21622263]
- Add missing testsuite package deps [Orabug: 21663841] [Orabug: 21753123]
- Fix check-module-loading testsuite target [Orabug: 21759323]
- Test logfiles should not be affected by the verbosity of the test run
  [Orabug: 21769905]

* Wed Aug 12 2015 - <nick.alcock@oracle.com> - 0.5.0-1
- Internal development release
- SPARC64 support. [Orabug: 19005071]
- Translator support for 4.1 kernel.

* Tue Jun 30 2015 - <nick.alcock@oracle.com> - 0.4.6-4
- Add DTrace release and SCM version info via dtrace -Vv [Orabug: 21351062]
- Add source-tree-independent testsuite RPM (not distributed)
- Fix the testsuite module-loading pre-checks to actually work
  [Orabug: 21344988]
- Various build system fixes

* Tue Jun 23 2015 - <nick.alcock@oracle.com> - 0.4.6-3
- Internal development release
- Fix deadlocks and failures to latch processes for symbol lookup caused
  by failure to correctly track their state over time, in 0.4.6-1+ only.

* Mon Jun 22 2015 - <nick.alcock@oracle.com> - 0.4.6-2
- Internal development release
- Fix a rare race causing stalls of fork()ed children of traced processes
  under load, in 0.4.6-1 only [Orabug: 21284447]

* Thu Jun 18 2015 - <nick.alcock@oracle.com> - 0.4.6-1
- Internal development release
- Support multiple kernels with a single userspace tree, loading system
  D libraries from directories named like /usr/lib64/dtrace/3.8.
  [Orabug: 21279908]
- Processes being userspace-traced can now receive SIGTRAP.
  [Orabug: 21279300]
- dtrace-utils-devel now depends on the same version of dtrace-utils.
  [Orabug: 21280259]
- No longer lose track of processes that exec() while their dynamic linker
  state is being inspected. [Orabug: 21279300]
- No longer assume that the symbol table of processes that are no longer
  being monitored is unchanged since it was last inspected. [Orabug: 21279300]
- Properly remove breakpoints from fork()ed children. [Orabug: 21279300]

* Mon Feb 16 2015 - <nick.alcock@oracle.com> - 0.4.5-3
- The dependencies are adjusted to pick up the renamed dtrace headers package.
  [Orabug: 20508087]

* Tue Nov 18 2014 - <nick.alcock@oracle.com> - 0.4.5-2
- A number of crashes when out of memory are fixed. [Orabug: 20014606]

* Thu Oct 23 2014 - <nick.alcock@oracle.com> - 0.4.5-1
- Automatically load provider modules from /etc/dtrace-modules, if present
  [Orabug: 19821254]
- Fix intermittent crash on failure of initial grabs or creations of processes
  via dtrace -c, -p, or u*() functions [Orabug: 19679998]
- Reliably track and compensate for processes undergoing execve()s
  [Orabug: 19046684]
- Handle processes hit by stopping signals correctly [Orabug: 18674244]
- Fix a sign-extension bug in breakpoint instruction poking [Orabug: 18674244]
- Fix some broken tests (Kris Van Hees) [Orabug: 19616155]
- Robustify DTrace against changes to glibc's internal data structures
  [Orabug: 19882050]
- Fix DIF subr names in dtrace -S output [Orabug: 19881997]

* Tue Jul 22 2014 - <nick.alcock@oracle.com> - 0.4.4-2
- Ensure that the DOF ELF object does not require execstack
  (Kris Van Hees) [Orabug: 19217436]

* Tue Jul  8 2014 - <nick.alcock@oracle.com> - 0.4.4-1
- New -xcppargs option as part of fixes for the testsuite on OL7
  [Orabug: 19054052]

* Tue May 13 2014 - <nick.alcock@oracle.com> - 0.4.3-1
- Fix array underrun when no textual mapping for the executable can be found
  [Orabug: 18550863]
- Fix unlikely buffer overrun at process-map-read time [Orabug: 18550863]
- Fix traversal of realloc()ed pointer which could lead to textual mappings
  being spuriously missed [Orabug: 18550863]
- Fix error-path dereference of uninitialized variable in error message
  [Orabug: 18550863]

* Thu May  1 2014 - <nick.alcock@oracle.com> - 0.4.2-2
- Interrupting dtrace with a SIGINT while monitored processes are dying no
  longer hangs dtrace on a condition variable [Orabug: 18689795]
- Symbol lookups on processes that died at the same instant now always fail
  and no longer access freed memory [Orabug: 18550863]

* Wed Apr 16 2014 - <nick.alcock@oracle.com> - 0.4.2-1
- killing dtrace while a ustack() is in progress no longer risks killing
  crucial system daemons [Orabug: 18600515]
- Fix a leak of filehandles to executables [Orabug: 18600594]
- Fix ustack() of multithreaded processes [Orabug: 18412802]
- Get the pid and ppid right for multithreaded processes [Orabug: 18412802]
- Fix an uninitialized memory read looking up certain kernel symbols
  [Orabug: 18603463]
- Fixes for newer versions of make, ld, and bison [Orabug: 18551552]

* Tue Jan  7 2014 - <nick.alcock@oracle.com> - 0.4.1-1
- Install showUSDT in docdir. (Kris van Hees) [Orabug: 17968414]
- Install ctf_module_dump. [Orabug: 17968381]
- A lexer bug was fixed causing spurious errors if D scripts contained a
  pragma or comment at intervals of 8192 characters, and preventing
  the use of scripts >16KiB entirely. [Orabug: 17742866]
- Fix devinfo_t's dev_statname and dev_pathanme for cases where the
  device does not have partitions. (Kris van Hees) [Orabug: 17973698]
- A variety of memory leaks and uninitialized memory reads are fixed.
  [Orabug: 17743019]
- Improve drti.o to minimize overhead when DTrace is not running.
  [Orabug: 17973604]
- Emit errors from drti.o on stderr, not stdout. [Orabug: 17973604]
- Use O_CLOEXEC when opening files in drti.o. [Orabug: 17973604]
- Fix RPM dependencies; automatically install and modprobe the dtrace
  modules as needed. [Orabug: 17804881]

* Wed Oct 16 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-8
- Fix format of RPM changelog
- Add missing RPM changelog entries

* Wed Oct 16 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-7
- never released, necessary for release management

* Wed Oct 16 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-6
- Fix visibility of .SUNW_dof sections in dtrace -G object files.
  (Kris van Hees) [Orabug: 17476663]
- Fix typos in changelog and specfile copyright date

* Tue Sep 17 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-5
- avoid deadlocking when doing process operations during dtrace -l.
  [Orabug: 17442388]

* Fri Aug 16 2013 Kris van Hees <kris.van.hees@oracle.com> - 0.4.0-4
- Support for USDT in shared libraries.

* Fri Aug 16 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-3
- never released, necessary for release management

* Fri Aug 16 2013 Kris van Hees <kris.van.hees@oracle.com> - 0.4.0-2
- never released, necessary for release management

* Tue Jul 23 2013 Nick Alcock <nick.alcock@oracle.com> - 0.4.0-1
- ustack() support and symbol lookups.
- USDT support.  dtrace -G works.
- evaltime option now works.
- DTrace headers largely moved to dtrace-modules-headers.
- DTRACE_OPT_* environment variables now set options.
  DTRACE_DEBUG=signal emits debugging output on SIGUSR1 receipt.

* Fri Aug 31 2012 Nick Alcock <nick.alcock@oracle.com> - 0.3.0-1
- CTF support.
- Fixed install path for dtrace libraries.
- Fixed -c and -p options.
- Faster startup.
- Split out a -devel package.

* Mon Mar 19 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.5-2
- Call ldconfig at appropriate times.

* Tue Mar 13 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.5
- libdtrace is now a shared library, with non-stable API/ABI.

* Thu Feb 16 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.4
- Updated README; new NEWS and PROBLEMS; synch with module version

* Thu Feb  9 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.3
- Fixes for reproducibility of test results under load
- Fix -G when setting the syslibdir

* Mon Feb  6 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.2
- Fix spurious failures of tst.resize*.d.

* Tue Jan 31 2012 Nick Alcock <nick.alcock@oracle.com> - 0.2.1
- Fix 'make check-installed' with an unbuilt source tree.

* Thu Jan 26 2012 Kris van Hees <kris.van.hees@oracle.com> - 0.2.0
- Branch for 0.2.0 release.

* Fri Oct  7 2011 Philip Copeland <philip.copeland@oracle.com> - 0.1-0.3
- The systemtap package in the 'wild' creates a dtrace.1 manpage
  which is bizarre since it doesn't have an associated dtrace
  binary. This will cause a conflict and the rpm will not install
  Since that man page is superfluous I've had to add a %post
  section here to move our manpage into position in such a way as
  to keep the rpm database happy. Technically this is a bit evil.

* Thu Oct 06 2011 Nick Alcock <nick.alcock@oracle.com> - 0.2
- Fix copyright.

* Tue Sep 27 2011 Nick Alcock <nick.alcock@oracle.com> - 0.1
- Branch for initial release.

* Mon Jun 27 2011 Pearly Zhao <pearly.zhao@oracle.com> - 0.0.1
- Initial build for dtrace.
