/*
Copyright (c) 2022 Advanced Micro Devices, Inc. All rights reserved.
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include <hip_test_common.hh>
#include <performance_common.hh>
#include <resource_guards.hh>

/**
 * @addtogroup event event
 * @{
 * @ingroup PerformanceTest
 */

class HipEventSynchronizeBenchmark : public Benchmark<HipEventSynchronizeBenchmark> {
 public:
  void operator()(unsigned flag) {
    hipEvent_t event;
    HIP_CHECK(hipEventCreateWithFlags(&event, flag));
    HIP_CHECK(hipEventRecord(event));

    TIMED_SECTION(kTimerTypeCpu) { HIP_CHECK(hipEventSynchronize(event)); }

    HIP_CHECK(hipEventDestroy(event));
  }
};

static void RunBenchmark(unsigned flag) {
  HipEventSynchronizeBenchmark benchmark;
  if (flag == hipEventDefault) {
    benchmark.AddSectionName("Default event");
  } else {
    benchmark.AddSectionName("Blocking sync event");
  }
  benchmark.Run(flag);
}

/**
 * Test Description
 * ------------------------
 *  - Executes `hipEventSynchronize`
 *    -# Checked on events created with flags:
 *      - hipEventDefault
 *      - hipEventBlockingSync
 * Test source
 * ------------------------
 *  - performance/event/hipEventSynchronize.cc
 * Test requirements
 * ------------------------
 *  - HIP_VERSION >= 5.2
 */
TEST_CASE("Performance_hipEventSynchronize") {
  const auto flag = GENERATE(hipEventDefault, hipEventBlockingSync);
  RunBenchmark(flag);
}
