/* File:      CONSULT.P
** Author(s): Kostis F. Sagonas, David S. Warren
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: consult.P,v 1.79 2010/01/21 19:29:54 dwarren Exp $
** 
*/

:- compiler_options([xpp_on,sysmod]).
#include "standard.h"
#include "char_defs.h"
#include "flag_defs_xsb.h"
#include "extensions_xsb.h"
#include "incr_xsb_defs.h"

:- mode compile(+).
compile(Path) :- compile_opt_checked(Path, [], 2).	% compile with no options

:- mode compile(+,+).
compile(Paths,Options):-
	check_nonvar(Options,'compile/2',2),
	check_compile_options(Options),
	compile_opt_checked(Paths, Options, 2).

/* compile_opt_checked(+FileList,+Options,+ToLoc), compiles the list
of files/modules in FileList.  It assumes Options to be correct.
If ToLoc = 0, it compiles to .xwam
If ToLoc = 1, it compiles to private .xwam-loc (but does not rename it)
If ToLoc = 2, it compiles to private .xwam-loc and renames it to .xwam
*/

:- mode compile_opt_checked(+,+,+).
compile_opt_checked(X, _, _) :-
	var(X), !,
	instantiation_error(compile/[1,2],1).
compile_opt_checked([], _, _) :- !.
compile_opt_checked([H|T], Options, ToLoc) :- !,
	\+ (\+ (compile0(H,Options,ToLoc))),
	compile_opt_checked(T, Options, ToLoc).
compile_opt_checked(Path, Options, ToLoc) :-
    \+ (\+ (compile0(Path,Options,ToLoc))).

%------

compile0(Path, Options, ToLoc) :-
	atom(Path),
	search_module(Path, Dir, Mod, SExt, Base, _Obj),
	!,
	(SExt \== XSB_OBJ_EXTENSION_ATOM_SANS_DOT
	 -> compile_f(SExt, Base, Options, ToLoc, _, Mod, Dir),
	    (ToLoc =:= 2
	     ->	str_cat(Dir,Mod,FileBase),
		str_cat(FileBase,XSB_OBJ_EXTENSION_ATOM,OFileName),
		private_obj_filename(OFileName,UFileName),
		rename_private_obj_file(UFileName,OFileName)
	     ;	true
	    )
	 ;	permission_error(compile,'file/module',Path,compile/[1,2])
	).
compile0(Path, _, _) :-
	(var(Path)
	 ->	instantiation_error(compile/[1,2],1)
	 ; \+ atom(Path)
	 ->	type_error(atom,Path,compile/[1,2],1)
	 ;	existence_error('file/module',Path,compile/[1,2],1)
	).

compile_f(XSB_SRC_EXTENSION_ATOM_SANS_DOT, Base, Opts, ToLoc, ExList, Mod, _) :- !,
	compile(Mod, Base, XSB_SRC_EXTENSION_ATOM, Opts, ToLoc, ExList).
compile_f(pl, Base, Opts, ToLoc, ExList, Mod, _) :- !,
	compile(Mod, Base, '.pl', Opts, ToLoc, ExList).
compile_f(prolog, Base, Opts, ToLoc, ExList, Mod, _) :- !,
	compile(Mod, Base, '.prolog', Opts, ToLoc, ExList).
compile_f('c', Base, Opts, ToLoc, ExList, Mod, Dir) :- !,
	compile_cH(Mod, Base, '.c ',Opts, ToLoc, ExList, Dir).
compile_f('cpp', Base, Opts, ToLoc, ExList, Mod, Dir) :- !,
	compile_cH(Mod, Base, '.cpp ', Opts, ToLoc, ExList, Dir).
compile_f('', Base, Opts, ToLoc, ExList, Mod, _) :- !,
	str_cat(Base, XSB_SRC_EXTENSION_ATOM, FileName),
	make_filecopy(Base, FileName),
	compile(Mod, Base, XSB_SRC_EXTENSION_ATOM, Opts, ToLoc, ExList),
	rm(FileName).

make_filecopy(OFileName,NFileName) :-
	xsb_configuration(os_type, OS_type),
	atom(OS_type),
	(is_wind(OS_type)
	 ->	sys_filecopy(OFileName,NFileName,_)
	 ;	sys_link(OFileName,NFileName,_)
	).

is_wind(OS_type) :- str_sub('windows',OS_type), !.
is_wind(OS_type) :- str_sub('cygwin',OS_type).

/*======================================================================*/
/* Predicates consult/[1,2] are thereafter similar to reconsult/[1,2].	*/
/*======================================================================*/

:- mode consult(+).
consult(Path) :- reconsult(Path, []).

:- mode consult(+,+).
consult(Path, Options) :- \+ (\+ (reconsult(Path, Options))).

:- mode [+|+].
[File|Rest] :- reconsult(File), ( Rest == [] -> true ; call(Rest) ).

/*======================================================================*/
/*======================================================================*/

:- thread_shared consult_file_loaded(_).
:- dynamic consult_file_loaded(_).
:- mode_on_success(consult_file_loaded(+)).

:- mode reconsult(+).
reconsult(Path) :- reconsult(Path, []).

:- mode reconsult(+,+).
reconsult(Path, Options) :- reconsult(Path, Options, false).

reconsult(Path, Options, Ensure_loaded) :- 
    check_nonvar(Path,'reconsult/1',1),
    check_atom(Path,'reconsult/1',1),
    \+ (\+ (reconsult0(Path, Options, Ensure_loaded))),!.
reconsult(Path, Options, _Ensure_loaded) :- 
    misc_error(['Unknown error while (re)consuting ',Path,' with options ',Options]).

/* TLS: restricting to only one thread for now.  Its possible that if
 * the file hasn't been loaded before, we could load it, but I'm not 
 * doing that yet. 

   Added over-ride flag for LogTalk.  */

check_one_thread_w_override:- 
	((stat_flag(RECOMPILE_OVERRIDE,X),X == 1) -> 
	    true
	 ; check_one_thread(load,file,reconsult/3) ).

reconsult0(Path, Options, Ensure_loaded) :-
    (Path == user
     ->	check_one_thread_w_override,
	compile_load_user(Options)
     ; search_module(Path, Dir, Mod, SExt, Base, Obj)
     ->	expand_filename(Obj,ExpObj),
	(SExt \== XSB_OBJ_EXTENSION_ATOM_SANS_DOT,
	 i_want_compile(SExt, Base, Obj)
	 -> check_one_thread_w_override,
	    compile_f(SExt,Base,Options,1,_ExList,Mod,Dir),
	    private_obj_filename(Obj,LObj),
	    load_exec(LObj, Mod, Dir), % Mod would be atomized ModTerm
	    rename_private_obj_file(LObj,Obj),
	    (file_is_consultloaded(ExpObj)
	     ->	true
	     ;	record_file_consultloaded(ExpObj)
	    )
	 ;  (Ensure_loaded==true
	     ->	(file_is_consultloaded(ExpObj)
		 -> true
		 %% File not loaded, ok to load w. many threads
		 ;  load_exec(Obj, Mod, Dir),
		    record_file_consultloaded(ExpObj) 
		)
	     ;	%% Ensure_loaded == false 
	     check_one_thread_w_override,
	     load_exec(Obj, Mod, Dir),
	     (file_is_consultloaded(ExpObj)
	      -> true
	      ;	 record_file_consultloaded(ExpObj)
	     )
	    )
	)
     ;  existence_error('file/module',Path,consult/1,1)
    ).

i_want_compile(SExt, Base, Obj) :-
	( not(file_exists(Obj)) -> true
	;
	 needs_recompile(SExt, Base, Obj) ).

needs_recompile(XSB_SRC_EXTENSION_ATOM_SANS_DOT, Base, Obj) :-
	( str_cat(Base, XSB_SRC_EXTENSION_ATOM, PFileName),
	    path_sysop(newerthan,PFileName,Obj)
	)
	;
	(str_cat(Base, XSB_HDR_EXTENSION_ATOM, HFileName),
	  file_time(Obj, OTime),
	  %% If no header file exists, then HTime = 0
	  file_time(HFileName, HTime), 
	  OTime =< HTime
	).	  
needs_recompile(pl, Base, Obj) :-
	( str_cat(Base, '.pl', PFileName),
	  file_time(Obj, OTime), 
	  file_time(PFileName, PTime),
	  OTime =< PTime
	)
	;
	( str_cat(Base, XSB_HDR_EXTENSION_ATOM, HFileName),
	  file_time(Obj, OTime),
	  %% If no header file exists, then HTime = 0
	  file_time(HFileName, HTime), 
	  OTime =< HTime
	).	  
needs_recompile(prolog, Base, Obj) :-
	( str_cat(Base, '.prolog', PFileName),
	  file_time(Obj, OTime), 
	  file_time(PFileName, PTime),
	  OTime =< PTime
	)
	;
	( str_cat(Base, XSB_HDR_EXTENSION_ATOM, HFileName),
	  file_time(Obj, OTime),
	  %% If no header file exists, then HTime = 0
	  file_time(HFileName, HTime), 
	  OTime =< HTime
	).	  
needs_recompile('', Base, Obj) :-
	file_time(Obj, OTime), 
	file_time(Base, PTime),
	OTime =< PTime.
needs_recompile('c', Base, Obj) :-
	xsb_configuration(os_type, OS_type), atom(OS_type),
	( ( str_sub('solaris', OS_type) -> str_cat(Base, '.so', C_Obj)
          ; (str_sub('linux', OS_type) -> str_cat(Base, '.so', C_Obj))
          ; (str_sub('bsd', OS_type) -> str_cat(Base, '.so', C_Obj))
	  ; str_sub('windows', OS_type) -> str_cat(Base, '.dll', C_Obj)
	  ; str_sub('cygwin', OS_type) -> str_cat(Base, '.dll', C_Obj)
	  ; str_sub(darwin, OS_type) -> str_cat(Base, '.dylib', C_Obj)
	  ; str_cat(Base, '.o', C_Obj)
	  ),
	  ( file_exists_here_or_configbin_or_configlib(C_Obj,CC_Obj) -> 
	      str_cat(Base, '.c', CFileName),
	      path_sysop(newerthan, CFileName, CC_Obj)
	  ; true
	  )
	)
	; 
	( str_cat(Base, XSB_HDR_EXTENSION_ATOM, HFileName),
	    path_sysop(newerthan, HFileName, Obj)
	)
        ; object_file_magic_mismatch(Obj).

file_exists_here_or_configbin_or_configlib(C_Obj,CC_Obj) :-
	(file_exists(C_Obj)
	 ->	CC_Obj = C_Obj
	;
	    parse_filename(C_Obj, _, Basename, ObjExt),
	    slash(Slash),
	    str_cat(Slash,Basename,Base1),
	    str_cat(Base1,'.',Base2),
	    str_cat(Base2,ObjExt,ObjFileTail),
	    (
	      xsb_configuration(config_bindir,ConfigBinDir),
	      str_cat(ConfigBinDir,ObjFileTail,BindirObjfile),
	      file_exists(BindirObjfile),
	      CC_Obj = BindirObjfile
	    ;
	      xsb_configuration(config_libdir,ConfigLibDir),
	      str_cat(ConfigLibDir,ObjFileTail,LibdirObjfile),
	      file_exists(LibdirObjfile),
	      CC_Obj = LibdirObjfile
	    )
	).

/* As of 11/15, magic numbers have the form 17 18 19 N, where N is 
   6 if it is the start of a Prolog predicate block.
   7 if the object file is Prolog code (mutable)
   8 if the object file is foreign code and was compiled by the seq engine (mutable)
   9 if the object file is foreign code and was compiled by the MT engine (mutable)
   a if the object file is Prolog code (immutable)
   b if the object file is foreign code and was compiled by the seq engine (immutable)
   c if the object file is foreign code and was compiled by the MT engine (immutable)
*/
/* To be called only on .xwam files */
object_file_magic_mismatch(Obj):- 
	xsb_configuration(engine_mode,Emode),
	get_magic_num(Obj,[_,_,_,Four]),
	\+ (engine_mode_magic_num(Emode,_,LastDigit),Four == LastDigit).

:- mode engine_mode_magic_num(?,?,?).
engine_mode_magic_num('slg-wam',mutable,8).
engine_mode_magic_num('multi-threading',mutable,9).
engine_mode_magic_num('slg-wam',immutable,11).
engine_mode_magic_num('multi-threading',immutable,12).

:- mode get_magic_num(+,?).
get_magic_num(ObjFile,[One,Two,Three,Four]) :- 
	file_open(ObjFile,r,Istr),
	file_get(Istr,One),file_get(Istr,Two),file_get(Istr,Three),file_get(Istr,Four),
	file_close(Istr).

/*======================================================================*/
/*  search_module(+FileName, -Dir, -Mod, -SExt, -Base, -Obj)	        */
/*                                                                      */
/*	Given a FileName (must be an atom), search_module/6 finds the	*/
/*	location of the corresponding module and returns:		*/
/*		1) the directory (Dir),					*/
/*		2) the module name (Mod),				*/
/*		3) the extension (SExt) of the source file		*/
/*		   ('P'/'c'/''/pl/[other name]/[object file only]), 	*/
/*		4) the base name of the file (Base = Dir+Mod), and	*/
/*		5) the object file name (Obj)				*/
/* 	The call will fail if the specified file cannot be found.	*/
/*     	       	       	       	       	       	       	       	        */
/*======================================================================*/
:- mode search_module(+,?,?,?,?,?).
search_module(FileName, Dir, Mod, Ext, Base, Obj) :-
	almost_search_module(FileName, D, M, E, B),
	( D == '', (FileName == M ; E == XSB_SRC_EXTENSION_ATOM_SANS_DOT)
	 ->	% only a module name was given
		Mod = M,
		real_search_module(Mod, Dir, Base, Ext)
	 ; acceptable_extension(E)
	 ->	Dir = D, Mod = M, Ext = E, Base = B,
		(Dir \== '', Ext == XSB_OBJ_EXTENSION_ATOM_SANS_DOT
		 ->	add_xsb_obj_ext_if_nec(FileName,OFileName),
			file_exists(OFileName)
		 ;	true
		)
	 ;	Dir = D,
		Ext = '',
		str_cat(B,'.',B1),
		str_cat(B1,E,Base),
		Mod = Base
	),
	str_cat(Base, XSB_OBJ_EXTENSION_ATOM, Obj).

add_xsb_obj_ext_if_nec(FileName,OFileName) :-
	(concat_atom([_,'.',XSB_OBJ_EXTENSION_ATOM_SANS_DOT],FileName)
	 ->	OFileName = FileName
	 ;	concat_atom([FileName,'.',XSB_OBJ_EXTENSION_ATOM_SANS_DOT],OFileName)
	).

acceptable_extension('').
%%acceptable_extension('P').
acceptable_extension(XSB_SRC_EXTENSION_ATOM_SANS_DOT).
acceptable_extension(XSB_OBJ_EXTENSION_ATOM_SANS_DOT).
acceptable_extension('p').
acceptable_extension('pl').
acceptable_extension('c').
acceptable_extension('cpp').

/*======================================================================*/
/* real_search_module(+ModName, -Dir, -Base, -Ext)			*/
/*======================================================================*/

real_search_module(ModName, Dir, Base, Ext) :-
        libpath(Dir),
        str_cat(Dir, ModName, Base),
        existing_file_extension(Base, Ext).

/*======================================================================*/
/* load_exec(+File, +ModName, +Directory)				*/
/*	Changed to use the C loader (code_load/4) that automatically	*/
/*	imports all exported predicates of the module into the current	*/
/*	working module when its third argument is 1.  - Kostis (4/3/93)	*/
/* The loader returns int 4 when a foreign file has been loaded.  	*/
/*======================================================================*/

%% dsw changed (8/28/19) to use loader:load_object_file (which was generatized to support this.)
load_exec(File, ModName, _Directory) :-	
    split_xsb_filename(File,Directory,BaseMod,XExt),
    load_object_file([Directory,BaseMod,XExt],ModName,1).

split_xsb_filename(Filename,Dir,Mod,Ext) :-
    xsb_filename(Filename,FullFile),
    atom_codes(FullFile,FileCodes),
    (llappend(FileNoSuff,ExtCodes,FileCodes), ExtCodes = [0'.|_]
     ->	true
     ;	FileNoSuff = FileCodes,
	ExtCodes = []
    ),
    (llappend(DirCodes1,[0'/|ModCodes],FileNoSuff)
     ->	llappend(DirCodes1,[0'/],DirCodes)
     ;	ModCodes = FileNoSuff,
	DirCodes = "./"
    ),
    atom_codes(Dir,DirCodes),
    atom_codes(Mod,ModCodes),
    atom_codes(Ext,ExtCodes).


cd_throw(Dir,Ball) :-
	cd(Dir),
	throw(Ball).

%% 4th byte in loadable xwam file...
%%immutable(10).
%immutable(11).
%%immutable(12).

compile_load_user(Options) :-
	compile(user, user, XSB_SRC_EXTENSION_ATOM, Options, 0, _), !,
	str_cat(user,XSB_OBJ_EXTENSION_ATOM,UserObjFile),
	load_exec(UserObjFile, user, '.'),
	rm(UserObjFile).

/*****************************************************************
add_lib_dir(+LibSpecs) adds libraries to library_directory/1, if they
are not already there.  LibSpecs is a "comma list" of library
specifications.  A library specification is either an atom which is
the directory name, absolute or relative, or it is a list of atoms,
which when concatenated, is the directory name.
*****************************************************************/
:- mode add_lib_dir(+).
add_lib_dir([]) :- !.
add_lib_dir([A|B]) :- !,
	add_lib_dir(A),
	add_lib_dir(B).
add_lib_dir((A,B)) :- !,
	add_lib_dir(A),
	add_lib_dir(B).
add_lib_dir(Path) :-
	add_lib_dir1(Path).

add_lib_dir1(a(Path)) :- !,
	standardize_pathname(Path, FullPath),
	check_dir_exists(FullPath),
	retractall(library_directory(FullPath)), % remove if it was there
	asserta(library_directory(FullPath)).	 % put at front.
%% push onto the top of the stack - don't remove anything
add_lib_dir1(push(Path)) :- !,
	standardize_pathname(Path, FullPath),
	check_dir_exists(FullPath),
	asserta(library_directory(FullPath)).	 % put at front.
add_lib_dir1(Path) :-
	standardize_pathname(Path, FullPath),
	check_dir_exists(FullPath),
	(library_directory(FullPath)
	 -> true
	 ;  assert(library_directory(FullPath))
	).

check_dir_exists(FullPath) :-
	(is_directory(FullPath)
	 ->	true
	 ;	warning(('add_lib_dir/1 or /2: Directory does not exist: ',FullPath),'Runtime/P')
	).

:- mode add_lib_dir(+,+).
add_lib_dir(_Root,[]) :- !.
add_lib_dir(Root,[A|B]) :- !,
	add_lib_dir(Root,A),
	add_lib_dir(Root,B).
add_lib_dir(Root,(A,B)) :- !,
	add_lib_dir(Root,A),
	add_lib_dir(Root,B).
add_lib_dir(ancestordir(AncName),Path0) :- !,
	path_sysop(cwd_ancestordir,AncName,Root),
	(Path0 = a(Path)
        ->	add_lib_dir1(a([Root,'/',Path]))
        ;	add_lib_dir1([Root,'/',Path0])
	).
add_lib_dir(Root,Path0) :-
	(Path0 = a(Path)
        ->	add_lib_dir1(a([Root,'/',Path]))
        ;	add_lib_dir1([Root,'/',Path0])
	).

%% extend to multiple dirs, if needed
:- mode remove_lib_dir(+).
remove_lib_dir(Dir) :-
        standardize_pathname(Dir,StandardDir),
        retract(library_directory(StandardDir)).

:- mode removeall_lib_dir(+).
removeall_lib_dir(Dir) :-
        standardize_pathname(Dir,StandardDir),
        retractall(library_directory(StandardDir)).

:- dynamic library_directory/1.
:- mode has_lib_dir(?).
has_lib_dir(D) :-
        (var(D) -> library_directory(D)
        ; standardize_pathname(D,StandardD),
            library_directory(StandardD)
        ).

/*-------------define load_dyn for reading in dynamic predicates -----*/

:- mode read_canonical(+,?).
read_canonical(S_or_A,Term) :- 
	check_open_stream_or_alias(S_or_A,'read_canonical/2',1,input,Istr),
	file_read_canonical(Istr,Term,_).

:- mode read_canonical(?).
read_canonical(Term) :- 
	current_input_port(File), 
	file_read_canonical(File,Term,_).

:- mode cvt_canonical(+,+).
cvt_canonical(InFile,OutFile) :-
	atom(InFile), atom(OutFile),
	seeing(OldInFile),
	expand_filename(InFile,InFilename),
	see(InFilename),
	telling(OldOutFile),
	expand_filename(OutFile,OutFilename),
	tell(OutFilename),
	repeat,
	read(Term),
	(Term == end_of_file
	 -> 	told, tell(OldOutFile), seen, see(OldInFile),!
	 ;	expand_term(Term,Term1),
		write_canonical_list(Term1),
		fail
	).

write_canonical_list([]) :- !.
write_canonical_list([Term|Terms]) :- !,
	write_canonical_list(Term),write_canonical_list(Terms).
write_canonical_list(Term) :-
    (Term = (:-(op(A,B,C))),
     integer(A),atom(B),atom(C)
     ->	op(A,B,C)
     ;	true
    ),
    write_canonical(Term),
    writeln('.').

%-------------------
	   
load_dyn_gen(File,Options) :- 
    check_nonvar(File,'load_dyn_gen/1',1),
    check_atom(File,'load_dyn_gen/1',1),
    process_ldg_options(Options,Canonical,AorZ,Module,Limit),
    (Canonical == yes ->
	load_dync_nocheck(File,AorZ,Module,Limit)
      ; load_dyn_nocheck(File,AorZ,Module) ).

process_ldg_options([canonical(Canonical)|T],Canonical,AorZ,Module,Limit):- !,
    ( member(Canonical,[yes,no]) ->
	 process_ldg_options(T,Canonical,AorZ,Module,Limit)
       ; domain_error(load_domain_gen_canonical_option,Canonical,load_dyn_gen/2,2) ).
process_ldg_options([ordering(AorZ)|T],Canonical,AorZ,Module,Limit):- !,
    ( member(AorZ,[a,z]) ->
	 process_ldg_options(T,Canonical,AorZ,Module,Limit)
       ; domain_error(load_domain_gen_ordering_option,AorZ,load_dyn_gen/2,2) ).
process_ldg_options([module(Module)|T],Canonical,AorZ,Module,Limit):- !,
    ( atom(Module) ->
	 process_ldg_options(T,Canonical,AorZ,Module,Limit)
       ; type_error(atom,Module,load_dyn_gen/2,2) ).
process_ldg_options([error_limit(Limit)|T],Canonical,AorZ,Module,Limit):- !,
    ( integer(Limit) ->
	 process_ldg_options(T,Canonical,AorZ,Module,Limit)
       ; type_error(atom,Limit,load_dyn_gen/2,2) ).
process_ldg_options([H|_T],_Canonical,_AorZ,_Module,_Limit):- !,
    domain_error(load_domain_gen_option,H,load_dyn_gen/2,2).
process_ldg_options([],Canonical,AorZ,Module,Limit):-
    (var(AorZ) -> AorZ = z ; true),
    (var(Canonical) -> Canonical = no ; true),
    (var(Module) -> Module = usermod ; true),
    (var(Limit) -> current_prolog_flag(load_dync_error_limit,Limit) ; true).

%------------------- load_dyn

:- mode load_dyn(+).
load_dyn(File) :-
	load_dyn(File,1,usermod).

:- mode load_dyn(+,+).
load_dyn(File,DirIn) :-
	load_dyn(File,DirIn,usermod).

:- export load_dyna/1.
load_dyna(File) :-
	check_nonvar(File,'load_dyna/1',1),
	check_atom(File,'load_dyna/1',1),
	load_dyn(File,0,usermod).

%% TES: changed behavior to throw error rather than fail to make
%% compatable with other ISO type errors, e.g. open/[3,4].  
%% TES allow calling with a or z 
:- mode load_dyn(+,+,+).
load_dyn(File,DirIn,Module) :-
	check_nonvar(File,'load_dyn/1',1),
	check_atom(File,'load_dyn/1',1),
	check_one_thread(load,file,'load_dyn/2'),
	load_dyn_nocheck(File,DirIn,Module).

load_dyn_nocheck(File,DirIn,Module) :-
	load_dyn_convert(DirIn,Dir,load_dyn),
	get_fname(File,Dirctry,SExt,Base,Mod),
	(SExt == XSB_OBJ_EXTENSION_ATOM_SANS_DOT
	 ->	existence_error('Prolog file/module',Base,load_dyn/[1,2],1)
	 ;	load_dyn0(Dirctry,SExt,Base,Mod,Dir,Module)
	).

load_dyn_convert(a,0,_):- !.
load_dyn_convert(z,1,_):- !.
load_dyn_convert(0,0,_):- !.
load_dyn_convert(1,1,_):- !.
load_dyn_convert(Dir,_,Pred):- 
	domain_error('a or z',Dir,Pred,2).

get_fname(Filename,Dirctry,SExt,Base,Mod) :-
	( search_module(Filename,Dirctry,Mod,SExt,Base,_Obj) -> true
         ; existence_error('file/module',Filename,load_dyn/[1,2],1)
%	; warning('Cannot find file'(Filename)), fail
	).

load_dyn0(Dirctry,SExt,Base,Mod,Dir,Module) :-
	get_HRname(SExt,Base,Hfname,Rfname),
	expand_filename(Rfname,ARfname),
	init_for_load(ARfname, Mod),
	fmod_from_base(Base,FMod),
	(Hfname == []
         ->     true
         ;      load_dyn1(Hfname,Dirctry,Dir,FMod,Module) % ? into module?
        ),
	load_dyn1(ARfname,Dirctry,Dir,FMod,Module),
	set_module_type(Mod),
	set_module_type(Module),
	cleanup_for_load.

get_HRname(SExt,Base,Hfname,Rfname) :-
	(SExt == ''	% no suffix
	 ->     Rfname = Base,
		Hfname = []
	 ;      dotted_ext(SExt, DExt),
	        str_cat(Base, DExt, Rfname),
		(SExt == XSB_SRC_EXTENSION_ATOM_SANS_DOT
		 ->	str_cat(Base, XSB_HDR_EXTENSION_ATOM, Hfname0),
			(file_exists(Hfname0)
			 ->	Hfname = Hfname0
			 ;	Hfname = []
			)
		 ;	Hfname = []
		)
	).

dotted_ext(Ext,DExt) :- str_cat('.',Ext,DExt).

fmod_from_base(Base,FMod) :-
	(atom_codes(Base,BaseC),
	 llappend(_,[Slash|FModC],BaseC),
	 (Slash =:= CH_SLASH ; Slash =:= CH_BACKSLASH)
	  ->	 atom_codes(FMod,FModC)
	  ;	 FMod = Base
	 ).

llappend([X|L1],L2,[X|L3]) :- llappend(L1,L2,L3).
llappend([],L,L).

load_dyn1(Rfname,Dirctry,Dir,Base,Module) :-
	cputime(T0),
	seeing(OldFile),
	open_or_xpp_file(Rfname, XPP_process),
	current_input_port(XPPport), %% port from process, if it was launched
	(   catch(load_dyn_read_and_assert(Dir,Base,Module),Ball,close_and_throw(XPP_process,XPPport,Ball))
	%% If needs restart, close old file and skip to the second load_dyn1
	 -> ( conget(needs_restart,1)
	     ->	seen, see(OldFile),
		seeing(Str),
		file_set_character_set(Str,utf_8),
		fail
	     ; true
	    )
	;   conset(needs_restart, 0), conset(xpp_on,0)
	),
	!,
	execute_accumulated_commands(Dirctry),
	stat_set_flag(CURRENT_MODULE,0), % reset to usermod
	check_xpp_process_status(Rfname,XPP_process,'Dynamic loading aborted'),
	%% Release the file descriptor used to read from preprocessor
	(XPP_process == skip -> true
	;
	    file_close(XPPport)
	),
	seen,
	see(OldFile),
	(   banner_control(quietload)
	->  true
	;   time_message(T0,(Rfname,' dynamically loaded'))
	).
load_dyn1(Rfname,Dirctry,Dir,Base,Module) :- 
	%% If we are restarting due to gpp, then reset needs_restart.
	conget(needs_restart,1), conset(needs_restart, 0),
	load_dyn1(Rfname,Dirctry,Dir,Base,Module).

close_and_throw(XPP_process,XPPport,Ball) :-
	(XPP_process == skip
	 ->	true
	 ;	file_close(XPPport)
	),
	seen,
	stat_set_flag(CURRENT_MODULE,0),  % reset to usermod
	throw(Ball).

:- mode_on_success(load_dyn_trie_retracted(+)).
:- dynamic load_dyn_trie_retracted(_).
:- mode_on_success(load_dyn_retracted(+,+,+,+,+)).
:- dynamic load_dyn_retracted(_,_,_,_,_).
:- mode_on_success(load_dyn_file_loaded(+,+)).
:- dynamic load_dyn_file_loaded(_,_).
:- mode_on_success(load_dyn_pred_loaded(+,?,+,+)).
:- dynamic load_dyn_pred_loaded(_,_,_,_).
:- mode_on_success(load_dyn_file(+,+)).
:- dynamic load_dyn_file(_,_).
:- mode_on_success(load_dyn_module(+)).
:- dynamic load_dyn_module(_).
:- mode_on_success(load_dyn_apply(+)).
:- dynamic load_dyn_apply(_).  %arities of apply already seen

init_for_load(Filename,_Mod) :-
	conset(needs_restart, 0), conset(xpp_on, 0), % Cleanup before XPP
	file_time(Filename,Time),
        remove_file_dynloaded(Filename),
        record_file_dynloaded(Filename,Time),
	retractall(load_dyn_trie_retracted(_)),
	retractall(load_dyn_retracted(_,_,_,_,_)),
	get_mod_from_filename(Filename,PMod),
	assert(load_dyn_file(Filename,PMod)),
	retractall(load_dyn_pred_loaded(Filename,_,_,_)),
	retractall(load_dyn_module(_)),
	retractall(load_dyn_apply(_)),
	init_in_mod.

get_mod_from_filename(Filename,Mod) :-
    atom_codes(Filename,Filecodes),
    (llappend(FileNoSuff,[0'.|_Suff],Filecodes)
     ->	true
     ;	FileNoSuff = Filecodes
    ),
    (llappend(_Pre,[C|ModCodes],FileNoSuff),
     (C =:= 0'/ ; C =:= 0'\\)
      -> true
      ;	 ModCodes = FileNoSuff
     ),
     atom_codes(Mod,ModCodes).

init_in_mod :-
	retractall(in_mod(_,_,_)),
	assert(in_mod([_|_],usermod,known)).

cleanup_for_load :-
	conset(needs_restart, 0), conset(xpp_on, 0), % Cleanup after XPP
	retractall(load_dyn_trie_retracted(_)),
	retractall(load_dyn_retracted(_,_,_,_,_)),
	retractall(load_dyn_file(_,_)),
	retractall(load_dyn_module(_)),
	retractall(load_dyn_apply(_)).
%%	retractall(in_mod(_,_,_)).

%% set module psc to have module type
set_module_type(Mod) :-
    (Mod \== usermod
     ->	conpsc(Mod,MPsc),
	psc_set_type(MPsc,4)
     ;  true
    ).

%% This version is used for load_dyn1 only!
load_dyn_read_and_assert(Dir,Base,Module) :-
	read(Term1),
	expand_term(Term1,Terms),
	(is_list(Terms)
	 ->	member(Term,Terms) % since expand_term can return a list
	 ;	Term = Terms
	),
	(Term == end_of_file
	 ->	!, fail
	 ;	do_assert_and_fail(Term,Dir,Base,Module)
	).
%% Don''t loop, if restart is requested
load_dyn_read_and_assert(_,_,_) :- conget(needs_restart,1), !.
load_dyn_read_and_assert(Dir,Base,Module) :- load_dyn_read_and_assert(Dir,Base,Module).

%------------------- load_dync

:- mode load_dync(+).
load_dync(File) :-
	load_dync(File,1,usermod).	% assertz in usermod

:- mode load_dynca(+).
load_dynca(File) :-
	load_dync(File,0,usermod).	% asserta in usermod

:- export load_dynca/2.
:- mode load_dynca(+,+).
load_dynca(File,Module) :-
	load_dync(File,0,Module).

:- mode load_dync(+,+).
load_dync(File,DirIn) :-
	load_dync(File,DirIn,usermod).

:- mode load_dync(+,+,+).
load_dync(File,Dirin,Module) :-
	check_one_thread(load,file,'load_dync/[1,2,3]'),
	check_atom(File,'load_dync/[1,2,3]',1),
	check_one_thread(load,file,'load_dync/2'),
	stat_flag(LOAD_DYNC_ERROR_LIMIT,ErrLimit),
	load_dync_nocheck(File,Dirin,Module,ErrLimit).
	
load_dync_nocheck(File,Dirin,Module,ErrLimit) :-
	load_dyn_convert(Dirin,Dir,load_dync),
	get_fname(File,Dirctry,SExt,Base,Mod),
	load_dync0(Dirctry,SExt,Base,Mod,Dir,Module,ErrLimit).

load_dync0(Dirctry,SExt,Base,Mod,Dir,Module,ErrLimit) :-
	get_HRname(SExt,Base,Hfname,Rfname),
	expand_filename(Rfname,ARfname),
	init_for_load(ARfname, Mod),
	fmod_from_base(Base,FMod),
	(Hfname == []
         ->     true
         ;      load_dync1(Hfname,Dirctry,Dir,FMod,Module,ErrLimit)
        ),
	load_dync1(ARfname,Dirctry,Dir,FMod,Module,ErrLimit),
	set_module_type(Mod),
	set_module_type(Module),
	cleanup_for_load.

load_dync1(Rfname,Dirctry,Dir,Base,Module,ErrLimit) :-
	cputime(T0),
	seeing(OldFile),
	open_or_xpp_file(Rfname, XPP_process),
	current_input_port(XPPport),
	conset('_$load_dync_line_no',0),
	conset('_$load_dync_error_cnt',0),
%%	(Module == usermod
%%	 ->	true
%%	 ;	psc_insertmod(Module,4,ModPsc),
%%		stat_set_flag(CURRENT_MODULE,ModPsc)
%%	),
	file_read_canonical(-1000,0,0),	% initialize previous psc
	(   catch(load_dync_read_and_assert(XPPport,Dir,Base,Module,ErrLimit),Ball,
		  close_and_throw(XPP_process,XPPport,Ball))

	%% If needs restart, close old file and skip to the second load_dync1
	->  (conget(needs_restart,1) -> seen, see(OldFile), fail ; true)
	;   conset(needs_restart, 0), conset(xpp_on,0)
	),
	!,
	execute_accumulated_commands(Dirctry),
	stat_set_flag(CURRENT_MODULE,0),  % reset to usermod
	check_xpp_process_status(Rfname,XPP_process,'Dynamic loading aborted'),
	%% Release the file descriptor used to read from preprocessor
	(XPP_process == skip -> true
	;
	    file_close(XPPport)
	),
	seen,
	see(OldFile),
	(   banner_control(quietload)
	->  true
	;   time_message(T0,(Rfname,' dynamically loaded'))
	).
load_dync1(Rfname,Dirctry,Dir,Base,Module,ErrLimit) :- 
	%% If we are restarting due to gpp, then reset needs_restart.
	conget(needs_restart,1), conset(needs_restart, 0),
	load_dync1(Rfname,Dirctry,Dir,Base,Module,ErrLimit).


%% This version is used for load_dync1 only!
load_dync_read_and_assert(IPort,Dir,Base,Module,ErrLimit) :-
	coninc('_$load_dync_line_no'),
	file_read_canonical(IPort,Term,Opsc),	% Opsc: PSC ptr of prev (and this) if prev same; otw 0
	(Opsc =:= 0
	 ->	true
	 ; not_asserted_term(Term)
	 ->	true
	 ;	('_$trie_asserted'(Term) 
		 ->	term_psc(Term,Psc),
			(load_dyn_trie_retracted(Psc)
			 ->	true
			 ;	term_new(Psc,Gen),
				retractall(Gen),
				asserta(load_dyn_trie_retracted(Psc))
			),
			t_assert(Term, _Flag)
		 ;	load_dyn_retracted(Opsc,Prref,NArity,Index,Hashsize),
			se_assert_code_to_buff(Term),
			assert_buff_to_clref(Term,NArity,Prref,Dir,Index,Hashsize)
		),
		fail
	),
	(Term == end_of_file
	 ->	!,
		fail
	 ; Term == read_canonical_error
	 ->	conget('_$load_dync_line_no',BadLineNo),
		file_write0(STDMSG,'     read_canonical warning at (or about) term number (and line number if one term per line) '),
		file_write0(STDMSG,BadLineNo),
		file_nl(STDMSG),
		coninc('_$load_dync_error_cnt',ErrCnt),
		(ErrCnt > ErrLimit
		 ->	!,
                        syntax_error('',['More than ',25,' non canonical terms in the input ',Base,': loading aborted'])
%			file_write0(STDMSG,'Too many read_canonical warnings: loading aborted'),
%			file_nl(STDMSG)
		 ;	true
		),
		fail
	 ;	do_assert_and_fail(Term,Dir,Base,Module)
	).
load_dync_read_and_assert(_IPort,_Dir,_Base,_Module,_ErrLimit) :-
	conget(needs_restart,1), % if needs restart
	!,			% don't continue to read
	conset('_$load_dync_line_no',0),
	conset('_$load_dync_error_cnt',0).
load_dync_read_and_assert(IPort,Dir,Base,Module,ErrLimit) :-
	load_dync_read_and_assert(IPort,Dir,Base,Module,ErrLimit).


not_asserted_term(:-(_)).
not_asserted_term(?-(_)).
not_asserted_term(_Mod:_Term).
not_asserted_term((_:-_)).
%%not_asserted_term((_Mod:_Head:-_Body)).
%%%not_asserted_term(Term) :- '_$trie_asserted'(Term).

:- dynamic load_dyn_cmds/1.
:- index(load_dyn_cmds/1,0).

execute_accumulated_commands(Dir) :-
	cwd(OldDir),
	cd(Dir),
	(load_dyn_cmds(Cmd),
	 (catch(Cmd,Ball,cd_throw(OldDir,Ball))
	  ->	 fail
	  ;	 warning('Command failed.'),
		 fail
	 )
	 ;
	 cd(OldDir),
	 retractall(load_dyn_cmds(_))
	).

do_assert_and_fail([],_,_,_) :- !, fail.
do_assert_and_fail(:-(Cmd),_Dir,Base,Mod) :- !, % DIRECTIVE
	(load_dyn_module(RMod)
	 ->	true
	 ;	RMod = Mod
	),
	Cmd =.. [CmdName|CmdArgs],
	(CmdName == export, CmdArgs = [PAs]
	 ->	(load_dyn_module(RMod0)
		 ->	(RMod0 == Base
			 ->	true
                         ; RMod0 == Mod
			 ->     true
			 ;	permission_error(redefine,'assumed module, export must precede ANY definition/use',
						 p/n,load_dyn/1)
			),
			NRMod = RMod
		;       (RMod == usermod
			 -> assert(load_dyn_module(Base)),
			    NRMod = Base
			 ;  assert(load_dyn_module(RMod)),
			    NRMod = RMod
			)
		),
		add_to_in_mod(PAs,NRMod),
		assert(load_dyn_cmds(import(from(PAs,NRMod))))
	 ; proc_directive(CmdName,CmdArgs,RMod)
	 ->	true
	 ;	fix_body_goal(Cmd,RMod,Cmd1),
		assert(load_dyn_cmds(Cmd1))
	),
	fail.
do_assert_and_fail(?-(Cmd),_Dir,_Base,Mod) :- !, % COMMAND
	(load_dyn_module(RMod)
	 ->	true
	 ;	RMod = Mod
	),
	fix_body_goal(Cmd,RMod,Cmd1),
	assert(load_dyn_cmds(Cmd1)),
	fail.
do_assert_and_fail(:-(Head0,Body0),Dir,_Base,Mod) :- !, % RULE
	(load_dyn_module(RMod)
	 ->	true
	 ;	RMod = Mod
	),
	check_set_mod(RMod),
	fix_head_term(Head0,RMod,Head),
	(predicate_property(Head,built_in)
	 ->	warning('Cannot assert to builtin '(Head)),fail
	 ;	true
	),
	('_$multifile'(Head)
	 ->	(functor(Head,apply,AArity)
		 ->	(load_dyn_apply(AArity)
			 ->	true
			 ;	proc_multifile(Head),
				assert(load_dyn_apply(AArity))
			)
		 ;	true
		),			
		multifile_cvt_head(Head,MHead),
		conpsc(MHead,Psc), conpsc(Head,OPsc)
	 ; functor(Head,apply,AArity)  % an apply that has not been declared mf
	 ->	proc_multifile(Head),
		assert(load_dyn_apply(AArity)),
		multifile_cvt_head(Head,MHead),
		conpsc(MHead,Psc), conpsc(Head,OPsc)
	 ;	MHead = Head,
		conpsc(MHead,Psc), conpsc(Head0,OPsc)
	),
	('_$trie_asserted'(MHead) 
	 ->	warning('Asserting a nonfact into a trie, ignored'((Head:-Body0))),
		fail
	 ;	true
	),
	(load_dyn_retracted(OPsc,Prref,NArity,Index,Hashsize)
	 ->	true
	 ;	term_new(Psc,Gen),
		retractall(Gen),
		psc_arity(Psc,Arity),
		set_retracted(MHead,Arity,OPsc,Psc,Prref,Index,Hashsize),
		NArity is Arity+1
	),
	(var(Body0)
	 ->	Nbody = call(Body0)
	 ;	fix_body_goal(Body0,RMod,Body),
		goal_cut_trans(Body,Nbody,Cutpoint)
	),
	(Body \== Nbody
	 ->	MHead =.. Hlist,
		append(Hlist,[Cutpoint],Nhlist),
		Nhead0 =.. Nhlist, %%% FIX TO KEEP IN SAME MOD, CHECK THAT GOAL_CUT_TRANS PRESERVES MODS
		term_new_mod(RMod,Nhead0,Nhead),
		se_assert_code_to_buff((Nhead:-Nbody))
	 ;	se_assert_code_to_buff((MHead:-Body))
	),
	assert_buff_to_clref(MHead,NArity,Prref,Dir,Index,Hashsize),
	fail.
do_assert_and_fail(Head0,Dir,_Base,Mod) :- % FACT
	(load_dyn_module(RMod)
	 ->	true
	 ;	RMod = Mod
	),
	check_set_mod(RMod),
	fix_head_term(Head0,RMod,Head),
	(predicate_property(Head,built_in)
	 ->	warning('Cannot assert to builtin'(Head)),fail
	 ;	true
	),
	('_$multifile'(Head)
	 -> multifile_cvt_head(Head,MHead),
	    (('_$multifile_comp'(MHead,Head) ; load_dyn_cmds(Head))
	     ->	true % was multifile, and this one is, too
	     ;	permission_error('Non-multifile predicate',Head,'but previously loaded as multifile','load_dyn')
	    ),
	    (functor(Head,apply,AArity)
		 ->	(load_dyn_apply(AArity)
			 ->	true
			 ;	proc_multifile(Head),
				assert(load_dyn_apply(AArity))
			)
		 ;	true
	    ),			
	    conpsc(MHead,Psc), conpsc(Head0,OPsc)
	 ; functor(Head,apply,AArity)  % an apply that has not been declared mf
	 ->	proc_multifile(Head),
		assert(load_dyn_apply(AArity)),
		multifile_cvt_head(Head,MHead),
		conpsc(MHead,Psc), conpsc(Head0,OPsc)
	 ;	MHead = Head,
		conpsc(MHead,Psc), conpsc(Head0,OPsc)
	),
	('_$trie_asserted'(MHead) 
	 ->	(load_dyn_trie_retracted(OPsc)
		 ->	true
		 ;	term_new(Psc,Gen),
			retractall(Gen),
			asserta(load_dyn_trie_retracted(OPsc))
		),
		t_assert(MHead, _Flag)
	 ;	(load_dyn_retracted(OPsc,Prref,NArity,Index,Hashsize)
		 ->	true
		 ;	term_new(Psc,Gen),
			retractall(Gen),
			psc_arity(Psc,Arity),
			set_retracted(MHead,Arity,OPsc,Psc,Prref,Index,Hashsize),
			NArity is Arity+1
		),
		se_assert_code_to_buff(MHead),
		assert_buff_to_clref(MHead,NArity,Prref,Dir,Index,Hashsize)
	),
	fail.

se_assert_code_to_buff(Term) :- assert_code_to_buff(Term),fail.
se_assert_code_to_buff(_).


multifile_cvt_head(Head,MHead) :-
	Head =.. [Pred|Args],
	atom(Pred),		% groundedness inferencing too weak for head of list.
	load_dyn_file(_,ModName),
	get_p_mod(Pred,ModName,LinkPred),
	MHead =.. [LinkPred|Args].

/* In set_retracted/7, Head is the predicate which has been transformed by
   multifile processing.  So p(_,_) may have been transformed to 
   p_file1(_,_). 

   OPsc is the Original Psc (e.g. for p(_,_)), while Psc is the actually
   used Psc (e.g. for p_file1(_,_)).  OPsc is only used in 
   load_dyn_retracted(OPsc,Prref,NArity,Index,Hashsize).  So, after
   file_read_canonical(IPort,Term0,OPsc) reads a new Term0 (say p(_,_)),
   and the psc address is the same as that of the previous read term,
   then this term can be asserted directly.   
*/

set_retracted(Head,Arity,OPsc,Psc,Prref,Index,Hashsize) :-
	('_$index'(Head,Index,Hashsize) -> true
        ; Arity =:= 0
        ->  Index = 0,default_assert_size(Hashsize)
        ;   Index = 1,default_assert_size(Hashsize)
	),
	NArity is Arity+1,      % to hold the cut addr
	psc_type(Psc, SYMTYPE),
	(SYMTYPE =\= 1 ->  dynamic(Head)
        ;  true
	),
	db_get_prref(Psc,Prref),
	asserta(load_dyn_retracted(OPsc,Prref,NArity,Index,Hashsize)),
	load_dyn_file(Filename,_),
	asserta(load_dyn_pred_loaded(Filename,Head,Index,Hashsize)).

proc_directive(encoding,[Charset],_) :- !,
    (atom(Charset),
     valid_charset(Charset)
     ->	seeing(Str),
	file_set_character_set(Str,Charset)
     ;	warning((Charset,' is an illegal file encoding identifier, ignored !'))
    ).

proc_directive(export,[_],_RMod) :- !, % should have been handled above
	warning('export directive ignored.').
proc_directive(local,[X],RMod) :- !,
	add_to_in_mod(X,RMod).
proc_directive(dynamic,[X],RMod) :-
	proc_dynamic(X,RMod).
proc_directive(import,[from(X,Mod)],_RMod) :- !,
	add_to_in_mod(X,Mod).
proc_directive(import,[as(from(PredIn,Mod),PredAs)],RMod) :- !,
	mpa_to_skel(PredIn,PredInSkel),
	term_new_mod(Mod,PredInSkel,MPredInSkel),
	term_psc(MPredInSkel,PscIn),
	mpa_to_skel(PredAs,PredAsSkel),
	term_new_mod(RMod,PredAsSkel,MPredAsSkel),
	term_psc(MPredAsSkel,PscAs),
	psc_import_as(PscIn,PscAs).
proc_directive(document_import,[_],_RMod) :- !,
	true.
proc_directive(index,[X],RMod) :- !,
	proc_index(X,RMod).
proc_directive(index,[Ps,Arg,Size],RMod) :- !,
    check_ground(Arg,'index/3',2),
    check_integer(Size,'index/3',2),
    multifile_cvt(Ps,RMod,Skel),
    index(Skel,Arg,Size).
proc_directive(index,[Ps,trie],RMod) :- !,
	multifile_cvt(Ps,RMod,Skel),
	index(Skel,trie).
proc_directive(index,[Ps,Arg],RMod) :- !,
    multifile_cvt(Ps,RMod,Skel),
    check_ground(Arg,'index/3',2),
    index(Skel,Arg,0).
proc_directive(mode,[_X],_RMod) :- !.
	%warning('mode directive ignored.').
proc_directive(parallel,[_X],_RMod) :- !.
	%warning('parallel directive ignored.').
proc_directive(comment,[_,_],_RMod) :- !.
	%warning('comment directive ignored.').
proc_directive(table,[as(Pred,TabOpts)],RMod) :- !,
	proc_table(Pred,RMod),  % do first
	proc_table_options(TabOpts,Pred).
proc_directive(table,[Pred],RMod) :- !,
	proc_table(Pred,RMod).
proc_directive(table_all,[],_RMod) :- !, 
	warning('table_all directive ignored. Use table/n explicitly').
proc_directive(thread_shared,[Preds],_RMod) :- !,
	thread_shared(Preds).
proc_directive(thread_private,[Preds],_RMod) :- !,
	thread_private(Preds).
proc_directive(op,[P,T,S],_RMod) :- !,
    check_integer(P,'op/3',1),
    check_atom(T,'op/3',2),
    check_atom(S,'op/3',3),
    op(P,T,S).
proc_directive(hilog,[X],_RMod) :- !, check_ground(X,'hilog/1',1), add_hilog_symbol(X).
proc_directive(multifile,[Specs],_RMod) :- !,
	proc_multifile(Specs).
proc_directive(compiler_options,[Options],_RMod) :- !,
	(xpp_is_off ->  conset(migh_need_restart,1) ;   true),
	check_file_compiler_options(Options),
	setoptions1(Options).

%:- import (use_subsumptive_tabling)/1,(use_variant_tabling)/1,add_incr_table/2 from tables.
% Table options; only some handled.  Might try to do better.
proc_table_options((TabOpt1,TabOpt2),Pred) :- !,
	proc_table_options(TabOpt1,Pred),
	proc_table_options(TabOpt2,Pred).
proc_table_options(incremental,_Pred) :- !,
	domain_error(dynamic_table_option,incremental,(table)/1,1,
	['must be one of ','dynamic,dyn,subsumptive,variant,nonincremental,opaque,private,shared,approximate,intern']).
proc_table_options(Option,Pred) :- !,
	exec_table_option(Option,Pred).

add_to_in_mod((Preds1,Preds2),Mod) :- !,
	add_to_in_mod(Preds1,Mod),
	add_to_in_mod(Preds2,Mod).
add_to_in_mod(PSpec,Mod) :-
	(PSpec = Pred/Arity
	 ->	true
	 ;	functor(PSpec,Pred,Arity)
	),
	functor(GTerm,Pred,Arity),
	(in_mod(GTerm,OMod,Type)
	 ->	(Mod == OMod
		 ->	true
		 ;	functor(GTerm,P,N),
			(Type == known
			 ->	permission_error(redefine,'imported (or exported) predicate',
						 OMod:P/N,load_dyn/1)
			 ;	permission_error(redefine,'assumed module, import/export must precede use',
						 OMod:P/N,load_dyn/1)
			)
		)
	 ;	assert(in_mod(GTerm,Mod,known))
	).

/** old **
multifile_cvt(Ps,Mod,Skel) :-
       mpa_to_skel(Ps,Skel0),
       ('_$multifile'(Skel0)
        ->     load_dyn_file(Mod),
               functor(Skel0,Pred,Arity),
               get_p_mod(Pred,Mod,P_Mod),
               functor(Skel,P_Mod,Arity)
        ;      fix_head_term(Skel0,Mod,Skel)
       ).
**/
multifile_cvt(Ps,Mod,Skel) :-
	mpa_to_skel(Ps,Skel0),
	('_$multifile'(Skel0)
	 -> load_dyn_file(_,PMod),
	    functor(Skel0,Pred,Arity),
	    get_p_mod(Pred,PMod,P_File),
	    functor(Skel,Mod,P_File,Arity)
	 ;  fix_head_term(Skel0,Mod,Skel)
    ).

proc_index((Pred, Preds),Mod) :- !,
	proc_index(Pred,Mod),
	proc_index(Preds,Mod).
proc_index(Pname/Arity-Arg,Mod) :- !,
	multifile_cvt(Pname/Arity,Mod,Skel),
	check_integer(Arg,'index/3',2),
	index(Skel, Arg, 0).
proc_index(Pname/Arity,Mod) :- 
	multifile_cvt(Pname/Arity,Mod,Skel),
	index(Skel, 1, 0).

proc_table((Pred, Preds),Mod) :- !,
	proc_table(Pred,Mod),
	proc_table(Preds,Mod).
proc_table(Pname/Arity,Mod) :- 
	functor(Term,Pname,Arity),
	fix_head_term(Term,Mod,MTerm),
	table(MTerm).

proc_dynamic((Pred, Preds),Mod) :- !,
	proc_dynamic(Pred,Mod),
	proc_dynamic(Preds,Mod).
proc_dynamic(Pname/Arity,Mod) :- 
	functor(Term,Pname,Arity),
	fix_head_term(Term,Mod,MTerm),
	dynamic(MTerm).

proc_multifile((Spec1,Spec2)) :- !,
	proc_multifile(Spec1),
	proc_multifile(Spec2).
proc_multifile(Spec) :-
	mpa_to_skel(Spec,Spec1),
	functor(Spec1,Pred,Arity),
	load_dyn_file(_,ModName),
	get_p_mod(Pred,ModName,LinkPred),
	functor(LinkSkel,LinkPred,Arity),
	multifile([Pred/Arity,ModName,LinkSkel]).

:- mode thread_shared(?).
thread_shared((Pred, Preds)) :- !,
	thread_shared(Pred),
	thread_shared(Preds).
thread_shared(PredSpec) :- 
	mpa_to_skel(PredSpec,Skel),
	term_psc(Skel,PSC),
	psc_set_shared(PSC,1).

:- mode thread_private(?).
thread_private((Pred, Preds)) :- !,
	thread_private(Pred),
	thread_private(Preds).
thread_private(PredSpec) :- 
	mpa_to_skel(PredSpec,Skel),
	term_psc(Skel,PSC),
	psc_set_private(PSC,1).

:- mode ensure_loaded(+,+).
ensure_loaded(File,Dir):- 
	ensure_loaded(File,Dir,usermod).

:- mode ensure_loaded(+,+,+).
ensure_loaded(File,Dir,Module):- 
	check_nonvar(Dir,'ensure_loaded/2',1),
	ensure_loaded_novarcheck(File,Dir,Module).

:- index ensure_loaded_novarcheck/3-2.
ensure_loaded_novarcheck(File,dyn,Module) :- !, ensure_dyn_loaded(File,1,Module).
ensure_loaded_novarcheck(File,dyna,Module) :- !, ensure_dyn_loaded(File,0,Module).
ensure_loaded_novarcheck(File,dync,Module) :- !, ensure_dync_loaded(File,1,Module).
ensure_loaded_novarcheck(File,dynca,Module) :- !, ensure_dync_loaded(File,0,Module).
ensure_loaded_novarcheck(File,consult,_Module) :- !, ensure_loaded(File).
ensure_loaded_novarcheck(_File,Dir,_Module) :- 
	domain_error(load_action([dyn,dyna,dync,dynca,consult]),Dir,ensure_loaded/2,2).


/* In the ensure loadeds, I'm relying on underlying reconsult,
   load_dyn, etc to check for atoms. */

:- mode ensure_loaded(+).
ensure_loaded(File):- 
	check_nonvar(File,'ensure_loaded/1',1),
	ensure_loaded_novarcheck(File).

ensure_loaded_novarcheck([]) :- !.
ensure_loaded_novarcheck([File|Files]) :- !,  
	ensure_loaded(File),   % check elements of list for vars.
	ensure_loaded(Files).
ensure_loaded_novarcheck(File) :-
	reconsult(File,[],true).

:- mode ensure_dyn_loaded(+).
ensure_dyn_loaded(Files) :-
	ensure_dyn_loaded(Files,1,usermod).

:- mode ensure_dyn_loaded(+,+,+).
ensure_dyn_loaded(File,Dir,Module) :- 
	check_nonvar(File,'ensure_dyn_loaded/1',1),
	ensure_dyn_loaded_novarcheck(File,Dir,Module).

ensure_dyn_loaded_novarcheck([],_Dir,_Module) :- !.
ensure_dyn_loaded_novarcheck([File|Files],Dir,Module) :- !, 
	ensure_dyn_loaded(File,Dir,Module),
	ensure_dyn_loaded(Files,Dir,Module).
ensure_dyn_loaded_novarcheck(File,Dir,Module) :-
	(if_should_not_reload(File)
	 ->	true
	 ;	load_dyn_nocheck(File,Dir,Module)  % no thread check
	).

:- mode ensure_dync_loaded(+).
ensure_dync_loaded(Files) :-
	ensure_dync_loaded(Files,1,usermod).

ensure_dync_loaded(File,Dir,Module) :- 
	check_nonvar(File,'ensure_dync_loaded/1',1),
	ensure_dync_loaded_novarcheck(File,Dir,Module).

ensure_dync_loaded_novarcheck([],_Dir,_Module) :- !.
ensure_dync_loaded_novarcheck([File|Files],Dir,Module) :- !, 
	ensure_dync_loaded(File,Dir,Module),
	ensure_dync_loaded(Files,Dir,Module).
ensure_dync_loaded_novarcheck(File,Dir,Module) :-
        check_atom(File,'ensure_dync_loaded/1',1),
	(if_should_not_reload(File)
	 ->	true
	;	current_prolog_flag(load_dync_error_limit,ErrLimit),
		load_dync_nocheck(File,Dir,Module,ErrLimit)   % no thread check
	).

if_should_not_reload(File) :-
	get_fname(File,_Dirctry,SExt,Base,_Mod),
	get_HRname(SExt,Base,_Hfname,Filename),
	expand_filename(Filename,AFilename),
	file_time(AFilename,Ctime),
        file_is_dynloaded(AFilename,Ltime),
	Ltime >= Ctime,
	\+ need_more_index(AFilename).

need_more_index(Filename) :-
	load_dyn_pred_loaded(Filename,Head,OIndex,OHashsize),
	'_$index'(Head,Index,Hashsize),
	(Hashsize > OHashsize
	 ;
	 \+ indexes_subsumed(Index,OIndex)
	).

indexes_subsumed(X,X) :- !.
indexes_subsumed([],_).
indexes_subsumed([Ispec|Ispecs],OIndex) :-
	memberchk(Ispec,OIndex),
	indexes_subsumed(Ispecs,OIndex).
indexes_subsumed(Ispec,OIndex) :-
	memberchk(Ispec,OIndex).

%% The stack of files currently being loaded
:- thread_shared loading_stack(_,_).
:- dynamic loading_stack(_,_).
:- index loading_stack/2-1.

%% Push/pop the currently loaded file onto a stack.
%% A program that is being loaded can examine which file it is in.
%% Rand is a gensym generated symbol, which ensures that we don't remove wrong
%% entries from the stack
push_loaded_file(File,Rand) :-
	gensym('loadfile',Rand),
	standardize_pathname(File,StandardizedFile),
	asserta(loading_stack(Rand,StandardizedFile)).


pop_loaded_file(Rand) :-
	retract(loading_stack(Rand,_)),!.

:- mode current_loaded_file(?).
current_loaded_file(File) :- loading_stack(_,File), !.


:- mode if_file_loaded(+).
if_file_loaded(FileName) :-
	(atom(FileName)
        ->  search_module(FileName,_Dir,_Mod,_SExt,_Base,Obj),
            expand_filename(Obj,ExpObj),
            (file_is_consultloaded(ExpObj) -> true
            ; get_fname(FileName,_Dirctry,SExtd,Based,_Modd),
                get_HRname(SExtd,Based,_Hfname,Rfname),
                expand_filename(Rfname,ARfname),
                file_is_dynloaded(ARfname,_)
            )
        ;  instantiation_error(if_file_loaded/1,1)
	).

:- mode is_loaded(+).
is_loaded(FilePath) :-
        file_is_consultloaded(FilePath)
        ; file_is_dynloaded(FilePath,_).

% Add currently loading file's directory to library_directory
% Useful to pull related files in a location independent manner
add_this_lib_dir :-
        current_loaded_file(F),
        parse_filename(F,D,_,_),
        add_lib_dir(D).

:- mode record_file_consultloaded(+).
record_file_consultloaded(F) :-
        atom(F),
        standardize_pathname(F,StandardF),
        assert(consult_file_loaded(StandardF)).

:- mode remove_file_consultloaded(+).
remove_file_consultloaded(F) :-
        atom(F),
        standardize_pathname(F,StandardF),
        retractall(consult_file_loaded(StandardF)).

:- mode file_is_consultloaded(+).
file_is_consultloaded(F) :-
        atom(F),
        standardize_pathname(F,StandardF),
        consult_file_loaded(StandardF).

record_file_dynloaded(F,Time) :-
        atom(F),
        standardize_pathname(F,StandardF),
        assert(load_dyn_file_loaded(StandardF,time(Time))).

remove_file_dynloaded(F) :-
        atom(F),
        standardize_pathname(F,StandardF),
        retractall(load_dyn_file_loaded(StandardF,_)).

file_is_dynloaded(F,Time) :-
        atom(F),
        standardize_pathname(F,StandardF),
        load_dyn_file_loaded(StandardF,time(Time)).

standardize_pathname(F,StandardF) :-
        %%expand_filename(F,StandardF).
	check_ground(F,'standardize_pathname/2',1),
	path_sysop(standardize,F,StandardF).

%% expand and convert back slashes to forward slashes
:- mode xsb_filename(+,?).
xsb_filename(InFile,UnixFile) :-
	expand_filename(InFile,ExpFile),
	(str_sub('\\',ExpFile)
	 ->	atom_codes(ExpFile,ExpFileC),
		cvt_back_to_for(ExpFileC,FExpFileC),
		atom_codes(UnixFile,FExpFileC)
	 ;	UnixFile = ExpFile
	).

cvt_back_to_for([],[]).
cvt_back_to_for([Ci|ExpFileC],[Co|FExpFileC]) :-
	(Ci =:= 0'\\
	 ->	Co = 0'/
	 ;	Co = Ci
	),
	cvt_back_to_for(ExpFileC,FExpFileC).
	

/* Take a clause with all terms in usermod, and a module, and convert
   all terms in the clause to the right modules. */

:- dynamic in_mod/3.		% (OpenTerm,Mod,Type) Type=known,assumed

check_set_mod(Module) :-
	(load_dyn_module(_)
	 ->	true
	 ;	assert(load_dyn_module(Module))
	).

fix_head_term(Head,Module,MHead) :-
	(load_dyn_module(RMod)
	 ->	cvt_head_to_mod(Head,RMod,MHead)
	 ;	cvt_head_to_mod(Head,Module,MHead)
	).
	
%fix_body_goal(Body,usermod,Body) :- !.
fix_body_goal(Body,Module,MBody) :-
    (load_dyn_module(RMod)
	 ->	cvt_body_to_mod(Body,RMod,MBody)
	 ;	cvt_body_to_mod(Body,Module,MBody)
	).

cvt_head_to_mod(Head,Mod,MHead) :-
	(atom(Head)
	 ->	cvt_to_mod(Head,Mod,MHead,known)
	 ; number(Head)
	 ->	MHead = Head
	 ;	Head =.. [Pred|Args],
		cvt_term_to_mod_list(Args,1,0,unknown,MArgs),
		Head1 =.. [Pred|MArgs],
		cvt_to_mod(Head1,Mod,MHead,known)
	).

%% Mod is known to be right (Type=known), or just to use if unknown (Type=assumed)
cvt_to_mod(Head,Mod,MHead,Type) :-
	(in_mod(Head,OMod,OType)
	 ->	(Type == known, OMod \== Mod % Must be Mod, but previously assumed another
		 ->	functor(Head,P,N),
			(OType == known
			 ->	permission_error(redefine,'imported (or exported) predicate',
						 OMod:P/N,load_dyn/1)
			 ;	permission_error(redefine,'assumed module, import/export/definition must precede use',
						 OMod:P/N,load_dyn/1)
			)
		 ;	true
		),
		(OMod == usermod
		 ->	MHead = Head
		 ;	term_new_mod(OMod,Head,MHead)
		)
	 ; functor(Head,Pred,Arity), Arity=\=0, standard_symbol(Pred,Arity,SMod)
	 ->	cvt_to_mod_and_add(Head,SMod,Type,MHead)
	 ;	cvt_to_mod_and_add(Head,Mod,Type,MHead)
	).

cvt_to_mod_and_add(Head,Mod,Type,MHead) :-
	(atom(Head)
	 ->	(Mod == usermod
		 ->	MHead = Head
		 ;	assert(in_mod(Head,Mod,Type)),
			term_new_mod(Mod,Head,MHead)
		)
	 ;	functor(Head,Fun,Arity),
		functor(MGHead,Fun,Arity),
		assert(in_mod(MGHead,Mod,Type)),
		(Mod == usermod
		 ->	MHead = Head
		 ;	term_new_mod(Mod,Head,MHead)
		)
	).

cvt_body_to_mod(Body,_Mod,MBody) :-
	(var(Body)
	 ->	MBody = Body
	 ; number(Body)
	 ->	abort('ERROR: [load_dyn] number cannot be a goal.')
	),
	!.
cvt_body_to_mod(true,_,true) :- !.  %DSW
cvt_body_to_mod(fail,_,fail) :- !.  %DSW
cvt_body_to_mod((G1,G2),Mod,(MG1,MG2)) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2).
cvt_body_to_mod(';'(G1,G2),Mod,';'(MG1,MG2)) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2).
cvt_body_to_mod('->'(G1,G2),Mod,'->'(MG1,MG2)) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2).
cvt_body_to_mod(not(G1),Mod,not(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(\+(G1),Mod,\+(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(fail_if(G1),Mod,fail_if(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(once(G1),Mod,once(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(for_all(G1,G2),Mod,for_all(MG1,MG2)) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2).
cvt_body_to_mod(do_all(G1,if_none(G2,G3)),Mod,do_all(MG1,if_none(MG2,MG3))) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2),
	cvt_body_to_mod(G3,Mod,MG3).
cvt_body_to_mod(do_all(G1),Mod,do_all(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(do_all(G1,G2),Mod,do_all(MG1,MG2)) :- !,
	cvt_body_to_mod(G1,Mod,MG1),
	cvt_body_to_mod(G2,Mod,MG2).
cvt_body_to_mod(call(G1),Mod,call(MG1)) :- !,
	cvt_body_to_mod(G1,Mod,MG1).
cvt_body_to_mod(!,_,!) :- !.
cvt_body_to_mod(Body,Mod,MBody) :-
	(atom(Body)
	 ->	cvt_to_mod(Body,Mod,MBody,assumed)
	 ; number(Body)
	 ->	MBody = Body
	 ; Body = Mod1:NBody, nonvar(Mod1)
	 ->	(Mod1 == usermod
		 ->	MBody = Body
		 ;	term_new_mod(Mod1,NBody,MBody)
		)
	 ;  Body =.. [Pred|Args],
	    functor(Body,Pred,Arity),
	    (standard_metapredicate(Pred,Arity,MetaArgNo)
	     ->	cvt_term_to_mod_list(Args,1,MetaArgNo,Mod,MArgs)
	     ;	cvt_term_to_mod_list(Args,1,0,Mod,MArgs)
	    ),
	    Body1 =.. [Pred|MArgs],
	    cvt_to_mod(Body1,Mod,MBody,assumed)
	).

cvt_term_to_mod_list([],_,_,_,[]).
cvt_term_to_mod_list([Term|Terms],ArgNo,MArgNo,Mod,[MTerm|MTerms]) :-
	(ArgNo =:= MArgNo	% if meta arg
	->     cvt_body_to_mod(Term,Mod,MTerm)
	;      cvt_term_to_mod(Term,MTerm)
	),
	ArgNo1 is ArgNo + 1,
	cvt_term_to_mod_list(Terms,ArgNo1,MArgNo,Mod,MTerms).

cvt_term_to_mod(Term,MTerm) :-
	(var(Term)
	 ->	MTerm = Term
	 ; atom(Term)
	 ->	cvt_to_mod(Term,usermod,MTerm,assumed)
	 ; number(Term)
	 ->	MTerm = Term
%% following is not like compiler, so comment out
%%	 ; Term = Mod1:NTerm, ground(Mod1), nonvar(NTerm)
%%	 ->	(Mod1 == usermod
%%		 -> 	cvt_term_to_mod(NTerm,MTerm)
%%		 ;	term_new_mod(Mod1,NTerm,MTerm0),
%%		    	cvt_term_to_mod(MTerm0,MTerm)
%%		)
	 ;	Term =.. [Pred|Args],
		cvt_term_to_mod_list(Args,1,0,unknown,MArgs),
		Term1 =.. [Pred|MArgs],
		cvt_to_mod(Term1,usermod,MTerm,assumed)
	).

