/* File:      aggregs.P
** Author(s): David S. Warren, Swift
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** 
*/

:- compiler_options([sysmod,optimize,xpp_on]).
#include "table_status_defs.h"

/*=======================================================================

bagReduce(+UnaryPred,?Result,+BinOp,+OpId) reduces all values that
	(HiLog) UnaryPred is true of using BinOp with identity OpId.
	So, for example, bagReduce(p,Sum,plus,0) binds Sum to the sum
	of all values X such that p(X) is true (assuming a definition
	of: plus(X,Y,Z) :- Z is X+Y. and plus is declared a hilog
	symbol.) bagSum, bagCount, and bagAvg are defined using this
	meta-predicate.

bagPO(+UnaryPred,?Result,+PO) finds all the maximal elements (under
	the partial order PO) that unify with Result and that
	UnaryPred is true of.

filterReduce(?UnaryPred,+BinOp,+OpId,?Result) 
	is copied from bagReduce with two changes.  NOTE CHANGE IN
	ARGUMENT ORDER!!!
	1) it does not use HiLog.  Rather than trying to reduce Hilog
	terms of the form, e.g. p(X,Y)(Value), it assumes that BinOp
	is applied to the last argument of a non-Hilog term,
	e.g. p(X,Y,Value).  The Hilog version is more elegant, surely,
	but is only as robust as XSB's Hilog implementation.

	2) it aggregates answers using a variance rather than a
	unification. If we have two facts

	p(X)(1)
	p(a)(2)
	p(a)(3)

	or equivalently 

	p(X,1).
	p(a,2).
	p(a,3).

	and want to construct the maximum, bagReduce will return

	p(a)(3).

	while filterReduce will return 

	p(X,1).
	p(a,3).

	Clause indexing should be more efficient for filterReduce, but
	indexing of answers within the table should be more efficient for
	bagReduce.

========================================================================*/

bagMin(Call,Var) :- bagReduce(Call,Var,minimum,zzz(zzz)).

bagMax(Call,Var) :- bagReduce(Call,Var,maximum,_).

bagSum(Call,Var) :- bagReduce(Call,Var,sum,0).

bagCount(Call,Var) :- bagReduce(Call,Var,successor,0).

bagAvg(Call,Avg) :- 
	bagReduce(Call,[Sum|Count],sumcount,[0|0]),
	Avg is Sum/Count.


% Keep all lubs in a partial order (HiLog) PO(/2).
bagPO(Call,Var,PO) :- bagPO1(Call,Var,PO), fail.
bagPO(Call,Var,PO) :- bagPO1(Call,Var,PO).
:- table bagPO1/3.
:- use_variant_tabling bagPO1(_,_,_).
bagPO1(Call,Var,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	Call(Var),
	((get_returns(Cs,Oskel,Leaf),(Ovar==Var -> true;PO(Var,Ovar)))
	 ->	fail
	 ;	(findall(p(Cs,Ovar,Leaf),
			 ('_$$get_returns'(Cs,Oskel,Leaf), PO(Ovar,Var)),
			 Rets),
		 sort(Rets, Rets1),
		 member(p(Cs,Ovar,Leaf),Rets1),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

% Reduce elements using associative Op with identity Id.
bagReduce(Call,_Res,Op,Id) :- bagReduce1(Call,_,Op,Id), fail.
bagReduce(Call,Res,Op,Id) :- bagReduce1(Call,Res0,Op,Id), Res=Res0.
:- table bagReduce1/4.
:- use_variant_tabling bagReduce1(_,_,_,_).
bagReduce1(Call,Res,Op,Id) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	copy_term(p(Call,Res,Skel),p(Call,Ovar,Oskel)),
	Call(Nvar),
	(get_returns(Cs,Oskel,Leaf)
	 ->	Op(Ovar,Nvar,Res),
		Res \== Ovar,
		delete_return(Cs,Leaf)
	 ;	Op(Id,Nvar,Res)
	).

% Define HiLog bagOf in terms of Prolog bagof. 
bagOf(Call,List) :- bagof(Var,Call(Var),List).

% Reduce elements using associative Op with identity Id.
% Template,Call,Res are like args for setof; Call may have existentials

bagReduce(Template,Call,Res,Op,Id) :-
	table_state(bagReduce1(Op,Id,Template,Call,Res),_PT,_,AnsSetStatus),
	(AnsSetStatus == complete
	 ->	bagReduce1(Op,Id,Template,Call,Res)
	 ; AnsSetStatus == undefined
	 ->	(bagReduce1(Op,Id,Template,Call,Res),fail
		 ;
		 bagReduce(Template,Call,Res,Op,Id)
		)
	 ; AnsSetStatus == incomplete
	 ->	get_ptcp(PTCP),
		get_producer_call(bagReduce1(Op,Id,Template,Call,Res),
		                  SubgPtr,_RetTmplt),
		is_incomplete(SubgPtr,PTCP),
		bagReduce1(Op,Id,Template,Call,Res)
	).

:- table bagReduce1/5.
:- use_variant_tabling bagReduce1(_,_,_,_,_).

:- import '_$$get_returns'/3 from tables.
bagReduce1(Op,Id,Templ,Call,Res) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,5,Skel,Cs),
	excess_vars(Call,Templ,[],VarList),
	copy_term(p(Call,VarList,Templ,Res,Skel),
		  p(OCall,VarList,OTempl,ORes,Oskel)),
	(atom(Op)
	 ->	Test =.. [Op,Var1,OTempl,Res]
	 ; Op = OpF/3
	 ->	Test =.. [OpF,Var1,OTempl,Res]
	 ; 	copy_term(Op,Test),
		arg(1,Test,Var1),
		arg(2,Test,OTempl),
		arg(3,Test,Res)
	),
	call(OCall),
%	(get_returns(Cs,Oskel,Leaf)
	('_$$get_returns'(Cs,Oskel,Leaf)
	 ->	Var1 = ORes, call_c(Test), %Op(ORes,OTempl,Res),
		Res \== ORes,
		delete_return(Cs,Leaf)
	 ;	Var1 = Id, call_c(Test)   %Op(Id,OTempl,Res)
%	 ;	Res = OTempl
	).

/*** The following (the else of VarList==[] test below) is a more
correct (but slower) implementation of bagPO.  It better handles
variables (in the set of non-existential variables in the goal) that
remain unbound in answers.  Consider the example,

v(a,2).  v(b,4).  v(_,3).

and goal bagPO(Y,v(X,Y),Min,(=<)).
This implementation gives:
X = a, Min = 2;
X = _, Min = 3;
no

This is still not exactly right semantically, since there really
should be the constraint on the _ answer that it is not a.  But this
is closer, and the important result is that the min given for b is 3,
not 4.
**************/
%%:- import writeln/2, nl/1 from standard.

bagPO(Templ,Call,_Res,PO) :- bagPO1(PO,_,Call,Templ),fail.
bagPO(Templ,Call,Res,PO) :- bagPO1(PO,_,Call,Templ),Res=Templ.

:- table bagPO1/4.

:- use_variant_tabling bagPO1(_,_,_,_).
bagPO1(PO,AVars,Call,Templ) :-  % templ is X, Call is pred(A,B,X)
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	excess_vars(Call,Templ,[],VarList),
	copy_term(p(Templ,AVars,VarList,Skel),p(OTempl,OAVars,OVarList,OSkel)),
	(atom(PO)
	 ->	POCA =.. [PO,Templ,OTempl],
		POCB =.. [PO,OTempl,Templ]
	 ; PO = PO1/2
	 ->	POCA =.. [PO1,Templ,OTempl],
		POCB =.. [PO1,OTempl,Templ]
	 ;	copy_term(PO,POCA), copy_term(PO,POCB),
		arg(1,POCA,Templ), arg(2,POCA,OTempl),
		arg(1,POCB,OTempl), arg(2,POCB,Templ)
	),
	POCE = (OTempl == Templ),
	%% following is to avoid building this term for every answer...
	Delete_old_as_nec_call = delete_old_as_nec(POCA,POCB,POCE,VarList,OVarList,AVars,OAVars,Cs,OSkel),
	call(Call),
	%%nl(userout),writeln(userout,newAnswer(Call,Templ)),
	(ground(VarList) -> AVars = [] ; excess_vars(VarList,Templ,[],AVars)), % optimize common case
	findall([],Delete_old_as_nec_call,[]).
%%	Rets = []. %%, writeln(userout,answerRet(Call,Templ)). % if noone said not to return it, then return this ans

delete_old_as_nec(POCA,POCB,POCE,Call,Call,NAVars,OAVars,Cs,OSkel) :-
	Template = p(OSkel,Leaf), % to do only once
	findall(Template,'_$$get_returns'(Cs,OSkel,Leaf),Returns),
	member(Template,Returns),
	%%writeln(userout,oldAnswer(Call)),
	(is_most_general_term(OAVars) % new answer subsumes old
	 ->	(is_most_general_term(NAVars) % variant
		 ->	(call_c(POCA) % variant, new better
			 ->	delete_return(Cs,Leaf),
				fail
			 ; call_c(POCB) % variant, old better
			 ->	true	% don't return this answer!
			 ;	fail % variant, incomparable
			)
		 ;	% new answer properly subsumes old
			(call_c(POCA)	% new subsumes, new better 
			 ->	delete_return(Cs,Leaf), % delete old
				fail
			 ; call_c(POCE) % new subsumes, old the same
			 ->	delete_return(Cs,Leaf), % delete old
				fail
			 ;	fail
			)
		)
	 ;	is_most_general_term(NAVars) % old properly subsumes new
	 ->	(call_c(POCA)	% old subsumes, new better
		 ->	fail	% leave general old
		 ; call_c(POCB) % old subsumes, old better
		 ->	true	% don't return this answer!
		 ;	fail
		)
	 ;	fail % incomparable
	).

/***** for testing:
:- table p(_,_,(<)/2).
p(A,B,) :- pred(A,B,C).
pred(1,f(1,2),2).
pred(1,f(_,1),4).
pred(1,f(1,2),1).
pred(1,f(1,1),5).
pred(1,f(_,1),3).
pred(1,f(1,_),2).
pred(1,f(_,_),4).
*****************/

/*=======================================================================*/

filterMin(Call,Var) :- filterReduce(Call,aggregs:min,neg_inf,Var).

filterMax(Call,Var) :- filterReduce(Call,aggregs:max,inf,Var).

:- fail, min(_,_,_), max(_,_,_).  % to shut compiler up! They are use above.
max(X,Y,Y):- \+ number(X),!.
max(X,Y,X):- \+ number(Y),!.
max(X,Y,Z):- (X > Y -> Z = X ; Z = Y).

min(X,Y,Y):- \+ number(X),!.
min(X,Y,X):- \+ number(Y),!.
min(One,Two,Min):- One > Two -> Min = Two ; Min = One.

/* Variant seems a better semantics, but is more expensive.  The issue
   is that in order to ensure variance we need to examine the table
   without binding further either the new answer or any answer in the
   table.  This means that our search for answers in the table can
   only be indexed by what is in the the original call -- otherwise if
   we indexed based on the new answer we might bind something in the
   table.  So we have to search through all answers for the call,
   rather than searching through all answers that unify with the new
   answer.

   At a coe level, the get_returns must unify answers with Oskel,
   which is the original call, rather than with (an abstraction of
   Skel), which is done in the unification routines (via copying
   Call).  */
% Reduce elements using associative Op with identity Id.
filterReduce(Call,Op,Id,Res) :- filterReduce1(Call,Op,Id,Res), fail.
filterReduce(Call,Op,Id,Res) :- filterReduce1(Call,Op,Id,Res).
:- table filterReduce1/4.
:- use_variant_tabling filterReduce1(_,_,_,_).
filterReduce1(Call,Op,Id,Res) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	copy_term(Skel,Oskel),
	get_opt_non_opt(Oskel,Osargs,Oopt),	
	get_non_opt(Skel,Sargs),	        % divide into opt arg and rest
	call(Call,Nvar),
%	writeln(New_call), 
	(get_returns(Cs,Oskel,Leaf),
%	 writeln(get_returns(Cs,Oskel,Leaf)),
						% divide into opt arg and rest
%	 writeln((Sargs,Osargs)),
	 variant(Sargs,Osargs)
	 ->	
%		Comp,
%		writeln(p(Comp,Res,Oopt)),
	        call(Op,Oopt,Nvar,Res),
		Res \== Oopt,
		delete_return(Cs,Leaf)
	 ;	
	        call(Op,Id,Nvar,Res)
	).

% Reduce elements using associative Op with identity Id.
filterReduce_u(Call,Op,Id,_Res) :- filterReduce1_u(Call,Op,Id,_), fail.
filterReduce_u(Call,Op,Id,Res) :- filterReduce1_u(Call,Op,Id,Res0), Res=Res0.
:- table filterReduce1_u/4.
:- use_variant_tabling filterReduce1_u(_,_,_,_).
filterReduce1_u(Call,Op,Id,Res) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	copy_term(p(Call,Res,Skel),p(Call,Ovar,Oskel)),
	call(Call,Nvar),
%	 writeln(getting_return_for(Skel,Oskel)),
	(get_returns(Cs,Oskel,Leaf)
	 ->	call(Op,Ovar,Nvar,Res),
		Res \== Ovar,
		delete_return(Cs,Leaf)
	 ;	call(Op,Id,Nvar,Res)
	).

% Keep all lubs in a partial order (HiLog) PO(/2).
filterPO(Call,Var,PO) :- filterPO1(Call,Var,PO), fail.
filterPO(Call,Var,PO) :- filterPO1(Call,Var,PO).
:- table filterPO1/3.
:- use_variant_tabling filterPO1(_,_,_).
filterPO1(Call,Var,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	call(Call,Var),
	((get_returns(Cs,Oskel,Leaf),
	    (Ovar==Var 
	    ->  true
            ;   % apply_last(PO,[Ovar,Var],PO_call),
%	        writeln(p(2,PO_call)),
		% call(PO_call)))
	        call(PO,Ovar,Var)))
	 ->	fail
	 ;	(findall(p(Cs,Ovar,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
%			  apply_last(PO,[Var,Ovar],PO_call_1),
%		          writeln(p(32,PO_call_1)),
%			  call(PO_call_1)
			  call(PO,Var,Ovar)
%			  writeln(p(33,PO_call_1))
			  ),
			 Rets),
		 sort(Rets,Rets1),
		 member(p(Cs,Ovar,Leaf),Rets1),
%		 writeln(deleting(Ovar)),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

% Keep all lubs in a partial order (HiLog) PO(/2).
filterPOA(Call,Var,PO,Abstract) :- filterPOA1(Call,Var,PO,Abstract), fail.
filterPOA(Call,Var,PO,Abstract) :- filterPOA1(Call,Var,PO,Abstract).
:- table filterPOA1/4.
:- use_variant_tabling filterPOA1(_,_,_,_).
filterPOA1(Call,Var,PO,Abstract) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	call(Call,VarTmp),
%	writeln(oskel(Oskel,VarTmp,Out)),
	((get_returns(Cs,Oskel,Leaf),
	    (Ovar==VarTmp
	    ->  true
            ;   call(PO,Ovar,VarTmp)))
	 ->	fail
	 ;	findall(p(Cs,Ovar,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
%			  writeln(get_returns(Cs,Oskel,Leaf)), 
%			  writeln(call(PO,VarTmp,Ovar)),
			  call(PO,VarTmp,Ovar)
			  ),
			 Rets),
		 sort(Rets,Rets1),
%		 writeln(Rets1),
		 filterOutCalls(Rets1,Calls),
%		 writeln(call(Abstract,Calls,VarTmp,Var)),
		 call(Abstract,Calls,VarTmp,Var),
%		 writeln(aftercall(Abstract,Calls,VarTmp,Var)),
		 ( member(p(Cs,Ovar,Leaf),Rets1),
%		   delete_return(Cs,Leaf),
		   fail
		  ;
		   true )
	).
%, writeln(success(Call,Var)).

:- export filterPOA/5.

% Keep all lubs in a partial order (HiLog) PO(/2).
filterPOA(Call,Var,PO,Abstract,CallAbs) :- filterPOA1(Call,Var,PO,Abstract,CallAbs), fail.
filterPOA(Call,Var,PO,Abstract,CallAbs) :- filterPOA1(Call,Var,PO,Abstract,CallAbs).
:- table filterPOA1/5.
:- use_variant_tabling filterPOA1(_,_,_,_,_).
filterPOA1(Call,Var,PO,Abstract,CallAbs) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	call(Call,VarTmp),
	call(CallAbs,VarTmp,Out),
	Oskel = ret(Out),
%	writeln(oskel(Oskel,VarTmp,Out)),
	((get_returns(Cs,Oskel,Leaf),
	    (Ovar==VarTmp
	    ->  true
            ;   call(PO,Ovar,VarTmp)))
	 ->	fail
	 ;	findall(p(Cs,Ovar,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
%			  writeln(get_returns(Cs,Oskel,Leaf)), 
%			  writeln(call(PO,VarTmp,Ovar)),
			  call(PO,VarTmp,Ovar)
			  ),
			 Rets),
		 sort(Rets,Rets1),
%		 writeln(Rets1),
		 filterOutCalls(Rets1,Calls),
%		 writeln(call(Abstract,Calls,VarTmp,Var)),
		 call(Abstract,Calls,VarTmp,Var),
%		 writeln(aftercall(Abstract,Calls,VarTmp,Var)),
		 ( member(p(Cs,Ovar,Leaf),Rets1),
%		   delete_return(Cs,Leaf),
		   fail
		  ;
		   true )
	).
%, writeln(success(Call,Var)).

filterOutCalls([],[]).
filterOutCalls([p(_,V,_)|R],[V|R1]):- 
	filterOutCalls(R,R1).

get_opt_non_opt(Call,Args,Opt):- 
	Call =.. [_Functor|List],
	get_opt_non_opt_1(List,Args,Opt).

get_non_opt(Call,Args):- 
	Call =.. [_Functor|List],
	get_non_opt_1(List,Args).

get_non_opt_1([_H],[]):-!.
get_non_opt_1([H|T],[H|T1]):-
	get_non_opt_1(T,T1).

get_opt_non_opt_1([H],[],H):-!.
get_opt_non_opt_1([H|T],[H|T1],O):-
	get_opt_non_opt_1(T,T1,O).

%-----------------

filterPO(Call,PO) :- filterPO1(Call,PO), fail.
filterPO(Call,PO) :- filterPO1(Call,PO).
:- table filterPO1/2.
:- use_variant_tabling filterPO1(_,_).
filterPO1(Call,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,2,Skel,Cs),
	copy_term(p(Call,Skel),p(Ocall,Oskel)),
	call(Call),
	((get_returns(Cs,Oskel,Leaf),
	  call(PO,Ocall,Call))
	 ->	fail
	 ;	(findall(p(Cs,Ocall,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
			  call(PO,Call,Ocall)
			  ),
			 Rets),
		 sort(Rets,Rets1),
		 member(p(Cs,Ocall,Leaf),Rets1),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

%-----------------
:- table table_once/1 as variant, opaque.
table_once(Call) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,1,Skel,Cs),
	copy_term(Skel,Oskel),
	call(Call),
	(get_returns(Cs,Oskel,_Leaf) -> fail ; true).

:- export table_random/1.
%:- import between/3 from basics.
:- import random/3 from random.
:- import get_call/3,get_answer_number/2,table_status/5 from tables.
%:- import writeln/1 from standard.
:- import conget/2,conset/2,coninc/1 from gensym.
table_random(Call) :- 
 	table_status(table_random_1(Call), _PredTypeCode, _CallTypeCode, AnsSetStatusCode,_SF),
	AnsSetStatusCode = UNDEFINED_ANSWER_SET,
        table_random_1(Call),
	fail.
table_random(Call) :- 
	get_call(table_random_1(Call),SF,_),
	get_answer_number(SF,Number),Num1 is Number + 1,
	random(1, Num1, Rand),
	conset(table_random_ctr,1),
	table_random_1(Call),
	 conget(table_random_ctr,Ctr),
	 (Ctr = Rand -> true 
           ; coninc(table_random_ctr),fail),
	 !.

:- table table_random_1/1 as variant, opaque.
table_random_1(Call):- call(Call).

/*
** A built-in used in the implementation of aggregates.                   
** Takes: Breg (where the choice point was saved), 
**    	  Arity of the skeleton term to return.
** Returns: Subgoal skeleton, Ptr to a subgoal structure.
**          Subgoal skeleton is of the form ret(X,Y,Z), where X,Y,Z are
**    	    all the free variables in the subgoal.
*/

breg_retskel(_BregOffset,_Arity,_RetTerm,_SubgoalPtr) :- '_$builtin'(154).


end_of_file.

apply_last(Call,Var,Newcall):-
	Call =.. List,
	append(List,Var,Newlist),
	Newcall =.. Newlist.

