/*
 * Copyright (c) 2007 Mockito contributors
 * This program is made available under the terms of the MIT License.
 */
package org.mockitousage.verification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.ArgumentMatchers.same;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.isA;
import static org.mockito.ArgumentMatchers.contains;
import static org.mockito.AdditionalMatchers.geq;
import static org.mockito.AdditionalMatchers.leq;
import static org.mockito.AdditionalMatchers.and;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.exceptions.verification.WantedButNotInvoked;
import org.mockito.exceptions.verification.opentest4j.ArgumentsAreDifferent;
import org.mockitousage.IMethods;
import org.mockitoutil.TestBase;

public class VerificationUsingMatchersTest extends TestBase {

    private IMethods mock;

    @Before
    public void setUp() {
        mock = Mockito.mock(IMethods.class);
    }

    @Test
    public void shouldVerifyExactNumberOfInvocationsUsingMatcher() {
        mock.simpleMethod(1);
        mock.simpleMethod(2);
        mock.simpleMethod(3);

        verify(mock, times(3)).simpleMethod(anyInt());
    }

    @Test
    public void shouldVerifyUsingSameMatcher() {
        Object one = new String("1243");
        Object two = new String("1243");
        Object three = new String("1243");

        assertNotSame(one, two);
        assertEquals(one, two);
        assertEquals(two, three);

        mock.oneArg(one);
        mock.oneArg(two);

        verify(mock).oneArg(same(one));
        verify(mock, times(2)).oneArg(two);

        try {
            verify(mock).oneArg(same(three));
            fail();
        } catch (WantedButNotInvoked e) {
        }
    }

    @Test
    public void shouldVerifyUsingMixedMatchers() {
        mock.threeArgumentMethod(11, "", "01234");

        try {
            verify(mock)
                    .threeArgumentMethod(and(geq(7), leq(10)), isA(String.class), contains("123"));
            fail();
        } catch (ArgumentsAreDifferent e) {
        }

        mock.threeArgumentMethod(8, new Object(), "01234");

        try {
            verify(mock)
                    .threeArgumentMethod(and(geq(7), leq(10)), isA(String.class), contains("123"));
            fail();
        } catch (ArgumentsAreDifferent e) {
        }

        mock.threeArgumentMethod(8, "", "no match");

        try {
            verify(mock)
                    .threeArgumentMethod(and(geq(7), leq(10)), isA(String.class), contains("123"));
            fail();
        } catch (ArgumentsAreDifferent e) {
        }

        mock.threeArgumentMethod(8, "", "123");

        verify(mock).threeArgumentMethod(and(geq(7), leq(10)), isA(String.class), contains("123"));
    }
}
