/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class NetworkPlayer : Object {
    public long id;
    public string name;
}

class Client : Object {
    public Gtk.Widget menu;

    /**
     * Signals relevant for game state.
     */
    public signal void on_join_game ();
    public signal void on_leave_game ();
    public signal void on_move_received (int move_sequence_num, string move_notation);

    /**
     * Parent for the purpose of showing dialog box
     */
    private Gtk.Window parent;

    /**
     * List of players logged in and waiting to play
     */
    private GLib.ListStore player_list;

    /**
     * User entries
     */
    private Gtk.Entry server_address_entry = new Gtk.Entry ();
    private Gtk.Entry server_port_entry = new Gtk.Entry ();
    private Gtk.Entry user_name_entry = new Gtk.Entry ();

    /**
     * Feedback
     */
    private Gtk.InfoBar status_info_bar = new Gtk.InfoBar ();
    private Gtk.Label status_info = new Gtk.Label (null);
    private string? error_message = null;

    /**
     * Communication with the asynchronous method 'dispatcher'
     */
    private Cancellable dispatcher_cancellable = new Cancellable ();
    private Queue<string> dispatcher_queue = new Queue<string> ();
    private SocketConnection connection = null;
    private DataInputStream input = null;
    private DataOutputStream output = null;

    /**
     * Information confirmed from the server
     */
    private long id = -1;
    private string user_name = null;
    private long opponent_id = -1;
    public string opponent_name = "";

    /**
     * Take note if we have recently sent any challenge
     */
    private long challenged_id = -1;

    public Client (Gtk.Window window) {
        parent = window;
        create_menu ();
        dispatcher.begin ();
    }

    private void send (string message) {
        if (connection != null) {
            try {
                output.put_string (message, null);
            }
            catch (Error err) {
                stderr.printf ("Send error: %s\n", err.message);
            }
        }
    }

    public void send_move (int seq, string move_notation) {
        if (opponent_id == -1) return;
        send ("MOVE %d %s\n".printf (seq, move_notation));
    }

    public void request_retreat () {
        opponent_id = -1;
        send ("LEAVE\n");
    }

    public void request_logout () {
        dispatch ("logout");
    }

    private void dispatch (string command) {
        dispatcher_queue.push_tail (command);
        dispatcher_cancellable.cancel ();
    }

    private async void dispatcher () {
        while (true) {
            dispatcher_cancellable.reset ();
            if (dispatcher_queue.is_empty () && connection == null) {
                // Previously we used Idle.add.
                // The resulting busy waiting causes significant cpu
                // consumption (12% on my machine. Now, only 0 - 2%)
                yield Util.wait_async (1000);
                continue;
            }
            switch (dispatcher_queue.pop_head ()) {
                case "login" : yield login (); break;
                case "logout" : yield logout (); break;
                default : yield process_messages (); break;
            }
        }
    }

    /**
     * Attempt login.
     * input: server_address_entry, server_port_entry, user_name_entry
     * output: connection (null if fail), input, output
     * messages sent: LOGIN ...
     */
    private async void login () {
        var server_name = server_address_entry.text.strip ();
        var user_name = user_name_entry.text.strip ();
        var server_port = int.parse (server_port_entry.text);
        if (server_name == "") {
            error_message = _("'Server' cannot be blank");
        }
        else if (user_name == "") {
            error_message = _("'Name' cannot be blank");
        }
        else if (server_port == 0) {
            error_message = _("'Port' must be valid");
        }
        else {
            try {
                if (connection != null) yield logout ();
                var resolver = Resolver.get_default ();
                var addresses = yield resolver.lookup_by_name_async (server_name, dispatcher_cancellable);
                var address = addresses.nth_data (0);
                debug ("server IP: %s", address.to_string ());
                var socket_address = new InetSocketAddress (address, (uint16) server_port);
                var client = new SocketClient ();
                connection = yield client.connect_async (socket_address, dispatcher_cancellable);
                input = new DataInputStream (connection.input_stream);
                output = new DataOutputStream (connection.output_stream);
                send ("LOGIN %s\n".printf (user_name_entry.text));
            } catch (IOError.CANCELLED err_cancelled) {
                connection = null;
            } catch (Error err) {
                connection = null;
                error_message = err.message;
            }
        }
        update_status ();
    }

    /**
     * Destroy connection.
     * output: set connection to null
     * error_message is retained in case the logout is performed because of disconnection
     * from the server.
     */
    private async void logout () {
        try {
            if (connection != null) {
                yield connection.close_async (100, dispatcher_cancellable);
                connection = null;
            }
        }
        catch (Error err) {
            stderr.printf ("Logout error: %s\n", err.message);
        }
        player_list.remove_all ();
        user_name = null;
        id = -1;
        opponent_id = -1;
        challenged_id = -1;
        on_leave_game ();
        update_status ();
    }

    /**
     * Process messages from the server.
     * The read loop may break if
     *  1. dispatcher_cancellable is triggered, or
     *  2. the connection is lost.
     */
    private async void process_messages () {
        if (connection == null) return;
        try {
            string line;
            size_t length;
            while ((line = yield input.read_line_async (100, dispatcher_cancellable, out length)) != null) {
                message ("From server: %s", line);
                interpret (sanitize (line));
                if (dispatcher_cancellable.is_cancelled ()) break;
            }
            // Disconnected from the server? If yes, logout.
            if (!dispatcher_cancellable.is_cancelled ()) {
                error_message = _("Disconnected from the server.");
                update_status ();
                yield logout ();
            }
        }
        catch (IOError.CANCELLED err_cancelled) {
            // A user command has been dispatched and must be attended
        }
        catch (Error err) {
            stderr.printf ("IO error: %s\n", err.message);
        }
    }

    private string sanitize (string s) {
        return s.strip().replace ("\n", "");
    }

    private void interpret (string line) throws Error {
        if (line == "") return;
        var msg = line.split (" ", 2);
        var arg = msg.length == 2 ? msg[1].strip () : "";
        switch (msg[0]) {
            case "LOGIN-OK" : login_accepted (arg); break;
            case "LOGIN-ERROR" : login_refused (); break;
            case "USER" : remote_user_registered (arg); break;
            case "EXIT" : remote_user_deregistered (arg); break;
            case "PLAY-AGAINST" : opponent_booked (arg); break;
            case "PLAY-END" : play_ended (); break;
            case "MOVE" : move_received (arg); break;
        }
    }

    /**
     * input: login_info in the form of "id name"
     * output: id, user_name
     * message sent: GET-USER-LIST
     * 
     */
    private void login_accepted (string login_info) throws Error {
        error_message = null;
        var info = login_info.split (" ", 2);
        if (info.length != 2) return;   // Malformed message from the server?
        id = long.parse (info[0]);
        user_name = info[1].strip ();
        send ("GET-USER-LIST\n");
        update_status ();
    }

    private void login_refused () {
        error_message = _("Login refused.");
        update_status ();
    }

    /**
     * Update list of users whenever a remote player logs in or changes her status.
     * Input: user_info in the form of "id status name"
     * Output: modified row, if id found; a new row if id is not found
     */
    private void remote_user_registered (string user_info) {
        var info = user_info.split (" ", 3);
        if (info.length != 3) return;
        var remote_user_id = long.parse (info[0]);
        if (remote_user_id == id) return; // Ignore if this is our own id
        var remote_user_name = info[2].strip ();

        remote_user_status_changed (remote_user_id, info[1] == "w", remote_user_name);
        update_status ();
    }

    /**
     * Remove remote user entry.
     * Input: remote user id
     * Output: row deleted if found
     */
    private void remote_user_deregistered (string id_string) {
        var remote_user_id = long.parse (id_string);
        remote_user_status_changed (remote_user_id, false);
        if (remote_user_id == opponent_id) {
            play_ended ();
        }
        update_status ();
    }

    /**
     * Add, remove or ignore remote user depending on the status.
     */
    private void remote_user_status_changed (long remote_user_id, bool available, string? remote_user_name = null) {
        // Search remote_user_id.
        // If found either delete if NOT available, or ignore otherwise.
        var n = 0;
        Object obj = null;
        while ((obj = player_list.get_item (n)) != null) {
            var p = obj as NetworkPlayer;
            if (p.id == remote_user_id) {
                if (!available) player_list.remove (n);
                break;
            }
            n++;
        }
        // If not found, insert if status is 'w'.
        if (obj == null && available) {
            var p = new NetworkPlayer ();
            p.id = remote_user_id;
            p.name = remote_user_name;
            player_list.append (p);
        }
    }

    /**
     * Receive or decline an opponent.
     * Input: remote user id
     * output: opponent_id, opponent_name
     */
    private void opponent_booked (string id_string) {
        var remote_user_id = long.parse (id_string);
        if (remote_user_id == id) return;  // Our own id. Ignore.
        // Get opponent's name
        var n = 0;
        Object obj = null;
        while ((obj = player_list.get_item (n)) != null) {
            var p = obj as NetworkPlayer;
            if (p.id == remote_user_id) {
                if (p.id == challenged_id || accept_challenge (p.name)) {
                    opponent_id = remote_user_id;
                    opponent_name = p.name;
                    challenged_id = -1;
                    on_join_game ();
                    update_status ();
                }
                else {
                    request_retreat ();
                }
                break;
            }
            n++;
        }
    }

    /**
     * Show a dialog to accept or decline a challenge
     */
    private bool accept_challenge (string challenger) {
        var dialog = new Gtk.MessageDialog (
            parent,
            Gtk.DialogFlags.MODAL,
            Gtk.MessageType.INFO,
            Gtk.ButtonsType.NONE,
            _("%s is inviting you to play."),
            challenger);
        dialog.add_button (_("Accept"), Gtk.ResponseType.OK);
        dialog.add_button (_("Decline"), Gtk.ResponseType.CANCEL);
        var response = dialog.run ();
        dialog.destroy ();
        return response == Gtk.ResponseType.OK;
    }

    /**
     * Play ended. Retreat from opponnet.
     */
    private void play_ended () {
        if (opponent_id != -1) {
            var dialog = new Gtk.MessageDialog (
                parent, Gtk.DialogFlags.MODAL, Gtk.MessageType.INFO, Gtk.ButtonsType.OK,
                _("Your opponent has left the game"));
            dialog.run ();
            dialog.destroy ();
        }
        opponent_id = -1;
        challenged_id = -1;
        on_leave_game ();
        update_status ();
    }

    /**
     * Opponent's move has arrived.
     * input: opponent id + sequence number + move notation
     * output: signal emitted
     */
    private void move_received (string arg) {
        if (opponent_id == -1) {
            message ("Sync error: no opponent, but a move was received.");
            return;
        }
        var info = arg.split (" ", 3);
        if (info.length != 3) {
            message ("Format error: \"id seq move\" expected.");
            return;
        }
        if (opponent_id != long.parse (info[0])) {
            message ("Sync error: unexpected opponent id.");
            return;
        }
        on_move_received (int.parse (info[1]), info[2]);
    }

    
    /**
     * Update status visually.
     */
    public void update_status () {
        // if network error:
        if (error_message != null) {
            status_info_bar.message_type = Gtk.MessageType.ERROR;
            status_info.label = error_message;
            status_info_bar.show ();
        }
        // if not logged in yet:
        else if (connection == null || user_name == null || id == -1) {
            status_info_bar.hide ();
        }
        // if already logged in, but not playing yet:
        else if (opponent_id == -1) {
            if (player_list.get_n_items () == 0) {
                status_info_bar.message_type = Gtk.MessageType.INFO;
                status_info.label = _("No other players are available");
                status_info_bar.show ();
            }
            else {
                status_info_bar.hide ();
            }
        }
    }

    /**
     * Create menu showing available remote players
     */
    private void create_menu () {
        var box = new Gtk.Box (Gtk.Orientation.VERTICAL, 0);
        box.get_style_context () .add_class ("listbox-menu");

        // Add configuration row
        var list_box = new Gtk.ListBox ();
        list_box.add (GameMenu.create_item (_("Play Online"), create_server_setter_button ()));
        var row = list_box.get_row_at_index (0);
        row.activatable = false;
        row.selectable = false;
        box.pack_start (list_box, false, false);

        // Add status row
        status_info_bar.get_content_area () .add (status_info);
        status_info.wrap = true;
        box.pack_start (status_info_bar, false, false);

        // Add player list
        player_list = new GLib.ListStore (typeof (NetworkPlayer));
        list_box = new Gtk.ListBox ();
        list_box.bind_model (player_list, (p) => {
            var widget = GameMenu.create_item ((p as NetworkPlayer).name);
            widget.show_all ();
            return widget;
        });
        box.pack_start (list_box, false, false);

        // Activate player list, allowing remote player to be challenged
        list_box.selection_mode = Gtk.SelectionMode.NONE;
        list_box.row_activated.connect ((row) => {
            var n = row.get_index ();
            var p = player_list.get_item (n) as NetworkPlayer;
            challenged_id = p.id;
            send ("CHALLENGE %ld\n".printf (p.id));
        });

        menu = box;
        update_status ();
    }

    /**
     * Create a switch and a gear button to pop up server configuration
     */
    private Gtk.Widget create_server_setter_button () {
        var box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);

        var button = new Gtk.MenuButton ();
        button.image = new Gtk.Image.from_icon_name ("emblem-system-symbolic", Gtk.IconSize.BUTTON);
        button.tooltip_text = _("Configure login");
        var popover = new Gtk.Popover (button);
        popover.add (create_configuration_widget (popover));
        button.popover = popover;

        var suis = new Gtk.Switch ();
        suis.active = false;

        // Notification for the switch and the gear
        suis.notify["active"].connect (() => {
            error_message = null;
            dispatch (suis.active ? "login" : "logout");
        });
        popover.closed.connect (() => {
            error_message = null;
            dispatch (suis.active ? "login" : "logout");
        });

        suis.valign = Gtk.Align.CENTER;
        box.pack_start (suis, false, false, 5);
        box.pack_end (button, false, false, 0);
        return box;
    }

    /**
     * Create widget to enter :
     *    Server name, Port number, and User name
     */
    private Gtk.Box create_configuration_widget (Gtk.Popover popover) {
        var grid = new Gtk.Grid ();
        grid.set_row_spacing (10);
        grid.set_column_spacing (10);
        server_address_entry.text = "localhost";
        server_address_entry.width_chars = 25;
        server_port_entry.text = "3812";
        server_port_entry.width_chars = 6;
        grid.attach (new Gtk.Label (_("Server:")), 0, 0, 1, 1);
        grid.attach (server_address_entry, 1, 0, 1, 1);
        grid.attach (new Gtk.Label (_("Port:")), 2, 0, 1, 1);
        grid.attach (server_port_entry, 3, 0, 1, 1);
        server_address_entry.hexpand = true;

        // User name
        user_name_entry.text = "";
        user_name_entry.placeholder_text = _("Name seen by other players");
        user_name_entry.width_chars = 30;
        grid.attach (new Gtk.Label (_("Name:")), 0, 1, 1, 1);
        grid.attach (user_name_entry, 1, 1, 3, 1);

        // Footnote about pasang-emas-server
        var cmd = Reloc.exe_name + " --server";
        var footnote = new Gtk.Label (
            _("Note: to run a game server, enter the command\n   <tt>%s</tt>\non a terminal.").printf (cmd)
        );
        footnote.use_markup = true;
        footnote.max_width_chars = 40;
        footnote.wrap = true;
        footnote.wrap_mode = Pango.WrapMode.WORD;

        // Allow pressing [Enter] to close the popup
        server_address_entry.activate.connect (() => {popover.closed ();});
        server_port_entry.activate.connect (() => {popover.closed ();});
        user_name_entry.activate.connect (() => {popover.closed ();});

        var box = new Gtk.Box (Gtk.Orientation.VERTICAL, 10);
        box.pack_start (grid);
        box.pack_start (new Gtk.Separator (Gtk.Orientation.HORIZONTAL));
        box.pack_start (footnote);
        box.get_style_context () .add_class ("box-top-level");
        box.show_all ();
        return box;
    }
}//class Client
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
