# Plugin Translation System

This document describes the translation system for Plugin v3, including both plugin UI translations and registry metadata translations.

---

## Overview

Plugin v3 supports two types of translations:

1. **Plugin UI translations** - For plugin interface text (buttons, messages, etc.)
2. **Registry metadata translations** - For plugin name and description in the registry

---

## Plugin UI Translations

### File Structure

```text
myplugin/
├── __init__.py
├── MANIFEST.toml
└── locale/
    ├── en.json          # English (source locale, generated by extraction tool)
    ├── de.json          # German
    ├── fr.json          # French
    └── ja.json          # Japanese
```

### Translation File Format

Simple flat JSON structure with dot notation for namespacing:

```json
{
  "ui.menu.submit_listens": "Submit Listens Now",
  "ui.menu.configure": "Configure ListenBrainz",
  "ui.button.login": "Login to ListenBrainz",
  "ui.button.logout": "Logout",

  "options.title": "ListenBrainz Options",
  "options.username": "Username",
  "options.password": "Password",

  "error.network": "Network error: {error}",
  "error.auth_failed": "Authentication failed",

  "status.submitting_listens": "Submitting: {artist} - {title}",
  "status.submitted": "Submitted {count} tracks",

  "message.login_success": "Successfully logged in as {username}"
}
```

### Usage in Plugin Code

```python
from picard.plugin3.api import PluginApi
from picard.i18n import gettext as _  # Picard's translations

def enable(api: PluginApi):
    # Use Picard's translations for standard UI
    menu_text = _("Plugins")  # Picard translates this

    # Use plugin translations for plugin-specific text
    login_text = api.tr("ui.button.login", "Login to ListenBrainz")
    logout_text = api.tr("ui.button.logout", "Logout")

    # With parameters
    status = api.tr("status.submitted", "Submitted {count} tracks", count=5)
    error = api.tr("error.network", "Network error: {error}", error="Timeout")
```

**Important:** While you can use Picard's `gettext()` for standard UI elements (like "Plugins", "Options", etc.), **avoid relying on Picard's translations for plugin-specific text**. Picard's translation strings may change or be removed between versions, which would break your plugin. Always use `api.tr()` for your plugin's own text.

**Signature:**
```python
api.tr(key: str, text: str = None, **kwargs) -> str
```

- `key`: Translation key (required)
- `text`: Optional source locale text (used as fallback and for extraction tool)
- `**kwargs`: Format parameters for string interpolation

### Translation Marker: `t_()`

For cases where you need to define translatable strings at module level or in data structures (before the API is available), use the `t_()` marker function:

```python
from picard.plugin3.api import t_

# Define translatable strings at module level
ERROR_MESSAGES = {
    404: t_('error.not_found', 'Not found'),
    500: t_('error.server', 'Server error'),
    403: t_('error.forbidden', 'Access denied'),
}

# Define plural forms
FILE_COUNT = t_('files.count', '{n} file', '{n} files')

# Use in class definitions
class MyAction(BaseAction):
    TITLE = t_("My Custom Action")

def enable(api):
    # Translate at runtime
    error_msg = api.tr(ERROR_MESSAGES[404], 'Not found')

    # Use plural forms
    count_msg = api.trn(*FILE_COUNT, n=5)
```

**How `t_()` works:**
- At runtime: Returns the key (or tuple for plurals) unchanged - zero overhead
- For extractor: Marks strings for extraction into translation files
- Signature: `t_(key, text=None, plural=None)`
- Returns:
  - If no plural: returns `key`
  - If plural: returns `(key, text, plural)` tuple for unpacking with `trn()`

**Benefits:**
- Define translations once, use multiple times
- No repetition of translation strings
- Works at module/class level before `enable()` is called
- Extractor finds and extracts all marked strings

**Example with repetition avoided:**
```python
from picard.plugin3.api import t_

# Define once
PLURALS = t_('items.count', 'There is {n} item.', 'There are {n} items.')

# Use multiple times without repetition
def callback(self, objs):
    msg1 = self.api.trn(*PLURALS, n=-1)
    msg2 = self.api.trn(*PLURALS, n=0)
    msg3 = self.api.trn(*PLURALS, n=1)
    msg4 = self.api.trn(*PLURALS, n=5)
```

### Key Naming Conventions

Use consistent prefixes for organization:

- `ui.*` - User interface elements (buttons, labels, menus)
- `options.*` - Settings/options page
- `error.*` - Error messages
- `status.*` - Status messages
- `message.*` - User notifications
- `help.*` - Help text

**Reserved prefixes:**

- `qt.*` - Reserved for Qt Designer `.ui` files (automatically generated)
  - Format: `qt.{ClassName}.{SourceText}`
  - Example: `qt.VariablesDialog.Variable`
  - Do not use this prefix in your code

### Placeholders

Use `{variable}` syntax for dynamic content:

```json
{
  "status.processing": "Processing {filename}...",
  "error.file_not_found": "File not found: {path}",
  "message.saved": "Saved {count} files successfully"
}
```

### Locale Fallback

1. If current locale matches source_locale → return text from code
2. Try current locale (e.g., `de_DE`)
3. Try language without region (e.g., `de`)
4. Fall back to text parameter if provided
5. Return `?key?` if not found (with warning logged)

---

## Registry Metadata Translations

### In MANIFEST.toml

Plugin name and description can be translated directly in MANIFEST.toml:

```toml
name = "ListenBrainz Submitter"
description = "Submit your music to ListenBrainz"

[name_i18n]
de = "ListenBrainz-Submitter"
fr = "Soumetteur ListenBrainz"
ja = "ListenBrainzサブミッター"

[description_i18n]
de = "Sende deine Musik zu ListenBrainz"
fr = "Soumettez votre musique sur ListenBrainz"
ja = "ListenBrainzに音楽をスクロブルする"
```

### In Registry TOML

The website extracts these translations and includes them in the registry:

```toml
[[plugins]]
id = "listenbrainz"
name = "ListenBrainz Submitter"
description = "Submit your music to ListenBrainz"

[plugins.name_i18n]
de = "ListenBrainz-Submitter"
fr = "Soumetteur ListenBrainz"
ja = "ListenBrainzサブミッター"

[plugins.description_i18n]
de = "Sende deine Musik zu ListenBrainz"
fr = "Soumettez votre musique sur ListenBrainz"
ja = "ListenBrainzに音楽をスクロブルする"
```

### Benefits

- Single source of truth (MANIFEST.toml)
- No separate translation files needed for registry
- Plugin developers manage translations
- Website automatically includes them
- Picard and website can display localized names/descriptions

---

## Plural Forms

For strings that need pluralization, use `api.trn()` (translate with number):

```python
# Signature
api.trn(key: str, singular: str = None, plural: str = None, n: int = 0, **kwargs) -> str

# Usage
count = 5
api.trn("status.files", "{n} file", "{n} files", n=count)
# Returns: "5 files"

count = 1
api.trn("status.files", "{n} file", "{n} files", n=count)
# Returns: "1 file"
```

**Important:**
- The `n` parameter is required for `trn()` to determine the correct plural form
- Use `{n}` in your format strings (not `{count}` or other names)
- The `n` value is automatically available in the format string as `{n}`

**Translation file formats:**

Both JSON and TOML formats are supported. For each locale, Picard will look for `.toml` first, then `.json`.
You can mix formats (e.g., `en.json` and `fr.toml`).

**Why use quoted keys in TOML?**

Translation keys use dot notation for organization (e.g., `"message.greeting"`, `"error.network.timeout"`).
In TOML, unquoted keys like `[message.greeting]` create nested tables: `{message: {greeting: ...}}`.
This breaks compatibility with JSON format and translation tools like Weblate.

Using flat keys with dots (like go-i18n does) ensures compatibility with translation management tools
and avoids issues where tools might incorrectly interpret nested structures.

**Always quote keys with dots in TOML:**
- ✅ Correct: `["message.greeting"]` → flat key `"message.greeting"`
- ❌ Wrong: `[message.greeting]` → nested `{message: {greeting: ...}}`

Picard will warn you if it detects nested structures in your TOML files.

**JSON format:**
```json
{
  "status.files": {
    "one": "{n} file",
    "other": "{n} files"
  }
}
```

**TOML format (use quoted keys):**
```toml
["status.files"]
one = "{n} file"
other = "{n} files"
```

**For languages with complex plural rules (e.g., Polish, Russian):**
```json
{
  "status.files": {
    "one": "{n} plik",
    "few": "{n} pliki",
    "many": "{n} plików",
    "other": "{n} plików"
  }
}
```

**Plural categories (CLDR):**
- `zero` - Used in Arabic for zero
- `one` - Singular (1 in English)
- `two` - Used in Arabic for two
- `few` - Used in Slavic languages (2-4 in Polish)
- `many` - Used in Slavic languages (5+ in Polish)
- `other` - Default/fallback

**Implementation:**
- Uses CLDR plural rules based on locale
- Falls back to `other` if specific form not available
- English only needs `one` and `other`

**Extraction tool behavior:**
```python
# Code
api.trn("status.files", "{n} file", "{n} files", n=count)

# Generates in locale/en.json
{
  "status.files": {
    "one": "{n} file",
    "other": "{n} files"
  }
}

# Without text parameters
api.trn("status.files", n=count)

# Generates placeholder
{
  "status.files": {
    "one": "?status.files?",
    "other": "?status.files?"
  }
}
```

---

## Extraction Tool

An extraction tool can scan code for `api.tr()` and `api.trn()` calls and generate the source locale file:

```bash
# Scan plugin code
$ picard-plugin-extract-translations myplugin/

# Generates locale/en.json (or source_locale from MANIFEST.toml)
{
  "ui.button.login": "Login to ListenBrainz",
  "ui.button.logout": "Logout",
  "status.submitted": "Submitted {count} tracks",
  "error.network": "Network error: {error}",
  "status.files": {
    "one": "{n} file",
    "other": "{n} files"
  },
  "some.missing.text": "?some.missing.text?"
}
```

**Extraction behavior:**
- If `api.tr("key", "Text")` → extracts `"key": "Text"`
- If `api.tr("key")` → generates `"key": "?key?"` (placeholder indicating missing text)
- If `api.trn("key", "singular", "plural")` → extracts as plural object with `one` and `other`
- If `api.trn("key")` → generates placeholder plural object
- Developers can then fill in the `?key?` placeholders manually

**Benefits:**
- Developers write readable code with inline text
- Tool extracts translations automatically
- Missing text is clearly marked with `?key?` placeholders
- Keys provide stability when text changes

---

## Source Locale

Declare source locale in MANIFEST.toml:

```toml
source_locale = "en_US"  # Default if not specified
# Can be: "en_GB", "de_DE", "pt_BR", "fr_CA", etc.
```

This allows:
- Plugin developers to write code in their native language
- Regional variants (en_US vs en_GB, pt_BR vs pt_PT, etc.)
- Inclusive development for non-English speakers

**Example with German source locale:**
```python
# German developer
api.tr("ui.button.login", "Bei ListenBrainz anmelden")
```

```toml
source_locale = "de_DE"
```

No `de_DE.json` needed - German text is in the code. Other locales in JSON:

```json
// en_US.json
{
  "ui.button.login": "Login to ListenBrainz"
}

// fr_FR.json
{
  "ui.button.login": "Connexion à ListenBrainz"
}
```

---

## Qt Integration

Plugin translations work alongside Qt's translation system:

```python
from PyQt6.QtWidgets import QDialog, QPushButton, QLabel

class MyPluginDialog(QDialog):
    def __init__(self, api):
        super().__init__()
        self.api = api

        # Qt's built-in translations (inherited from QObject)
        ok_button = QPushButton(self.tr("&OK"))
        cancel_button = QPushButton(self.tr("&Cancel"))

        # Plugin-specific translations
        title = self.api.tr("options.title", "ListenBrainz Options")
        label = QLabel(self.api.tr("options.username", "Username:"))
```

- `self.tr()` → Qt's translation system (for standard UI elements)
- `self.api.tr()` → Plugin's translation system (for plugin-specific text)
- Both use the same locale from `QLocale().name()`

### Qt Designer (.ui files)

For plugins using Qt Designer `.ui` files, translations work automatically:

**ui_variables_dialog.ui** (designed in Qt Designer):
```xml
<widget class="QTableWidget" name="metadata_table">
  <column>
    <property name="text">
      <string>Variable</string>
    </property>
  </column>
  <column>
    <property name="text">
      <string>Value</string>
    </property>
  </column>
</widget>
```

**ui_variables_dialog.py** (generated by `pyuic6`):
```python
def retranslateUi(self, VariablesDialog):
    _translate = QtCore.QCoreApplication.translate
    item = self.metadata_table.horizontalHeaderItem(0)
    item.setText(_translate("VariablesDialog", "Variable"))
    item = self.metadata_table.horizontalHeaderItem(1)
    item.setText(_translate("VariablesDialog", "Value"))
```

**Your plugin code:**
```python
from .ui_variables_dialog import Ui_VariablesDialog

class VariablesDialog(QDialog, Ui_VariablesDialog):
    def __init__(self, api, parent=None):
        super().__init__(parent)
        self.api = api
        self.setupUi(self)  # Automatically uses plugin translations!

        # Dynamic text still uses api.tr()
        self.setWindowTitle(self.api.tr("dialog.title", "Script Variables"))
```

**How it works:**

`PluginApi` automatically installs a `QTranslator` that bridges Qt's translation system to the plugin's translation system:

```python
class PluginApi:
    def __init__(self, manifest: PluginManifest, tagger) -> None:
        self._manifest = manifest
        self._tagger = tagger
        self._translations = {}
        self._source_locale = manifest.source_locale or "en_US"
        self._current_locale = QLocale().name()
        self._load_translations()

        # Automatically install Qt translator for .ui files
        self._install_qt_translator()

    def _install_qt_translator(self):
        """Install Qt translator for .ui file translations."""
        translator = PluginTranslator(self)
        QCoreApplication.installTranslator(translator)
        self._qt_translator = translator  # Keep reference to prevent GC

class PluginTranslator(QTranslator):
    """Qt translator that uses plugin's translation system."""

    def __init__(self, api: PluginApi):
        super().__init__()
        self.api = api

    def translate(self, context, source_text, disambiguation=None, n=-1):
        """Translate using plugin's translation system.

        Args:
            context: Qt context (usually class name, e.g., "VariablesDialog")
            source_text: Source text from .ui file (e.g., "Variable")
            disambiguation: Optional disambiguation string
            n: Plural form number (-1 for non-plural)

        Returns:
            Translated text or source_text if not found
        """
        # Generate key from context and source text
        # "VariablesDialog" + "Variable" → "qt.VariablesDialog.Variable"
        key = f"qt.{context}.{source_text}"

        # Try plugin translation, fall back to source text
        return self.api.tr(key, source_text)
```

**Translation file:**
```json
{
  "qt.VariablesDialog.Variable": "Variable",
  "qt.VariablesDialog.Value": "Value",
  "dialog.title": "Script Variables"
}
```

**German translation (locale/de.json):**
```json
{
  "qt.VariablesDialog.Variable": "Variable",
  "qt.VariablesDialog.Value": "Wert",
  "dialog.title": "Skriptvariablen"
}
```

**Benefits:**

✅ **Zero configuration** - Works automatically for all plugins
✅ **Unified system** - All translations in one place (JSON/TOML)
✅ **Extraction tool compatible** - Can extract from `.ui` files
✅ **Fallback** - Source text used if translation missing
✅ **No manual override** - `.ui` files work transparently

**Extraction tool behavior:**

The extraction tool can parse `.ui` files and generate keys:

```bash
$ picard-plugin-extract-translations myplugin/

# Scans:
# - Python files for api.tr() calls
# - .ui files for translatable strings

# Generates locale/en.json:
{
  "qt.VariablesDialog.Variable": "Variable",
  "qt.VariablesDialog.Value": "Value",
  "dialog.title": "Script Variables",
  "ui.button.login": "Login to ListenBrainz"
}
```

---

## Translation Workflow

### For Plugin Developers

1. Write code with `api.tr()` and `api.trn()` including inline text
2. Run extraction tool to generate source locale file (e.g., `locale/en.json`)
3. Create additional locale files (e.g., `de.json`, `fr.json`) or use Weblate
4. Add translations to MANIFEST.toml for name/description
5. Test with different locales

### For Translators

1. Use generated source locale file as reference
2. Create new locale file (e.g., `de.json`)
3. Translate all values (keep keys unchanged)
4. Test translations
5. Submit PR or send to plugin developer

### For Website

1. Scan plugin repositories
2. Extract `name_i18n` and `description_i18n` from MANIFEST.toml
3. Include in registry TOML
4. Serve to clients

---

## Best Practices

1. **Always provide text parameter** - Makes code readable: `api.tr("key", "Text")`
2. **Use consistent key naming** - Follow conventions (ui.*, error.*, etc.)
3. **Keep translations in sync** - All locale files should have same keys
4. **Use placeholders** - For dynamic content: `{variable}`
5. **Test with multiple locales** - Verify fallback behavior
6. **Use extraction tool** - Automate source locale file generation
7. **Keep strings short** - UI space is limited
8. **Avoid concatenation** - Use complete sentences with placeholders
9. **Use `api.trn()` for plurals** - Don't manually handle plural forms

---

## Locale Codes

Use standard locale codes:

- `en` - English
- `de` - German
- `fr` - French
- `es` - Spanish
- `it` - Italian
- `ja` - Japanese
- `zh_CN` - Chinese (Simplified)
- `zh_TW` - Chinese (Traditional)
- `pt_BR` - Portuguese (Brazilian)
- `ru` - Russian

---

## Example: Complete Translation Setup

### MANIFEST.toml
```toml
name = "ListenBrainz Submitter"
description = "Submit your music to ListenBrainz"
source_locale = "en_US"

[name_i18n]
de = "ListenBrainz-Submitter"
fr = "Soumetteur ListenBrainz"

[description_i18n]
de = "Sende deine Musik zu ListenBrainz"
fr = "Soumettez votre musique sur ListenBrainz"
```

### __init__.py
```python
from picard.plugin3.api import PluginApi
from picard.i18n import gettext as _

def enable(api: PluginApi):
    # Use Picard's translations for standard UI
    menu_text = _("Plugins")

    # Use plugin translations for plugin-specific text
    login_text = api.tr("ui.button.login", "Login to ListenBrainz")
    logout_text = api.tr("ui.button.logout", "Logout")

    # With parameters
    status = api.tr("status.submitted", "Submitted {count} tracks", count=5)

    # Plurals
    file_count = api.trn("status.files", "{n} file", "{n} files", n=count)
```

### locale/en.json (generated by extraction tool)
```json
{
  "ui.button.login": "Login to ListenBrainz",
  "ui.button.logout": "Logout",
  "status.submitted": "Submitted {count} tracks",
  "status.files": {
    "one": "{n} file",
    "other": "{n} files"
  }
}
```

### locale/de.json
```json
{
  "ui.button.login": "Bei ListenBrainz anmelden",
  "ui.button.logout": "Abmelden",
  "status.submitted": "{count} Titel übermittelt",
  "status.files": {
    "one": "{n} Datei",
    "other": "{n} Dateien"
  }
}
```

---

## Weblate Integration (Future)

For collaborative translation, plugins could integrate with Weblate:

1. Plugin repository connected to Weblate
2. Translators contribute via web interface
3. Translations automatically committed to repository
4. Plugin developer merges translation updates

**Community Feedback:**
> **rdswift:** "Perhaps the final plugin developer documentation could contain links to information pertaining to setup and use of a weblate server to help facilitate the process for plugins not managed by Picard, and ultimately provide more user-friendly plugins."

**Recommendation:** Include in plugin developer documentation:
- Link to Weblate setup guide
- Instructions for connecting plugin repository
- Configuration examples
- Best practices for managing translation PRs
- Information about MusicBrainz Weblate instance (if available for community plugins)

### TOML Format Support

**Weblate supports TOML files** via translate-toolkit (added in version 3.16.0). If we migrate from JSON to TOML for translation files, Weblate can handle them natively.

**Supported TOML variants:**
- **Plain TOML**: Simple key-value pairs and nested tables
- **Go i18n TOML**: Supports pluralized strings using CLDR plural categories (zero, one, two, few, many, other)

**Example Plain TOML translation file:**
```toml
# locale/de.toml
[ui]
button_login = "Bei ListenBrainz anmelden"
button_logout = "Abmelden"

[error]
auth_failed = "Authentifizierung fehlgeschlagen"
network = "Netzwerkfehler: {error}"

[status]
submitted = "{count} Titel übermittelt"
```

**Benefits of TOML over JSON:**
- Comments supported (translator notes)
- More human-readable structure
- Nested sections for organization
- Native support in Weblate via translate-toolkit
- Consistent with MANIFEST.toml format

**Considerations:**
- Requires updating PluginApi.gettext() to parse TOML instead of JSON
- Need to decide on flat keys (`"ui.button.login"`) vs nested tables (`[ui]` + `button_login`)
- Weblate can handle both approaches
- TOML support was added to translate-toolkit in 3.16.0 and integrated into Weblate 4.1+

**Note:** While translate-toolkit supports TOML, it's not explicitly listed in Weblate's format documentation table yet. However, since Weblate uses translate-toolkit for format support, TOML files should work. Testing recommended before full adoption.

**References:**
- [translate-toolkit TOML support](https://docs.translatehouse.org/projects/translate-toolkit/en/latest/formats/toml.html)
- [Weblate issue #2558 - Go i18n formats including TOML](https://github.com/WeblateOrg/weblate/issues/2558) (closed, implemented in 4.1)

---

## See Also

- **[MANIFEST.md](MANIFEST.md)** - MANIFEST.toml specification
- **[REGISTRY.md](REGISTRY.md)** - Registry metadata translations
- **[WEBSITE.md](WEBSITE.md)** - Website translation extraction
