/*
 *  CUPS add-on module for Canon UFR II printer.
 *  Copyright CANON INC. 2015
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <cups/cups.h>
#include <cups/ppd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "functions.h"

#define	SEPARATOR						","
#define	PPD_KEY_CNGSADDITIONALOPTIONS	"*CNGSAdditionalOptions:"
#define	REAR_ARGUMENT					"-"
#define PPD_KEY_SPACE_LEN 1

#define CNGPLP_OPTION_FILE_PATH					"/etc/cngplp/options/options.conf"
#define	CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS	"*CNGSAdditionalOptions:"
#define	CNGPLP_OPTION_KEY_CNCMCORRECTGRAY		"*CNCMCorrectGray"
#define	CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE		"TRUE"
#define CNGPLP_OPTION_KEY_SPACE_LEN 1

static char *get_gs_option(char *string, char **next);
static void free_gs_cmd_arg(char **cmd_arg, int arraycnt);
static char* get_ppd_value(char* pKey);
static char* get_option_value(char* pKey);

static
char *get_gs_option(char *string, char **next)
{
	char *gs_option = NULL;

	if ((string != NULL) && (next != NULL)) {

		char *tmpstr = NULL;
		tmpstr = strtok_r(string, SEPARATOR, next);
		if (tmpstr != NULL) {
			gs_option = strdup(tmpstr);
		}
	}

	return gs_option;
}

static
void free_gs_cmd_arg(char **cmd_arg, int arraycnt)
{
	if (cmd_arg != NULL) {
		int i = 0;

		for(i = 0 ; i < arraycnt ; i++)
		{
			if (cmd_arg[i] != NULL) {
				free(cmd_arg[i]);
				cmd_arg[i] = NULL;
			}
		}
		free(cmd_arg);
	}
}

static
char* get_ppd_value(char* pKey)
{
	char	*pValue = NULL;

	if (pKey != NULL) {
		FILE	*fp = NULL;
		char	*p_ppd_name = getenv("PPD");

		if (p_ppd_name != NULL) {
			fp = fopen(p_ppd_name, "r");
			if (fp != NULL) {
				char	line_buf[LINE_BUF_SIZE];

				while (fgets(line_buf, sizeof(line_buf), fp)) {
					if (strncmp(line_buf, pKey, strlen(pKey)) == 0) {
						pValue = (char*)calloc((size_t)BUFSIZE, sizeof(char));
						if (pValue != NULL) {
							(void)GetValue((line_buf + PPD_KEY_SPACE_LEN + (int)strlen(pKey)), pValue, BUFSIZE);
						}
						break;
					}
				}
				fclose(fp);
			}
		}
	}
	return pValue;
}

static
char* get_option_value(char* pKey)
{
	char	*pValue = NULL;

	if (pKey != NULL) {
		FILE		*fp = NULL;

		fp = fopen(CNGPLP_OPTION_FILE_PATH, "r");
		if (fp != NULL) {
			char	line_buf[LINE_BUF_SIZE];

			while (fgets(line_buf, sizeof(line_buf), fp)) {
				if (strncmp(line_buf, pKey, strlen(pKey)) == 0) {
					pValue = (char*)calloc((size_t)BUFSIZE, sizeof(char));
					if (pValue != NULL) {
						(void)GetValue((line_buf + CNGPLP_OPTION_KEY_SPACE_LEN + (int)strlen(pKey)), pValue, BUFSIZE);
					}
					break;
				}
			}
			fclose(fp);
		}
	}
	return pValue;
}

char **init_cmd_arg(char *org_arg[], int org_arg_cnt, int *cmd_arg_num)
{
	char **cmd_arg = NULL;
	int add_arg_cnt = 0;
	bool success = true;
	char unifiedoption[LINE_BUF_SIZE] = {0};

	if (org_arg == NULL) {
		success = false;
	}

	if (success != false) {
		char *ppdvalue = NULL;
		char *optionvalue = NULL;

		ppdvalue = get_ppd_value( PPD_KEY_CNGSADDITIONALOPTIONS );

		if ( ppdvalue != NULL ) {
			char *tmpstr = NULL;

			add_arg_cnt = add_arg_cnt + 1;
			tmpstr = strstr(ppdvalue, SEPARATOR);

			while (tmpstr != NULL) {
				add_arg_cnt = add_arg_cnt + 1;
				tmpstr = tmpstr + 1;
				tmpstr = strstr(tmpstr, SEPARATOR);
			}
		}

		if (ppdvalue != NULL) {
			strncat(unifiedoption, ppdvalue, BUFSIZE);

			free(ppdvalue);
			ppdvalue = NULL;
		}

		optionvalue = get_option_value( CNGPLP_OPTION_KEY_CNGSADDITIONALOPTIONS );

		if ( optionvalue != NULL ) {
			char *next = NULL;
			char *tmpstr = NULL;
			char *ppdvalue_pos = NULL;

			tmpstr = get_gs_option(optionvalue, &next);
			while (tmpstr != NULL) {

				ppdvalue_pos = strstr(unifiedoption, tmpstr);
				if (ppdvalue_pos == NULL) {
					add_arg_cnt = add_arg_cnt +  + 1;
					strncat(unifiedoption, SEPARATOR, strlen(SEPARATOR)+1);
					strncat(unifiedoption, tmpstr, strlen(tmpstr)+1);
				}

				free(tmpstr);
				tmpstr = next;
				tmpstr = get_gs_option(tmpstr, &next);
			}

			free(optionvalue);
			optionvalue = NULL;
		}
	}

	if (success != false) {
		*cmd_arg_num = (org_arg_cnt + add_arg_cnt);
		cmd_arg = (char **)calloc((org_arg_cnt + add_arg_cnt), sizeof(char *));

		if (cmd_arg == NULL) {
			success = false;
		}
	}

	if (success != false) {
		int idx_org_arg = 0;
		int idx_cmd_arg = 0;

		for(idx_org_arg = 0 ; idx_org_arg < org_arg_cnt ; idx_org_arg++){
			if (org_arg[idx_org_arg] != NULL){
				if (strcasecmp( org_arg[idx_org_arg], REAR_ARGUMENT) == 0 ){
					int idx_add_arg = 0;
					char *next = NULL;
					char *tmpstr = unifiedoption;

					for (idx_add_arg = 0 ; idx_add_arg < add_arg_cnt ; idx_add_arg++ ){
						cmd_arg[idx_cmd_arg] = get_gs_option(tmpstr, &next);
						tmpstr = next;

						if (cmd_arg[idx_cmd_arg] == NULL) {
							success = false;
							break;
						}

						idx_cmd_arg = idx_cmd_arg + 1;
					}
				}

				if (success != false) {
					cmd_arg[idx_cmd_arg] = strdup(org_arg[idx_org_arg]);
					if (cmd_arg[idx_cmd_arg] == NULL) {
						success = false;
						break;
					}
				} else {
					break;
				}

			} else {
				cmd_arg[idx_cmd_arg] = NULL;
			}

			idx_cmd_arg = idx_cmd_arg + 1;
		}
	}

	if (success == false) {
		if (cmd_arg != NULL) {
			free_gs_cmd_arg(cmd_arg, (org_arg_cnt + add_arg_cnt));
			cmd_arg = NULL;
		}
	}

	return cmd_arg;
}

void set_correct_gray(char *p_buf, int buf_len)
{
	if( NULL != p_buf ) {
		char *optionvalue = get_option_value( CNGPLP_OPTION_KEY_CNCMCORRECTGRAY );
		if( NULL != optionvalue ) {
			if( strcmp(optionvalue, CNGPLP_OPTION_CNCMCORRECTGRAY_TRUE) == 0 ) {
				int new_len = strlen(p_buf) + strlen(CNGPLP_OPTION_KEY_CNCMCORRECTGRAY);
				if( new_len < (buf_len - 1) )
				{
					strncat(p_buf, CNGPLP_OPTION_KEY_CNCMCORRECTGRAY, strlen(CNGPLP_OPTION_KEY_CNCMCORRECTGRAY)+1);
				}
			}
			free(optionvalue);
			optionvalue = NULL;
		}
	}
}

