;;;*************************************************************************************************
;;;*                                       gnet-spice-msw.scm                                      *
;;;*                                      --------------------                                     *
;;;* Description : gEDA               - GPL Electronic Design Automation                           *
;;;*               gnetlist           - gEDA Netlist                                               *
;;;*               gnet-spice-msw.scm - gnetlist SPICE backend                                     *
;;;* Started     : 1998-01-01                                                                      *
;;;* Updated     : Refer below to version string declaration                                       *
;;;* Authors     : Ales Hvezda Copyright (C) 1998-2010                                             *
;;;*               gEDA Contributors Copyright (C) 1998-2010                                       *
;;;*               S. Gieltjes : SPICE netlist backend                                             *
;;;*               W. Kazubski : Modified to use scaling parameters for devices other than MOSFETS *
;;;*               Stuart Brorson (SDB) : Hacked to add support advanced SPICE netlist generation  *
;;;*               Mike Waters (2016-2022) : New features, refactoring and debugging               *
;;;*************************************************************************************************

;;;*************************************************************************************************
;;;*   This program is free software; you can redistribute it and/or modify it under the terms     *
;;;*   of the GNU General Public License as published by the Free Software Foundation; either      *
;;;*   version 2 of the License, or (at your option) any later version.                            *
;;;*                                                                                               *
;;;*   This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;   *
;;;*   without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   *
;;;*   See the GNU General Public License for more details.                                        *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Get the current version of this Guile script.
;;;
;;; Return Values :
;;;   The current version string.

(define spice-msw:version "v1.1.12 (2022-01-08)")

;;;*************************************************************************************************
;;; Things To Do :
;;;
;;; 2022-01-06 Embed_mode isn't implemented here., should it be?
;;; 2022-01-02 What is a SPICE-IO?
;;; 2016-10-24 The Josephson Junction procedure "write-joseph-jn" uses the prefix B for the refdes
;;;            but in SPICE this is the prefix for a non-linear dependent source.
;;; 2016-09-08 Search for "???" to find things needing attention and resolve them.
;;;*************************************************************************************************

;;;*************************************************************************************************
;;; Usage :
;;;
;;;   1. Invoke this guile backend from gnetlist :
;;;        lepton-netlist -g spice-msw -o circuit.ckt circuit.sch
;;;   2. Send command line arguments to the guile backend :
;;;        lepton-netlist -O <OPTN1> -O <OPTN2> -g spice-msw circuit.sch
;;;   3. Send the gnetlist output to standard out :
;;;        lepton-netlist -g spice-msw -o - circuit.sch
;;;
;;; Options :
;;;
;;;   include_mode - Enable .INCLUDE directives instead of inserting file contents
;;;   nomunge_mode - Enable testing of package prefix and prepending the correct prefix if required
;;;   sort_mode    - Enable sorting of packages alphabetically according to refdes's
;;;   embed_mode   - Enable inserting of file contents when a .INCLUDE directive is encountered
;;;   no_end_mode  - Disable appending of .END or .ENDS directives at the end of the netlist file
;;;*************************************************************************************************

;;;*************************************************************************************************
;;;                                     Program Description
;;;
;;; Overview :
;;;
;;;   The top level utility "lepton-netlist" sets the seen (eg. initializes liblepton, loads
;;;   modules) then it calls "netlist.scm" which opens and ingests schematic file/s then calls the
;;;   required backend eg. gnet-spice.scm.
;;;
;;; Program Sections :
;;;
;;;   1. The program entry point (at the top of the file).
;;;   2. High-level procedures for program control.
;;;   3. Procedures for performing file manipulation.
;;;   4. Procedures for creating the netlist file.
;;;   5. Sundry utility procedures.
;;;
;;; Variables :
;;;
;;;   An electronic schematic is made up of various interconnected components. Listed below are often
;;;   used terms found in this program :
;;;    - package   : an electronic component
;;;    - attribute : packages have attributes eg. label, value, etc.
;;;    - net       : the interconnection between components
;;;    - refdes    : the component label eg. R1, Q2 or L3
;;;    -
;;;*************************************************************************************************

;;;*************************************************************************************************
;;; New Features (in gnet-spice-msw.scm) :
;;;
;;;   1. Program is now case insensitive w.r.t. the input (schematic) file contents.
;;;   2. Refactored code to make it more readable eg. matched indenting of matching parenthesis.
;;;   3. In verbose mode the debug spew has been tidied up.
;;;   4. Documentation of procedure argument list and return values where appropriate.
;;;   5. Documentation of program call options.
;;;   6. Fixed a bug where JFETs could be identified as unknown components.
;;;   7. Removed unnecessary procedures eg. "write-footer" and "empty-string?".
;;;   8. Inserted the contents of spice-common.scm in this file. Easier to debug and maintain.
;;;   9. Extend nomunge_mode to include all components eg. resistor, capacitors, etc.
;;;  10. Add support for current controlled switches.
;;;  11. Procedure "get-file-type" now only stops before EOF if a .SUBCKT or .MODEL is encountered.
;;;*************************************************************************************************

;;;*************************************************************************************************
;;; Change Log :
;;;
;;; 2022-01-08 Implement updated mechanism for handling command line arguments.
;;; 2022-01-08 Update this file in line with changes made to the whole "lepton-netlist" code base.
;;; 2022-01-07 Rename and re-order procedures to facilitate readability & therefore maintainability.
;;; 2021-12-30 The variable "package" is often a refdes so in these instances "refdes" is now used.
;;; 2021-12-13 Added procedure "get-command-line" since "netlist.scm" doesn't export it.
;;; 2021-12-07 All references to procedure "debug-spew" originally defined in "netlist.scm" replaced
;;;            with the procedure "spice-msw:dbg-msg".
;;; 2016-11-05 Removed unecessary space characters from component definition lines in netlist.
;;; 2016-09-08 Procedure "get-file-type" now overlooks SPICE .DIRECTIVE commands (eg. .PARAM) and
;;;            only stop before EOF if it encounters a .SUBCKT or .MODEL line.
;;; 2016-10-24 Extend procedure "write-def-compnt" to include as many components as possible.
;;; 2016-10-24 Removed procedure "get-cmpnt-value", it didn't do enough to justify it's existence.
;;; 2016-10-23 Drop support for Guile 1.8.x (gnetlist has dropped support for it).
;;; 2016-10-22 Removed procedure "write-footer", unnecessary obfuscation.
;;; 2016-10-21 Dropped case sensitivity by using string-ci? instead of string? when testing strings.
;;; 2016-10-20 Document procedure argument lists and return values.
;;; 2016-10-20 Moved to a case insensitive model. This backend should now be less finicky.
;;; 2016-10-16 Move the script entry point to the top of the file (seems more logical to me).
;;; 2016-09-17 Refactor code : choose more readable procedure names where necessary.
;;; 2016-09-17 Removed procedure "component-model" as it wasn't used.
;;; 2016-09-17 Removed procedure "component-optional-value" as it wasn't used.
;;; 2016-09-16 Insert contents of spice-common.scm in this file. Easier to debug.
;;; 2016-09-12 Removed procedure "empty-string?" and replaced it with call to builtin "string-null?"
;;; 2016-09-10 Refactor code : use short hand procedure definitions (remove lambda lines),
;;; 2016-09-10 Refactor code : match indenting of matched parenthesis.
;;; 2016-09-10 Refactor code : general tidy up of file formatting.
;;; 2011-06-12 Updated the Problematic name=? symbols to name=unknown and removed the FIXME check
;;;            for them. This should be a step closer to place holder consistancy. (CC)
;;; 2011-01-13 Add four lines of code (and some comments) that allow formatting strings to be used
;;;            for netlisting NGspice device models. (CC)
;;; 2011-01-03 Combine write-ic and write-subcircuit with a fix to the unbound type variable. Fully
;;;            document a check for the special "?" value explaining why it fails silently. Clean up
;;;            write-net-names-on-component to make it a bit more flexible. Combine write-probe-item
;;;            and write-net-names-on-component. Add a range utility function.  (CC)
;;; 2008-01-09 Fix slotted part handling to work without a modified pinseq.  (pcjc2)
;;; 2007-04-28 Fixed slotted part stuff so that it uses pinseq to emit pins.  (SDB)
;;; 2007-02-10 Various bugfixes. Also incorporated slotted part netlist patch from Jeff Mallatt. (SDB)
;;; 2006-04-11 Changed the .END and .ENDS cards to lowercase.
;;;            This fixes bug 1442912. Carlos Nieves Onega.
;;; 2006-03-10 Added "m" attribute to PMOS and NMOS per request of Peter Kaiser.
;;; 2005-12-27 Fix bug discovered by John Doty: spice-IO pins with refdes greater than P9 were
;;;            sorted incorrectly (as strings). Now they are sorted as numbers.
;;; 2005-09-11 Incorporated patch from Paul Bunyk to enable netlisting of Josephson junctions and
;;;            "K" mutual inductances. Also enabled netlisting of "COIL" devices as inductors.
;;; 2005-06-12 Changed order of writing out netlist and .model/.subckt cards to facilitate use of
;;;            numparam with ngspice. Change supplied by Dominique Michel.
;;; 2005-05-16 Modified behavior of .INCLUDE directive. Now by default it just spits out the string
;;;            instead of putting the contents of the file into the SPICE netlist. You can force
;;;            insertion of the file using the -e flag.
;;; 2005-03-16 Fixed CCCS bug (typo in Vsense) noticed by David Logan
;;; 2004-10-09 Added patches for voltage controlled switches from Peter Kaiser.
;;; 2004-08-29 Changed sense source naming in controlled sources because the old convention was
;;;            confusing ngspice.
;;; 2004-08-22 Added command line as first line of file.
;;; 2004-03-24 Bugfixes made to JFET stuff during Feb.  Change released now.
;;; 2003-12-29 Change res & cap to incorporate modelname & "area=" attrib.
;;; 2003-12-29 Minor bugfixes.
;;; 2003-12-29 Two small enhancements requested by Peter Kaiser.
;;; 2003-12-25 Bugfix : Unswizzled emission of pins from user-defined .subckts. (Now correctly uses
;;;                     pinseq to define emission order of pins.) Also added ability to emit
;;;                     attributes for semiconductors (e.g. area, off, ic, etc.)  Added in response
;;;                     to user requests.
;;; 2003-10-14 Bugfixes : - Added empty-string? and hacked get-file-type to handle case where a
;;;                         model file has an empty line before .SUBCKT or .MODEL.
;;;                       - Also modified write-net-names-on-component to gracefully handle case
;;;                         where not every pin has a pinseq attribute. Now only outputs pins with
;;;                         valid pinseq attribute.
;;; 2003-09-09 Rearranged code for more organization (I was beginning to get lost ....).
;;;            Incorporated changes to handle external SPICE files more intelligently. Changed spew
;;;            to be configurable by setting "-v" from the command line. Placed new function
;;;            debug-spew into gnetlist.scm. Added -I command line flag.
;;; 2003-08-29 Include patches from Ken Healy to sort netlist, code by SDB to use gnetlist command
;;;            line args in Scheme fcns, as well as from Theo Deckers to fix strange problem with
;;;            '.SUBCKT quoting.
;;; 2003-03-31 Hacked to enable creating .SUBCKT schematics for hierarchical circuit modeling.
;;; 2003-03-17 Hacked to allow for .SUBCKT files to model ics. Changed write-ic. Added
;;             get-file-type. Added write-subcircuit.
;;; 2003-03-05 Started hacking. (SDB)
;;;*************************************************************************************************

;;; Define modules needed by this program
(use-modules (ice-9 rdelim)
             (ice-9 match)
             (srfi srfi-1)
             (srfi srfi-13)
             (netlist attrib compare)
             (netlist error)
             (netlist option)
             (netlist schematic)
             (netlist schematic toplevel)
)

;;; Flags that may be passed to this back-end
(define include-mode? #f)
(define nomunge-mode? #f)
(define sort-mode?    #f)
(define embed-mode?   #f)
(define no-end-mode?  #f)

;;; Closure storing calling flags
;;; (the procedure "calling-flag?" is defined in "/usr/share/lepton-eda/scheme/netlist.scm")
(let
  ( (flags (gnetlist:get-calling-flags)) )

  (set! include-mode? (calling-flag? "include_mode" flags))
  (set! nomunge-mode? (calling-flag? "nomunge_mode" flags))
  (set! sort-mode?    (calling-flag? "sort_mode"    flags))
  (set! embed-mode?   (calling-flag? "embed_mode"   flags))
  (set! no-end-mode?  (calling-flag? "no_end_card"  flags))
)

;;;*************************************************************************************************
;;;*                                                                                               *
;;;*          Section 1 - Program Entry Point                                                      *
;;;*                                                                                               *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Spice netlist generation entry point.
;;;
;;; The algorithm is as follows :
;;;
;;;   1. Determine if there's a .SUBCKT block in the schematic or if it is just a normal schematic.
;;;      If there's a .SUBCKT block :
;;;       - Write out subcircuit header (a comment identifying the netlister).
;;;       - Find all spice-IO pins. Get a list of the packages.
;;;       - Put them in order (ordered by package refdes)
;;;       - Get the list of nets attached to the spice-IO pins.
;;;       - Write out .SUBCKT line
;;;      If a normal schematic :
;;;       - Write out top header (a comment identifying the netlister).
;;;   2. Loop through all components looking for components with a "file" attribute. Every time a
;;;      "file" attribute is found :
;;;       - Open the file and find out what kind of file it is (.SUBCKT or .MODEL).
;;;       - Determine if the file has previously been processed. If not stick the following info.
;;;         into the file-info list : (model-name file-name file-type), otherwise just continue.
;;;   3. Loop through all components again, and write out a SPICE card for each.
;;;   4. Afterwards, for each item in the file-info list, open the file and write it's contents into
;;;      the netlist.
;;;   5. If the schematic type is .SUBCKT :  write out .ENDS otherwise write out .END
;;;   6. Close the SPICE netlist file and return.
;;;
;;; Argument List :
;;;   netlist-filename - The name of the SPICE netlist file

(define (spice-msw netlist-filename)

  (let*
    (
      (refdes-list (schematic-package-names (toplevel-schematic)))
      (schem-type  (spice-msw:get-schem-type refdes-list))
      (model-name  (substring schem-type 8 (string-length schem-type)))
      (file-list   (list))
    )

    (spice-msw:dbg-msg "\nRunning the SPICE backend to gnetlist :\n\n")

    ;; First decide if this is regular schematic or a .SUBCKT lower level
    (if (string=? schem-type "normal schematic")
      ;; This is a regular schematic
      (begin
        (spice-msw:dbg-msg "Found a normal type schematic\n\n")
        (display (string-append "* " (spice-msw:get-command-line) "\n"))
        (spice-msw:write-netlist-header)
      )
      ;; This is a .SUBCKT type schematic
      (let*
        (
          (io-pin-pkgs         (spice-msw:get-spice-io-pins  refdes-list (list)))
          (io-pin-pkgs-ordered (spice-msw:sort-spice-io-pins io-pin-pkgs))
          (io-nets-list        (spice-msw:get-io-nets        io-pin-pkgs-ordered (list)))
        )

        ;; Write out a .SUBCKT header and .SUBCKT line
        (spice-msw:dbg-msg "Found a .SUBCKT type schematic\n\n")
        (spice-msw:write-subckt-header)
        (let
          ( (io-nets-string (spice-msw:list-2-str io-nets-list)) )

          (display (string-append schem-type " " io-nets-string "\n"))
        )
      )
    )

    ;; Search all the devices and compile a model file list from all the "file" attributes
    (spice-msw:dbg-msg "Make first pass through design and create list of all model files referenced :\n")
    (set! file-list (spice-msw:create-file-list refdes-list file-list))
    (spice-msw:dbg-msg "\nDone creating file-list\n\n")

    ;; Loop through the file list and add the file contents or a reference to the file in the netlist
    (spice-msw:dbg-msg "Now process the items in the model file list :\n")
    (spice-msw:process-files file-list)
    (spice-msw:dbg-msg "\nDone processing items in the model file list\n\n")

    ;; Write components to the netlist file (sorting components if required)
    (spice-msw:dbg-msg "Make second pass through design and write out a SPICE card for each component found :\n")
    (if sort-mode?
      (spice-msw:write-components file-list (sort refdes-list spice-msw:refdes-sort)) ;; Sort on refdes
      (spice-msw:write-components file-list refdes-list)                              ;; Don't sort
    )
    (spice-msw:dbg-msg "\n")

    ;; Write .END(S) depending upon whether schematic is a "normal schematic" or .SUBCKT
    (if (not (string=? schem-type "normal schematic"))
      (display (string-append ".ENDS " model-name "\n"))
;;      (if (not (calling-flag? "no_end_mode" (gnetlist:get-calling-flags)))
     (if (not no-end-mode?)
        (display ".END\n")
      )
    )
  )

  (spice-msw:dbg-msg "Done writing SPICE cards\n\n")
)

;;;*************************************************************************************************
;;;*                                                                                               *
;;;*          Section 2 - High-level procedures for program control                                *
;;;*                                                                                               *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Get the command line used to envoked this process.
;;;
;;; Note : The module netlist.scm implements this procedure but doesn't export it.
;;;
;;; Return Values :
;;;   The command line as a string

(define (spice-msw:get-command-line)

"Outputs gnetlist command line as a string. Only basename of the command is output.
If gnetlist is called from libtool wrapper, the name is changed to canonical."

  (define libtool-prefix "lt-")
  (define lt-prefix-length (string-length libtool-prefix))
  (define (remove-lt-prefix name)
    (if (string-prefix? libtool-prefix name)
        (string-drop name lt-prefix-length)
        name
    )
  )
  (let
    (
      (name (remove-lt-prefix (basename (car (program-arguments)))))
    )
    (string-join (cons name (cdr (program-arguments))) " ")
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Determine the schematic type ie. a normal schematic or a .SUBCKT lower level.
;;;
;;; Search for a "SPICE-SUBCIRCUIT-LL" device instantiated somewhere in the schematic. If it is a
;;; .SUBCKT return ".SUBCKT model-name" else return "normal schematic".
;;;
;;; Argument List :
;;;   refdes-list - A list of package (component) labels (refdes's)

(define (spice-msw:get-schem-type refdes-list)

  (if (not (null? refdes-list))
    (let*
      (
        (refdes (car refdes-list))
        (device (gnetlist:get-package-attribute refdes "device"))
      )

      (if (string-ci=? device "SPICE-SUBCIRCUIT-LL")      ;; Look for a subcircuit label
        (string-append ".SUBCKT " (gnetlist:get-package-attribute refdes "model-name"))
        (spice-msw:get-schem-type (cdr refdes-list))  ;; Iterate to the next refdes
      )
    )
    "normal schematic"
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; This procedure takes a list of package labels (refdes's). It runs through the package labels and
;;; for each label gets the package attributes. If there is a "FILE" attribute, it gets the file
;;; info. & uses it to build a file-list. When done, it returns the file-list.
;;;
;;; Argument List :
;;;   refdes-list - A list of package (component) labels (refdes's)
;;;   file-list   - A list of files referred to in the schematic file

(define (spice-msw:create-file-list refdes-list file-list)

  (if (null? refdes-list)
    file-list                            ;; Return the file-list
    (let*
      (
        (refdes     (car refdes-list))  ;; Get the next package (ie. refdes)
        (device     (string))
        (model      (string))
        (value      (string))
        (model-file (string))
      )

      (set! device     (gnetlist:get-package-attribute refdes "device") )
      (set! model      (gnetlist:get-package-attribute refdes "model-name") )
      (set! value      (gnetlist:get-package-attribute refdes "value") )
      (set! model-file (gnetlist:get-package-attribute refdes "file") )

      ;; Now run a series of checks to see if we should add this file to the file-list
      ;; Check to see if "file" attribute is non-empty
      (if (not (string-ci=? model-file "unknown"))
        (begin
          (spice-msw:dbg-msg "\n  spice-msw:create-file-list : ")
          (spice-msw:dbg-msg (string-append "Found file attribute for package " refdes " = " model-file "\n"))

          ;; Now check if the file is already in the file-list
          (if (not (spice-msw:in-file-list? model-file file-list))
            ;; File is new, open it and find out what type it is
            (let
              ( (file-type (spice-msw:get-file-type model-file)) )

              (spice-msw:dbg-msg "  spice-msw:create-file-list : ")
              (spice-msw:dbg-msg (string-append "File is not in the list and has type " file-type "\n"))
              ;; if #f : Check to see if file-type is known.
              (if (not (string-ci=? file-type "OTHER"))
                (begin
                  (spice-msw:dbg-msg "  spice-msw:create-file-list : Adding it to the list of model files\n")

                  (set! file-list (append (list (list model model-file file-type)) file-list) )
                )
                (spice-msw:dbg-msg "  spice-msw:create-file-list : File type is OTHER so ignore it\n")
              )
            )

            ;; File is already in list. Print debug spew if desired.
            (spice-msw:dbg-msg "  spice-msw:create-file-list : File is already in the model file list.\n")
          )
        )
      )

      ;; Completed checking and processing of this package, iterate to the next one
      (spice-msw:create-file-list (cdr refdes-list) file-list)
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~-
;;; This is a helper function which returns #t if a file is already in file-list, otherwise #f.
;;;
;;; Note : It is assumed that file-list is of the form :
;;;   ( (model1 file-name1 file-type1)  (model2 file-name2 file-type2) . . . . )
;;;
;;; Argument List :
;;;   file-name - The file name to search for
;;;   file-list - The list of file info. to search
;;;
;;; Return Values :
;;;   Success - #t (the file was    found in the list)
;;;   Failure - #f (the file wasn't found in the list)

(define (spice-msw:in-file-list? file-name file-list)

  (if (or (null? file-list) (null? file-name))
    #f                                             ;; file-list or file-name is empty
    (let
      ( (list-element (car file-list)) )           ;; Get a list element and process it

      (if (null? list-element)
        #f                                         ;; list-element is empty (should never get here)
        (let
          ( (list-file-name (cadr list-element)) )

          (if (string=? list-file-name file-name)
            #t                                     ;; file-name was found in file-list
            (spice-msw:in-file-list? file-name (cdr file-list))  ;; Iterate . . .
          )
        )
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~-
;;; Search a file and determine it's type based on it's contents.
;;;
;;; Note : The function opens input-file and closes it when it is done.
;;;
;;; Argument List :
;;;   file-name - The file to test
;;;
;;; Return Values :
;;;   ".MODEL"  - The file contains a model
;;;   ".SUBCKT" - The file contains a sub-circuit
;;;   "OTHER"   - The file is neither of the above

(define (spice-msw:get-file-type file-name)

  (if (file-exists? file-name)
    (let
      ( (in-file (open-input-file file-name)) )

      (let while ( (file-line (read-line in-file)) )

        (cond

          ((eof-object? file-line)       ;; Arrived at end of line without finding .MODEL or .SUBCKT
            "OTHER"
          )

          ((string-null? file-line)      ;; Found empty line, iterate before doing anything else
            (while (read-line in-file))
          )

          ((string=? (string (string-ref file-line 0)) "*")  ;; Found comment, iterate
            (while (read-line in-file))
          )

          ((string=? (string (string-ref file-line 0)) ".")  ;; The first char is a '.'
            (begin
              (cond
                ((string-prefix-ci? ".SUBCKT" file-line)    ;; Found .SUBCKT line
                  ".SUBCKT"
                )
                ((string-prefix-ci? ".MODEL"  file-line)    ;; Found .MODEL  line
                  ".MODEL"
                )
                (else
                  (while (read-line in-file))
                )
              )
            )
          )

          (else
            (while (read-line in-file))
          )
        )
      )
    )
    (begin
      (spice-msw:dbg-msg (string-append "ERROR: File '" file-name "' not found.\n"))
      (primitive-exit 1)
    )
  )
)

;;;*************************************************************************************************
;;;*                                                                                               *
;;;*          Section 3 - Procedures for performing file manipulation                              *
;;;*                                                                                               *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Loop through the model-file list looking for a triplet corresponding to a particular model-name.
;;;
;;; Argument List :
;;;   model-name - The model name being sort
;;;   file-list  - A list of files referred to in the schematic file
;;;
;;; Return Values :
;;;   Success - The file list item associated with the model name
;;;   Failure - #f

(define (spice-msw:get-file-list-item model-name file-list)

  (if (null? file-list)
    (quote #f)
    (let*
      (
        (list-item       (car   file-list))
        (item-model-name (car   list-item))
        (item-file-name  (cadr  list-item))
        (item-file-type  (caddr list-item))
      )

      (if (string-ci=? item-model-name model-name)
         list-item                                                 ;; Found model-name
         (spice-msw:get-file-list-item model-name (cdr file-list)) ;; Keep looking
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Loop through the model file list and for each file invoke handle-spice-file.
;;;
;;; Argument List :
;;;   file-list - A list of model files referred to in the schematic file

(define (spice-msw:process-files file-list)

  (if (not (null? file-list))
    (let*
      (
        (list-element (car  file-list))    (model-name (car   list-element))
        (file-name    (cadr list-element)) (file-type  (caddr list-element))
      )

      (spice-msw:process-spice-file file-name)
      (spice-msw:process-files (cdr file-list))
    )
    (display "\n")
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; This wraps insert-text-file.
;;;
;;; If "include_mode" has been enabled just write an .INCLUDE card with the file name. If not call
;;; "insert-text-file" to insert the file's contents into the netlist file.
;;;
;;; Argument List :
;;;   file-name - The file name to be processed

(define (spice-msw:process-spice-file file-name)

  (spice-msw:dbg-msg (string-append "\n  spice-msw:process-spice-file : " file-name "\n"))

  (if include-mode?
    (display (string-append ".INCLUDE " file-name "\n"))  ;; Use .INCLUDE card
    (spice-msw:insert-text-file file-name)                ;; Insert file contents
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Open a file, get the contents and insert it into the SPICE file.
;;;
;;; This function is usually used to include SPICE models contained in files into the netlist.
;;;
;;; Argument List :
;;;   file-name - The name of the file

(define (spice-msw:insert-text-file file-name)

  (if (file-exists? file-name)
    (let
      ( (in-file (open-input-file file-name)) )

      (display (string-append "* Include SPICE model file : " file-name "\n") )

      (let while ((line (read-line in-file)))
        (if (not (eof-object? line))
          (begin
            (display (string-append line "\n"))
            (while (read-line in-file))
          )
        )
      )

      (close-port in-file)
    )
    (begin
      (spice-msw:dbg-msg (string-append "ERROR: File '" file-name "' not found.\n"))
      (primitive-exit 1)
    )
  )
)

;;;*************************************************************************************************
;;;*                                                                                               *
;;;*          Section 4 - Procedures for creating the netlist file                                 *
;;;*                                                                                               *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write out spice netlist header.

(define (spice-msw:write-netlist-header)

  (display                "*************************************************************\n")
  (display                "*       SPICE file generated by lepton-netlist using        *\n")
  (display (string-append "*       the backend : spice-msw, " spice-msw:version "       *\n"))
  (display                "*************************************************************\n\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~-
;;; Write out .SUBCKT netlist header.

(define (spice-msw:write-subckt-header)

  (display "*************************************\n")
  (display "*        Begin .SUBCKT model        *\n")
  (display "*************************************\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; This function is passed a list of refdes's. It uses each refdes to get the corresponding
;;; "device" attribute. Depending upon the device, it then invokes one or another of the spice line
;;; output procedures to output a line of the spice netlist.
;;;
;;; Write the refdes, to the pin# connected net and component value and optional extra attributes
;;; check if the component is a special spice component.
;;;
;;; Argument List :
;;;   file-list - A list of files referred to in the schematic file
;;;   refdes-list - A list of package (component) labels (ie. refdes's)

(define (spice-msw:write-components file-list refdes-list)

  (if (not (null? refdes-list))
    (let*
      (
        (refdes (car refdes-list))
        (device (gnetlist:get-package-attribute refdes "device"))
      )

      ;; Super debug stuff -- outputs line describing device being processed.
      (spice-msw:dbg-msg "\n  spice-msw:write-components    : ")
      (spice-msw:dbg-msg (string-append "Check package refdes = " refdes ", device = " device "\n"))

      (cond
        ((string-ci=? device "NONE"               ))  ;; Do nothing for graphical symbols
        ((string-ci=? device "SPICE-SUBCIRCUIT-LL"))  ;; Do nothing for subcircuit declaration
        ((string-ci=? device "SPICE-IO"           ))  ;; Do nothing for SPICE IO pins
        ((string-ci=? device "SPICE-CCVS"         ) (spice-msw:write-ccvs        refdes))
        ((string-ci=? device "SPICE-CCCS"         ) (spice-msw:write-cccs        refdes))
        ((string-ci=? device "SPICE-VCVS"         ) (spice-msw:write-vcvs        refdes))
        ((string-ci=? device "SPICE-VCCS"         ) (spice-msw:write-vccs        refdes))
        ((string-ci=? device "VOLTAGE_SOURCE"     ) (spice-msw:write-ivs         refdes))
        ((string-ci=? device "CURRENT_SOURCE"     ) (spice-msw:write-ics         refdes))
        ((string-ci=? device "SPICE-NULLOR"       ) (spice-msw:write-nullor      refdes))
        ((string-ci=? device "DIODE"              ) (spice-msw:write-diode       refdes))
        ((string-ci=? device "PMOS_TRANSISTOR"    ) (spice-msw:write-pmos        refdes))
        ((string-ci=? device "NMOS_TRANSISTOR"    ) (spice-msw:write-nmos        refdes))
        ((string-ci=? device "SUBCKT_PMOS"        ) (spice-msw:write-pmos-subckt refdes))
        ((string-ci=? device "SUBCKT_NMOS"        ) (spice-msw:write-nmos-subckt refdes))
        ((string-ci=? device "PNP_TRANSISTOR"     ) (spice-msw:write-pnp         refdes))
        ((string-ci=? device "SPICE-PNP"          ) (spice-msw:write-pnp         refdes))
        ((string-ci=? device "NPN_TRANSISTOR"     ) (spice-msw:write-npn         refdes))
        ((string-ci=? device "SPICE-NPN"          ) (spice-msw:write-npn         refdes))
        ((string-ci=? device "PFET_TRANSISTOR"    ) (spice-msw:write-pfet        refdes))
        ((string-ci=? device "NFET_TRANSISTOR"    ) (spice-msw:write-nfet        refdes))
        ((string-ci=? device "MESFET_TRANSISTOR"  ) (spice-msw:write-mesfet      refdes))
        ((string-ci=? device "SPICE-VC-SWITCH"    ) (spice-msw:write-vc-switch   refdes))
        ((string-ci=? device "SPICE-CC-SWITCH"    ) (spice-msw:write-cc-switch   refdes))
        ((string-ci=? device "AOP-STANDARD"       ) (spice-msw:write-def-compnt  refdes device file-list))
        ((string-ci=? device "RESISTOR"           ) (spice-msw:write-resistor    refdes))
        ((string-ci=? device "CAPACITOR"          ) (spice-msw:write-capacitor   refdes))
        ((string-ci=? device "POLARIZED_CAPACITOR") (spice-msw:write-capacitor   refdes))
        ((string-ci=? device "INDUCTOR"           ) (spice-msw:write-inductor    refdes))
        ((string-ci=? device "COIL"               ) (spice-msw:write-inductor    refdes)) ;; Added to enable netlisting of coil-*.sym
        ((string-ci=? device "JOSEPHSON_JUNCTION" ) (spice-msw:write-joseph-jn   refdes))
        ((string-ci=? device "COUPLED_INDUCTOR"   ) (spice-msw:write-coupled-ind refdes))
        ((string-ci=? device "MODEL"              ) (spice-msw:write-model       refdes))
        ((string-ci=? device "OPTIONS"            ) (spice-msw:write-options     refdes))
        ((string-ci=? device "DIRECTIVE"          ) (spice-msw:write-directive   refdes))
        ((string-ci=? device "INCLUDE"            ) (spice-msw:write-include     refdes))
        ((string-ci=? device "TESTPOINT"          ) (spice-msw:write-probe       refdes))
        (else
          ;; The package couldn't be identified using it's device attribute so attempt to write the
          ;;  package using it's refdes prefix to identify it
          (spice-msw:write-def-compnt refdes device file-list)
        )
      )

      (spice-msw:write-components file-list (cdr refdes-list))
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write out the default component based on the first character of the refdes (label).
;;; Eg. for the refdes Q1, the prefix 'Q' is the default for a bipolar transistor.
;;;
;;; This function does the following :
;;;
;;;   1. Gets the refdes (package label).
;;;   2. Checks the refdes prefix against a list of possible values and calls the corresponding
;;;      procedure which writes out the component.
;;;   3. Otherwise, if the refdes prefix isn't recognized just output the refdes, the attached nets
;;;      and the "value" attribute.
;;;
;;; Argument List :
;;;   refdes    - The refdes (label) of a component (eg. R1)
;;;   device    - The device attribute value for the package
;;;   file-list - A list of files referred to in the schematic file

(define (spice-msw:write-def-compnt refdes device file-list)

  (let
    ( (first-char (string (string-ref refdes 0))) )  ;; Extract first char of refdes

    (cond
      ((string-ci=? first-char "A") (spice-msw:write-ic           refdes file-list))
      ((string-ci=? first-char "C") (spice-msw:write-capacitor    refdes))
      ((string-ci=? first-char "D") (spice-msw:write-diode        refdes))
      ((string-ci=? first-char "E") (spice-msw:write-vcvs         refdes))
      ((string-ci=? first-char "F") (spice-msw:write-cccs         refdes))
      ((string-ci=? first-char "G") (spice-msw:write-vccs         refdes))
      ((string-ci=? first-char "H") (spice-msw:write-ccvs         refdes))
      ((string-ci=? first-char "I") (spice-msw:write-ics          refdes))
      ((string-ci=? first-char "J") (spice-msw:write-compnt-model refdes #f "<unknown>" (list)))
      ((string-ci=? first-char "K") (spice-msw:write-coupled-ind  refdes))
      ((string-ci=? first-char "L") (spice-msw:write-inductor     refdes))
      ((string-ci=? first-char "M") (spice-msw:write-compnt-model refdes #f "<unknown>" (list)))
      ((string-ci=? first-char "Q") (spice-msw:write-compnt-model refdes #f "<unknown>" (list)))
      ((string-ci=? first-char "R") (spice-msw:write-resistor     refdes))
      ((string-ci=? first-char "S") (spice-msw:write-vc-switch    refdes))
      ((string-ci=? first-char "M") (spice-msw:write-compnt-model refdes #f "<unknown>" (list)))
      ((string-ci=? first-char "U") (spice-msw:write-ic           refdes file-list))
      ((string-ci=? first-char "V") (spice-msw:write-ivs          refdes))
      ((string-ci=? first-char "W") (spice-msw:write-cc-switch    refdes))
      ((string-ci=? first-char "X") (spice-msw:write-ic           refdes file-list))
      ((string-ci=? first-char "Z") (spice-msw:write-mesfet       refdes))
      (else
        (spice-msw:dbg-msg "  spice-msw:write-def-compnt     : ")
        (spice-msw:dbg-msg (string-append "Found unknown device " device "\n"))

        (display (string-append refdes " "))  ;; Write a component's refdes
        (spice-msw:write-compnt-nets refdes)
        (display (gnetlist:get-package-attribute refdes "value"))
        (newline)
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write out component followed by model or model file associated with the component.
;;;
;;; This function does the following :
;;;   1.  Writes out the correct refdes prefix (if specified and necessary).
;;;   2.  Writes out the refdes and nets
;;;   3.  Looks for "model-name" attribute. Writes it out if it exists.
;;;   4.  If there is no "model-name" attribute, it writes out the "value" attribute. If there is no
;;;       "value" attribute, it writes out "unknown" and returns, causing the spice simulator to
;;;       puke when the netlist is run. This is important because the spice simulator needs to have
;;;       some indication of what model to look for.
;;;   5.  Outputs optional attributes attached to device, if any.
;;;   6.  Outputs a new line
;;;   7.  Looks for a "model" attribute. If it exists, it writes a .MODEL line like :
;;;         .MODEL model-name type ( model )
;;;
;;; Argument List :
;;;   refdes      - The refdes of a component eg. Q1
;;;   prefix      - The correct refdes prefix for this package type
;;;   type        - The SPICE model type eg. NPN or JFT
;;;   attrib-list - A list of attributes for the package

(define (spice-msw:write-compnt-model refdes prefix type attrib-list)

  (let
    (
      (model-name (gnetlist:get-package-attribute refdes "model-name"))
      (model      (gnetlist:get-package-attribute refdes "model"     ))
      (value      (gnetlist:get-package-attribute refdes "value"     ))
      (area       (gnetlist:get-package-attribute refdes "area"      ))
      (off        (gnetlist:get-package-attribute refdes "off"       ))
      (model-file (gnetlist:get-package-attribute refdes "file"      ))
    )

    ;; Write out the refdes prefix, if required
    (if prefix (spice-msw:write-compnt-prefix refdes prefix) )

    ;; Next we write out the refdes and nets.
    (display (string-append refdes " "))  ;; Write a component's refdes
    (spice-msw:write-compnt-nets refdes)

    ;; Look for "model-name" attribute. Write it out if it exists otherwise look for "value" attribute
    (if (not (string-ci=? model-name "unknown"))
      (display model-name)  ;; Display the model-name
      (display value)       ;; Otherwise display the value
    )

    ;; Next write out attributes if they exist
    ;; First attribute is area. It is written as a simple string
    (if (not (string-ci=? area "unknown"))
      (display (string-append " " area))
    )

    ;; Next attribute is off. It is written as a simple string
    (if (not (string-ci=? off "unknown"))
      (display (string-append " " off))
    )

    ;; Write out remaining attributes
    (spice-msw:write-attrib-list refdes attrib-list)

    ;; Now write out newline in preparation for writing out model.
    (newline)

    ;; Now write out any model which is pointed to by the part.
    (cond

      ;; One line model and model name exist
      ( (not (or (string-ci=? model "unknown") (string-ci=? model-name "unknown")))
        (begin
          (spice-msw:dbg-msg "  spice-msw:write-compnt-model     : ")
          (spice-msw:dbg-msg (string-append "Found model and model-name for " refdes "\n"))
        )
        (display (string-append ".MODEL " model-name " " type " (" model ")\n"))
      )

      ;; One line model and component value exist
      ( (not (or (string-ci=? model "unknown") (string-ci=? value "unknown")))
        (begin
          (spice-msw:dbg-msg "  spice-msw:write-compnt-model     : ")
          (spice-msw:dbg-msg (string-append "Found model and value for " refdes "\n"))
        )
        (display (string-append ".MODEL " model-name " " type " (" value ")\n"))
      )

      ;; Model file and model name exist
      ( (not (or (string-ci=? model-file "unknown") (string-ci=? model-name "unknown")))
        (begin
          (spice-msw:dbg-msg "  spice-msw:write-compnt-model     : ")
          (spice-msw:dbg-msg (string-append "Found file and model-name for " refdes "\n"))
        )
      )

      ;; Model file and component value exist
      ( (not (or (string-ci=? model-file "unknown") (string-ci=? value "unknown")))
        (begin
          (spice-msw:dbg-msg "  spice-msw:write-compnt-model     : ")
          (spice-msw:dbg-msg (string-append "Found file and value for " refdes "\n"))
        )
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a diode.
;;;
;;; Write out a valid diode refdes & then call the function which writes the rest of the line.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. D1

(define (spice-msw:write-diode refdes)

  (spice-msw:dbg-msg "  spice-msw:write-diode         : Found a diode\n")

  (let
    ( (attrib-list (list "IC" "TEMP")) )

    (spice-msw:write-compnt-model refdes "D" "D" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write out a valid IC or sub-circuit line.
;;;
;;; The algorithm is as follows :
;;;  1. Figure out what type of model goes with this part from file-list. If it isn't listed
;;;     look for a MODEL attribute. If a MODEL attribute is attached write out SPICE card and then
;;;     write out .MODEL on next line. If no MODEL attribute is attached just write out what little
;;;     we know then return.
;;;  2. If the model-name is in the file-list, get the associated file-type. Compare it against
;;;     the component's refdes. If model-type is .MODEL or .SUBCKT and refdes doesn't begin with a U
;;;     or X respectively, prepend the correct prefix to the refdes.
;;;  3. Print out the rest of the line.
;;;
;;; Argument List :
;;;   refdes    - The refdes of a component eg. U1
;;;   file-list - A list of files referred to in the schematic file

(define (spice-msw:write-ic refdes file-list)

  ;; First do local assignments
  (let
    (
      (first-char (string (string-ref refdes 0)))  ;; extract first char of refdes
      (model-name (gnetlist:get-package-attribute refdes "model-name"))
      (model      (gnetlist:get-package-attribute refdes "model"))
      (value      (gnetlist:get-package-attribute refdes "value"))
      (type       (gnetlist:get-package-attribute refdes "type"))
      (model-file (gnetlist:get-package-attribute refdes "file"))
      (list-item  (list))
    )

    (cond
      ( (string-ci=? first-char "U")
        (spice-msw:dbg-msg "  spice-msw:write-ic            : Found a Integrated Circuit\n")
      )
      ( (string-ci=? first-char "X")
        (spice-msw:dbg-msg "  spice-msw:write-ic            : Found a Sub-Circuit\n")
      )
    )

    ;; First, if model-name is empty, we use value attribute instead.
    ;; We do this by sticking the contents of "value" into "model-name".
    (if (string-ci=? model-name "unknown") (set! model-name value) )

    ;; Now get item from file-list using model-name as key
    (set! list-item (spice-msw:get-file-list-item model-name file-list) )

    ;; Check if list-item is null
    (if (or (null? list-item) (eq? list-item #f))

      ;; list-item is null.  Evidently, we didn't discover any files holding this model.
      ;; Instead we look for model attribute
      (if (not (string-ci=? model "unknown"))
        (begin                                     ;; Model attribute exists, write card and model
          (spice-msw:dbg-msg "  spice-msw:write-ic            : ")
          (spice-msw:dbg-msg "Model info not found in model file list, but model attribute exists, write out spice card and .MODEL line\n")
          (display (string-append refdes " "))  ;; Write a component's refdes
          (spice-msw:write-compnt-nets refdes)
          (display (string-append model-name "\n" ))
          (display (string-append ".MODEL " model-name " "))
          (if (not (string-ci=? type "unknown")) (display (string-append type " ")) )  ;; If no type then just skip it.
          (display (string-append "(" model ")\n"))
        )
        (begin                                     ;; No model attribute either, just write out card
          (spice-msw:dbg-msg "  spice-msw:write-ic            : ")
          (spice-msw:dbg-msg "Model info not found in model file list, no model attribute either\n")
          (display (string-append refdes " "))  ;; Write a component's refdes
          (spice-msw:write-compnt-nets refdes)
          (display (string-append model-name "\n" ))
        )
      )

      ;; list-item is not null. Therefore we process line depending upon contents of list-item
      (let
        ( (file-type (caddr list-item)) )

        (cond
          ;; File contains a model
          ((string-ci=? file-type ".MODEL")
            (begin
              (spice-msw:dbg-msg "  spice-msw:write-ic            : ")
              (spice-msw:dbg-msg (string-append "Found .MODEL with model-file and model-name for " refdes "\n"))
              (spice-msw:write-compnt-prefix refdes "U")  ;; Write out the refdes prefix, if required
              (display (string-append refdes " "))  ;; Write a component's refdes
              (spice-msw:write-compnt-nets refdes)
              (display (string-append model-name "\n" ))
            )
          )
          ;; File contains a subcircuit
          ((string-ci=? file-type ".SUBCKT")
            (begin
              (spice-msw:dbg-msg "  spice-msw:write-ic            : ")
              (spice-msw:dbg-msg (string-append "Found .SUBCKT with model-file and model-name for " refdes "\n"))
              (spice-msw:write-compnt-prefix refdes "X")  ;; Write out the refdes prefix, if required
              (display (string-append refdes " "))  ;; Write a component's refdes
              (spice-msw:write-compnt-nets refdes)
              (display (string-append model-name "\n" ))
            )
          )
        )
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write an NPN bipolar transistor.
;;;
;;; Write a valid transistor refdes & then call the function which writes the rest of the line.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. Q1

(define (spice-msw:write-npn refdes)

  (spice-msw:dbg-msg "  spice-msw:write-npn           : Found a NPN Bipolar Transistor\n")

  (let
    ( (attrib-list (list "IC" "TEMP")) )

    (spice-msw:write-compnt-model refdes "Q" "NPN" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a PNP bipolar transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. Q1

(define (spice-msw:write-pnp refdes)

  (spice-msw:dbg-msg "  spice-msw:write-pnp           : Found a PNP Bipolar Transistor\n")

  (let
    ( (attrib-list (list "IC" "TEMP")) )

    (spice-msw:write-compnt-model refdes "Q" "PNP" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write an N-channel JFET transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. J1

(define (spice-msw:write-nfet refdes)

  (spice-msw:dbg-msg "  spice-msw:write-nfet          : Found a N-channel JFET\n")

  (let
    ( (attrib-list (list "IC" "TEMP") ))

    (spice-msw:write-compnt-model refdes "J" "NJF" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a P-channel JFET transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. J1

(define (spice-msw:write-pfet refdes)

  (spice-msw:dbg-msg "  spice-msw:write-pfet          : Found a P-channel JFET\n")

  (let
    ( (attrib-list (list "IC" "TEMP")) )

    (spice-msw:write-compnt-model refdes "J" "PJF" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a PMOS transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. M1

(define (spice-msw:write-pmos refdes)

  (spice-msw:dbg-msg "  spice-msw:write-pmos          : Found a PMOS Transistor\n")

  (let
    ( (attrib-list (list "L" "W" "AS" "AD" "PD" "PS" "NRD" "NRS" "TEMP" "IC" "M")) )

    (spice-msw:write-compnt-model refdes "M" "PMOS" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a NMOS transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. M1

(define (spice-msw:write-nmos refdes)

  (spice-msw:dbg-msg "  spice-msw:write-nmos          : Found a NMOS Transistor\n")

  (let
    ( (attrib-list (list "L" "W" "AS" "AD" "PD" "PS" "NRD" "NRS" "TEMP" "IC" "M")) )

    (spice-msw:write-compnt-model refdes "M" "NMOS" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a subckt PMOS transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. X1

(define (spice-msw:write-pmos-subckt refdes)

  (spice-msw:dbg-msg "  spice-msw:write-pmos-subckt   : Found a PMOS Transistor sub-circuit\n")

  (let
    ( (attrib-list (list "L" "W" "AS" "AD" "PD" "PS" "NRD" "NRS" "TEMP" "IC" "M")) )

    (spice-msw:write-compnt-model refdes "X" "PMOS" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a subckt NMOS transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. X1

(define (spice-msw:write-nmos-subckt refdes)

  (spice-msw:dbg-msg "  spice-msw:write-nmos-subckt   : Found a NMOS Transistor sub-circuit\n")

  (let
    ( (attrib-list (list "L" "W" "AS" "AD" "PD" "PS" "NRD" "NRS" "TEMP" "IC" "M")) )

    (spice-msw:write-compnt-model refdes "X" "NMOS" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a MESFET transistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. Z1

(define (spice-msw:write-mesfet refdes)

  (spice-msw:dbg-msg "  spice-msw:write-mesfet        : Found a MESFET Transistor\n")

  (let
    ( (attrib-list (list "IC")) )

    (spice-msw:write-compnt-model refdes "Z" "MESFET" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a voltage controled switch.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. S1

(define (spice-msw:write-vc-switch refdes)

  (spice-msw:dbg-msg "  spice-msw:write-vc-switch     : Found a Voltage Controlled Switch\n")

  (let
    ( (attrib-list (list " ")) )

    (spice-msw:write-compnt-model refdes "S" "SW" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a current controled switch.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. W1

(define (spice-msw:write-cc-switch refdes)

  (spice-msw:dbg-msg "  spice-msw:write-cc-switch     : Found a Voltage Controlled Switch\n")

  (let
    ( (attrib-list (list " ")) )

    (spice-msw:write-compnt-model refdes "W" "CSW" attrib-list)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a resistor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. R1

(define (spice-msw:write-resistor refdes)

  (spice-msw:dbg-msg "  spice-msw:write-resistor      : Found a Resistor\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "R")

  ;; First write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; Next write mandatory resistor value if it exists.
  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (if (not (string-ci=? value "unknown"))
      (display value)
    )
  )

  ;; Next write our model name if it exists
  (let
    ( (model-name (gnetlist:get-package-attribute refdes "model-name")) )

    (if (not (string-ci=? model-name "unknown"))
      (display (string-append " " model-name))
    )
  )

  ;; next create list of attributes which can be attached to a resistor.
  ;; I include non-standard "area" attrib here per popular demand.
  (let
    ( (attrib-list (list "AREA" "L" "W" "TEMP")) )

    (spice-msw:write-attrib-list refdes attrib-list)  ;; write the attributes (if any) separately
  )

  ;; finally output a new line
  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a capacitor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. C1

(define (spice-msw:write-capacitor refdes)

  (spice-msw:dbg-msg "  spice-msw:write-capacitor     : Found a Capacitor\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "C")

  ;; first write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; next write capacitor value, if any.  Note that if the
  ;; component value is not assigned nothing will be written out.
  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (if (not (string-ci=? value "unknown"))
      (display value)
    )
  )

  ;; next write capacitor model name, if any.  This is applicable to
  ;; semiconductor caps used in chip design.
  (let
    ( (model-name (gnetlist:get-package-attribute refdes "model-name")) )

    (if (not (string-ci=? model-name "unknown"))
      (display (string-append " " model-name))
    )
  )

  ;; Next write out attributes if they exist.  Use
  ;; a list of attributes which can be attached to a capacitor.
  ;; I include non-standard "area" attrib here per request of Peter Kaiser.
  (let
    ( (attrib-list (list "AREA" "L" "W" "IC")) )

    (spice-msw:write-attrib-list refdes attrib-list)
  )

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write an inductor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. L1

(define (spice-msw:write-inductor refdes)

  (spice-msw:dbg-msg "  spice-msw:write-inductor      : Found a Inductor\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "L")

  ;; first write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; next write inductor value, if any.  Note that if the
  ;; component value is not assigned, then it will write "unknown"
  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (display value)
  )

  ;; create list of attributes which can be attached to a inductor
  (let
    ( (attrib-list (list "L" "W" "IC")) )

    (spice-msw:write-attrib-list refdes attrib-list)
  )

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write an independent voltage source.
;;;
;;; The behavior of the voltage source is held in the "value" attribute.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. V1

(define (spice-msw:write-ivs refdes)

  (spice-msw:dbg-msg "  spice-msw:write-ivs           : Found a Independent Voltage Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "V")

  ;; First write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; Next write voltage value, if any. Note that if the voltage value is not assigned, then it will
  ;; write "unknown".
  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (display value)
  )

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write an independent current source.
;;;
;;; The behavior of the current source is held in the "value" attribute
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. I1

(define (spice-msw:write-ics refdes)

  (spice-msw:dbg-msg "  write-ics                     : Found a Independent Current Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "I")

  ;; First write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; Next write current value, if any. Note that if the current value is not assigned, then it will
  ;; write "unknown".
  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (display value)
  )

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a current controlled voltage source and implement the necessary current measuring voltage
;;; source.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. H1

(define (spice-msw:write-ccvs refdes)

  (spice-msw:dbg-msg "  spice-msw:write-ccvs          : Found a Current Controlled Voltage Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "H")

  ;; Implement the controlled current source
  (display (string-append refdes " "))
  (spice-msw:write-pin-pair-nets refdes "1" "2")
  (display (string-append "Vsense_" refdes  " " (spice-msw:component-value refdes) "\n" ))

  ;; Implement the current measuring voltage source
  (display (string-append "Vsense_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "3" "4")
  (display "DC 0\n")

  ;; It's possible to leave the output voltage source unconnected, SPICE won't complain about
  ;; unconnected nodes
  (display (string-append "IOut_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "1" "2")
  (display "DC 0\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a current controlled current source and implement the necessary current measuring voltage
;;; source.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. F1

(define (spice-msw:write-cccs refdes)

  (spice-msw:dbg-msg "  spice-msw:write-cccs          : Found a Current Controlled Current Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "F")

  ;; Implement the controlled current source
  (display (string-append refdes " "))
  (spice-msw:write-pin-pair-nets refdes "1" "2")
  (display (string-append "Vsense_" refdes " " (gnetlist:get-package-attribute refdes "value") "\n"))

  ;; Implement the current measuring voltage source
  (display (string-append "Vsense_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "3" "4")
  (display "DC 0\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a voltage controlled current source and implement the necessary voltage measuring current
;;; source.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. G1

(define (spice-msw:write-vccs refdes)

  (spice-msw:dbg-msg "  spice-msw:write-vccs          : Found a Voltage Controlled Current Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "G")

  ;; Implement the controlled current source
  (display (string-append refdes " "))
  (spice-msw:write-compnt-nets refdes)
  (display (string-append (spice-msw:component-value refdes) "\n"))

  ;; Implement the voltage measuring current source
  ;; Imagine copying the voltage of a voltage source with an internal impedance, SPICE complains
  ;; about unconnected nets if this current source is not here.
  (display (string-append "IMeasure_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "3" "4")
  (display "DC 0\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a voltage controlled voltage source and implement the necessary voltage measuring current
;;; source.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. E1

(define (spice-msw:write-vcvs refdes)

  (spice-msw:dbg-msg "  spice-msw:write-vcvs          : Found a Voltage Controlled Voltage Source\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "E")

  ;; Implement the controlled voltage source
  (display (string-append refdes " "))
  (spice-msw:write-compnt-nets refdes)
  (display (string-append (gnetlist:get-package-attribute refdes "value") "\n" ))

  ;; Implement the voltage measuring current source
  ;; Imagine copying the voltage of a voltage source with an internal impedance, SPICE complains
  ;; about unconnected nets if this current source is not here.
  (display (string-append "Isense_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "3" "4")
  (display "DC 0\n")

  ;; With an output current source it is possible to leave the output voltage source unconnected,
  ;; SPICE won't complain about unconnected nodes
  (display (string-append "IOut_" refdes " "))
  (spice-msw:write-pin-pair-nets refdes "1" "2")
  (display "DC 0\n")
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Create a nullor, make sure it consists of a voltage controlled source.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. E1

(define (spice-msw:write-nullor refdes)

  (spice-msw:dbg-msg "  spice-msw:write-nullor        : Found a Nullor\n")

  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    ;; Write out the refdes prefix, if required
    (spice-msw:write-compnt-prefix refdes "E")

    ;; Implement the controlled voltage source
    (display (string-append refdes " "))
    (spice-msw:write-compnt-nets refdes)
    (display (string-append (if (string-ci=? value "unknown") "1000Meg" value) "\n"))

    ;; Implement the voltage measuring current source.
    ;; Imagine yourself copying the voltage of a voltage source with an internal impedance, SPICE
    ;; complains about unconnected nets if this current source is not here.
    (display (string-append "IMeasure_" refdes " "))
    (spice-msw:write-pin-pair-nets refdes "3" "4")
    (display "DC 0\n")

    ;; With an output current source it is possible to leave the output voltage source unconnected,
    ;; SPICE won't complain about unconnected nodes
    (display (string-append "IOut_" refdes " "))
    (spice-msw:write-pin-pair-nets refdes "1" "2")
    (display "DC 0\n")
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a Josephson junction.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. B1

(define (spice-msw:write-joseph-jn refdes)

  (spice-msw:dbg-msg "  spice-msw:write-joseph-jn     : Found a Josephson junction\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "B")

   ;; First write out refdes and attached nets
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; Next, add a dummy node for JJ phase. Unlike in Xic netlister, give it
  ;; a reasonable name, not a number, eg. refdes
  (display (string-append refdes " "))

  ;; Next write JJ model name, if any.
  (let
    ( (model-name (gnetlist:get-package-attribute refdes "model-name")) )

    (if (not (string-ci=? model-name "unknown")) (display (string-append model-name " " )))
  )

  ;; Next write out attributes if they exist. Use a list of attributes which can be attached to a
  ;; junction.
  (spice-msw:write-attrib-list refdes (list "AREA"))

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a coupled (mutual) inductor.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. K1

(define (spice-msw:write-coupled-ind refdes)

  (spice-msw:dbg-msg "  spice-msw:write-coupled-ind   : Found a coupled (mutual) inductor\n")

  ;; Write out the refdes prefix, if required
  (spice-msw:write-compnt-prefix refdes "K")

  ;; First write out refdes and attached nets (none)
  (display (string-append refdes " "))  ;; Write a component's refdes
  (spice-msw:write-compnt-nets refdes)

  ;; Next two inductor names and value
  (let
    (
      (inductors (gnetlist:get-package-attribute refdes "inductors"))
      (value     (gnetlist:get-package-attribute refdes "value"))
    )

    (if (not (string-ci=? inductors "unknown")) (display (string-append inductors " " )))
    (if (not (string-ci=? value "unknown"))     (display (string-append value " " )))
  )

  (newline)
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a voltage probe.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. R1

(define (spice-msw:write-probe refdes)

  (spice-msw:dbg-msg "  spice-msw:write-probe         : Found a probe\n")

  (let
    ( (value (gnetlist:get-package-attribute refdes "value")) )

    (if (string-ci=? value "unknown") (set! value "TRAN") )

    (display (string-append "* Probe device " refdes " on nets "))
    (spice-msw:write-compnt-nets refdes)
    (newline)
    (display (string-append ".print " value " +"))
    (spice-msw:write-compnt-nets refdes
    (string-join (map (lambda (x) "V(~a)") (gnetlist:get-pins refdes)) " " 'infix) ) ;; make format string
    (newline)
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Include SPICE statements from a directive block.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. R1

(define (spice-msw:write-directive refdes)

  (spice-msw:dbg-msg "  spice-msw:write-directive     : Found a SPICE directive\n")

  ;; Collect variables used in creating spice code
  (let
    (
      (value (gnetlist:get-package-attribute refdes "value"))
      (file  (gnetlist:get-package-attribute refdes "file"))
    )

    (cond
      ;; First look to see if there is a value.
      ((not (string-ci=? value "unknown"))
        (begin
          (display (string-append value "\n"))
          (spice-msw:dbg-msg (string-append "Appending value = \"" value "\" to output file.\n"))
        )
      )

      ;; Since there is no value, look for file.
      ((not (string-ci=? file "unknown"))
        (begin
          (spice-msw:insert-text-file file)  ;; Note that we don't wait until the end here.  Is that OK?
          (spice-msw:dbg-msg (string-append "Inserting contents of file = " file " into output file.\n"))
        )
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Include a file using an .INCLUDE directive.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. .INCLUDE

(define (spice-msw:write-include refdes)

  (spice-msw:dbg-msg "  spice-msw:write-include       : Found a .INCLUDE directive\n")

  (let
    ( (file (gnetlist:get-package-attribute refdes "file")) )

    (if (not (string-ci=? file "unknown"))
      (if embed-mode?
        (begin
          (spice-msw:insert-text-file file)
          (spice-msw:dbg-msg (string-append "embedding contents of file " file " into netlist.\n"))
        )
        (begin
          (display (string-append ".INCLUDE " file "\n"))
          (spice-msw:dbg-msg "placing .include directive string into netlist.\n")
        )
      )
      (spice-msw:dbg-msg "silently skip \"unknown\" file.\n")
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Include an option using an .OPTIONS directive.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. .OPTIONS

(define (spice-msw:write-options refdes)

  (spice-msw:dbg-msg "  spice-msw:write-options       : Found a .OPTIONS directive\n")

  (display (string-append ".OPTIONS " (gnetlist:get-package-attribute refdes "value")  "\n"))
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Include a spice model (instantiated as a model box on the schematic).
;;;
;;; Two types of model can be included:
;;;   1. An embedded model, which is a one- or multi-line string held in the attribute "model".
;;;      In this case, the following attributes are mandatory :
;;;       - model (ie. list of parameter=value strings)
;;;       - model-name
;;;       - type
;;;      In this case, the function creates and formats the correct spice model line(s).
;;;   2. A model held in a file whose name is held in the attribute "file"
;;;      In this case, the following attribute are mandatory :
;;;       - file (i.e. list of parameter=value strings)
;;;      In this case, the function just opens the file and dumps the contents into the netlist.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. .MODEL

(define (spice-msw:write-model refdes)

  (spice-msw:dbg-msg "  spice-msw:write-model         : Found a .MODEL directive")

  ;; Collect variables used in creating spice code
  (let
    (
      (model-name (gnetlist:get-package-attribute refdes "model-name"))
      (model-file (gnetlist:get-package-attribute refdes "file"))
      (model      (gnetlist:get-package-attribute refdes "model"))
      (type       (gnetlist:get-package-attribute refdes "type"))
    )

    ;; Now, depending upon what combination of model, model-file, and model-name
    ;; exist (as described above) write out lines into spice netlist.
    (cond

      ;; one model and model name exist
      ((not (or (string-ci=? model "unknown") (string-ci=? model-name "unknown")))
        (spice-msw:dbg-msg (string-append " and model and model-name for " refdes "\n"))
        (display (string-append ".MODEL " model-name " " type " (" model ")\n"))
      )

      ;; model file exists
      ((not (or (string-ci=? model-file "unknown") ))
        (spice-msw:dbg-msg (string-append " and model-file for " refdes "\n"))
      )

      (else
        (spice-msw:dbg-msg " but no model, model-name or model-file\n")
      )
    )
  )
)

;;;*************************************************************************************************
;;;*                                                                                               *
;;;*          Section 5 - Sundry utility procedures                                                *
;;;*                                                                                               *
;;;*************************************************************************************************

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a debug message to standard error.
;;;
;;; Wrapper around "message" procedure. If the -v / --verbose flag is set, otherwise does nothing.
;;;
;;; Calling form : (spice-msw:dbg-msg "verbose debug text")

(define spice-msw:dbg-msg
  (lambda (msg-str)

    (if (netlist-option-ref 'verbose)
      (message msg-str)
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write a prefix if first char of refdes is improper.
;;;
;;; Eg. if MOSFET is named T1 then it becomes MT1 in the SPICE file.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. R1
;;;   prefix - The desired prefix character

(define (spice-msw:write-compnt-prefix refdes prefix)

  (let
    ( (different-prefix (not (string-ci=? (substring refdes 0 1) prefix))) )

    (spice-msw:dbg-msg "  spice-msw:write-compnt-prefix : Check refdes ")
    (spice-msw:dbg-msg (string-append "prefix = " (substring refdes 0 1)))
    (spice-msw:dbg-msg (string-append " (correct prefix = " prefix ")\n"))

    (if (and different-prefix (not nomunge-mode?)) (display prefix))
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Given a refdes, and optionally a format string, write out the nets attached to the component's
;;; pins.
;;;
;;; If it's not called with a format string it looks for one in the net-format attribute, otherwise
;;; it writes out the pins unformatted. This is used to write out non-slotted parts.
;;;
;;; Argument List :
;;;   refdes - Component label
;;;   format - Format string (optional argument)

(define (spice-msw:write-compnt-nets refdes . format)

  ;;------------------------------------------------------------------------------------------------
  ;; Get the netname connected to a pin sequence number.
  ;;
  ;; Argument List :
  ;;   pinseq - A pin sequence number
  ;;
  ;; Return Values :
  ;;  The netlabel connected to the pin OR false if error occurs

  (define (get-netname pinseq)

    (set! pinseq (number->string pinseq))

    (let*
      (
        (pinnumber (gnetlist:get-attribute-by-pinseq refdes pinseq "pinnumber"))
        (netname   (car (spice-msw:get-compnt-pin-net refdes pinnumber)))
      )

      (if (string-prefix? "unnamed_net" netname)
        (set! netname (string-drop netname (string-length "unnamed_net")))
      )

      ;;=========  Super debug stuff  ========
      (if #f
        (let*
          (
            (pinlabel (gnetlist:get-attribute-by-pinseq refdes pinseq "pinlabel" ))
            (pintype  (gnetlist:get-attribute-by-pinseq refdes pinseq "pintype"  ))
          )

          (spice-msw:dbg-msg "  spice-msw:write-compnt-nets :\n")
          (spice-msw:dbg-msg (string-append "    refdes    = " refdes    "\n"))
          (spice-msw:dbg-msg (string-append "    pinseq    = " pinseq    "\n"))
          (spice-msw:dbg-msg (string-append "    pinnumber = " pinnumber "\n"))
          (spice-msw:dbg-msg (string-append "    pinlabel  = " pinlabel  "\n"))
          (spice-msw:dbg-msg (string-append "    pintype   = " pintype   "\n"))
          (spice-msw:dbg-msg (string-append "    netname   = " netname   "\n"))
        )
      )
      ;;======================================

      (if (string-ci=? pinnumber "ERROR_INVALID_PIN")
        (begin
          (spice-msw:dbg-msg (string-append refdes " : found pin with no pinseq attribute. Ignoring.\n"))
          #f
        )
        netname
      )
    )
  )

  ;;------------------------------------------------------------------------------------------------

  ;; First do local assignments
  (let
    ( (netnames (filter-map get-netname (spice-msw:range 1 (length (gnetlist:get-pins refdes))))) )

    (if (null? format) ;; Format argument takes priority, otherwise use attribute
      (set! format (gnetlist:get-package-attribute refdes "net-format"))
      (set! format (car format))
    )
    (if (string-ci=? format "unknown")
      (display (string-join netnames " " 'suffix))             ;; Write out nets.
      (apply simple-format (cons #t (cons format netnames)))   ;; Write out nets with format string
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write netnames connected to two pins.
;;;
;;; Argument List :
;;;   refdes - The refdes of a component eg. R1
;;;   pin-a  - The name of pin-a
;;;   pin-b  - The name of pin-b

(define (spice-msw:write-pin-pair-nets refdes pin-a pin-b)

  (display (string-append
    (car (spice-msw:get-net refdes (gnetlist:get-attribute-by-pinseq refdes pin-a "pinnumber"))) " "))
  (display (string-append
    (car (spice-msw:get-net refdes (gnetlist:get-attribute-by-pinseq refdes pin-b "pinnumber"))) " "))
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Write out a specified list of attributes in the form of <variable>=<value>.
;;;
;;; Argument List :
;;;   refdes      - The label of a component eg. R1
;;;   attrib-list - A list of attributes

(define (spice-msw:write-attrib-list refdes attrib-list)

  (if (not (null? attrib-list))
    (let
      ( (attrib (gnetlist:get-package-attribute refdes (car attrib-list))) )

      (if (not (string-ci=? attrib "unknown"))
        (display (string-append " " (car attrib-list) "=" attrib))
      )
      (spice-msw:write-attrib-list refdes (cdr attrib-list))
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Iterate through the schematic and compile a list of all SPICE-IO pins found. This is used when
;;; writing out a .SUBCKT lower level netlist.
;;;
;;; Argument List :
;;;   refdes-list - A list of package (component) labels
;;;
;;; Return Values :
;;;   spice-io-pin-list - A list of the SPICE-IO pins

(define (spice-msw:get-spice-io-pins refdes-list spice-io-pin-list)

  (if (null? refdes-list)
    spice-io-pin-list
    (let*
      (
        (refdes (car refdes-list))
        (device (gnetlist:get-package-attribute refdes "device"))
      )

      (if (string-ci=? device "SPICE-IO")  ;; Look for subcircuit label
         ;; Found a spice-IO pin
         (spice-msw:get-spice-io-pins (cdr ls) (cons refdes spice-io-pin-list))
         ;; No spice-IO pin found. Iterate . . . .
         (spice-msw:get-spice-io-pins (cdr ls) spice-io-pin-list)
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Take a list of refdes's and sorts it in order of refdes.
;;;
;;; Argument List :
;;;   refdes-list - A list of package (component) labels (refdes's)

(define (spice-msw:sort-spice-io-pins refdes-list)

  ;; Yes, this isn't good Scheme form. Tough! Writing this out in a functional programming form
  ;; would be totally confusing! Note that this function requires that each spice-IO pin have the
  ;; same, single character prefix (i.e. 'P')
  (let*
    (
      (char-prefixes           (map car (map string->list refdes-list)))  ;; Pull off first char (prefix)
      (prefixes                (map string char-prefixes))                ;; Make list of strings from prefixes
      (split-numbers-list      (map cdr (map string->list refdes-list)))  ;; Pull off refdes numbers as list elements
      (string-numbers-list     (map list->string split-numbers-list))     ;; Recombine split up (multidigit) number strings
      (numbers-list            (map string->number string-numbers-list))  ;; Convert strings to numbers for sorting
      (sorted-numbers-list     (sort numbers-list <))                     ;; Sort refdes numbers as numbers
      (sorted-str-numbers-list (map number->string sorted-numbers-list))  ;; Create sorted list of refdes strings
    )

    (map-in-order string-append  prefixes sorted-str-numbers-list)  ;; Laminate prefixes back onto refdes numbers & return
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Given a list of spice-IO packages (refdes's), return the list of nets attached to the IOs.
;;;
;;; Argument List :
;;;   refdes-list - A list of package (component) labels (refdes's)
;;;
;;; Return Values :
;;;   net-list - A list of nets

(define (spice-msw:get-io-nets refdes-list net-list)

  (if (null? refdes-list)
    net-list        ;; End iteration & return net-list if the refdes-list is empty
    (let*
      (
        (refdes (car refdes-list))            ;; otherwise process package . . .
        (net    (car (get-nets refdes "1")))  ;; get the net attached to pin 1
      )

      ;; Now iterate
      (spice-msw:get-io-nets (cdr refdes-list) (cons net net-list))
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; A wrapper for the procedure "get-nets" from "netlist.scm". The net labeled "GND" is converted
;;; to "0".
;;;
;;; Argument List :
;;;   refdes   - The refdes of a component eg. R1
;;;   pinlabel - The component pinlabel connected to the net

(define (spice-msw:get-compnt-pin-net refdes pinlabel)

  (let
    ( (netname (get-nets refdes pinlabel)) )  ;; Procedure "get-nets" defined in "netlist.scm"

    (cond
      ((string-ci=? (car netname) "GND") (cons "0" #t))
      (else                              (cons (car netname) #t))
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Sort procedure for ordering refdes's alphabetically but keep A? packages at the end of the list
;;; so SPICE simulation directives operate correctly.
;;;
;;; This function was written by Ken Healy to facilitate SPICE netlisting for GNU-CAP, which wants
;;; A? refdes cards (ie. SPICE directives) to appear last in the SPICE netlist.
;;;
;;; Argument List :
;;;   x - First  item to compare
;;;   y - Second item to compare

(define (spice-msw:refdes-sort x y)

  (define (string-tail string start) (substring string start (string-length string)))

  (let
    (
      (xdes (string-ref  x 0))
      (ydes (string-ref  y 0))
      (xnum (string-tail x 1))
      (ynum (string-tail y 1))
    )

    (if (char-ci=? xdes ydes)
      (if (string-ci<? xnum ynum) #t #f)
      (if (char-ci=? xdes #\A) #f
        (if (char-ci=? ydes #\A) #t
          (if (char-ci<? xdes ydes) #t #f)
        )
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Take a list and turns it into a string.
;;;
;;; The difference between this and list->string is that this function can handle lists made up of
;;; multi-char strings.
;;;
;;; Argument List :
;;;   str-list - A list of strings

(define (spice-msw:list-2-str str-list)

  (let while ( (str (string)) (local-list str-list) )
    (if (null? local-list)
      str                                                    ;; Return when list is empty
      (begin                                                 ;; Otherwise keep appending elements
        (set! str (string-append (car local-list) " " str))  ;; Append next element to str
        (while str (cdr local-list))                         ;; Iterate with remainder of local-list
      )
    )
  )
)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Return a list of all the integers from start to stop, with an optional step size.
;;;
;;; Argument List :
;;;   start - the first integer in the list
;;;   stop  - the last integer in the list
;;;   step  - the integer step size (optional argument)

(define (spice-msw:range start stop . step)

  (if (null? step)
    (iota (+ (- stop start) 1) start)
    (begin
      (set! step (car step))
      (iota (+ (ceiling (/ (- stop start) step)) 1) start step)
    )
  )
)

;;;*************************************************************************************************
