<?php
/**
 * Efficient concatenated text storage.
 *
 * @license GPL-2.0-or-later
 * @file
 */

use MediaWiki\MediaWikiServices;

/**
 * To speed up conversion from 1.4 to 1.5 schema, text rows can refer to the
 * leftover cur table as the backend. This avoids expensively copying hundreds
 * of megabytes of data during the conversion downtime.
 */
class HistoryBlobCurStub {
	/** @var int */
	public $mCurId;

	/**
	 * @param int $curid The cur_id pointed to
	 */
	public function __construct( $curid = 0 ) {
		$this->mCurId = $curid;
	}

	/**
	 * Sets the location (cur_id) of the main object to which this object
	 * points
	 *
	 * @param int $id
	 */
	public function setLocation( $id ) {
		$this->mCurId = $id;
	}

	/**
	 * @return string|false
	 */
	public function getText() {
		$dbr = MediaWikiServices::getInstance()->getConnectionProvider()->getReplicaDatabase();
		$row = $dbr->newSelectQueryBuilder()
			->select( [ 'cur_text' ] )
			->from( 'cur' )
			->where( [ 'cur_id' => $this->mCurId ] )
			->caller( __METHOD__ )->fetchRow();
		return $row ? $row->cur_text : false;
	}
}

// Blobs generated by MediaWiki < 1.5 on PHP 4 were serialized with the
// class name coerced to lowercase. We can improve efficiency by adding
// autoload entries for the lowercase variants of these classes (T166759).
// The code below is never executed, but it is picked up by the AutoloadGenerator
// parser, which scans for class_alias() calls.
/*
class_alias( HistoryBlobCurStub::class, 'historyblobcurstub' );
*/
