// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: BUSL-1.1

package minimal

import (
	"testing"

	logicalKv "github.com/hashicorp/vault-plugin-secrets-kv"
	"github.com/hashicorp/vault/audit"
	logicalDb "github.com/hashicorp/vault/builtin/logical/database"
	"github.com/hashicorp/vault/builtin/plugin"
	"github.com/hashicorp/vault/helper/builtinplugins"
	"github.com/hashicorp/vault/helper/testhelpers/corehelpers"
	"github.com/hashicorp/vault/http"
	"github.com/hashicorp/vault/sdk/logical"
	"github.com/hashicorp/vault/sdk/physical/inmem"
	"github.com/hashicorp/vault/vault"
	"github.com/mitchellh/copystructure"
)

// NewTestSoloCluster is a simpler version of NewTestCluster that only creates
// single-node clusters.  It is intentionally minimalist, if you need something
// from vault.TestClusterOptions, use NewTestCluster instead.  It should work fine
// with a nil config argument.  There is no need to call Start or Cleanup or
// TestWaitActive on the resulting cluster.
func NewTestSoloCluster(t testing.TB, config *vault.CoreConfig) *vault.TestCluster {
	logger := corehelpers.NewTestLogger(t)

	mycfg := &vault.CoreConfig{}

	if config != nil {
		// It's rude to modify an input argument as a side-effect
		copy, err := copystructure.Copy(config)
		if err != nil {
			t.Fatal(err)
		}
		mycfg = copy.(*vault.CoreConfig)
	}
	if mycfg.Physical == nil {
		// Don't use NewTransactionalInmem because that would enable replication,
		// which we don't care about in our case (use NewTestCluster for that.)
		inm, err := inmem.NewInmem(nil, logger)
		if err != nil {
			t.Fatal(err)
		}
		mycfg.Physical = inm
	}
	if mycfg.CredentialBackends == nil {
		mycfg.CredentialBackends = map[string]logical.Factory{
			"plugin": plugin.Factory,
		}
	}
	if mycfg.LogicalBackends == nil {
		mycfg.LogicalBackends = map[string]logical.Factory{
			"plugin":   plugin.Factory,
			"database": logicalDb.Factory,
			// This is also available in the plugin catalog, but is here due to the need to
			// automatically mount it.
			"kv": logicalKv.Factory,
		}
	}
	if mycfg.AuditBackends == nil {
		mycfg.AuditBackends = map[string]audit.Factory{
			"file":   audit.NewFileBackend,
			"socket": audit.NewSocketBackend,
			"syslog": audit.NewSyslogBackend,
		}
	}
	if mycfg.BuiltinRegistry == nil {
		mycfg.BuiltinRegistry = builtinplugins.Registry
	}

	cluster := vault.NewTestCluster(t, mycfg, &vault.TestClusterOptions{
		NumCores:    1,
		HandlerFunc: http.Handler,
		Logger:      logger,
	})
	t.Cleanup(cluster.Cleanup)
	return cluster
}
