// SPDX-License-Identifier: MIT

use netlink_packet_utils::{Emitable, Parseable};

use crate::link::{
    InfoData, InfoKind, InfoMacSec, LinkAttribute, LinkFlag, LinkHeader,
    LinkInfo, LinkLayerType, LinkMessage, LinkMessageBuffer, MacSecCipherId,
    MacSecOffload, MacSecValidate,
};
use crate::AddressFamily;

#[test]
fn test_macsec_link_info() {
    let raw: Vec<u8> = vec![
        0x00, 0x00, 0x01, 0x00, 0x09, 0x00, 0x00, 0x00, 0x43, 0x10, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x12, 0x00, 0x0b, 0x00, 0x01, 0x00,
        0x6d, 0x61, 0x63, 0x73, 0x65, 0x63, 0x00, 0x00, 0x6c, 0x00, 0x02, 0x00,
        0x0c, 0x00, 0x01, 0x00, 0xf2, 0xca, 0xda, 0x49, 0x34, 0x51, 0x00, 0x01,
        0x05, 0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x04, 0x00,
        0x01, 0x00, 0x00, 0x01, 0x00, 0x02, 0x80, 0x00, 0x05, 0x00, 0x06, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x09, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x0d, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
    ];

    let expected = LinkMessage {
        header: LinkHeader {
            interface_family: AddressFamily::Unspec,
            index: 9,
            link_layer_type: LinkLayerType::Ether,
            flags: vec![
                LinkFlag::Broadcast,
                LinkFlag::LowerUp,
                LinkFlag::Multicast,
                LinkFlag::Running,
                LinkFlag::Up,
            ],
            change_mask: vec![],
        },
        attributes: vec![LinkAttribute::LinkInfo(vec![
            LinkInfo::Kind(InfoKind::MacSec),
            LinkInfo::Data(InfoData::MacSec(vec![
                InfoMacSec::Sci(72146879057152754),
                InfoMacSec::IcvLen(16),
                #[allow(deprecated)]
                InfoMacSec::CipherSuite(MacSecCipherId::DefaultGcmAes128),
                InfoMacSec::EncodingSa(0),
                InfoMacSec::Encrypt(1),
                InfoMacSec::Protect(1),
                InfoMacSec::IncSci(1),
                InfoMacSec::Es(0),
                InfoMacSec::Scb(0),
                InfoMacSec::ReplayProtect(0),
                InfoMacSec::Validation(MacSecValidate::Strict),
                InfoMacSec::Offload(MacSecOffload::Off),
            ])),
        ])],
    };

    assert_eq!(
        expected,
        LinkMessage::parse(&LinkMessageBuffer::new(&raw)).unwrap()
    );

    let mut buf = vec![0; expected.buffer_len()];

    expected.emit(&mut buf);

    assert_eq!(buf, raw);
}
