/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.load.kotlin.header

import kotlin.reflect.jvm.internal.impl.load.java.JvmAnnotationNames
import kotlin.reflect.jvm.internal.impl.load.java.JvmBytecodeBinaryVersion
import kotlin.reflect.jvm.internal.impl.load.kotlin.JvmMetadataVersion
import kotlin.reflect.jvm.internal.impl.load.kotlin.header.KotlinClassHeader.MultifileClassKind.DELEGATING
import kotlin.reflect.jvm.internal.impl.load.kotlin.header.KotlinClassHeader.MultifileClassKind.INHERITING

class KotlinClassHeader(
        val kind: KotlinClassHeader.Kind,
        val metadataVersion: JvmMetadataVersion,
        val bytecodeVersion: JvmBytecodeBinaryVersion,
        val data: Array<String>?,
        val incompatibleData: Array<String>?,
        val strings: Array<String>?,
        val extraString: String?,
        val extraInt: Int
) {
    // See kotlin.Metadata
    enum class Kind(val id: Int) {
        UNKNOWN(0),
        CLASS(1),
        FILE_FACADE(2),
        SYNTHETIC_CLASS(3),
        MULTIFILE_CLASS(4),
        MULTIFILE_CLASS_PART(5);

        companion object {
            private val entryById = values().associateBy(Kind::id)

            @JvmStatic
            fun getById(id: Int) = entryById[id] ?: UNKNOWN
        }
    }

    enum class MultifileClassKind {
        DELEGATING,
        INHERITING;
    }

    val multifileClassName: String?
        get() = if (kind == Kind.MULTIFILE_CLASS_PART) extraString else null

    // TODO: use in incremental compilation
    val multifileClassKind: MultifileClassKind?
        get() = if (kind == Kind.MULTIFILE_CLASS || kind == Kind.MULTIFILE_CLASS_PART) {
            if ((extraInt and JvmAnnotationNames.METADATA_MULTIFILE_PARTS_INHERIT_FLAG) != 0)
                INHERITING
            else
                DELEGATING
        }
        else
            null

    val isPreRelease: Boolean
        get() = (extraInt and JvmAnnotationNames.METADATA_PRE_RELEASE_FLAG) != 0

    val isScript: Boolean
        get() = (extraInt and JvmAnnotationNames.METADATA_SCRIPT_FLAG) != 0

    override fun toString() = "$kind version=$metadataVersion"
}
