// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package bootstrap

import (
	"context"
	"database/sql"
	"io/ioutil"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/dchest/uniuri"
	"github.com/golang/mock/gomock"
	"github.com/sirupsen/logrus"
)

var noContext = context.TODO()

func init() {
	logrus.SetOutput(ioutil.Discard)
}

func TestBootstrap(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: true,
		Admin:   true,
		Hash:    uniuri.NewLen(32),
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(nil, sql.ErrNoRows)
	store.EXPECT().Create(gomock.Any(), dummyUser).Return(nil)

	err := New(store).Bootstrap(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestBootstrap_GenerateHash(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: false,
		Admin:   true,
		Hash:    "",
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(nil, sql.ErrNoRows)
	store.EXPECT().Create(gomock.Any(), dummyUser).Return(nil)

	err := New(store).Bootstrap(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
	if got, want := len(dummyUser.Hash), 32; got != want {
		t.Errorf("Want generated hash length %d, got %d", want, got)
	}
}

func TestBootstrap_Empty(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "",
	}

	store := mock.NewMockUserStore(controller)
	err := New(store).Bootstrap(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestBootstrap_Exists_WithoutUpdates(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: true,
		Admin:   true,
		Hash:    uniuri.NewLen(32),
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(dummyUser, nil)
	err := New(store).Bootstrap(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestBootstrap_Exists_WithUpdates(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: true,
		Admin:   true,
		Hash:    uniuri.NewLen(32),
	}
	existingUser := &core.User{
		Login:   "octocat",
		Machine: false,
		Admin:   false,
		Hash:    uniuri.NewLen(32),
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(existingUser, nil)
	store.EXPECT().Update(gomock.Any(), existingUser).Return(nil)
	err := New(store).Bootstrap(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestBootstrap_MissingTokenError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: true,
		Admin:   true,
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(nil, sql.ErrNoRows)

	err := New(store).Bootstrap(noContext, dummyUser)
	if err != errMissingToken {
		t.Errorf("Expect missing token error")
	}
}

func TestBootstrap_CreateError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login:   "octocat",
		Machine: true,
		Admin:   true,
		Hash:    uniuri.NewLen(32),
	}

	store := mock.NewMockUserStore(controller)
	store.EXPECT().FindLogin(gomock.Any(), dummyUser.Login).Return(nil, sql.ErrNoRows)
	store.EXPECT().Create(gomock.Any(), dummyUser).Return(sql.ErrConnDone)

	err := New(store).Bootstrap(noContext, dummyUser)
	if err != sql.ErrConnDone {
		t.Errorf("Expect error creating user")
	}
}
