/*****************************************************************************

Copyright (c) 1994, 2015, Oracle and/or its affiliates. All Rights Reserved.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; version 2 of the License.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1335 USA

*****************************************************************************/

/**************************************************************//**
@file include/ut0ut.ic
Various utilities

Created 5/30/1994 Heikki Tuuri
*******************************************************************/

#include <algorithm>

/** Calculate the minimum of two pairs.
@param[out]	min_hi	MSB of the minimum pair
@param[out]	min_lo	LSB of the minimum pair
@param[in]	a_hi	MSB of the first pair
@param[in]	a_lo	LSB of the first pair
@param[in]	b_hi	MSB of the second pair
@param[in]	b_lo	LSB of the second pair */
UNIV_INLINE
void
ut_pair_min(
	ulint*	min_hi,
	ulint*	min_lo,
	ulint	a_hi,
	ulint	a_lo,
	ulint	b_hi,
	ulint	b_lo)
{
	if (a_hi == b_hi) {
		*min_hi = a_hi;
		*min_lo = std::min(a_lo, b_lo);
	} else if (a_hi < b_hi) {
		*min_hi = a_hi;
		*min_lo = a_lo;
	} else {
		*min_hi = b_hi;
		*min_lo = b_lo;
	}
}

/******************************************************//**
Compares two ulints.
@return 1 if a > b, 0 if a == b, -1 if a < b */
UNIV_INLINE
int
ut_ulint_cmp(
/*=========*/
	ulint	a,	/*!< in: ulint */
	ulint	b)	/*!< in: ulint */
{
	if (a < b) {
		return(-1);
	} else if (a == b) {
		return(0);
	} else {
		return(1);
	}
}

/** Compare two pairs of integers.
@param[in]	a_h	more significant part of first pair
@param[in]	a_l	less significant part of first pair
@param[in]	b_h	more significant part of second pair
@param[in]	b_l	less significant part of second pair
@return comparison result of (a_h,a_l) and (b_h,b_l)
@retval -1 if (a_h,a_l) is less than (b_h,b_l)
@retval 0 if (a_h,a_l) is equal to (b_h,b_l)
@retval 1 if (a_h,a_l) is greater than (b_h,b_l) */
UNIV_INLINE
int
ut_pair_cmp(
	ulint	a_h,
	ulint	a_l,
	ulint	b_h,
	ulint	b_l)
{
	if (a_h < b_h) {
		return(-1);
	}
	if (a_h > b_h) {
		return(1);
	}
	return(ut_ulint_cmp(a_l, b_l));
}

/*************************************************************//**
Calculates fast the 2-logarithm of a number, rounded upward to an
integer.
@return logarithm in the base 2, rounded upward */
UNIV_INLINE
ulint
ut_2_log(
/*=====*/
	ulint	n)	/*!< in: number != 0 */
{
	ulint	res;

	res = 0;

	ut_ad(n > 0);

	n = n - 1;

	for (;;) {
		n = n / 2;

		if (n == 0) {
			break;
		}

		res++;
	}

	return(res + 1);
}

/*************************************************************//**
Calculates 2 to power n.
@return 2 to power n */
UNIV_INLINE
ulint
ut_2_exp(
/*=====*/
	ulint	n)	/*!< in: number */
{
	return((ulint) 1 << n);
}
