/*
 * Copyright 2007 Juan Lang
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdarg.h>
#include <stdio.h>

#include "ntstatus.h"
#define WIN32_NO_STATUS
#include "windef.h"
#include "winbase.h"
#include "winreg.h"
#include "wincrypt.h"
#include "winternl.h"
#include "wine/debug.h"
#include "crypt32_private.h"

WINE_DEFAULT_DEBUG_CHANNEL(crypt);

static const char *trust_status_to_str(DWORD status)
{
    static const struct
    {
        DWORD flag;
        char text[32];
    }
    messages[] =
    {
        { CERT_TRUST_IS_NOT_TIME_VALID, "expired" },
        { CERT_TRUST_IS_NOT_TIME_NESTED, "bad time nesting" },
        { CERT_TRUST_IS_REVOKED, "revoked" },
        { CERT_TRUST_IS_NOT_SIGNATURE_VALID, "bad signature" },
        { CERT_TRUST_IS_NOT_VALID_FOR_USAGE, "bad usage" },
        { CERT_TRUST_IS_UNTRUSTED_ROOT, "untrusted root" },
        { CERT_TRUST_REVOCATION_STATUS_UNKNOWN, "unknown revocation status" },
        { CERT_TRUST_IS_CYCLIC, "cyclic chain" },
        { CERT_TRUST_INVALID_EXTENSION, "unsupported critical extension" },
        { CERT_TRUST_INVALID_POLICY_CONSTRAINTS, "bad policy" },
        { CERT_TRUST_INVALID_BASIC_CONSTRAINTS, "bad basic constraints" },
        { CERT_TRUST_INVALID_NAME_CONSTRAINTS, "bad name constraints" },
        { CERT_TRUST_HAS_NOT_SUPPORTED_NAME_CONSTRAINT, "unsupported name constraint" },
        { CERT_TRUST_HAS_NOT_DEFINED_NAME_CONSTRAINT, "undefined name constraint" },
        { CERT_TRUST_HAS_NOT_PERMITTED_NAME_CONSTRAINT, "disallowed name constraint" },
        { CERT_TRUST_HAS_EXCLUDED_NAME_CONSTRAINT, "excluded name constraint" },
        { CERT_TRUST_IS_OFFLINE_REVOCATION, "revocation server offline" },
        { CERT_TRUST_NO_ISSUANCE_CHAIN_POLICY, "no issuance policy" },
    };
    static char buf[1024];
    int i, pos = 0;

    for (i = 0; i < ARRAY_SIZE(messages); i++)
    {
        if (status & messages[i].flag)
            pos += sprintf(buf + pos, "\n\t%s", messages[i].text);
    }

    return buf;
}

static const char *get_cert_common_name(PCCERT_CONTEXT cert)
{
    static char buf[1024];
    const char *name = NULL;
    CERT_NAME_INFO *nameInfo;
    DWORD size;
    BOOL ret = CryptDecodeObjectEx(X509_ASN_ENCODING, X509_NAME,
     cert->pCertInfo->Subject.pbData, cert->pCertInfo->Subject.cbData,
     CRYPT_DECODE_NOCOPY_FLAG | CRYPT_DECODE_ALLOC_FLAG, NULL, &nameInfo,
     &size);

    if (ret)
    {
        PCERT_RDN_ATTR commonName = CertFindRDNAttr(szOID_COMMON_NAME,
         nameInfo);

        if (commonName)
        {
            CertRDNValueToStrA(commonName->dwValueType,
             &commonName->Value, buf, sizeof(buf));
            name = buf;
        }
        LocalFree(nameInfo);
    }
    return name;
}

static void check_and_store_certs(HCERTSTORE cached, HCERTSTORE new, HCERTSTORE to)
{
    DWORD root_count = 0;
    CERT_CHAIN_ENGINE_CONFIG chainEngineConfig =
     { sizeof(chainEngineConfig), 0 };
    HCERTCHAINENGINE engine;

    TRACE("\n");

    CertDuplicateStore(to);
    engine = CRYPT_CreateChainEngine(to, CERT_SYSTEM_STORE_CURRENT_USER, &chainEngineConfig);
    if (engine)
    {
        PCCERT_CONTEXT cert = NULL;

        do {
            cert = CertEnumCertificatesInStore(new, cert);
            if (cert)
            {
                CERT_CHAIN_PARA chainPara = { sizeof(chainPara), { 0 } };
                PCCERT_CHAIN_CONTEXT chain;
                BOOL ret;

                ret = CertGetCertificateChain(engine, cert, NULL, cached,
                 &chainPara, CERT_CHAIN_CACHE_ONLY_URL_RETRIEVAL, NULL, &chain);
                if (!ret)
                    TRACE("rejecting %s: %s\n", get_cert_common_name(cert),
                     "chain creation failed");
                else
                {
                    DWORD allowedErrors = CERT_TRUST_IS_UNTRUSTED_ROOT |
                     CERT_TRUST_IS_NOT_VALID_FOR_USAGE |
                     CERT_TRUST_INVALID_BASIC_CONSTRAINTS |
                     CERT_TRUST_IS_NOT_TIME_VALID;

                    /* The certificate chain verification only allows certain
                     * invalid CA certs if they're installed locally:  CA
                     * certs missing the key usage extension, and CA certs
                     * missing the basic constraints extension.  Of course
                     * there's a chicken and egg problem:  we have to accept
                     * them here in order for them to be accepted later.
                     * Expired, locally installed certs are also allowed here,
                     * because we don't know (yet) what date will be checked
                     * for an item signed by one of these certs.
                     * Thus, accept certs with any of the allowed errors.
                     */
                    if (chain->TrustStatus.dwErrorStatus & ~allowedErrors)
                        TRACE("rejecting %s: %s\n", get_cert_common_name(cert),
                         trust_status_to_str(chain->TrustStatus.dwErrorStatus &
                         ~CERT_TRUST_IS_UNTRUSTED_ROOT));
                    else
                    {
                        DWORD i, j;

                        for (i = 0; i < chain->cChain; i++)
                            for (j = 0; j < chain->rgpChain[i]->cElement; j++)
                                if (CertAddCertificateContextToStore(to,
                                 chain->rgpChain[i]->rgpElement[j]->pCertContext,
                                 CERT_STORE_ADD_NEW, NULL))
                                    root_count++;
                    }
                    CertFreeCertificateChain(chain);
                }
            }
        } while (cert);
        CertFreeCertificateChainEngine(engine);
    }
    TRACE("Added %ld root certificates\n", root_count);
}

static const BYTE authenticode[] = {
0x30,0x82,0x03,0xd6,0x30,0x82,0x02,0xbe,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x04,0x05,0x00,0x30,
0x50,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0d,
0x30,0x0b,0x06,0x03,0x55,0x04,0x0a,0x13,0x04,0x4d,0x53,0x46,0x54,0x31,0x32,0x30,
0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,
0x74,0x20,0x41,0x75,0x74,0x68,0x65,0x6e,0x74,0x69,0x63,0x6f,0x64,0x65,0x28,0x74,
0x6d,0x29,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,
0x79,0x30,0x1e,0x17,0x0d,0x39,0x35,0x30,0x31,0x30,0x31,0x30,0x38,0x30,0x30,0x30,
0x31,0x5a,0x17,0x0d,0x39,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,
0x5a,0x30,0x50,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,
0x31,0x0d,0x30,0x0b,0x06,0x03,0x55,0x04,0x0a,0x13,0x04,0x4d,0x53,0x46,0x54,0x31,
0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x20,0x41,0x75,0x74,0x68,0x65,0x6e,0x74,0x69,0x63,0x6f,0x64,0x65,
0x28,0x74,0x6d,0x29,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,
0x69,0x74,0x79,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,
0x82,0x01,0x01,0x00,0xdf,0x08,0xba,0xe3,0x3f,0x6e,0x64,0x9b,0xf5,0x89,0xaf,0x28,
0x96,0x4a,0x07,0x8f,0x1b,0x2e,0x8b,0x3e,0x1d,0xfc,0xb8,0x80,0x69,0xa3,0xa1,0xce,
0xdb,0xdf,0xb0,0x8e,0x6c,0x89,0x76,0x29,0x4f,0xca,0x60,0x35,0x39,0xad,0x72,0x32,
0xe0,0x0b,0xae,0x29,0x3d,0x4c,0x16,0xd9,0x4b,0x3c,0x9d,0xda,0xc5,0xd3,0xd1,0x09,
0xc9,0x2c,0x6f,0xa6,0xc2,0x60,0x53,0x45,0xdd,0x4b,0xd1,0x55,0xcd,0x03,0x1c,0xd2,
0x59,0x56,0x24,0xf3,0xe5,0x78,0xd8,0x07,0xcc,0xd8,0xb3,0x1f,0x90,0x3f,0xc0,0x1a,
0x71,0x50,0x1d,0x2d,0xa7,0x12,0x08,0x6d,0x7c,0xb0,0x86,0x6c,0xc7,0xba,0x85,0x32,
0x07,0xe1,0x61,0x6f,0xaf,0x03,0xc5,0x6d,0xe5,0xd6,0xa1,0x8f,0x36,0xf6,0xc1,0x0b,
0xd1,0x3e,0x69,0x97,0x48,0x72,0xc9,0x7f,0xa4,0xc8,0xc2,0x4a,0x4c,0x7e,0xa1,0xd1,
0x94,0xa6,0xd7,0xdc,0xeb,0x05,0x46,0x2e,0xb8,0x18,0xb4,0x57,0x1d,0x86,0x49,0xdb,
0x69,0x4a,0x2c,0x21,0xf5,0x5e,0x0f,0x54,0x2d,0x5a,0x43,0xa9,0x7a,0x7e,0x6a,0x8e,
0x50,0x4d,0x25,0x57,0xa1,0xbf,0x1b,0x15,0x05,0x43,0x7b,0x2c,0x05,0x8d,0xbd,0x3d,
0x03,0x8c,0x93,0x22,0x7d,0x63,0xea,0x0a,0x57,0x05,0x06,0x0a,0xdb,0x61,0x98,0x65,
0x2d,0x47,0x49,0xa8,0xe7,0xe6,0x56,0x75,0x5c,0xb8,0x64,0x08,0x63,0xa9,0x30,0x40,
0x66,0xb2,0xf9,0xb6,0xe3,0x34,0xe8,0x67,0x30,0xe1,0x43,0x0b,0x87,0xff,0xc9,0xbe,
0x72,0x10,0x5e,0x23,0xf0,0x9b,0xa7,0x48,0x65,0xbf,0x09,0x88,0x7b,0xcd,0x72,0xbc,
0x2e,0x79,0x9b,0x7b,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xba,0x30,0x81,0xb7,0x30,
0x0d,0x06,0x03,0x55,0x1d,0x0a,0x04,0x06,0x30,0x04,0x03,0x02,0x07,0x80,0x30,0x32,
0x06,0x03,0x55,0x04,0x03,0x04,0x2b,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x41,0x75,0x74,0x68,0x65,0x6e,0x74,0x69,0x63,0x6f,0x64,0x65,0x28,
0x74,0x6d,0x29,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,
0x74,0x79,0x30,0x72,0x06,0x03,0x55,0x1d,0x01,0x04,0x6b,0x30,0x69,0x80,0x10,0x1a,
0x1b,0xe7,0x5b,0x9f,0xfd,0x8c,0x2a,0xc3,0x39,0xae,0x0c,0x62,0x2e,0x53,0x32,0xa1,
0x52,0x30,0x50,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,
0x31,0x0d,0x30,0x0b,0x06,0x03,0x55,0x04,0x0a,0x13,0x04,0x4d,0x53,0x46,0x54,0x31,
0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,0x73,
0x6f,0x66,0x74,0x20,0x41,0x75,0x74,0x68,0x65,0x6e,0x74,0x69,0x63,0x6f,0x64,0x65,
0x28,0x74,0x6d,0x29,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,
0x69,0x74,0x79,0x82,0x01,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x04,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x2d,0xc9,0xe2,0xf6,0x12,0x9e,
0x5d,0x56,0x67,0xfa,0xfa,0x4b,0x9a,0x7e,0xdc,0x29,0x56,0x5c,0x80,0x14,0x02,0x28,
0x85,0x6e,0x26,0xf3,0xcd,0x58,0xda,0x50,0x80,0xc5,0xf8,0x19,0xb3,0xa6,0x7c,0xe2,
0x9d,0x6b,0x5f,0x3b,0x8f,0x22,0x74,0xe6,0x18,0x04,0xfc,0x47,0x40,0xd8,0x7a,0x3f,
0x30,0x66,0xf0,0x12,0xa4,0xd1,0xeb,0x1d,0xe7,0xb6,0xf4,0x98,0xab,0x53,0x22,0x86,
0x51,0x58,0xee,0x23,0x09,0x76,0xe4,0x1d,0x45,0x5c,0x4b,0xff,0x4c,0xe3,0x02,0x50,
0x01,0x13,0xcc,0x41,0xa4,0x52,0x97,0xd4,0x86,0xd5,0xc4,0xfe,0x83,0x83,0x65,0x7d,
0xea,0xbe,0xa2,0x68,0x3b,0xc1,0xb1,0x29,0x98,0xbf,0xa2,0xa5,0xfc,0x9d,0xd3,0x84,
0xee,0x70,0x17,0x50,0xf3,0x0b,0xfa,0x3c,0xef,0xa9,0x27,0x8b,0x91,0xb4,0x48,0xc8,
0x45,0xa0,0xe1,0x01,0x42,0x4b,0x44,0x76,0x04,0x1c,0xc2,0x19,0xa2,0x8e,0x6b,0x20,
0x98,0xc4,0xdd,0x02,0xac,0xb4,0xd2,0xa2,0x0e,0x8d,0x5d,0xb9,0x36,0x8e,0x4a,0x1b,
0x5d,0x6c,0x1a,0xe2,0xcb,0x00,0x7f,0x10,0xf4,0xb2,0x95,0xef,0xe3,0xe8,0xff,0xa1,
0x73,0x58,0xa9,0x75,0x2c,0xa2,0x49,0x95,0x85,0xfe,0xcc,0xda,0x44,0x8a,0xc2,0x12,
0x44,0xd2,0x44,0xc8,0xa5,0xa2,0x1f,0xa9,0x5a,0x8e,0x56,0xc2,0xc3,0x7b,0xcf,0x42,
0x60,0xdc,0x82,0x1f,0xfb,0xce,0x74,0x06,0x7e,0xd6,0xf1,0xac,0x19,0x6a,0x4f,0x74,
0x5c,0xc5,0x15,0x66,0x31,0x6c,0xc1,0x62,0x71,0x91,0x0f,0x59,0x5b,0x7d,0x2a,0x82,
0x1a,0xdf,0xb1,0xb4,0xd8,0x1d,0x37,0xde,0x0d,0x0f };
static const BYTE rootauthority[] = {
0x30,0x82,0x04,0x12,0x30,0x82,0x02,0xfa,0xa0,0x03,0x02,0x01,0x02,0x02,0x0f,0x00,
0xc1,0x00,0x8b,0x3c,0x3c,0x88,0x11,0xd1,0x3e,0xf6,0x63,0xec,0xdf,0x40,0x30,0x0d,
0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x04,0x05,0x00,0x30,0x70,0x31,
0x2b,0x30,0x29,0x06,0x03,0x55,0x04,0x0b,0x13,0x22,0x43,0x6f,0x70,0x79,0x72,0x69,
0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x37,0x20,0x4d,0x69,0x63,
0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x2e,0x31,0x1e,0x30,0x1c,
0x06,0x03,0x55,0x04,0x0b,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x21,0x30,0x1f,
0x06,0x03,0x55,0x04,0x03,0x13,0x18,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,
0x1e,0x17,0x0d,0x39,0x37,0x30,0x31,0x31,0x30,0x30,0x37,0x30,0x30,0x30,0x30,0x5a,
0x17,0x0d,0x32,0x30,0x31,0x32,0x33,0x31,0x30,0x37,0x30,0x30,0x30,0x30,0x5a,0x30,
0x70,0x31,0x2b,0x30,0x29,0x06,0x03,0x55,0x04,0x0b,0x13,0x22,0x43,0x6f,0x70,0x79,
0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x37,0x20,0x4d,
0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x2e,0x31,0x1e,
0x30,0x1c,0x06,0x03,0x55,0x04,0x0b,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x21,
0x30,0x1f,0x06,0x03,0x55,0x04,0x03,0x13,0x18,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,
0x79,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,
0x01,0x00,0xa9,0x02,0xbd,0xc1,0x70,0xe6,0x3b,0xf2,0x4e,0x1b,0x28,0x9f,0x97,0x78,
0x5e,0x30,0xea,0xa2,0xa9,0x8d,0x25,0x5f,0xf8,0xfe,0x95,0x4c,0xa3,0xb7,0xfe,0x9d,
0xa2,0x20,0x3e,0x7c,0x51,0xa2,0x9b,0xa2,0x8f,0x60,0x32,0x6b,0xd1,0x42,0x64,0x79,
0xee,0xac,0x76,0xc9,0x54,0xda,0xf2,0xeb,0x9c,0x86,0x1c,0x8f,0x9f,0x84,0x66,0xb3,
0xc5,0x6b,0x7a,0x62,0x23,0xd6,0x1d,0x3c,0xde,0x0f,0x01,0x92,0xe8,0x96,0xc4,0xbf,
0x2d,0x66,0x9a,0x9a,0x68,0x26,0x99,0xd0,0x3a,0x2c,0xbf,0x0c,0xb5,0x58,0x26,0xc1,
0x46,0xe7,0x0a,0x3e,0x38,0x96,0x2c,0xa9,0x28,0x39,0xa8,0xec,0x49,0x83,0x42,0xe3,
0x84,0x0f,0xbb,0x9a,0x6c,0x55,0x61,0xac,0x82,0x7c,0xa1,0x60,0x2d,0x77,0x4c,0xe9,
0x99,0xb4,0x64,0x3b,0x9a,0x50,0x1c,0x31,0x08,0x24,0x14,0x9f,0xa9,0xe7,0x91,0x2b,
0x18,0xe6,0x3d,0x98,0x63,0x14,0x60,0x58,0x05,0x65,0x9f,0x1d,0x37,0x52,0x87,0xf7,
0xa7,0xef,0x94,0x02,0xc6,0x1b,0xd3,0xbf,0x55,0x45,0xb3,0x89,0x80,0xbf,0x3a,0xec,
0x54,0x94,0x4e,0xae,0xfd,0xa7,0x7a,0x6d,0x74,0x4e,0xaf,0x18,0xcc,0x96,0x09,0x28,
0x21,0x00,0x57,0x90,0x60,0x69,0x37,0xbb,0x4b,0x12,0x07,0x3c,0x56,0xff,0x5b,0xfb,
0xa4,0x66,0x0a,0x08,0xa6,0xd2,0x81,0x56,0x57,0xef,0xb6,0x3b,0x5e,0x16,0x81,0x77,
0x04,0xda,0xf6,0xbe,0xae,0x80,0x95,0xfe,0xb0,0xcd,0x7f,0xd6,0xa7,0x1a,0x72,0x5c,
0x3c,0xca,0xbc,0xf0,0x08,0xa3,0x22,0x30,0xb3,0x06,0x85,0xc9,0xb3,0x20,0x77,0x13,
0x85,0xdf,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xa8,0x30,0x81,0xa5,0x30,0x81,0xa2,
0x06,0x03,0x55,0x1d,0x01,0x04,0x81,0x9a,0x30,0x81,0x97,0x80,0x10,0x5b,0xd0,0x70,
0xef,0x69,0x72,0x9e,0x23,0x51,0x7e,0x14,0xb2,0x4d,0x8e,0xff,0xcb,0xa1,0x72,0x30,
0x70,0x31,0x2b,0x30,0x29,0x06,0x03,0x55,0x04,0x0b,0x13,0x22,0x43,0x6f,0x70,0x79,
0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x37,0x20,0x4d,
0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x2e,0x31,0x1e,
0x30,0x1c,0x06,0x03,0x55,0x04,0x0b,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,0x31,0x21,
0x30,0x1f,0x06,0x03,0x55,0x04,0x03,0x13,0x18,0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,
0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,
0x79,0x82,0x0f,0x00,0xc1,0x00,0x8b,0x3c,0x3c,0x88,0x11,0xd1,0x3e,0xf6,0x63,0xec,
0xdf,0x40,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x04,0x05,
0x00,0x03,0x82,0x01,0x01,0x00,0x95,0xe8,0x0b,0xc0,0x8d,0xf3,0x97,0x18,0x35,0xed,
0xb8,0x01,0x24,0xd8,0x77,0x11,0xf3,0x5c,0x60,0x32,0x9f,0x9e,0x0b,0xcb,0x3e,0x05,
0x91,0x88,0x8f,0xc9,0x3a,0xe6,0x21,0xf2,0xf0,0x57,0x93,0x2c,0xb5,0xa0,0x47,0xc8,
0x62,0xef,0xfc,0xd7,0xcc,0x3b,0x3b,0x5a,0xa9,0x36,0x54,0x69,0xfe,0x24,0x6d,0x3f,
0xc9,0xcc,0xaa,0xde,0x05,0x7c,0xdd,0x31,0x8d,0x3d,0x9f,0x10,0x70,0x6a,0xbb,0xfe,
0x12,0x4f,0x18,0x69,0xc0,0xfc,0xd0,0x43,0xe3,0x11,0x5a,0x20,0x4f,0xea,0x62,0x7b,
0xaf,0xaa,0x19,0xc8,0x2b,0x37,0x25,0x2d,0xbe,0x65,0xa1,0x12,0x8a,0x25,0x0f,0x63,
0xa3,0xf7,0x54,0x1c,0xf9,0x21,0xc9,0xd6,0x15,0xf3,0x52,0xac,0x6e,0x43,0x32,0x07,
0xfd,0x82,0x17,0xf8,0xe5,0x67,0x6c,0x0d,0x51,0xf6,0xbd,0xf1,0x52,0xc7,0xbd,0xe7,
0xc4,0x30,0xfc,0x20,0x31,0x09,0x88,0x1d,0x95,0x29,0x1a,0x4d,0xd5,0x1d,0x02,0xa5,
0xf1,0x80,0xe0,0x03,0xb4,0x5b,0xf4,0xb1,0xdd,0xc8,0x57,0xee,0x65,0x49,0xc7,0x52,
0x54,0xb6,0xb4,0x03,0x28,0x12,0xff,0x90,0xd6,0xf0,0x08,0x8f,0x7e,0xb8,0x97,0xc5,
0xab,0x37,0x2c,0xe4,0x7a,0xe4,0xa8,0x77,0xe3,0x76,0xa0,0x00,0xd0,0x6a,0x3f,0xc1,
0xd2,0x36,0x8a,0xe0,0x41,0x12,0xa8,0x35,0x6a,0x1b,0x6a,0xdb,0x35,0xe1,0xd4,0x1c,
0x04,0xe4,0xa8,0x45,0x04,0xc8,0x5a,0x33,0x38,0x6e,0x4d,0x1c,0x0d,0x62,0xb7,0x0a,
0xa2,0x8c,0xd3,0xd5,0x54,0x3f,0x46,0xcd,0x1c,0x55,0xa6,0x70,0xdb,0x12,0x3a,0x87,
0x93,0x75,0x9f,0xa7,0xd2,0xa0 };
static const BYTE rootcertauthority[] = {
0x30,0x82,0x05,0x99,0x30,0x82,0x03,0x81,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x79,
0xad,0x16,0xa1,0x4a,0xa0,0xa5,0xad,0x4c,0x73,0x58,0xf4,0x07,0x13,0x2e,0x65,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x5f,
0x31,0x13,0x30,0x11,0x06,0x0a,0x09,0x92,0x26,0x89,0x93,0xf2,0x2c,0x64,0x01,0x19,
0x16,0x03,0x63,0x6f,0x6d,0x31,0x19,0x30,0x17,0x06,0x0a,0x09,0x92,0x26,0x89,0x93,
0xf2,0x2c,0x64,0x01,0x19,0x16,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,
0x31,0x2d,0x30,0x2b,0x06,0x03,0x55,0x04,0x03,0x13,0x24,0x4d,0x69,0x63,0x72,0x6f,
0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,
0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x30,
0x1e,0x17,0x0d,0x30,0x31,0x30,0x35,0x30,0x39,0x32,0x33,0x31,0x39,0x32,0x32,0x5a,
0x17,0x0d,0x32,0x31,0x30,0x35,0x30,0x39,0x32,0x33,0x32,0x38,0x31,0x33,0x5a,0x30,
0x5f,0x31,0x13,0x30,0x11,0x06,0x0a,0x09,0x92,0x26,0x89,0x93,0xf2,0x2c,0x64,0x01,
0x19,0x16,0x03,0x63,0x6f,0x6d,0x31,0x19,0x30,0x17,0x06,0x0a,0x09,0x92,0x26,0x89,
0x93,0xf2,0x2c,0x64,0x01,0x19,0x16,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,
0x74,0x31,0x2d,0x30,0x2b,0x06,0x03,0x55,0x04,0x03,0x13,0x24,0x4d,0x69,0x63,0x72,
0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,
0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,
0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,0x82,0x02,0x01,
0x00,0xf3,0x5d,0xfa,0x80,0x67,0xd4,0x5a,0xa7,0xa9,0x0c,0x2c,0x90,0x20,0xd0,0x35,
0x08,0x3c,0x75,0x84,0xcd,0xb7,0x07,0x89,0x9c,0x89,0xda,0xde,0xce,0xc3,0x60,0xfa,
0x91,0x68,0x5a,0x9e,0x94,0x71,0x29,0x18,0x76,0x7c,0xc2,0xe0,0xc8,0x25,0x76,0x94,
0x0e,0x58,0xfa,0x04,0x34,0x36,0xe6,0xdf,0xaf,0xf7,0x80,0xba,0xe9,0x58,0x0b,0x2b,
0x93,0xe5,0x9d,0x05,0xe3,0x77,0x22,0x91,0xf7,0x34,0x64,0x3c,0x22,0x91,0x1d,0x5e,
0xe1,0x09,0x90,0xbc,0x14,0xfe,0xfc,0x75,0x58,0x19,0xe1,0x79,0xb7,0x07,0x92,0xa3,
0xae,0x88,0x59,0x08,0xd8,0x9f,0x07,0xca,0x03,0x58,0xfc,0x68,0x29,0x6d,0x32,0xd7,
0xd2,0xa8,0xcb,0x4b,0xfc,0xe1,0x0b,0x48,0x32,0x4f,0xe6,0xeb,0xb8,0xad,0x4f,0xe4,
0x5c,0x6f,0x13,0x94,0x99,0xdb,0x95,0xd5,0x75,0xdb,0xa8,0x1a,0xb7,0x94,0x91,0xb4,
0x77,0x5b,0xf5,0x48,0x0c,0x8f,0x6a,0x79,0x7d,0x14,0x70,0x04,0x7d,0x6d,0xaf,0x90,
0xf5,0xda,0x70,0xd8,0x47,0xb7,0xbf,0x9b,0x2f,0x6c,0xe7,0x05,0xb7,0xe1,0x11,0x60,
0xac,0x79,0x91,0x14,0x7c,0xc5,0xd6,0xa6,0xe4,0xe1,0x7e,0xd5,0xc3,0x7e,0xe5,0x92,
0xd2,0x3c,0x00,0xb5,0x36,0x82,0xde,0x79,0xe1,0x6d,0xf3,0xb5,0x6e,0xf8,0x9f,0x33,
0xc9,0xcb,0x52,0x7d,0x73,0x98,0x36,0xdb,0x8b,0xa1,0x6b,0xa2,0x95,0x97,0x9b,0xa3,
0xde,0xc2,0x4d,0x26,0xff,0x06,0x96,0x67,0x25,0x06,0xc8,0xe7,0xac,0xe4,0xee,0x12,
0x33,0x95,0x31,0x99,0xc8,0x35,0x08,0x4e,0x34,0xca,0x79,0x53,0xd5,0xb5,0xbe,0x63,
0x32,0x59,0x40,0x36,0xc0,0xa5,0x4e,0x04,0x4d,0x3d,0xdb,0x5b,0x07,0x33,0xe4,0x58,
0xbf,0xef,0x3f,0x53,0x64,0xd8,0x42,0x59,0x35,0x57,0xfd,0x0f,0x45,0x7c,0x24,0x04,
0x4d,0x9e,0xd6,0x38,0x74,0x11,0x97,0x22,0x90,0xce,0x68,0x44,0x74,0x92,0x6f,0xd5,
0x4b,0x6f,0xb0,0x86,0xe3,0xc7,0x36,0x42,0xa0,0xd0,0xfc,0xc1,0xc0,0x5a,0xf9,0xa3,
0x61,0xb9,0x30,0x47,0x71,0x96,0x0a,0x16,0xb0,0x91,0xc0,0x42,0x95,0xef,0x10,0x7f,
0x28,0x6a,0xe3,0x2a,0x1f,0xb1,0xe4,0xcd,0x03,0x3f,0x77,0x71,0x04,0xc7,0x20,0xfc,
0x49,0x0f,0x1d,0x45,0x88,0xa4,0xd7,0xcb,0x7e,0x88,0xad,0x8e,0x2d,0xec,0x45,0xdb,
0xc4,0x51,0x04,0xc9,0x2a,0xfc,0xec,0x86,0x9e,0x9a,0x11,0x97,0x5b,0xde,0xce,0x53,
0x88,0xe6,0xe2,0xb7,0xfd,0xac,0x95,0xc2,0x28,0x40,0xdb,0xef,0x04,0x90,0xdf,0x81,
0x33,0x39,0xd9,0xb2,0x45,0xa5,0x23,0x87,0x06,0xa5,0x55,0x89,0x31,0xbb,0x06,0x2d,
0x60,0x0e,0x41,0x18,0x7d,0x1f,0x2e,0xb5,0x97,0xcb,0x11,0xeb,0x15,0xd5,0x24,0xa5,
0x94,0xef,0x15,0x14,0x89,0xfd,0x4b,0x73,0xfa,0x32,0x5b,0xfc,0xd1,0x33,0x00,0xf9,
0x59,0x62,0x70,0x07,0x32,0xea,0x2e,0xab,0x40,0x2d,0x7b,0xca,0xdd,0x21,0x67,0x1b,
0x30,0x99,0x8f,0x16,0xaa,0x23,0xa8,0x41,0xd1,0xb0,0x6e,0x11,0x9b,0x36,0xc4,0xde,
0x40,0x74,0x9c,0xe1,0x58,0x65,0xc1,0x60,0x1e,0x7a,0x5b,0x38,0xc8,0x8f,0xbb,0x04,
0x26,0x7c,0xd4,0x16,0x40,0xe5,0xb6,0x6b,0x6c,0xaa,0x86,0xfd,0x00,0xbf,0xce,0xc1,
0x35,0x02,0x03,0x01,0x00,0x01,0xa3,0x51,0x30,0x4f,0x30,0x0b,0x06,0x03,0x55,0x1d,
0x0f,0x04,0x04,0x03,0x02,0x01,0xc6,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,
0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,
0x16,0x04,0x14,0x0e,0xac,0x82,0x60,0x40,0x56,0x27,0x97,0xe5,0x25,0x13,0xfc,0x2a,
0xe1,0x0a,0x53,0x95,0x59,0xe4,0xa4,0x30,0x10,0x06,0x09,0x2b,0x06,0x01,0x04,0x01,
0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x02,0x01,0x00,0xc5,0x11,0x4d,
0x03,0x3a,0x60,0xdd,0x5d,0x52,0x11,0x77,0x8f,0xb2,0xbb,0x36,0xc8,0xb2,0x05,0xbf,
0xb4,0xb7,0xa8,0xd8,0x20,0x9d,0x5c,0x13,0x03,0xb6,0x1c,0x22,0xfa,0x06,0x13,0x35,
0xb6,0xc8,0x63,0xd4,0x9a,0x47,0x6f,0x26,0x57,0xd2,0x55,0xf1,0x04,0xb1,0x26,0x5f,
0xd6,0xa9,0x50,0x68,0xa0,0xbc,0xd2,0xb8,0x6e,0xcc,0xc3,0xe9,0xac,0xdf,0x19,0xcd,
0x78,0xac,0x59,0x74,0xac,0x66,0x34,0x36,0xc4,0x1b,0x3e,0x6c,0x38,0x4c,0x33,0x0e,
0x30,0x12,0x0d,0xa3,0x26,0xfe,0x51,0x53,0x00,0xff,0xaf,0x5a,0x4e,0x84,0x0d,0x0f,
0x1f,0xe4,0x6d,0x05,0x2e,0x4e,0x85,0x4b,0x8d,0x6c,0x33,0x6f,0x54,0xd2,0x64,0xab,
0xbf,0x50,0xaf,0x7d,0x7a,0x39,0xa0,0x37,0xed,0x63,0x03,0x0f,0xfc,0x13,0x06,0xce,
0x16,0x36,0xd4,0x54,0x3b,0x95,0x1b,0x51,0x62,0x3a,0xe5,0x4d,0x17,0xd4,0x05,0x39,
0x92,0x9a,0x27,0xa8,0x5b,0xaa,0xbd,0xec,0xbb,0xbe,0xe3,0x20,0x89,0x60,0x71,0x6c,
0x56,0xb3,0xa5,0x13,0xd0,0x6d,0x0e,0x23,0x7e,0x95,0x03,0xed,0x68,0x3d,0xf2,0xd8,
0x63,0xb8,0x6b,0x4d,0xb6,0xe8,0x30,0xb5,0xe1,0xca,0x94,0x4b,0xf7,0xa2,0xaa,0x5d,
0x99,0x30,0xb2,0x3d,0xa7,0xc2,0x51,0x6c,0x28,0x20,0x01,0x24,0x27,0x2b,0x4b,0x00,
0xb7,0x9d,0x11,0x6b,0x70,0xbe,0xb2,0x10,0x82,0xbc,0x0c,0x9b,0x68,0xd0,0x8d,0x3b,
0x24,0x87,0xaa,0x99,0x28,0x72,0x9d,0x33,0x5f,0x59,0x90,0xbd,0xf5,0xde,0x93,0x9e,
0x3a,0x62,0x5a,0x34,0x39,0xe2,0x88,0x55,0x1d,0xb9,0x06,0xb0,0xc1,0x89,0x6b,0x2d,
0xd7,0x69,0xc3,0x19,0x12,0x36,0x84,0xd0,0xc9,0xa0,0xda,0xff,0x2f,0x69,0x78,0xb2,
0xe5,0x7a,0xda,0xeb,0xd7,0x0c,0xc0,0xf7,0xbd,0x63,0x17,0xb8,0x39,0x13,0x38,0xa2,
0x36,0x5b,0x7b,0xf2,0x85,0x56,0x6a,0x1d,0x64,0x62,0xc1,0x38,0xe2,0xaa,0xbf,0x51,
0x66,0xa2,0x94,0xf5,0x12,0x9c,0x66,0x22,0x10,0x6b,0xf2,0xb7,0x30,0x92,0x2d,0xf2,
0x29,0xf0,0x3d,0x3b,0x14,0x43,0x68,0xa2,0xf1,0x9c,0x29,0x37,0xcb,0xce,0x38,0x20,
0x25,0x6d,0x7c,0x67,0xf3,0x7e,0x24,0x12,0x24,0x03,0x08,0x81,0x47,0xec,0xa5,0x9e,
0x97,0xf5,0x18,0xd7,0xcf,0xbb,0xd5,0xef,0x76,0x96,0xef,0xfd,0xce,0xdb,0x56,0x9d,
0x95,0xa0,0x42,0xf9,0x97,0x58,0xe1,0xd7,0x31,0x22,0xd3,0x5f,0x59,0xe6,0x3e,0x6e,
0x22,0x00,0xea,0x43,0x84,0xb6,0x25,0xdb,0xd9,0xf3,0x08,0x56,0x68,0xc0,0x64,0x6b,
0x1d,0x7c,0xec,0xb6,0x93,0xa2,0x62,0x57,0x6e,0x2e,0xd8,0xe7,0x58,0x8f,0xc4,0x31,
0x49,0x26,0xdd,0xde,0x29,0x35,0x87,0xf5,0x30,0x71,0x70,0x5b,0x14,0x3c,0x69,0xbd,
0x89,0x12,0x7d,0xeb,0x2e,0xa3,0xfe,0xd8,0x7f,0x9e,0x82,0x5a,0x52,0x0a,0x2b,0xc1,
0x43,0x2b,0xd9,0x30,0x88,0x9f,0xc8,0x10,0xfb,0x89,0x8d,0xe6,0xa1,0x85,0x75,0x33,
0x7e,0x6c,0x9e,0xdb,0x73,0x13,0x64,0x62,0x69,0xa5,0x2f,0x7d,0xca,0x96,0x6d,0x9f,
0xf8,0x04,0x4d,0x30,0x92,0x3d,0x6e,0x21,0x14,0x21,0xc9,0x3d,0xe0,0xc3,0xfd,0x8a,
0x6b,0x9d,0x4a,0xfd,0xd1,0xa1,0x9d,0x99,0x43,0x77,0x3f,0xb0,0xda };
static const BYTE rootcertauthority2010[] = {
0x30,0x82,0x05,0xed,0x30,0x82,0x03,0xd5,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x28,
0xcc,0x3a,0x25,0xbf,0xba,0x44,0xac,0x44,0x9a,0x9b,0x58,0x6b,0x43,0x39,0xaa,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x81,
0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,
0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,
0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,
0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,
0x61,0x74,0x69,0x6f,0x6e,0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,
0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,
0x72,0x69,0x74,0x79,0x20,0x32,0x30,0x31,0x30,0x30,0x1e,0x17,0x0d,0x31,0x30,0x30,
0x36,0x32,0x33,0x32,0x31,0x35,0x37,0x32,0x34,0x5a,0x17,0x0d,0x33,0x35,0x30,0x36,
0x32,0x33,0x32,0x32,0x30,0x34,0x30,0x31,0x5a,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,
0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,
0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,
0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,
0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,
0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,
0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,
0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x20,
0x32,0x30,0x31,0x30,0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,
0x02,0x82,0x02,0x01,0x00,0xb9,0x08,0x9e,0x28,0xe4,0xe4,0xec,0x06,0x4e,0x50,0x68,
0xb3,0x41,0xc5,0x7b,0xeb,0xae,0xb6,0x8e,0xaf,0x81,0xba,0x22,0x44,0x1f,0x65,0x34,
0x69,0x4c,0xbe,0x70,0x40,0x17,0xf2,0x16,0x7b,0xe2,0x79,0xfd,0x86,0xed,0x0d,0x39,
0xf4,0x1b,0xa8,0xad,0x92,0x90,0x1e,0xcb,0x3d,0x76,0x8f,0x5a,0xd9,0xb5,0x91,0x10,
0x2e,0x3c,0x05,0x8d,0x8a,0x6d,0x24,0x54,0xe7,0x1f,0xed,0x56,0xad,0x83,0xb4,0x50,
0x9c,0x15,0xa5,0x17,0x74,0x88,0x59,0x20,0xfc,0x08,0xc5,0x84,0x76,0xd3,0x68,0xd4,
0x6f,0x28,0x78,0xce,0x5c,0xb8,0xf3,0x50,0x90,0x44,0xff,0xe3,0x63,0x5f,0xbe,0xa1,
0x9a,0x2c,0x96,0x15,0x04,0xd6,0x07,0xfe,0x1e,0x84,0x21,0xe0,0x42,0x31,0x11,0xc4,
0x28,0x36,0x94,0xcf,0x50,0xa4,0x62,0x9e,0xc9,0xd6,0xab,0x71,0x00,0xb2,0x5b,0x0c,
0xe6,0x96,0xd4,0x0a,0x24,0x96,0xf5,0xff,0xc6,0xd5,0xb7,0x1b,0xd7,0xcb,0xb7,0x21,
0x62,0xaf,0x12,0xdc,0xa1,0x5d,0x37,0xe3,0x1a,0xfb,0x1a,0x46,0x98,0xc0,0x9b,0xc0,
0xe7,0x63,0x1f,0x2a,0x08,0x93,0x02,0x7e,0x1e,0x6a,0x8e,0xf2,0x9f,0x18,0x89,0xe4,
0x22,0x85,0xa2,0xb1,0x84,0x57,0x40,0xff,0xf5,0x0e,0xd8,0x6f,0x9c,0xed,0xe2,0x45,
0x31,0x01,0xcd,0x17,0xe9,0x7f,0xb0,0x81,0x45,0xe3,0xaa,0x21,0x40,0x26,0xa1,0x72,
0xaa,0xa7,0x4f,0x3c,0x01,0x05,0x7e,0xee,0x83,0x58,0xb1,0x5e,0x06,0x63,0x99,0x62,
0x91,0x78,0x82,0xb7,0x0d,0x93,0x0c,0x24,0x6a,0xb4,0x1b,0xdb,0x27,0xec,0x5f,0x95,
0x04,0x3f,0x93,0x4a,0x30,0xf5,0x97,0x18,0xb3,0xa7,0xf9,0x19,0xa7,0x93,0x33,0x1d,
0x01,0xc8,0xdb,0x22,0x52,0x5c,0xd7,0x25,0xc9,0x46,0xf9,0xa2,0xfb,0x87,0x59,0x43,
0xbe,0x9b,0x62,0xb1,0x8d,0x2d,0x86,0x44,0x1a,0x46,0xac,0x78,0x61,0x7e,0x30,0x09,
0xfa,0xae,0x89,0xc4,0x41,0x2a,0x22,0x66,0x03,0x91,0x39,0x45,0x9c,0xc7,0x8b,0x0c,
0xa8,0xca,0x0d,0x2f,0xfb,0x52,0xea,0x0c,0xf7,0x63,0x33,0x23,0x9d,0xfe,0xb0,0x1f,
0xad,0x67,0xd6,0xa7,0x50,0x03,0xc6,0x04,0x70,0x63,0xb5,0x2c,0xb1,0x86,0x5a,0x43,
0xb7,0xfb,0xae,0xf9,0x6e,0x29,0x6e,0x21,0x21,0x41,0x26,0x06,0x8c,0xc9,0xc3,0xee,
0xb0,0xc2,0x85,0x93,0xa1,0xb9,0x85,0xd9,0xe6,0x32,0x6c,0x4b,0x4c,0x3f,0xd6,0x5d,
0xa3,0xe5,0xb5,0x9d,0x77,0xc3,0x9c,0xc0,0x55,0xb7,0x74,0x00,0xe3,0xb8,0x38,0xab,
0x83,0x97,0x50,0xe1,0x9a,0x42,0x24,0x1d,0xc6,0xc0,0xa3,0x30,0xd1,0x1a,0x5a,0xc8,
0x52,0x34,0xf7,0x73,0xf1,0xc7,0x18,0x1f,0x33,0xad,0x7a,0xec,0xcb,0x41,0x60,0xf3,
0x23,0x94,0x20,0xc2,0x48,0x45,0xac,0x5c,0x51,0xc6,0x2e,0x80,0xc2,0xe2,0x77,0x15,
0xbd,0x85,0x87,0xed,0x36,0x9d,0x96,0x91,0xee,0x00,0xb5,0xa3,0x70,0xec,0x9f,0xe3,
0x8d,0x80,0x68,0x83,0x76,0xba,0xaf,0x5d,0x70,0x52,0x22,0x16,0xe2,0x66,0xfb,0xba,
0xb3,0xc5,0xc2,0xf7,0x3e,0x2f,0x77,0xa6,0xca,0xde,0xc1,0xa6,0xc6,0x48,0x4c,0xc3,
0x37,0x51,0x23,0xd3,0x27,0xd7,0xb8,0x4e,0x70,0x96,0xf0,0xa1,0x44,0x76,0xaf,0x78,
0xcf,0x9a,0xe1,0x66,0x13,0x02,0x03,0x01,0x00,0x01,0xa3,0x51,0x30,0x4f,0x30,0x0b,
0x06,0x03,0x55,0x1d,0x0f,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x0f,0x06,0x03,0x55,
0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,
0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0xd5,0xf6,0x56,0xcb,0x8f,0xe8,0xa2,0x5c,0x62,
0x68,0xd1,0x3d,0x94,0x90,0x5b,0xd7,0xce,0x9a,0x18,0xc4,0x30,0x10,0x06,0x09,0x2b,
0x06,0x01,0x04,0x01,0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x02,0x01,
0x00,0xac,0xa5,0x96,0x8c,0xbf,0xbb,0xae,0xa6,0xf6,0xd7,0x71,0x87,0x43,0x31,0x56,
0x88,0xfd,0x1c,0x32,0x71,0x5b,0x35,0xb7,0xd4,0xf0,0x91,0xf2,0xaf,0x37,0xe2,0x14,
0xf1,0xf3,0x02,0x26,0x05,0x3e,0x16,0x14,0x7f,0x14,0xba,0xb8,0x4f,0xfb,0x89,0xb2,
0xb2,0xe7,0xd4,0x09,0xcc,0x6d,0xb9,0x5b,0x3b,0x64,0x65,0x70,0x66,0xb7,0xf2,0xb1,
0x5a,0xdf,0x1a,0x02,0xf3,0xf5,0x51,0xb8,0x67,0x6d,0x79,0xf3,0xbf,0x56,0x7b,0xe4,
0x84,0xb9,0x2b,0x1e,0x9b,0x40,0x9c,0x26,0x34,0xf9,0x47,0x18,0x98,0x69,0xd8,0x1c,
0xd7,0xb6,0xd1,0xbf,0x8f,0x61,0xc2,0x67,0xc4,0xb5,0xef,0x60,0x43,0x8e,0x10,0x1b,
0x36,0x49,0xe4,0x20,0xca,0xad,0xa7,0xc1,0xb1,0x27,0x65,0x09,0xf8,0xcd,0xf5,0x5b,
0x2a,0xd0,0x84,0x33,0xf3,0xef,0x1f,0xf2,0xf5,0x9c,0x0b,0x58,0x93,0x37,0xa0,0x75,
0xa0,0xde,0x72,0xde,0x6c,0x75,0x2a,0x66,0x22,0xf5,0x8c,0x06,0x30,0x56,0x9f,0x40,
0xb9,0x30,0xaa,0x40,0x77,0x15,0x82,0xd7,0x8b,0xec,0xc0,0xd3,0xb2,0xbd,0x83,0xc5,
0x77,0x0c,0x1e,0xae,0xaf,0x19,0x53,0xa0,0x4d,0x79,0x71,0x9f,0x0f,0xaf,0x30,0xce,
0x67,0xf9,0xd6,0x2c,0xcc,0x22,0x41,0x7a,0x07,0xf2,0x97,0x42,0x18,0xce,0x59,0x79,
0x10,0x55,0xde,0x6f,0x10,0xe4,0xb8,0xda,0x83,0x66,0x40,0x16,0x09,0x68,0x23,0x5b,
0x97,0x2e,0x26,0x9a,0x02,0xbb,0x57,0x8c,0xc5,0xb8,0xba,0x69,0x62,0x32,0x80,0x89,
0x9e,0xa1,0xfd,0xc0,0x92,0x7c,0x7b,0x2b,0x33,0x19,0x84,0x2a,0x63,0xc5,0x00,0x68,
0x62,0xfa,0x9f,0x47,0x8d,0x99,0x7a,0x45,0x3a,0xa7,0xe9,0xed,0xee,0x69,0x42,0xb5,
0xf3,0x81,0x9b,0x47,0x56,0x10,0x7b,0xfc,0x70,0x36,0x84,0x18,0x73,0xea,0xef,0xf9,
0x97,0x4d,0x9e,0x33,0x23,0xdd,0x26,0x0b,0xba,0x2a,0xb7,0x3f,0x44,0xdc,0x83,0x27,
0xff,0xbd,0x61,0x59,0x2b,0x11,0xb7,0xca,0x4f,0xdb,0xc5,0x8b,0x0c,0x1c,0x31,0xae,
0x32,0xf8,0xf8,0xb9,0x42,0xf7,0x7f,0xdc,0x61,0x9a,0x76,0xb1,0x5a,0x04,0xe1,0x11,
0x3d,0x66,0x45,0xb7,0x18,0x71,0xbe,0xc9,0x24,0x85,0xd6,0xf3,0xd4,0xba,0x41,0x34,
0x5d,0x12,0x2d,0x25,0xb9,0x8d,0xa6,0x13,0x48,0x6d,0x4b,0xb0,0x07,0x7d,0x99,0x93,
0x09,0x61,0x81,0x74,0x57,0x26,0x8a,0xab,0x69,0xe3,0xe4,0xd9,0xc7,0x88,0xcc,0x24,
0xd8,0xec,0x52,0x24,0x5c,0x1e,0xbc,0x91,0x14,0xe2,0x96,0xde,0xeb,0x0a,0xda,0x9e,
0xdd,0x5f,0xb3,0x5b,0xdb,0xd4,0x82,0xec,0xc6,0x20,0x50,0x87,0x25,0x40,0x3a,0xfb,
0xc7,0xee,0xcd,0xfe,0x33,0xe5,0x6e,0xc3,0x84,0x09,0x55,0x03,0x25,0x39,0xc0,0xe9,
0x35,0x5d,0x65,0x31,0xa8,0xf6,0xbf,0xa0,0x09,0xcd,0x29,0xc7,0xb3,0x36,0x32,0x2e,
0xdc,0x95,0xf3,0x83,0xc1,0x5a,0xcf,0x8b,0x8d,0xf6,0xea,0xb3,0x21,0xf8,0xa4,0xed,
0x1e,0x31,0x0e,0xb6,0x4c,0x11,0xab,0x60,0x0b,0xa4,0x12,0x23,0x22,0x17,0xa3,0x36,
0x64,0x82,0x91,0x04,0x12,0xe0,0xab,0x6f,0x1e,0xcb,0x50,0x05,0x61,0xb4,0x40,0xff,
0x59,0x86,0x71,0xd1,0xd5,0x33,0x69,0x7c,0xa9,0x73,0x8a,0x38,0xd7,0x64,0x0c,0xf1,
0x69 };
static const BYTE rootcertauthority2011[] = {
0x30,0x82,0x05,0xed,0x30,0x82,0x03,0xd5,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x3f,
0x8b,0xc8,0xb5,0xfc,0x9f,0xb2,0x96,0x43,0xb5,0x69,0xd6,0x6c,0x42,0xe1,0x44,0x30,
0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x81,
0x88,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,
0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,
0x74,0x6f,0x6e,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,
0x64,0x6d,0x6f,0x6e,0x64,0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,
0x61,0x74,0x69,0x6f,0x6e,0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,
0x4d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,
0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,
0x72,0x69,0x74,0x79,0x20,0x32,0x30,0x31,0x31,0x30,0x1e,0x17,0x0d,0x31,0x31,0x30,
0x33,0x32,0x32,0x32,0x32,0x30,0x35,0x32,0x38,0x5a,0x17,0x0d,0x33,0x36,0x30,0x33,
0x32,0x32,0x32,0x32,0x31,0x33,0x30,0x34,0x5a,0x30,0x81,0x88,0x31,0x0b,0x30,0x09,
0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,
0x04,0x08,0x13,0x0a,0x57,0x61,0x73,0x68,0x69,0x6e,0x67,0x74,0x6f,0x6e,0x31,0x10,
0x30,0x0e,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6d,0x6f,0x6e,0x64,
0x31,0x1e,0x30,0x1c,0x06,0x03,0x55,0x04,0x0a,0x13,0x15,0x4d,0x69,0x63,0x72,0x6f,
0x73,0x6f,0x66,0x74,0x20,0x43,0x6f,0x72,0x70,0x6f,0x72,0x61,0x74,0x69,0x6f,0x6e,
0x31,0x32,0x30,0x30,0x06,0x03,0x55,0x04,0x03,0x13,0x29,0x4d,0x69,0x63,0x72,0x6f,
0x73,0x6f,0x66,0x74,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,
0x69,0x63,0x61,0x74,0x65,0x20,0x41,0x75,0x74,0x68,0x6f,0x72,0x69,0x74,0x79,0x20,
0x32,0x30,0x31,0x31,0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,
0x02,0x82,0x02,0x01,0x00,0xb2,0x80,0x41,0xaa,0x35,0x38,0x4d,0x13,0x72,0x32,0x68,
0x22,0x4d,0xb8,0xb2,0xf1,0xff,0xd5,0x52,0xbc,0x6c,0xc7,0xf5,0xd2,0x4a,0x8c,0x36,
0xee,0xd1,0xc2,0x5c,0x7e,0x8c,0x8a,0xae,0xaf,0x13,0x28,0x6f,0xc0,0x73,0xe3,0x3a,
0xce,0xd0,0x25,0xa8,0x5a,0x3a,0x6d,0xef,0xa8,0xb8,0x59,0xab,0x13,0x23,0x68,0xcd,
0x0c,0x29,0x87,0xd1,0x6f,0x80,0x5c,0x8f,0x44,0x7f,0x5d,0x90,0x01,0x52,0x58,0xac,
0x51,0xc5,0x5f,0x2a,0x87,0xdc,0xdc,0xd8,0x0a,0x1d,0xc1,0x03,0xb9,0x7b,0xb0,0x56,
0xe8,0xa3,0xde,0x64,0x61,0xc2,0x9e,0xf8,0xf3,0x7c,0xb9,0xec,0x0d,0xb5,0x54,0xfe,
0x4c,0xb6,0x65,0x4f,0x88,0xf0,0x9c,0x48,0x99,0x0c,0x42,0x0b,0x09,0x7c,0x31,0x59,
0x17,0x79,0x06,0x78,0x28,0x8d,0x89,0x3a,0x4c,0x03,0x25,0xbe,0x71,0x6a,0x5c,0x0b,
0xe7,0x84,0x60,0xa4,0x99,0x22,0xe3,0xd2,0xaf,0x84,0xa4,0xa7,0xfb,0xd1,0x98,0xed,
0x0c,0xa9,0xde,0x94,0x89,0xe1,0x0e,0xa0,0xdc,0xc0,0xce,0x99,0x3d,0xea,0x08,0x52,
0xbb,0x56,0x79,0xe4,0x1f,0x84,0xba,0x1e,0xb8,0xb4,0xc4,0x49,0x5c,0x4f,0x31,0x4b,
0x87,0xdd,0xdd,0x05,0x67,0x26,0x99,0x80,0xe0,0x71,0x11,0xa3,0xb8,0xa5,0x41,0xe2,
0xa4,0x53,0xb9,0xf7,0x32,0x29,0x83,0x0c,0x13,0xbf,0x36,0x5e,0x04,0xb3,0x4b,0x43,
0x47,0x2f,0x6b,0xe2,0x91,0x1e,0xd3,0x98,0x4f,0xdd,0x42,0x07,0xc8,0xe8,0x1d,0x12,
0xfc,0x99,0xa9,0x6b,0x3e,0x92,0x7e,0xc8,0xd6,0x69,0x3a,0xfc,0x64,0xbd,0xb6,0x09,
0x9d,0xca,0xfd,0x0c,0x0b,0xa2,0x9b,0x77,0x60,0x4b,0x03,0x94,0xa4,0x30,0x69,0x12,
0xd6,0x42,0x2d,0xc1,0x41,0x4c,0xca,0xdc,0xaa,0xfd,0x8f,0x5b,0x83,0x46,0x9a,0xd9,
0xfc,0xb1,0xd1,0xe3,0xb3,0xc9,0x7f,0x48,0x7a,0xcd,0x24,0xf0,0x41,0x8f,0x5c,0x74,
0xd0,0xac,0xb0,0x10,0x20,0x06,0x49,0xb7,0xc7,0x2d,0x21,0xc8,0x57,0xe3,0xd0,0x86,
0xf3,0x03,0x68,0xfb,0xd0,0xce,0x71,0xc1,0x89,0x99,0x4a,0x64,0x01,0x6c,0xfd,0xec,
0x30,0x91,0xcf,0x41,0x3c,0x92,0xc7,0xe5,0xba,0x86,0x1d,0x61,0x84,0xc7,0x5f,0x83,
0x39,0x62,0xae,0xb4,0x92,0x2f,0x47,0xf3,0x0b,0xf8,0x55,0xeb,0xa0,0x1f,0x59,0xd0,
0xbb,0x74,0x9b,0x1e,0xd0,0x76,0xe6,0xf2,0xe9,0x06,0xd7,0x10,0xe8,0xfa,0x64,0xde,
0x69,0xc6,0x35,0x96,0x88,0x02,0xf0,0x46,0xb8,0x3f,0x27,0x99,0x6f,0xcb,0x71,0x89,
0x29,0x35,0xf7,0x48,0x16,0x02,0x35,0x8f,0xd5,0x79,0x7c,0x4d,0x02,0xcf,0x5f,0xeb,
0x8a,0x83,0x4f,0x45,0x71,0x88,0xf9,0xa9,0x0d,0x4e,0x72,0xe9,0xc2,0x9c,0x07,0xcf,
0x49,0x1b,0x4e,0x04,0x0e,0x63,0x51,0x8c,0x5e,0xd8,0x00,0xc1,0x55,0x2c,0xb6,0xc6,
0xe0,0xc2,0x65,0x4e,0xc9,0x34,0x39,0xf5,0x9c,0xb3,0xc4,0x7e,0xe8,0x61,0x6e,0x13,
0x5f,0x15,0xc4,0x5f,0xd9,0x7e,0xed,0x1d,0xce,0xee,0x44,0xec,0xcb,0x2e,0x86,0xb1,
0xec,0x38,0xf6,0x70,0xed,0xab,0x5c,0x13,0xc1,0xd9,0x0f,0x0d,0xc7,0x80,0xb2,0x55,
0xed,0x34,0xf7,0xac,0x9b,0xe4,0xc3,0xda,0xe7,0x47,0x3c,0xa6,0xb5,0x8f,0x31,0xdf,
0xc5,0x4b,0xaf,0xeb,0xf1,0x02,0x03,0x01,0x00,0x01,0xa3,0x51,0x30,0x4f,0x30,0x0b,
0x06,0x03,0x55,0x1d,0x0f,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x0f,0x06,0x03,0x55,
0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,
0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x72,0x2d,0x3a,0x02,0x31,0x90,0x43,0xb9,0x14,
0x05,0x4e,0xe1,0xea,0xa7,0xc7,0x31,0xd1,0x23,0x89,0x34,0x30,0x10,0x06,0x09,0x2b,
0x06,0x01,0x04,0x01,0x82,0x37,0x15,0x01,0x04,0x03,0x02,0x01,0x00,0x30,0x0d,0x06,
0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x02,0x01,
0x00,0x7f,0x72,0xcf,0x0f,0xb7,0xc5,0x15,0xdb,0x9b,0xc0,0x49,0xca,0x26,0x5b,0xfe,
0x9e,0x13,0xe6,0xd3,0xf0,0xd2,0xdb,0x97,0x5f,0xf2,0x4b,0x3f,0x4d,0xb3,0xae,0x19,
0xae,0xed,0xd7,0x97,0xa0,0xac,0xef,0xa9,0x3a,0xa3,0xc2,0x41,0xb0,0xe5,0xb8,0x91,
0x9e,0x13,0x81,0x24,0x03,0xe6,0x09,0xfd,0x3f,0x57,0x40,0x39,0x21,0x24,0x56,0xd1,
0x10,0x2f,0x4b,0x40,0xa9,0x36,0x86,0x4b,0xb4,0x53,0x57,0x9a,0xfb,0xf1,0x7e,0x89,
0x8f,0x11,0xfe,0x18,0x6c,0x51,0xaa,0xe8,0xed,0x09,0x95,0xb5,0xe5,0x71,0xc9,0xa1,
0xe9,0x87,0x75,0xa6,0x15,0x7f,0xc9,0x7e,0x37,0x54,0x5e,0x74,0x93,0xc5,0xc3,0x67,
0xcc,0x0d,0x4f,0x6b,0xa8,0x17,0x0c,0x6d,0x08,0x92,0x7e,0x8b,0xdd,0x81,0xaa,0x2d,
0x70,0x21,0xc3,0x3d,0x06,0x14,0xbb,0xbf,0x24,0x5e,0xa7,0x84,0xd7,0x3f,0x0f,0x21,
0x22,0xbd,0x4b,0x00,0x06,0xdb,0x97,0x1c,0xd8,0x5e,0xd4,0xc5,0x0b,0x5c,0x87,0x6e,
0x50,0xa4,0xe8,0xc3,0x38,0xa4,0xfb,0xcb,0x2c,0xc5,0x92,0x66,0x9b,0x85,0x5e,0xcb,
0x7a,0x6c,0x93,0x7c,0x80,0x29,0x58,0x5b,0x57,0xb5,0x40,0x69,0xba,0x08,0x79,0xa6,
0x64,0x62,0x15,0x9d,0x87,0x96,0x45,0xb5,0x66,0x23,0x20,0x03,0x8b,0x1c,0x73,0xa0,
0xd3,0xa2,0x79,0x33,0xe0,0x50,0x59,0x86,0xdb,0x2f,0xe5,0x02,0x25,0xea,0x73,0x2a,
0x9f,0x00,0x14,0xc8,0x36,0xc7,0x92,0x3b,0xe9,0x4e,0x00,0xec,0xd8,0x56,0x09,0xb9,
0x33,0x49,0x12,0xd2,0x54,0x0b,0x01,0xab,0xac,0x47,0xb6,0x91,0x29,0x7d,0x4c,0xb4,
0x75,0x80,0x52,0x01,0xe8,0xca,0x82,0xf6,0x9f,0xcc,0xac,0x9c,0x8f,0x17,0xea,0x2f,
0x26,0xb0,0xab,0x72,0xac,0x0b,0xfe,0x9e,0x51,0x1e,0xc7,0x43,0x55,0x67,0x4f,0x51,
0xb3,0x57,0xd6,0xb6,0xec,0xee,0x52,0xb7,0x3a,0xe9,0x4e,0xe1,0xd7,0x81,0x88,0xbc,
0x4f,0x8e,0x75,0xbb,0x4b,0xa8,0xf0,0x35,0xaa,0x26,0xd4,0x67,0x67,0x49,0xb2,0x70,
0x4c,0x3b,0x93,0xdc,0x1d,0xdf,0x78,0x90,0x86,0x72,0xb2,0x38,0xa4,0xd1,0xdc,0x92,
0x4d,0xc9,0x58,0xeb,0x2b,0x12,0x5c,0xd4,0x3b,0xae,0x8c,0x6b,0xb0,0x83,0xe5,0x01,
0x3f,0xf8,0x09,0x32,0xf6,0x93,0x35,0x34,0x22,0xaf,0xdd,0x37,0x0d,0x77,0x09,0x80,
0x2b,0xcd,0x48,0x00,0xf1,0x8c,0x99,0x19,0x47,0x05,0x01,0xe9,0xd1,0xbf,0xd1,0x4e,
0xd0,0xe6,0x28,0x43,0x37,0x99,0xa4,0x0a,0x4a,0x08,0xd9,0x9a,0x71,0x73,0xd2,0xaa,
0xcd,0x31,0x13,0x63,0x76,0xa1,0x37,0x6f,0x92,0x38,0x1e,0x7d,0x12,0x3c,0x66,0x32,
0xe7,0xcb,0x6d,0xe1,0xfc,0x52,0x89,0xdd,0xca,0xd6,0x66,0x05,0x9a,0x96,0x61,0xbe,
0xa2,0x28,0xc7,0x1c,0xa3,0xa7,0x36,0x50,0x3c,0x3a,0xa4,0xdf,0x4a,0x6e,0xe6,0x87,
0x3b,0xce,0xeb,0xf0,0xe0,0x81,0x37,0x9d,0x13,0x3c,0x52,0x8e,0xbd,0xb9,0x1d,0x34,
0xc6,0x1d,0xd5,0x0a,0x6a,0x3d,0x98,0x29,0x70,0x8c,0x89,0x2a,0xd1,0xab,0x82,0x10,
0x48,0x1f,0xdc,0xf4,0xef,0xa5,0xc5,0xbb,0x55,0x1a,0x38,0x63,0x84,0x4e,0xb7,0x6c,
0xad,0x95,0x54,0xec,0x65,0x22,0x10,0x49,0x17,0xb8,0xc0,0x1e,0xc7,0x0f,0xac,0x54,
0x47 };

static const struct CONST_BLOB {
    const BYTE *pb;
    DWORD       cb;
} msRootCerts[] = {
    { authenticode, sizeof(authenticode) },
    { rootauthority, sizeof(rootauthority) },
    { rootcertauthority, sizeof(rootcertauthority) },
    { rootcertauthority2010, sizeof(rootcertauthority2010) },
    { rootcertauthority2011, sizeof(rootcertauthority2011) },
};

static void add_ms_root_certs(HCERTSTORE to, HCERTSTORE cached)
{
    PCCERT_CONTEXT cert, existing;
    DWORD i;

    TRACE("\n");

    for (i = 0; i < ARRAY_SIZE(msRootCerts); i++)
    {
        if (!CertAddEncodedCertificateToStore(to, X509_ASN_ENCODING,
            msRootCerts[i].pb, msRootCerts[i].cb, CERT_STORE_ADD_NEW, &cert))
        {
            WARN("adding root cert %ld failed: %08lx\n", i, GetLastError());
            continue;
        }
        if ((existing = CertFindCertificateInStore(cached, X509_ASN_ENCODING, 0, CERT_FIND_EXISTING, cert, NULL)))
        {
            CertDeleteCertificateFromStore(existing);
            CertFreeCertificateContext(existing);
        }
        CertFreeCertificateContext(cert);
    }
}

/* Reads certificates from the list of known locations into store.  Stops when
 * any location contains any certificates, to prevent spending unnecessary time
 * adding redundant certificates, e.g. when both a certificate bundle and
 * individual certificates exist in the same directory.
 */
static void read_trusted_roots_from_known_locations(HCERTSTORE store, HCERTSTORE cached, BOOL *delete)
{
    HCERTSTORE new;
    DWORD needed, size;
    struct enum_root_certs_params params = { NULL, 2048, &needed };
    HCRYPTPROV prov;
    HCRYPTHASH hash;
    BYTE hashval[20];
    DWORD hashlen;
    CRYPT_HASH_BLOB hash_blob = { sizeof(hashval), hashval };
    CRYPT_DATA_BLOB exists_blob = { 0, NULL };
    PCCERT_CONTEXT cert, existing;
    unsigned int existing_count = 0, new_count = 0;
    unsigned int cached_count = 0;

    new = CertOpenStore( CERT_STORE_PROV_MEMORY, X509_ASN_ENCODING, 0, CERT_STORE_CREATE_NEW_FLAG, NULL );
    if (!new) return;

    existing = NULL;
    while ((existing = CertEnumCertificatesInStore( cached, existing )))
        ++cached_count;

    CryptAcquireContextW( &prov, NULL, NULL, PROV_RSA_FULL, CRYPT_VERIFYCONTEXT );
    params.buffer = CryptMemAlloc( params.size );
    while (!CRYPT32_CALL( enum_root_certs, &params ))
    {
        if (needed > params.size)
        {
            CryptMemFree( params.buffer );
            params.buffer = CryptMemAlloc( needed );
            params.size = needed;
            continue;
        }
        CryptCreateHash( prov, CALG_SHA1, 0, 0, &hash );
        CryptHashData( hash, params.buffer, needed, 0 );
        hashlen = sizeof(hashval);
        CryptGetHashParam( hash, HP_HASHVAL, hashval, &hashlen, 0 );
        CryptDestroyHash( hash );
        if ((existing = CertFindCertificateInStore( cached, X509_ASN_ENCODING, 0,
                                                    CERT_FIND_SHA1_HASH, &hash_blob, NULL )))
        {
            /* Skip certificate which is already cached. CERT_FIRST_USER_PROP_ID is set once the cached cert
             * is found among host imports. */
            if (!CertGetCertificateContextProperty( existing, CERT_FIRST_USER_PROP_ID, NULL, &size ))
            {
                if (!CertSetCertificateContextProperty( existing, CERT_FIRST_USER_PROP_ID, 0, &exists_blob ))
                    ERR( "Failed to set property.\n" );
                ++existing_count;
            }
            CertFreeCertificateContext( existing );
            continue;
        }
        CertAddEncodedCertificateToStore( new, X509_ASN_ENCODING, params.buffer, needed, CERT_STORE_ADD_ALWAYS, &cert );
        /* Add to cached so we can catch duplicates and check_and_store_certs() has the full chains. */
        CertAddCertificateContextToStore( cached, cert, CERT_STORE_ADD_ALWAYS, NULL );
        if (!CertSetCertificateContextProperty( cert, CERT_FIRST_USER_PROP_ID, 0, &exists_blob ))
            ERR("Failed to set property.\n");
        CertFreeCertificateContext( cert );
        ++new_count;
    }
    CryptMemFree( params.buffer );
    CryptReleaseContext( prov, 0 );

    if (existing_count < cached_count)
    {
        /* Some certs were removed on host. Clean up the cache and add all the certificates so cert chains
         * get revalidated. The certs present on host are now in 'cached' store and are marked with
         * CERT_FIRST_USER_PROP_ID property. */
        TRACE( "Some keys were removed, reimporting, cached %u, existing %u, new %u.\n",
                cached_count, existing_count, new_count );
        *delete = TRUE;
        existing_count = 0;
        existing = NULL;
        while ((existing = CertEnumCertificatesInStore( cached, existing )))
        {
            if (!CertGetCertificateContextProperty( existing, CERT_FIRST_USER_PROP_ID, NULL, &size ))
                continue;
            CertAddCertificateContextToStore( new, existing, CERT_STORE_ADD_NEW, NULL );
            ++new_count;
        }
    }
    if (new_count)
    {
        /* Clear custom property so it is not serialized and seen by apps. */
        cert = NULL;
        while ((cert = CertEnumCertificatesInStore( new, cert )))
            CertSetCertificateContextProperty( cert, CERT_FIRST_USER_PROP_ID, 0, NULL );
        check_and_store_certs( cached, new, store );
    }

    CertCloseStore( new, 0 );
    TRACE( "existing %u, new %u.\n", existing_count, new_count );
}

static HCERTSTORE create_root_store(HCERTSTORE cached, BOOL *delete)
{
    HCERTSTORE memStore = CertOpenStore(CERT_STORE_PROV_MEMORY,
     X509_ASN_ENCODING, 0, CERT_STORE_CREATE_NEW_FLAG, NULL);


    *delete = FALSE;
    if (memStore)
    {
        add_ms_root_certs(memStore, cached);
        read_trusted_roots_from_known_locations(memStore, cached, delete);
    }

    TRACE("returning %p\n", memStore);
    return memStore;
}

void CRYPT_ImportSystemRootCertsToReg(void)
{
    HCERTSTORE store = NULL, reg = NULL;
    HKEY key = NULL;
    LONG rc;
    HANDLE hsem;
    BOOL delete;

    static BOOL root_certs_imported = FALSE;

    if (root_certs_imported)
        return;

    hsem = CreateSemaphoreW( NULL, 0, 1, L"crypt32_root_semaphore");
    if (!hsem)
    {
        ERR("Failed to create semaphore\n");
        return;
    }

    if(GetLastError() == ERROR_ALREADY_EXISTS)
    {
        WaitForSingleObject(hsem, INFINITE);
        goto done;
    }

    rc = RegCreateKeyExW(HKEY_LOCAL_MACHINE, L"Software\\Microsoft\\SystemCertificates\\Root\\Certificates", 0, NULL, 0,
            KEY_ALL_ACCESS, NULL, &key, 0);
    if (rc)
        goto done;

    if (!(reg = CertOpenStore(CERT_STORE_PROV_MEMORY, X509_ASN_ENCODING, 0, CERT_STORE_CREATE_NEW_FLAG, NULL)))
    {
        ERR("Failed to create memory store.\n");
        goto done;
    }
    CRYPT_RegReadSerializedFromReg(key, CERT_STORE_CERTIFICATE_CONTEXT_FLAG, reg, CERT_STORE_ADD_ALWAYS);

    if (!(store = create_root_store(reg, &delete)))
    {
        ERR("Failed to create root store\n");
        goto done;
    }
    if (delete && RegDeleteTreeW(key, NULL))
        ERR("Error deleting key.\n");
    if (!CRYPT_SerializeContextsToReg(key, 0, pCertInterface, store))
        ERR("Failed to import system certs into registry, %08lx\n", GetLastError());

done:
    RegCloseKey(key);
    CertCloseStore(store, 0);
    CertCloseStore(reg, 0);
    root_certs_imported = TRUE;
    ReleaseSemaphore(hsem, 1, NULL);
    CloseHandle(hsem);
}
