package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"go/format"
	"html/template"
	"io"
	"os/exec"
	"path"
	"sort"
	"strings"
)

func main() {
	stdPaths, err := getStdlibPaths()
	if err != nil {
		panic(err)
	}
	allPackages := make([]*packageInfo, len(stdPaths))
	for i, p := range stdPaths {
		info := &packageInfo{
			Name: path.Base(p),
			Path: p,
			Freq: -1,
		}
		if freq, ok := stdlibFreqStats[p]; ok {
			info.Freq = freq
		}
		allPackages[i] = info
	}

	packagesByName := make(map[string][]*packageInfo)
	for _, info := range allPackages {
		packagesByName[info.Name] = append(packagesByName[info.Name], info)
	}

	// Some too generic names should be avoided.
	ignoreNames := map[string]struct{}{
		"doc":    {},
		"user":   {},
		"syntax": {},
		"pe":     {},
		"quick":  {},
	}
	var tmplData templateData
	tmplData.Packages = make([]*packageInfo, len(allPackages))
	copy(tmplData.Packages, allPackages)
	for pkgName, infoList := range packagesByName {
		if _, ok := ignoreNames[pkgName]; ok {
			continue
		}
		sort.Slice(infoList, func(i, j int) bool {
			return infoList[i].Freq > infoList[j].Freq
		})
		info := infoList[0]
		if info.Freq < 10 {
			continue
		}
		tmplData.PathByName = append(tmplData.PathByName, info)
	}

	sort.Slice(tmplData.PathByName, func(i, j int) bool {
		return tmplData.PathByName[i].Freq > tmplData.PathByName[j].Freq
	})
	sort.Slice(tmplData.Packages, func(i, j int) bool {
		return tmplData.Packages[i].Name < tmplData.Packages[j].Name
	})

	var outBuf bytes.Buffer
	if err := outputTemplate.Execute(&outBuf, tmplData); err != nil {
		panic(err)
	}
	formattedBuf, err := format.Source(outBuf.Bytes())
	if err != nil {
		panic(err)
	}
	fmt.Print(string(formattedBuf))
}

func getStdlibPaths() ([]string, error) {
	out, err := exec.Command("go", "list", "std").CombinedOutput()
	if err != nil {
		return nil, fmt.Errorf("go list error (%v): %s", err, out)
	}

	paths := strings.Split(string(out), "\n")
	filtered := paths[:0]
	for _, p := range paths {
		if p == "" {
			continue
		}
		if strings.HasPrefix(p, "vendor/") {
			continue
		}
		if strings.HasPrefix(p, "internal/") || strings.HasSuffix(p, "/internal") || strings.Contains(p, "/internal/") {
			continue
		}
		filtered = append(filtered, p)
	}

	return filtered, nil
}

// Package is `go list --json` output structure.
type Package struct {
	Dir        string   // directory containing package sources
	ImportPath string   // import path of package in dir
	GoFiles    []string // .go source files (excluding CgoFiles, TestGoFiles, XTestGoFiles)
}

func golistJSON(pkgPath string) (*Package, error) {
	out, err := exec.Command("go", "list", "--json", pkgPath).CombinedOutput()
	if err != nil {
		return nil, fmt.Errorf("go list error (%v): %s", err, out)
	}

	var pkg Package
	if err := json.NewDecoder(bytes.NewReader(out)).Decode(&pkg); err != io.EOF && err != nil {
		return nil, err
	}
	return &pkg, nil
}

type templateData struct {
	PathByName []*packageInfo
	Packages   []*packageInfo
}

type packageInfo struct {
	Name string
	Path string
	Freq int
}

var outputTemplate = template.Must(template.New("").Parse(`// Code generated by "script/gen.go"; DO NOT EDIT.

package stdinfo

var generatedPathByName = map[string]string{
	{{- range $.PathByName}}
		"{{.Name}}": "{{.Path}}", // Freq={{.Freq}}
	{{- end}}
}

var generatedPackagesList = []Package{
	{{- range $.Packages}}
		{Name: "{{.Name}}", Path: "{{.Path}}", Freq: {{.Freq}}},
	{{- end}}
}
`))

var stdlibFreqStats = map[string]int{
	"fmt":                  15795,
	"testing":              12807,
	"context":              10797,
	"time":                 8900,
	"strings":              8852,
	"os":                   5712,
	"bytes":                4129,
	"io":                   3981,
	"net/http":             3691,
	"sync":                 3492,
	"errors":               3107,
	"strconv":              3076,
	"reflect":              3025,
	"path/filepath":        2843,
	"encoding/json":        2537,
	"sort":                 2382,
	"io/ioutil":            2164,
	"net":                  2025,
	"math":                 1746,
	"net/url":              1411,
	"regexp":               1320,
	"runtime":              1318,
	"log":                  1149,
	"flag":                 1002,
	"path":                 993,
	"unsafe":               992,
	"math/rand":            981,
	"syscall":              902,
	"sync/atomic":          804,
	"bufio":                695,
	"net/http/httptest":    676,
	"os/exec":              676,
	"encoding/binary":      476,
	"crypto/tls":           475,
	"go/token":             471,
	"unicode/utf8":         404,
	"encoding/base64":      383,
	"go/ast":               373,
	"crypto/x509":          357,
	"encoding/hex":         340,
	"unicode":              309,
	"go/types":             309,
	"crypto/rand":          256,
	"math/big":             230,
	"crypto/sha256":        227,
	"text/template":        211,
	"html/template":        173,
	"io/fs":                162,
	"go/parser":            160,
	"database/sql":         157,
	"compress/gzip":        150,
	"os/signal":            139,
	"encoding/pem":         137,
	"hash":                 137,
	"crypto":               132,
	"go/build":             121,
	"runtime/debug":        121,
	"math/bits":            120,
	"go/constant":          120,
	"encoding/xml":         118,
	"text/tabwriter":       116,
	"crypto/md5":           110,
	"crypto/rsa":           103,
	"go/format":            88,
	"crypto/sha1":          85,
	"database/sql/driver":  81,
	"crypto/x509/pkix":     80,
	"container/heap":       78,
	"archive/tar":          77,
	"crypto/ecdsa":         75,
	"crypto/cipher":        74,
	"hash/crc32":           70,
	"encoding/gob":         65,
	"crypto/elliptic":      60,
	"crypto/subtle":        54,
	"archive/zip":          54,
	"crypto/aes":           53,
	"testing/quick":        51,
	"mime":                 51,
	"os/user":              51,
	"runtime/pprof":        47,
	"net/textproto":        47,
	"crypto/hmac":          45,
	"image":                45,
	"hash/fnv":             45,
	"debug/elf":            44,
	"net/http/httputil":    44,
	"crypto/sha512":        41,
	"encoding":             41,
	"math/cmplx":           40,
	"image/color":          38,
	"html":                 37,
	"expvar":               34,
	"net/http/pprof":       33,
	"embed":                32,
	"go/importer":          31,
	"encoding/csv":         31,
	"mime/multipart":       30,
	"container/list":       27,
	"encoding/asn1":        27,
	"log/syslog":           27,
	"go/printer":           27,
	"go/scanner":           25,
	"crypto/ed25519":       25,
	"debug/dwarf":          23,
	"text/scanner":         23,
	"compress/flate":       22,
	"compress/zlib":        21,
	"image/png":            20,
	"runtime/trace":        20,
	"unicode/utf16":        19,
	"net/http/httptrace":   19,
	"net/rpc":              19,
	"debug/macho":          16,
	"testing/iotest":       15,
	"go/doc":               15,
	"crypto/dsa":           13,
	"text/template/parse":  13,
	"debug/pe":             12,
	"net/http/cookiejar":   12,
	"regexp/syntax":        11,
	"image/jpeg":           11,
	"testing/fstest":       11,
	"crypto/des":           8,
	"net/mail":             7,
	"hash/adler32":         7,
	"compress/bzip2":       7,
	"image/draw":           7,
	"net/smtp":             6,
	"time/tzdata":          6,
	"image/gif":            5,
	"go/build/constraint":  5,
	"encoding/base32":      5,
	"image/color/palette":  4,
	"plugin":               4,
	"runtime/metrics":      3,
	"debug/gosym":          3,
	"hash/crc64":           3,
	"crypto/rc4":           3,
	"compress/lzw":         3,
	"index/suffixarray":    2,
	"container/ring":       2,
	"mime/quotedprintable": 2,
	"net/http/fcgi":        2,
	"net/http/cgi":         1,
	"debug/plan9obj":       1,
	"hash/maphash":         1,
}
