// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief gamma gamma -> p pbar
  class TASSO_1983_I191417 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1983_I191417);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // book histos
      book(_h["cProton"], 1,1,1);
      book(_h["cTheta1"], 2,1,1);
      book(_h["cTheta2"], 2,1,2);
      for (const string& en : _h["cProton"].binning().edges<0>()) {
        if (isCompatibleWithSqrtS(stod(en)*GeV)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
      _axis = YODA::Axis<double>({0., 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7});
      _edges = _h["cTheta2"].binning().edges<0>();
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const Particles& part = apply<FinalState>(event,"FS").particles();
      if (part.size() != 2) vetoEvent;
      double cTheta(0.);
      bool foundP(false), foundM(false);
      for (const Particle& p : part) {
        if (p.pid()==PID::PROTON) {
          foundP = true;
          cTheta = abs(p.mom().z()/p.mom().p3().mod());
        }
        else if (p.pid()==PID::ANTIPROTON) {
          foundM = true;
        }
      }
      if (!foundP || !foundM)  vetoEvent;
      if (cTheta<=0.6)  _h["cProton"]->fill(_sqs);
      if (sqrtS() <= 2.8) {
        const size_t idx = _axis.index(cTheta);
        if (0 < idx && idx < _axis.numBins()) {
          _h[sqrtS()> 2.4? "cTheta2"s : "cTheta1"s]->fill(_edges[idx-1]);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/nanobarn/sumOfWeights());
      for (const string& label : vector<string>{"cTheta1"s,"cTheta2"s}) {
        for (auto& b : _h[label]->bins()) {
          const size_t idx = b.index();
          b.scaleW(1./_axis.width(idx));
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    string _sqs = "";
    vector<string> _edges;
    YODA::Axis<double> _axis;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1983_I191417);
}
