// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief ND measurement of exclusive hadronic final states
  class ND_1991_I321108 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ND_1991_I321108);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // book hists
      vector<string> energies({"0.661", "0.671", "0.681", "0.691", "0.701", "0.711", "0.725", "0.735", "0.745", "0.755",
        "0.765", "0.805", "0.81", "0.815", "0.825", "0.83", "0.831", "0.841", "0.85", "0.851", "0.861", "0.87", "0.871",
        "0.881", "0.89", "0.891", "0.901", "0.91", "0.911", "0.921", "0.93", "0.943", "0.95", "0.955", "0.956", "0.963",
        "0.97", "0.973", "0.98", "0.983", "0.99", "0.993", "1.0", "1.003", "1.005", "1.02", "1.036", "1.04", "1.05",
        "1.059", "1.06", "1.07", "1.075", "1.08", "1.09", "1.099", "1.1", "1.11", "1.12", "1.13", "1.139", "1.14", "1.15",
        "1.16", "1.17", "1.179", "1.18", "1.19", "1.2", "1.21", "1.219", "1.22", "1.23", "1.24", "1.25", "1.259", "1.26",
        "1.27", "1.28", "1.29", "1.299", "1.3", "1.31", "1.32", "1.325", "1.33", "1.339", "1.34", "1.35", "1.36", "1.37",
        "1.375", "1.379", "1.38", "1.39", "1.395", "1.4", "1.5", "1.6"
      });
      for (const string& en : energies) {
        if (isCompatibleWithSqrtS(stod(en)*GeV)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
      book(_h["nOmegaPi0"],  1,1,1);
      book(_h["nOmegaPi1"],  2,1,1);
      book(_h["nOmegaPi2"],  3,1,1);
      book(_h["nOmegaPi3"],  4,1,1);
      book(_h["nOmegaPi4"], 10,1,3);
      book(_h["n2Pi"],      11,1,1);
      book(_h["n3Pi0"],      5,1,1);
      book(_h["n3Pi1"],     10,1,4);
      book(_h["n4PiC0"],     7,1,1);
      book(_h["n4PiC1"],    10,1,1);
      book(_h["n4PiN0"],     8,1,1);
      book(_h["n4PiN1"],    10,1,2);
      book(_h["nEtaPiPi"],   6,1,1);
      book(_h["nKC"],       12,1,1);
      book(_h["nKN"],       13,1,1);
      book(_h["n5Pi"],      14,1,1);
    }


    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      if (ntotal==2) {
        if (nCount[211]==1&&nCount[-211]==1) _h["n2Pi"]->fill(_sqs);
        if (nCount[321]==1&&nCount[-321]==1) _h["nKC"]->fill(_sqs);
        if (nCount[310]==1&&nCount[130]==1)  _h["nKN"]->fill(_sqs);
      }
      else if (ntotal==3) {
        if (nCount[211]==1&&nCount[-211]==1&&nCount[111]==1) {
          _h["n3Pi0"]->fill(_sqs);
          _h["n3Pi1"]->fill(_sqs);
        }
      }
      else if(ntotal==4) {
        if (nCount[211]==2&&nCount[-211]==2) {
          _h["n4PiC0"]->fill(_sqs);
          _h["n4PiC1"]->fill(_sqs);
        }
        else if(nCount[211]==1&&nCount[-211]==1&&nCount[111]==2) {
          _h["n4PiN0"]->fill(_sqs);
          _h["n4PiN1"]->fill(_sqs);
        }
      }
      else if(ntotal==5) {
        if (nCount[211]==2&&nCount[-211]==2&&nCount[111]==1) _h["n5Pi"]->fill(_sqs);
      }

      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles()) {
        if (p.children().empty()) continue;
        // find the eta
        if (p.pid()==221) {
          map<long,int> nRes = nCount;
          int ncount = ntotal;
          findChildren(p,nRes,ncount);
          // eta pi+pi-
          if (ncount!=2) continue;
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==211) {
              if (val.second !=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) _h["nEtaPiPi"]->fill(_sqs);
        }
        else if(p.pid()==223) {
          map<long,int> nRes = nCount;
          int ncount = ntotal;
          findChildren(p,nRes,ncount);
          // eta pi+pi-
          if (ncount!=1) continue;
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==111) {
              if (val.second !=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            _h["nOmegaPi0"]->fill(_sqs);
            _h["nOmegaPi1"]->fill(_sqs);
            _h["nOmegaPi2"]->fill(_sqs);
            _h["nOmegaPi3"]->fill(_sqs);
            _h["nOmegaPi4"]->fill(_sqs);
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/sumOfWeights()/nanobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ND_1991_I321108);
}
