// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Analyses/MC_CENT_PPB_Projections.hh"
#include "Rivet/Tools/Percentile.hh"

namespace Rivet {


  class MC_CENT_PPB_ETA : public Analysis {
  public:

    RIVET_DEFAULT_ANALYSIS_CTOR(MC_CENT_PPB_ETA);

    /// Book histograms and initialise projections before the run
    void init() {

      MSG_INFO("CENT parameter set to " << getOption<string>("cent","REF"));

      // The centrality projection.
      declareCentrality(MC_SumETFwdPbCentrality(),
                        "MC_CENT_PPB_CALIB", "SumETPb", "CENT");

      // The trigger projection.
      declare(MC_pPbMinBiasTrigger(), "Trigger");

      // The particles to be analysed.
      declare(ChargedFinalState(Cuts::abseta < 2.7 && Cuts::pT > 0.1*GeV), "CFS");

      // The centrality bins and the corresponding histograms.
      std::vector< std::pair<double, double> > centralityBins =
        { {0, 1}, {1, 5}, {5, 10}, {10, 20},
          {20, 30}, {30, 40}, {40, 60}, {60, 90} };
      // std::vector< std::tuple<int, int, int> > refData =
      //   { {2, 1, 8}, {2, 1, 7}, {2, 1, 6}, {2, 1, 5},
      //     {2, 1, 4}, {2, 1, 3}, {2, 1, 2}, {2, 1, 1} };
      std::vector< std::tuple<size_t, size_t, size_t> > refData;
      refData.reserve(8);
      for (size_t i = 8; i > 0; --i ) {
        refData.push_back(std::tuple<size_t, size_t, size_t>(2, 1, i));
      }

      // The centrality-binned histograms.
      _hEta = book<Histo1D>("CENT", centralityBins, refData);

    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      if ( !apply<TriggerProjection>(event, "Trigger")() ) vetoEvent;

      _hEta->init(event);
      for ( const auto &p : apply<ChargedFinalState>(event,"CFS").particles() )
        _hEta->fill(p.eta());
    }


    /// Finalize
    void finalize() {
      // Scale by the inverse sum of event weights in each centrality bin.
      _hEta->normalizePerEvent();
    }


  private:

    /// The histograms binned in centrality.
    Percentile<Histo1D> _hEta;

  };


  RIVET_DECLARE_PLUGIN(MC_CENT_PPB_ETA);

}
