//! Implementation of Ambient set.

use nix::errno::Errno;

use crate::caps::{errors::CapsError, nr, runtime, Capabilities, Capability};

pub fn clear() -> Result<(), CapsError> {
    Errno::result(unsafe { libc::prctl(nr::PR_CAP_AMBIENT, nr::PR_CAP_AMBIENT_CLEAR_ALL, 0, 0, 0) })
        .map(std::mem::drop)
        .map_err(CapsError)
}

pub fn drop(cap: Capability) -> Result<(), CapsError> {
    Errno::result(unsafe {
        libc::prctl(
            nr::PR_CAP_AMBIENT,
            nr::PR_CAP_AMBIENT_LOWER,
            libc::c_uint::from(cap.index()),
            0,
            0,
        )
    })
    .map(std::mem::drop)
    .map_err(CapsError)
}

pub fn has_cap(cap: Capability) -> Result<bool, CapsError> {
    let ret = Errno::result(unsafe {
        libc::prctl(
            nr::PR_CAP_AMBIENT,
            nr::PR_CAP_AMBIENT_IS_SET,
            libc::c_uint::from(cap.index()),
            0,
            0,
        )
    })
    .map_err(CapsError)?;

    match ret {
        0 => Ok(false),
        _ => Ok(true),
    }
}

pub fn raise(cap: Capability) -> Result<(), CapsError> {
    Errno::result(unsafe {
        libc::prctl(
            nr::PR_CAP_AMBIENT,
            nr::PR_CAP_AMBIENT_RAISE,
            libc::c_uint::from(cap.index()),
            0,
            0,
        )
    })
    .map(std::mem::drop)
    .map_err(CapsError)
}

pub fn read() -> Result<Capabilities, CapsError> {
    let mut res = Capabilities::empty();

    #[expect(clippy::disallowed_methods)]
    for flag in runtime::thread_all_supported() {
        let cap = flag.try_into().unwrap();

        if has_cap(cap)? {
            res |= flag;
        }
    }

    Ok(res)
}

pub fn set(value: Capabilities) -> Result<(), CapsError> {
    #[expect(clippy::disallowed_methods)]
    for flag in runtime::thread_all_supported() {
        let cap = flag.try_into().unwrap();

        if value.contains(flag) {
            raise(cap)?;
        } else {
            drop(cap)?;
        }
    }

    Ok(())
}
