/*
  Copyright (C) 2009, 2010, 2012, 2013 and 2015 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <string.h>

#include <memory>

#include <glib.h>

#include "fstab_parser.h"
#include "mounter.h"

#include <c++-gtk-utils/pipes.h>
#include <c++-gtk-utils/fdstream.h>
#include <c++-gtk-utils/thread.h>
#include <c++-gtk-utils/shared_handle.h>
#include <c++-gtk-utils/shared_ptr.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#endif


// used by GIO async continuations
extern "C"  {
  static void mount_gtk_async_wrapper(GObject*, GAsyncResult* res, void* data) {
    std::unique_ptr<Callback::CallbackArg<GAsyncResult*>>
      cb{static_cast<Callback::CallbackArg<GAsyncResult*>*>(data)};
    // we have to suppress exceptions because this executes in a glib
    // main loop.  If the program were to use thread cancellation we
    // would also need a cancel-block
    try {
      cb->dispatch(res);
    }
    catch (...) {
      g_critical("mount_gtk_async_wrapper(): "
		 "exception thrown in executing continuation callback\n");
    }
  }
}

void MounterCB::mounter_mounts_changed(GUnixMountMonitor*, void* data) {
  Mounter* instance = static_cast<Mounter*>(data);
  instance->fill_cache();
  instance->monitor_cb();
}

Mounter::Mounter(): monitor(g_unix_mount_monitor_new()) {

#if GLIB_CHECK_VERSION(2,18,0)
  g_unix_mount_monitor_set_rate_limit(monitor, 500);
#endif
  g_signal_connect(G_OBJECT(monitor.get()), "mounts_changed",
		   G_CALLBACK(MounterCB::mounter_mounts_changed), this);
  fill_cache();
}

void Mounter::fill_cache() {

  mounted_device_map.clear();

  GList* mount_list = g_list_first(g_unix_mounts_get(0));
  GList* temp = mount_list;
  while (temp) {
    mounted_device_map.insert(std::pair<std::string, std::string>{
	  (char*)g_unix_mount_get_device_path(static_cast<GUnixMountEntry*>(temp->data)),
	  (char*)g_unix_mount_get_mount_path(static_cast<GUnixMountEntry*>(temp->data))});
    g_unix_mount_free(static_cast<GUnixMountEntry*>(temp->data));
    temp = g_list_next(temp);
  }
  g_list_free(mount_list);
}

std::string Mounter::get_mount_point(const std::string& device) const {
  std::map<std::string, std::string>::const_iterator iter =
    mounted_device_map.find(device);
  if (iter == mounted_device_map.end()) return std::string();
  return iter->second;
}


// 'mount' can block for a considerable period when mounting a network
// file system in an attempt to cater for network perturbation.
// Therefore, to avoid blocking the GTK+ user interface,
// Mounter::raw_mount() starts a new thread to execute
// Mounter::raw_mount_thread() which will exec() 'mount'.  We observe
// the POSIX requirement for multi-threaded programs that the only
// functions which may be called in the child process between the
// fork() and the exec() are async-signal-safe ones.  We communicate
// with the UI thread with Callback::post().
void Mounter::raw_mount(const std::string& device,
			Direction direction,
			Callback::Callback* on_fail) {

  std::unique_ptr<Callback::Callback> fail_cb{on_fail};

  // now block off the signals with which the main thread may be interested
  // rather than this one
  sigset_t sig_mask;
  sigemptyset(&sig_mask);
  sigaddset(&sig_mask, SIGCHLD);
  sigaddset(&sig_mask, SIGQUIT);
  sigaddset(&sig_mask, SIGTERM);
  sigaddset(&sig_mask, SIGINT);
  sigaddset(&sig_mask, SIGHUP);
  pthread_sigmask(SIG_BLOCK, &sig_mask, 0);

  char* device_arg = strdup(device.c_str());

  if (!Thread::Thread::start(Callback::make(&Mounter::raw_mount_thread,
					    device_arg,
					    direction,
					    fail_cb.release()),
			     false).get()) {
    free(device_arg);
    if (fail_cb.get()) Callback::post(fail_cb.release());
    write_error("Cannot start new thread to mount/unmount non-block device\n", false);
  }

  // now unblock the signals so that the initial (GUI) thread can receive them
  pthread_sigmask(SIG_UNBLOCK, &sig_mask, 0);
}

void Mounter::raw_mount_thread(char* device_arg,
			       Direction direction,
			       Callback::Callback* on_fail) {

  std::unique_ptr<char, CFree> device{device_arg};
  std::unique_ptr<Callback::Callback> fail_cb{on_fail};
  // we are only in this function if the device to be mounted is not a
  // block device.  In that case we need to mount the device via its
  // mount point specified in /etc/fstab rather than by its device
  // name, or some versions of mount will complain with FUSE file
  // systems for a non-root mounter
  std::unique_ptr<char, CFree> mount_point{get_fstab_mount_point_for_device(device_arg)};
  if (!mount_point.get()) {
    if (fail_cb.get()) Callback::post(fail_cb.release());
    return; // get_fstab_mount_point_for_device() will already have
            // displayed an error message
  }

  PipeFifo fork_pipe{PipeFifo::block};

  // now fork to create the process which invokes 'mount'
  pid_t pid = fork();

  if (pid == -1) {
    if (fail_cb.get()) Callback::post(fail_cb.release());
    Callback::post(Callback::make(write_error,
				  "Fork error in Mounter::raw_mount_thread(\n",
				  false));
    return;
  }

  if (!pid) { // child process

    // unblock signals for mount
    sigset_t sig_mask;
    sigemptyset(&sig_mask);
    sigaddset(&sig_mask, SIGCHLD);
    sigaddset(&sig_mask, SIGQUIT);
    sigaddset(&sig_mask, SIGTERM);
    sigaddset(&sig_mask, SIGINT);
    sigaddset(&sig_mask, SIGHUP);
    // this child process is single threaded, so we can use sigprocmask()
    // rather than pthread_sigmask() (and should do so as sigprocmask()
    // is guaranteed to be async-signal-safe)
    // this process will not be receiving interrupts so we do not need
    // to test for EINTR on the call to sigprocmask()
    sigprocmask(SIG_UNBLOCK, &sig_mask, 0);

    fork_pipe.connect_to_stderr();

    if (direction == do_mount)
      execlp("mount", "mount", mount_point.get(), static_cast<char*>(0));
    else
      execlp("umount", "umount", mount_point.get(), static_cast<char*>(0));

    // if we reached this point, then the execlp() call must have failed
    const char message[] = "Exec failed in Mounter::raw_mount_thread().\n"
                           "Is mount installed?\n";
    write(2, message, sizeof(message) - 1);
    _exit(EXIT_SUCCESS - 1);
  }

  // read from the pipe
  fork_pipe.make_readonly();

  fdistream filein{fork_pipe.get_read_fd(), false}; // fork_pipe will manage the file
                                                    // descriptor, not filein
  std::string line;
  std::string err;
  // if we use std::getline() we will be safe against
  // multi-byte encodings such as UTF-8
  while (std::getline(filein, line)) {
    err += line;
    err += '\n';
  }

  pid_t ret;
  int status;
  do {
    ret = waitpid(pid, &status, 0);
  } while (ret == -1 && errno == EINTR);
  
  bool success = false;
  if (ret != -1                                    // waitpid() did not return an error
      && WIFEXITED(status)                         // mount exited normally
      && WEXITSTATUS(status) == EXIT_SUCCESS) {    // mount returned success
    success = true;
  }
  if (!success) {
    if (!err.empty()) err += '\n';
    err += gettext("Error mounting or unmounting following device: ");
    err += device.get();
    err += '\n';
    if (fail_cb.get()) Callback::post(fail_cb.release());
    Callback::post(Callback::make(write_error_del,
				  const_cast<const char*>(strdup(err.c_str())),
				  true));
  }
}

void Mounter::udisks_mount(const GobjHandle<UDisksObject>& object,
			   Direction direction,
			   Callback::Callback* on_fail) {

  std::unique_ptr<Callback::Callback> fail_cb{on_fail};

  GobjHandle<UDisksFilesystem> fs{udisks_object_get_filesystem(object)};
  if (!fs.get()) {
    fail_cb->dispatch();
    write_error(gettext("The block device specified in the mount table is not mountable"),
		true);
  }
  else {
    if (direction == do_mount) {
      udisks_filesystem_call_mount(fs,
				   g_variant_new("a{sv}",
						 static_cast<void*>(0)),
				   0,
				   mount_gtk_async_wrapper,
				   Callback::lambda<GAsyncResult*>([fs, on_fail] (GAsyncResult* res) {
	std::unique_ptr<Callback::Callback> fail_cb{on_fail};
        GError* error = 0;
	udisks_filesystem_call_mount_finish(fs,
					    0,
					    res,
					    &error);
	if (error) {
	  fail_cb->dispatch();
	  std::string err{"Failed to mount device: "};
	  err += error->message;
	  write_error(err.c_str(), true);
	  g_error_free(error);
	}
      }));
    }
    else {
      udisks_filesystem_call_unmount(fs,
				     g_variant_new("a{sv}",
						   static_cast<void*>(0)),
				     0,
				     mount_gtk_async_wrapper,
				     Callback::lambda<GAsyncResult*>([fs, on_fail] (GAsyncResult* res) {
	std::unique_ptr<Callback::Callback> fail_cb{on_fail};
        GError* error = 0;
	udisks_filesystem_call_unmount_finish(fs,
					      res,
					      &error);
	if (error) {
	  fail_cb->dispatch();
	  std::string err{"Failed to unmount device: "};
	  err += error->message;
	  write_error(err.c_str(), true);
	  g_error_free(error);
	}
      }));
    }
    fail_cb.release();
  }
}

void Mounter::get_udisks_object(const std::string& device,
				Direction direction,
				Callback::Callback* on_fail) {
  // check pre-conditions
  struct stat statbuf;
  if (stat(device.c_str(), &statbuf) || !S_ISBLK(statbuf.st_mode)) {
    raw_mount(device, direction, on_fail);
    return;
  }

  dev_t rdev = statbuf.st_rdev;
  std::unique_ptr<Callback::Callback> fail_cb{on_fail};
  udisks_client_new(0,
		    mount_gtk_async_wrapper,
		    Callback::lambda<GAsyncResult*>([direction, rdev, on_fail] (GAsyncResult* res) {

    std::unique_ptr<Callback::Callback> fail_cb{on_fail};
    GError* error = 0;
    GobjHandle<UDisksClient> client{udisks_client_new_finish(res, &error)};
    if (!client.get()) {
      fail_cb->dispatch();
      std::string err{"Failed to create udisks client object: "};
      err += error->message;
      write_error(err.c_str(), true);
      g_error_free(error);
    }
    else {
      GobjHandle<UDisksObject> val;
      GList* objects = g_dbus_object_manager_get_objects(
        udisks_client_get_object_manager(client)			  
      );
      for (GList* elt = objects; elt; elt = elt->next) {
	if (!val.get()) {
	  UDisksObject* current = UDISKS_OBJECT(elt->data);
	  UDisksBlock* block = udisks_object_peek_block(current);
	  if (block && rdev == udisks_block_get_device_number(block)) {
	    val.reset(static_cast<UDisksObject*>(g_object_ref(current)));
	  }
	}
	g_object_unref(elt->data);
      }
      g_list_free(objects);
      if (!val.get()) {
	fail_cb->dispatch();
	write_error(gettext("Failed to find the block device specified in the mount table"),
		    true);
      }
      else {
	udisks_mount(val, direction, fail_cb.release());
      }
    }
  }));
  fail_cb.release();
}

void Mounter::mount(const std::string& device, Callback::Callback* on_fail) {
  std::unique_ptr<Callback::Callback> fail_cb{on_fail};
  if (is_mounted(device)) {
    fail_cb->dispatch();
    return;
  }
  get_udisks_object(device, do_mount, fail_cb.release());
}

void Mounter::unmount(const std::string& device, Callback::Callback* on_fail) {
  std::unique_ptr<Callback::Callback> fail_cb{on_fail};
  if (!is_mounted(device)) {
    fail_cb->dispatch();
    return;
  }
  get_udisks_object(device, do_unmount, fail_cb.release());
}
