/* ----- decode/653x4621, derived from supercop/crypto_decode/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *decode_653x4621_checksums[] = {
  "8c4b6e117d8f9958d52355a14318bfe558207ead06003ef2387d4efee125675c",
  "3cf98a091d5f8a7de0f454731fbb8eee5d09052d746697e3be6fb56bdca4152c",
} ;

static void (*crypto_decode)(void *,const unsigned char *);
#define crypto_decode_STRBYTES ntruprime_decode_653x4621_STRBYTES
#define crypto_decode_ITEMS ntruprime_decode_653x4621_ITEMS
#define crypto_decode_ITEMBYTES ntruprime_decode_653x4621_ITEMBYTES

static void *storage_decode_653x4621_x;
static unsigned char *test_decode_653x4621_x;
static void *storage_decode_653x4621_s;
static unsigned char *test_decode_653x4621_s;
static void *storage_decode_653x4621_x2;
static unsigned char *test_decode_653x4621_x2;
static void *storage_decode_653x4621_s2;
static unsigned char *test_decode_653x4621_s2;

#define precomputed_decode_653x4621_NUM 4

static const int16_t precomputed_decode_653x4621_x[precomputed_decode_653x4621_NUM][crypto_decode_ITEMS] = {
  {-690,-971,-2016,828,-347,-97,1902,1074,-2150,-975,1948,1570,-128,1334,1762,-2061,-540,-1222,-643,891,-1039,-2007,-1475,-8,726,-1712,419,-568,263,1926,-621,-1322,-463,1079,745,222,1166,-1751,-494,2160,-750,-1468,-126,-171,1456,2039,173,-50,-757,-861,775,1488,1907,835,-1378,-2055,-76,-2135,-1914,397,-730,821,-852,8,-1789,-471,640,168,-2187,1620,-945,-1434,242,-983,-1672,1239,-211,134,2176,-2185,214,-1251,-2103,-1608,-1585,-1745,-503,-1680,-1171,-1246,-622,-1026,-1330,-1482,-2147,24,-69,-642,-1609,2264,96,888,-69,1611,1152,184,1910,234,1675,-454,-786,-1791,-1116,-98,-407,-314,-2184,-1644,1751,-1757,-1276,-383,1545,-1908,-1325,-287,797,281,748,-402,-1558,1065,1423,1107,1366,-936,636,729,2124,-1013,-1017,188,-966,-1342,1997,-1481,-1310,-152,-1191,-144,2135,2168,-801,524,1400,620,25,1025,-783,1917,1167,-509,-1214,-829,-776,1808,1634,-1680,-635,-2258,370,-1787,-118,-1914,658,175,-867,-356,2012,232,-419,-460,147,-1260,1384,22,1806,1729,2023,433,1978,120,-213,258,134,1126,-1681,-1287,-2253,-507,-343,-181,2131,-1217,2205,1989,1226,1292,567,-1453,2243,-595,2027,1882,-1010,-1198,-2242,684,-1172,1847,-610,2224,1674,1223,-696,-171,-311,-1374,-337,-6,1589,-1973,1672,-737,1930,2129,-1123,1802,-2185,-839,650,-2029,804,2068,817,1922,-291,-1760,-1888,315,1417,-155,-2310,-1521,944,-1992,777,-486,577,-1965,2158,1164,-470,1175,88,1628,-556,1388,5,2247,162,1928,1171,-2080,1670,2127,171,1924,-1305,-883,-643,-2264,2081,1225,-1924,554,1088,1027,1657,407,52,-1021,-406,-1112,-2194,-1473,1902,1646,-367,-449,1298,1090,-2180,982,37,2224,1939,-60,-255,1288,734,-1585,-796,-2026,-2197,-236,-2222,2159,822,-940,-2232,-205,-266,-898,-737,-1424,-1840,778,-1274,1921,-1505,-1720,2114,-1176,107,771,-1977,-1802,2292,2270,-1256,2,-2224,1563,1998,1239,40,-136,-1514,1604,-1596,675,-1789,1378,1406,-2086,1598,187,419,-1000,1529,1398,2157,-30,290,2089,-2196,1291,2222,-705,1955,-1366,-870,-2041,-2075,1785,1715,1636,-698,21,929,703,819,-2259,-1086,-1978,-993,-2307,-1556,-124,1765,1073,-29,-1001,501,1794,-725,-1970,1523,509,2045,-1259,942,435,-748,2126,953,-573,899,-55,-848,-1283,1507,1635,-2299,-1601,824,877,1975,1661,721,1735,-1960,-2137,-1802,-333,1300,-1406,-1264,1445,1094,-455,-1956,1863,2239,1682,-371,2298,-788,-995,-1843,-1439,-1706,-592,696,-2200,703,967,-1007,-1482,635,1456,646,-980,562,-2223,904,1602,-1783,-786,2085,381,-1051,-1891,-1738,1639,908,-1624,758,505,-1461,952,-1442,-628,-2078,174,-1346,-2060,854,-2216,-1642,-1414,-614,-1592,1128,301,-754,2147,2302,-1913,-614,-1775,1600,-1512,-656,-586,2098,1280,-2230,-2194,-1182,-1679,853,371,-602,-2118,1123,1749,-2109,2049,-2086,2244,1176,1322,222,2257,-1394,338,-2014,1993,-2293,-1236,788,1125,412,-2157,1111,-1981,1856,-2305,-2142,1639,1349,1141,-448,-2201,-851,-619,1773,491,1218,475,1985,-1676,-1279,193,2197,1534,139,1635,2245,-2166,-1616,1434,475,1136,-84,-1010,-155,1122,1188,2279,1525,-2070,1209,1960,-1632,611,-1538,1771,-1650,1864,-2007,138,-99,-450,-1193,1714,366,1839,959,940,-34,1328,-1711,2272,-2272,-1756,-163,-518,-1616,-1780,1771,-2084,1816,-1685,-1122,2285,-1380,-397,1196,2145,1471,-1310,268,-1322,124,-1898,1757,741,-2255,-2132,464,177,-1889,-2060,-1575,2261,-437,2307,-2256,462,767,-743,1982,-642,-2038,1658,1972,563,1591,964,809,-718,-38,-2239,-1806,-967,60,2051,-1863,-1504,2000,-539,-2304,-2184,698,-442,-415,1753,1119,917,-1043},
  {-919,-42,-413,-764,157,-1768,1828,-1751,-2208,-769,1848,1587,-698,-2254,-2079,-1189,-240,1863,509,-1015,1159,-2310,-1113,61,1277,2261,1715,227,1489,-1625,1104,-2308,-152,-1424,2278,2095,342,678,-521,787,-905,1485,1517,707,-1535,522,1487,903,2270,446,-1981,-344,1727,-592,-1719,2294,2067,141,-67,254,-135,343,-1617,-223,766,-1993,883,-89,-1128,-2229,1670,-1298,-1127,819,-1095,-1852,786,267,-432,-1068,-135,-603,1705,1722,-584,-642,-717,-269,1267,-589,-1773,1682,-1582,-1854,-1266,-1497,-2035,-330,1175,-322,-1084,-287,1222,303,1358,1505,-23,-1461,-1098,-31,2031,389,-1521,-1387,-2080,535,-1533,-1787,1215,93,510,544,1655,2058,-804,-1255,349,-794,1801,-779,-399,-1447,-483,-1208,-60,1756,-973,-953,991,-387,-1340,43,640,-853,1743,-1205,-1932,1772,1860,-1891,113,1145,98,2299,-1023,101,11,-336,-1733,948,-1421,-1490,-1432,470,-1625,-1669,-502,-1431,-538,-120,-757,1498,-1263,-1092,438,1410,-1623,689,-1236,1585,-2067,-2296,1429,-1790,443,-1309,-1796,366,1667,-1717,1607,9,1002,-1367,-1161,-655,1484,994,-1590,-2140,546,2026,1216,827,-915,-852,-1096,-1320,782,260,525,-720,417,922,1330,-2059,-899,-342,-235,-2248,728,1587,1085,1516,1814,962,1583,1013,-1439,-987,2025,694,1637,-639,-2225,268,-1799,2115,-320,-1085,1363,784,1710,-1454,1106,-1728,565,-2023,-2168,-1715,280,541,822,196,2013,496,1163,-1422,2243,496,-891,-1655,-918,-79,1838,-479,-698,1310,40,-265,1000,1345,1035,503,2069,-1919,-119,147,832,1287,-473,1037,162,-83,-1493,-2246,-230,-2088,1286,-1075,2190,-1333,-442,-413,1979,1996,-2106,-1581,-597,2136,1314,2177,-1605,-892,-1261,573,1551,-697,340,-784,1758,1619,-1383,289,584,-2200,-1655,-1049,646,1740,-241,-810,907,2277,-1885,-303,-1809,-167,-413,86,1092,-680,-705,2131,-1791,-417,1398,-513,-1674,-699,-1311,1667,839,807,204,1983,584,-1076,-930,1247,-1283,-287,-2147,-1213,-1629,2040,1801,-627,-645,458,-754,-1168,841,1725,2057,1993,-1737,195,760,381,-1701,1142,-726,-1929,-356,-1326,-2020,1882,-1384,-854,569,-288,-1101,593,-1145,-1170,-1075,-1186,2226,539,-1247,328,1821,-1014,-1831,-2263,1242,543,2205,-1888,1706,-433,-1742,-1630,-588,-458,-189,1362,1741,1505,1351,-790,875,961,-2279,-830,650,-1694,-791,-2301,1276,-1327,1561,-1161,-1251,-1553,-1728,2101,138,367,-446,344,-2090,587,-1694,150,267,-1061,2141,2007,-736,2198,-1313,-119,-735,-218,106,448,-1914,798,-1602,1447,1335,233,-2166,-1462,-1052,-330,1699,507,-54,1477,967,1092,-1389,1239,-1730,-692,-1225,1560,-1974,-2075,2192,-1878,757,483,1877,1661,-2055,237,183,2216,467,-1026,-1588,660,-2139,-1227,-1855,-831,1324,-694,989,-639,2251,382,2137,-1176,-1868,-562,-1273,139,1302,1140,-889,-1576,828,-724,1001,1814,1405,532,-207,-1539,2075,-1737,-58,-326,2145,-1506,-1478,2148,139,-397,-1969,2010,1636,115,720,-2236,1726,1583,477,-2028,-1868,-264,1632,-1063,1935,-593,1078,1883,906,2251,1921,1111,-1907,-507,-789,1291,1586,-1415,1704,-175,287,353,126,614,-1479,70,1392,2075,911,-124,1917,1377,-1053,-309,-1751,514,-1678,-199,864,634,-2191,73,1259,-1242,-443,-1872,1664,1749,-1701,-1818,578,-1838,564,-1402,380,-1318,-967,720,257,-1512,-531,144,2045,1165,-1099,1545,2099,-1917,-396,-1203,1546,-247,833,524,-1908,229,-809,-77,-2001,-1771,1610,1160,-1977,-1644,2013,-178,786,-1003,1492,1358,-700,-441,-1461,-1432,-975,1413,-607,-1236,-312,-1029,-718,131,-1825,-8,-510,1685,372,506,1032,2230,-2073,-2258,1193,590,-2285,-975,-1085,-1821,-821},
  {-1096,-1034,968,-629,1503,1194,621,1545,-2186,1949,-1940,1261,1943,2308,2234,1180,-1658,-312,-1555,-2259,734,1916,-696,-118,-1121,1610,905,-911,-1609,1439,1786,-1780,-714,-12,124,2031,158,1820,-755,-32,-1689,-2122,1431,-128,1002,-831,987,-2113,1690,-966,1049,-944,2205,953,837,-2027,1412,2024,1827,-1589,-2189,1363,-231,560,-1776,1042,-579,705,1268,-366,-1755,-1469,1160,-1134,-592,2187,1909,-1591,214,-205,-1207,-512,-810,1641,-1928,1839,362,-255,-1209,203,2017,827,78,-1677,492,-270,1832,782,1177,-1071,1439,-1307,-64,769,-1145,1689,-1891,-171,-42,-213,2136,1896,-585,1166,473,857,-525,1816,2096,-1318,-253,1545,1860,-1639,2150,2025,-1381,843,-322,-1026,1321,-585,1701,726,1293,300,588,-1420,575,558,1154,406,1075,-1839,-2059,497,16,1909,59,2284,616,-1661,617,1847,173,-87,-1518,-744,850,52,-914,-334,-2027,-2244,-2213,-110,-1654,-1570,1410,-861,-1023,800,-1562,252,-1834,-2310,-1739,-248,-1594,1630,1107,836,-317,306,-2022,-1860,-921,-193,-115,1146,-1066,1416,-1835,-2009,-1816,4,-2223,-784,-469,304,-1160,-589,-737,-558,-1205,1668,-537,861,-438,1963,-698,499,-448,378,1217,-2038,153,-2181,1293,-1000,326,-239,2125,-1702,2069,27,185,1453,-542,380,-781,-1738,112,598,161,1034,-623,-640,-910,-941,450,-1894,428,-1311,-1182,1720,-1889,909,92,-1454,376,-2302,380,-832,-2007,1817,-1467,-302,1471,-935,2194,-536,956,-312,-153,-354,-2202,1848,-1403,924,-286,479,-2033,2096,2279,1901,500,891,-1806,-1744,2231,2072,1622,2183,404,855,-1390,-2090,-1793,-238,-454,228,-1452,-1337,140,-536,-2270,-347,1996,-526,264,1445,1133,1204,1182,-1661,-1274,-2177,-279,837,1492,-173,49,-1675,-1059,1790,-114,2236,-354,-91,1821,2214,-245,165,-1359,-535,1208,1092,319,-107,2260,1915,1519,2207,-1056,-2257,-2213,505,-1891,-1404,1405,552,-1890,1364,-1608,459,-2090,1957,1563,691,-1,1344,-1845,-190,-1117,1505,2255,-1922,55,1993,-2204,2225,863,1150,-184,1613,2134,890,-1246,-586,2152,487,-35,-2018,1239,-1943,-725,-1179,-1534,-2117,132,-872,679,79,2103,64,-2267,-1372,-1955,709,25,-2153,-484,213,-334,2193,1204,-2064,1806,1990,-1447,1737,-839,-1280,-806,-71,1605,-562,-1184,526,1784,1957,508,-220,611,-242,-862,984,-1437,472,-527,-1304,-1961,-159,-350,-303,1908,942,172,-1177,1557,-1584,173,-1765,-912,-2233,-2004,-1039,-1325,-1018,889,391,2209,-9,2001,487,-533,-644,1081,-1700,458,581,-1022,83,-1571,-933,-2058,-133,1102,2240,835,-580,1797,1316,1576,-1233,-798,-1245,544,1577,-242,157,1662,-1364,-2265,-130,-219,759,2215,1679,-348,-346,-1242,-392,-2185,569,-769,-2158,279,-373,1529,586,820,-159,926,-1248,-1087,-1351,964,1018,-2099,697,-1980,-1798,1356,842,761,1706,1215,-1193,1757,1176,-1250,849,1548,2094,-775,1971,-87,-1078,-2117,230,1978,1566,645,1787,-1909,405,-1002,-1765,1341,330,-944,-2224,1377,-260,-1706,-1073,-1781,-1851,-1228,-1581,-1019,1213,-580,609,-1734,2159,74,-440,287,1094,754,1179,349,1217,-811,-2014,1755,-1254,1711,-1068,2013,-2008,-987,-1510,1844,344,-1120,-2000,-1443,1826,-1688,324,1291,-995,1460,50,1593,-1064,1159,-208,1735,-1977,-793,-282,-1174,1019,445,2111,2281,303,80,1605,1592,2211,-1541,538,1135,-1838,779,-1214,-1071,-1540,577,173,-1356,-1970,285,2230,-885,354,-1390,836,-1417,1405,583,-1046,498,148,-2090,-1030,195,-1679,-642,264,1108,-1090,-966,965,2158,-1485,-533,-1257,-738,792,-1172,-1659,716,-1408,-389,-383,-1642,-886,912,-1157,401,633,-2051,-1013,2049,-954,-333},
  {-1153,-1319,-935,-119,-1063,-374,1434,1893,-948,933,1362,-1647,-2254,2031,577,-905,1516,520,-2289,1298,-2221,1151,-2080,-1649,-683,-1522,1093,-341,-250,2067,1293,2171,-2184,-176,-2310,1955,-2011,-1077,-1786,-432,157,690,2196,-415,1120,1031,838,-1849,-1776,1186,2049,-261,-1914,-1775,592,-1042,-1951,-272,-1851,1305,821,-790,-2134,-562,413,813,1117,216,1321,797,320,63,556,1346,1477,1756,-1363,-1722,1551,1383,-101,235,1398,857,1320,958,1799,212,-150,-230,1304,1700,-1171,309,-2085,-1961,1931,751,-384,616,2182,-259,572,1903,-1153,1761,2303,313,-2085,461,-1272,-340,568,266,1159,-1232,-2255,-1351,-2033,2254,-836,-1083,1906,569,-956,-1409,1779,-1065,733,1297,348,430,1365,504,-996,-1275,-1110,-777,100,273,541,-823,984,313,-818,-2034,-1146,-1946,1178,1150,-239,605,1069,336,-192,-1438,-1391,-2233,917,-337,673,2252,988,754,-1837,966,809,-1215,-690,1317,-934,1308,2154,-606,-119,1600,-1557,-1259,-513,125,970,1928,2303,1476,-1344,150,782,-2144,-1276,-510,2102,1571,-2151,333,2041,1809,-2194,619,-1322,-1213,2300,684,-1648,-979,821,-452,1763,2104,-1406,1853,-1118,2269,-1905,2146,-1645,-412,-1522,-707,14,-1851,39,-1961,-800,799,-1815,540,546,-1011,2305,-20,-2098,-1556,754,-1350,-2264,1951,1805,-400,2253,-1304,-791,1195,-996,-1615,-1314,1602,1300,-1322,391,-263,-1286,315,-1331,1291,1105,-204,1917,-1657,388,164,2299,-1383,179,1661,839,1526,-500,-64,-473,626,1827,-979,-2206,-1034,-2274,-1506,1675,-1497,-623,-960,-1391,2038,-1925,1571,640,853,587,1623,-128,-566,-16,1644,-1704,-530,2102,1361,-1479,-528,983,1495,-1746,1641,-1706,-80,-429,1847,-1322,369,1174,-2109,-1389,-1045,-292,1127,1088,1506,982,511,456,-1427,1677,-2031,1161,-721,446,775,-2111,-2302,-710,-2176,2,-224,547,-425,-897,-1696,1652,-2223,-1592,-1331,-1393,1221,1354,-1151,-548,-856,-981,295,1183,-103,1206,348,-1300,1869,239,-1975,-1665,611,575,2088,-1145,646,100,1807,-640,-1546,69,1905,1701,148,670,-1122,-46,-288,1369,1848,131,1909,-972,-2130,-2137,-1074,1340,70,1101,-1002,-2176,1343,-1323,-120,-79,265,-1741,-55,2152,-316,134,-1003,1158,-1203,177,1524,-257,-1643,-894,-745,-532,-1523,723,-1329,1617,1025,1479,-805,-1304,-1963,-2187,-1095,-2158,1546,1794,-1366,-358,-791,1985,-1707,-1508,-869,-365,98,315,-282,837,2226,-1990,523,90,752,-1480,-686,1920,-750,1330,-98,9,-1953,977,1364,781,-2238,1354,-2115,-472,-1674,-891,1431,-565,1672,1256,-1481,480,-207,-1199,-545,1723,1882,1172,-854,1561,1843,-1078,-657,-853,-1158,561,854,-1493,1550,2017,795,-1556,-2279,1987,217,695,-1691,-142,950,-771,1382,914,-260,-1278,-1281,2025,-1802,-1152,1906,-1160,-2101,-2245,1867,1054,1494,1078,-877,-293,-1470,835,1108,145,318,-2217,2247,1608,116,1460,2166,1384,-1978,703,-300,-698,-1778,-185,-2115,787,-145,-2293,-1372,1598,1351,-2200,1420,-315,-1766,1039,144,-390,-1688,1769,1991,-1955,-1199,1760,2169,-903,842,2079,1146,157,-1449,-105,-229,1332,-869,-2277,-1475,651,-1058,421,-679,-1970,-353,518,1397,212,-2103,1699,2293,719,-2160,-955,-501,534,958,-55,6,426,-987,2044,1589,699,1659,-1252,-2170,2067,-51,196,1153,-261,2264,2154,1857,472,1280,-837,1944,-2114,-567,1736,-1474,1139,506,-1181,1963,1424,743,-1202,213,-1891,1943,-2300,-874,-2264,711,979,-363,-1169,1770,1448,1549,1442,1721,-2298,-1328,-848,113,-1553,-823,-1053,-752,333,-1030,1235,2169,-1663,-434,-991,1371,1359,963,-275,-1245,-618,-2151,-598,1536,2035,-337,371,186,986},
} ;

static const unsigned char precomputed_decode_653x4621_s[precomputed_decode_653x4621_NUM][crypto_decode_STRBYTES] = {
  {83,112,128,68,12,18,76,172,107,34,170,165,146,249,141,158,42,190,16,187,90,98,41,84,58,54,31,223,41,185,197,176,80,253,131,148,247,119,22,54,218,100,39,219,143,181,119,100,166,49,235,216,46,210,151,254,157,95,3,225,43,203,104,119,108,173,92,197,13,28,209,201,91,155,183,64,79,92,223,225,163,181,117,128,134,217,13,115,123,10,204,143,224,101,41,147,117,165,3,135,204,135,222,129,44,232,172,113,209,237,79,158,254,252,203,196,80,246,242,13,229,227,121,103,148,168,182,189,214,148,83,252,26,254,34,240,213,83,47,133,231,39,104,70,236,132,126,45,93,190,195,208,207,217,72,167,122,48,158,18,10,11,125,113,28,99,102,123,47,177,7,235,172,244,201,67,221,204,248,77,85,121,235,18,218,124,47,219,56,122,38,104,153,26,8,80,104,36,200,33,204,37,218,34,242,49,186,8,194,120,224,254,209,165,140,109,78,28,139,33,226,184,249,44,237,216,87,7,181,124,88,210,111,249,251,15,115,245,48,185,213,219,124,190,31,115,209,207,243,24,254,1,17,162,220,120,175,168,204,94,222,5,41,194,88,181,164,198,116,90,222,220,71,69,221,234,179,148,100,162,217,68,170,82,242,242,187,88,120,163,175,236,70,128,245,4,88,1,24,64,192,202,174,31,105,187,219,182,189,186,181,42,86,12,195,61,73,29,206,165,51,109,176,151,35,127,166,190,231,88,27,157,222,6,230,71,25,211,142,2,134,9,3,23,13,79,148,83,222,253,95,124,81,13,228,217,17,32,47,105,75,131,59,213,139,55,79,233,53,61,153,160,73,253,222,190,27,76,171,98,168,127,208,164,164,109,76,59,244,37,182,153,98,85,99,107,117,255,32,212,98,44,25,154,94,207,244,134,12,13,221,111,62,57,12,254,21,196,100,16,64,67,39,52,97,104,2,204,197,200,127,217,75,63,125,140,147,241,185,190,14,108,154,37,84,205,46,28,35,103,191,150,22,211,163,45,54,227,115,119,210,207,50,144,197,116,104,107,239,136,247,4,206,156,12,39,118,164,34,173,28,48,22,167,5,40,105,74,45,16,203,191,204,83,2,150,134,51,52,46,115,130,128,22,108,144,136,65,36,92,34,28,14,38,207,15,12,186,171,105,155,186,137,64,103,212,18,64,135,106,49,115,138,99,207,182,146,65,48,144,53,109,249,196,34,129,207,28,230,51,27,57,214,0,223,5,182,178,179,6,179,160,43,251,149,34,251,248,65,196,138,82,127,157,23,47,181,192,37,151,207,53,162,141,5,23,72,16,99,99,160,97,107,255,219,33,251,3,199,230,237,45,141,145,62,180,110,22,210,48,65,141,33,103,87,162,254,80,198,147,250,116,152,137,120,173,144,202,159,164,127,233,7,77,123,10,43,180,183,136,173,214,197,240,108,226,156,194,186,131,68,151,204,176,4,166,210,97,134,8,154,73,63,172,104,131,0,110,21,195,185,220,176,47,231,18,129,89,161,90,227,216,47,82,84,48,184,72,202,186,27,103,239,42,219,245,123,190,76,131,225,2,164,163,12,123,241,85,162,236,4,75,75,142,73,59,16,243,45,189,96,130,25,11,91,118,233,95,115,239,0,123,166,148,96,166,117,73,208,127,30,216,197,100,125,210,238,166,23,174,98,59,16,148,79,124,112,101,197,148,252,108,83,49,176,1,85,171,54,170,80,181,235,148,58,78,45,218,65,249,12,148,119,84,4,221,85,204,202,210,122,135,163,59,131,126,243,229,237,156,23,232,94,51,31,85,32,215,191,174,206,80,32,218,48,121,28,222,59,131,159,20,242,158,126,62,18,154,164,82,36,14,212,97,135,36,237,164,170,116,226,25,53,116,15,40,66,31,204,141,189,83,167,106,154,254,181,27,67,52,172,20,81,96,240,118,188,101,16,38,90,81,169,94,63,63,111,11,12,219,84,2,251,121,182,213,215,52,146,24,122,208,31,131,102,94,2,161,51,244,173,99,43,73,212,29,116,253,5,0,123,143,201,227,113,158,138,192,135,18,207,148,86,19,249,126,140,178,46,52,92,10,233,195,134,215,169,190,64,99,129,33,118,41,186,94,100,43,150,154,243,232,53,216,86,225,81,16,8,225,217,191,11,224,4,3,1,111,95,192,128,250,150,43,28,163,183,15,93,239,192,95,215,179,252,136,119,158,131,189,67,201,10,110},
  {155,240,235,9,41,65,141,122,167,168,35,216,36,249,212,11,255,69,198,90,54,226,20,51,34,92,142,242,160,91,112,49,108,129,157,171,24,186,66,102,52,156,40,202,215,178,254,155,216,101,31,161,3,51,27,164,144,227,247,210,56,25,176,42,29,102,66,167,187,186,240,106,132,165,1,80,245,192,104,154,46,101,111,92,114,163,222,239,94,103,209,124,0,42,93,87,159,157,145,58,133,169,125,76,15,14,12,230,119,182,4,96,244,23,95,155,152,227,204,125,242,71,77,13,117,105,95,239,206,3,74,225,25,187,68,187,34,180,140,164,71,237,131,199,242,249,196,212,145,155,234,166,117,5,118,5,79,57,179,187,29,213,154,8,58,53,179,1,34,114,113,135,241,229,164,87,250,120,253,167,169,253,3,185,150,159,230,177,166,223,16,147,211,138,136,182,154,6,114,255,88,199,19,63,125,211,4,211,150,66,209,39,199,238,247,192,115,201,65,103,195,211,141,211,68,198,244,93,150,76,123,225,70,226,63,199,180,4,251,103,119,55,44,107,230,22,206,71,197,244,229,16,126,191,97,235,169,170,71,236,206,52,187,84,47,43,32,70,7,59,137,196,234,101,246,162,84,71,239,172,36,7,191,16,113,134,102,175,195,34,49,245,161,201,107,175,209,103,119,132,3,112,195,254,128,76,254,202,216,163,105,25,154,69,228,204,152,236,54,157,65,204,128,123,148,133,200,28,21,249,16,252,202,41,40,124,189,195,75,154,220,111,150,212,211,189,248,13,5,212,190,168,88,90,143,187,134,187,17,51,18,140,54,143,146,121,114,163,165,202,173,105,137,227,154,105,2,150,142,173,237,157,36,182,113,102,231,69,101,244,29,6,243,113,66,82,193,56,17,211,1,105,192,244,198,156,193,242,142,15,125,59,140,176,71,91,216,122,100,168,237,93,120,19,148,100,69,8,129,203,14,42,249,69,84,119,126,27,156,118,18,227,40,129,99,136,126,129,96,39,141,235,96,93,160,203,118,161,182,176,31,15,143,237,210,65,110,24,195,228,63,147,134,135,208,251,2,16,86,152,147,43,13,148,164,109,170,93,226,74,66,0,190,223,133,226,245,6,126,66,106,178,78,81,211,198,204,224,91,214,213,114,94,101,58,120,140,237,59,194,162,89,182,236,181,156,143,12,152,67,117,143,53,237,160,69,185,252,198,33,41,180,45,166,64,72,34,35,206,238,171,42,25,154,96,207,0,54,27,212,35,63,151,47,190,9,134,28,151,33,171,219,230,161,122,7,45,92,137,233,165,67,93,179,64,83,86,17,226,252,29,171,141,78,145,15,122,23,126,214,46,199,177,21,211,133,49,224,129,8,66,121,148,2,134,187,31,247,39,101,119,52,30,177,51,207,231,235,70,136,49,193,219,90,45,36,203,82,231,183,122,153,154,43,145,171,243,136,228,33,158,127,109,208,88,157,8,239,238,48,2,110,3,147,126,17,228,237,190,111,156,127,32,143,79,182,211,222,47,148,241,90,1,70,212,129,73,137,199,143,193,41,209,33,226,249,191,216,174,187,250,235,159,68,85,137,162,160,226,15,243,188,66,161,31,142,28,28,59,214,40,219,218,30,239,95,201,78,177,41,161,70,187,215,212,151,100,250,248,206,249,200,248,79,47,248,11,86,234,164,45,183,94,14,79,86,43,107,63,222,26,120,89,159,11,213,175,101,38,112,182,10,219,153,148,35,242,30,141,210,38,219,222,242,241,115,109,237,171,55,184,120,81,146,63,191,127,106,161,131,134,15,111,251,161,222,50,232,49,136,98,120,70,203,170,43,221,149,227,75,15,208,197,6,137,42,102,242,140,94,233,204,236,141,89,89,117,125,79,83,26,79,101,74,166,237,224,251,165,99,168,150,238,48,233,155,209,240,143,133,9,150,3,132,36,220,169,96,76,189,37,49,156,178,25,19,49,92,237,147,235,188,58,66,14,69,125,166,177,28,131,136,191,78,155,120,203,188,90,57,95,227,193,64,78,123,67,185,46,87,248,98,136,21,231,15,146,9,29,176,138,241,118,55,253,59,22,190,48,59,46,208,21,39,127,108,116,82,245,168,250,139,107,191,187,8,156,140,167,158,156,55,23,8,97,121,4,16,123,236,149,226,194,138,169,144,14,248,175,154,105,227,229,220,87,195,67,108,239,52,239,7,178,37,160,28,145,115,193,81,187,73,60,27},
  {138,253,43,148,213,32,54,221,195,78,201,204,31,175,250,38,2,228,138,155,126,6,158,149,181,107,154,177,30,91,234,110,238,14,243,31,94,63,193,165,249,67,107,233,11,86,226,240,224,211,125,94,86,37,170,0,160,166,198,230,254,252,221,101,78,92,222,157,178,32,224,78,70,249,19,29,254,194,193,118,157,203,127,156,47,142,203,240,234,53,52,66,121,171,138,226,50,21,138,106,148,199,33,35,160,253,66,212,89,229,244,162,65,31,178,89,127,244,150,3,204,217,93,96,143,187,190,85,248,144,200,175,215,33,157,22,132,204,233,68,116,143,36,67,134,237,85,133,139,246,99,206,136,40,150,200,158,110,74,152,204,89,117,168,25,32,36,48,29,58,245,78,6,169,133,214,241,102,224,210,27,225,161,124,250,187,238,74,19,184,18,190,36,59,112,43,213,153,239,87,227,93,29,143,83,130,244,157,77,82,241,22,198,143,151,59,44,34,153,108,119,17,51,240,200,217,214,94,130,179,5,91,34,21,119,211,101,199,253,140,232,95,109,123,14,45,28,251,218,100,230,154,144,65,194,0,67,153,152,2,174,39,56,238,193,243,146,47,197,11,137,175,211,172,196,253,135,191,182,234,67,12,133,221,83,53,196,134,61,27,34,252,195,158,168,31,215,105,106,219,189,206,229,211,153,208,205,100,190,237,95,189,118,207,23,29,110,147,83,126,223,13,192,139,218,43,128,238,36,96,103,250,86,142,151,193,84,125,165,227,217,219,54,16,91,65,139,223,134,169,147,174,121,125,100,4,137,109,192,113,181,196,25,4,133,165,220,179,180,147,119,73,183,159,128,238,160,197,213,158,144,110,254,126,203,117,205,35,178,224,24,27,65,237,99,138,56,102,112,66,226,94,13,166,147,229,15,80,106,252,173,238,36,105,109,220,238,72,175,44,13,246,152,172,147,139,158,93,59,237,249,63,96,119,95,115,189,159,117,29,168,127,128,80,224,72,139,127,57,16,159,227,141,192,208,26,137,129,98,224,35,8,45,188,223,255,253,29,137,30,91,251,142,195,225,182,4,110,62,86,102,131,146,65,176,0,189,189,122,158,15,66,117,183,146,238,122,163,202,181,134,7,74,27,98,78,239,55,126,208,233,220,143,194,182,150,100,225,63,231,189,25,148,93,152,237,112,113,201,114,83,52,180,21,129,154,45,59,96,95,63,107,242,109,253,216,49,31,70,228,1,15,37,15,126,187,227,228,129,133,199,162,57,92,203,109,0,51,100,224,107,165,48,188,216,198,146,118,133,234,75,68,182,138,233,4,125,191,68,197,160,80,37,22,211,214,161,211,109,85,185,83,241,79,94,31,254,252,175,40,217,220,90,215,36,246,125,43,202,91,220,65,212,135,58,133,78,19,143,241,89,61,106,70,170,191,129,235,160,165,92,231,236,106,35,89,10,142,224,116,227,173,157,17,206,137,94,70,224,38,78,227,227,246,158,94,221,62,228,171,0,170,130,132,31,70,164,156,29,131,239,70,233,229,112,38,122,197,251,145,205,210,163,211,72,146,79,45,221,151,167,187,27,18,81,239,49,14,26,173,15,181,65,232,96,250,19,197,167,124,38,139,157,249,62,114,184,146,163,225,50,232,211,215,209,16,0,155,203,25,35,245,247,154,87,234,221,85,179,214,95,235,205,63,207,57,128,144,43,65,132,59,10,195,173,46,25,52,205,96,24,174,116,135,80,211,63,172,64,76,162,177,89,90,2,102,23,3,239,35,198,80,21,90,167,225,112,222,139,135,8,206,17,249,98,59,167,98,57,194,230,103,223,235,85,214,128,36,125,6,194,186,128,238,20,98,218,239,96,92,28,122,65,186,167,69,215,65,125,15,189,253,53,50,108,89,223,188,181,186,115,67,127,107,208,222,154,145,126,247,151,91,208,20,52,124,30,250,196,134,239,130,184,127,131,204,196,234,105,75,220,77,169,209,252,219,146,143,10,214,101,15,220,174,21,223,25,142,56,245,150,38,204,116,67,178,27,143,91,25,135,208,124,221,210,3,198,178,48,158,127,195,119,32,196,174,63,25,187,8,206,246,222,139,97,106,100,255,37,109,89,215,218,123,79,88,195,59,57,1,91,195,31,105,38,87,191,11,214,253,35,76,217,171,234,48,196,115,121,26,24,20,235,56,77,101,50,187,2,83,81,108,85,240,253,245,88,244,155,14,26,100,95,36},
  {216,228,162,130,47,135,15,106,1,176,3,206,169,22,160,28,168,154,77,103,26,10,119,156,95,150,72,227,81,168,160,3,220,120,149,186,200,241,106,109,251,145,213,175,15,161,181,141,158,131,20,139,183,186,186,115,229,180,94,231,107,57,116,65,58,63,169,41,246,33,199,92,218,212,69,193,143,121,158,116,222,123,79,93,34,124,31,228,16,178,192,205,114,175,154,156,2,230,28,88,179,175,51,27,64,17,56,5,152,99,24,236,14,174,75,16,234,158,217,208,17,138,171,16,11,141,50,217,14,97,134,61,65,121,177,255,137,28,149,42,166,228,17,0,216,123,8,175,84,5,30,146,145,159,142,132,128,113,204,42,81,183,122,24,53,0,202,65,131,196,26,33,248,55,29,187,80,30,172,184,6,224,71,6,178,120,130,192,114,239,81,184,214,92,42,128,49,135,145,93,12,46,45,220,149,14,15,76,239,140,47,227,221,51,251,214,123,10,99,102,230,164,199,61,169,246,31,163,81,138,30,43,184,188,143,114,17,189,233,0,236,41,109,6,140,218,70,184,128,96,77,27,176,236,73,140,172,27,44,124,20,111,96,9,25,135,32,101,69,12,192,233,52,249,248,176,218,98,37,55,99,152,150,168,37,18,10,93,22,1,192,213,194,132,167,233,189,169,22,88,215,152,154,63,114,36,231,233,209,57,214,53,107,96,14,31,29,246,165,77,190,187,64,24,109,145,47,145,14,121,246,30,226,244,179,80,229,49,133,10,228,252,43,199,184,140,122,179,184,171,182,120,41,174,248,168,218,248,155,38,169,114,123,84,82,229,86,61,125,96,248,207,134,155,133,61,200,133,94,182,113,39,30,223,191,71,7,148,17,111,122,153,188,2,176,170,235,49,195,27,105,96,228,15,1,95,233,132,170,55,178,54,58,45,141,123,46,172,104,228,248,159,195,34,102,149,79,158,225,209,61,9,163,226,29,195,222,240,182,133,190,15,140,6,48,53,105,27,187,31,55,206,122,223,132,205,7,205,7,130,177,83,248,119,161,164,138,183,4,228,193,146,49,64,227,35,53,47,148,0,133,50,28,63,44,177,4,230,127,89,178,152,73,146,187,226,158,74,27,193,94,134,80,70,1,24,137,66,219,13,11,87,198,81,128,30,6,170,74,119,215,197,80,137,52,200,10,180,179,35,45,51,216,1,107,99,179,21,22,227,88,84,206,87,89,225,143,69,165,192,92,191,74,196,103,169,136,224,37,83,126,11,181,254,75,109,93,162,104,159,115,159,40,147,16,121,107,86,223,83,1,253,78,231,62,227,208,89,165,55,71,140,11,227,44,45,199,249,71,231,66,2,121,156,102,79,159,83,115,234,156,227,246,64,151,3,184,85,165,179,165,40,91,73,234,61,193,222,120,220,7,224,83,75,76,223,203,73,20,146,138,214,8,32,175,60,254,53,42,47,57,9,97,120,120,68,59,136,243,156,158,110,49,171,86,123,3,13,136,133,53,65,173,76,174,72,43,198,181,127,125,9,216,9,196,118,48,254,233,2,104,7,81,245,185,170,12,44,175,36,6,220,65,141,49,152,49,250,170,31,75,240,93,241,38,117,194,209,66,61,169,51,216,157,127,9,197,4,141,100,128,186,179,217,59,70,228,30,112,189,112,145,28,107,95,172,113,1,162,24,231,176,232,120,135,250,111,132,200,195,169,3,22,235,147,22,244,124,11,220,75,242,174,201,100,218,253,110,135,135,3,133,171,157,206,107,111,131,239,91,48,74,233,217,0,99,17,92,131,0,83,135,95,247,115,163,3,184,193,54,124,86,104,37,171,100,155,36,143,126,74,88,100,3,77,92,199,71,236,1,98,118,200,223,238,84,5,250,142,178,247,219,233,244,190,63,12,93,42,231,252,2,245,144,183,142,27,87,169,254,178,138,48,167,146,252,235,89,64,62,103,112,10,3,124,194,56,210,203,155,207,232,183,59,129,238,46,247,170,195,60,221,174,184,209,0,224,93,253,254,184,167,28,159,141,78,133,200,30,140,1,154,223,139,124,148,92,168,149,125,6,156,213,126,170,196,201,249,135,181,24,22,13,50,182,13,230,97,65,156,217,96,181,39,221,178,118,122,99,249,226,164,240,142,1,50,107,74,217,28,226,69,44,191,219,55,202,8,19,38,62,185,202,27,9,244,194,3,33,186,12,170,3,69,251,210,161,183,21,253,195,144,35,147},
} ;

static void test_decode_653x4621_impl(long long impl)
{
  unsigned char *x = test_decode_653x4621_x;
  unsigned char *s = test_decode_653x4621_s;
  unsigned char *x2 = test_decode_653x4621_x2;
  unsigned char *s2 = test_decode_653x4621_s2;
  long long xwords = crypto_decode_ITEMS;
  long long xlen;
  long long slen = crypto_decode_STRBYTES;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_decode_653x4621_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_decode = ntruprime_dispatch_decode_653x4621(impl);
    printf("decode_653x4621 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_decode_653x4621_implementation(impl),ntruprime_dispatch_decode_653x4621_compiler(impl));
  } else {
    crypto_decode = ntruprime_decode_653x4621;
    printf("decode_653x4621 selected implementation %s compiler %s\n",ntruprime_decode_653x4621_implementation(),ntruprime_decode_653x4621_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 1024;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {
      xlen = xwords*crypto_decode_ITEMBYTES;

      output_prepare(x2,x,xlen);
      input_prepare(s2,s,slen);
      secret(s,slen);
      crypto_decode(x,s);
      public(s,slen);
      public(x,xlen);
      endianness(x,xwords,crypto_decode_ITEMBYTES);
      checksum(x,xlen);
      output_compare(x2,x,xlen,"crypto_decode");
      input_compare(s2,s,slen,"crypto_decode");

      double_canary(x2,x,xlen);
      double_canary(s2,s,slen);
      secret(s2,slen);
      crypto_decode(x2,s2);
      public(s2,slen);
      public(x2,xlen);
      endianness(x2,xwords,crypto_decode_ITEMBYTES);
      if (memcmp(x2,x,xlen) != 0) fail("failure: crypto_decode is nondeterministic\n");
    }
    checksum_expected(decode_653x4621_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_decode_653x4621_NUM;++precomp) {
    output_prepare(x2,x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
    input_prepare(s2,s,crypto_decode_STRBYTES);
    memcpy(s,precomputed_decode_653x4621_s[precomp],crypto_decode_STRBYTES);
    memcpy(s2,precomputed_decode_653x4621_s[precomp],crypto_decode_STRBYTES);
    crypto_decode(x,s);
    if (memcmp(x,precomputed_decode_653x4621_x[precomp],crypto_decode_ITEMS*crypto_decode_ITEMBYTES)) {
      fail("failure: crypto_decode fails precomputed test vectors\n");
      printf("expected x: ");
      for (long long pos = 0;pos < crypto_decode_ITEMS*crypto_decode_ITEMBYTES;++pos) printf("%02x",((unsigned char *) precomputed_decode_653x4621_x[precomp])[pos]);
      printf("\n");
      printf("received x: ");
      for (long long pos = 0;pos < crypto_decode_ITEMS*crypto_decode_ITEMBYTES;++pos) printf("%02x",x[pos]);
      printf("\n");
    }
    output_compare(x2,x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES,"crypto_decode");
    input_compare(s2,s,crypto_decode_STRBYTES,"crypto_decode");
  }
}

void test_decode_653x4621(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"decode")) return;
  if (targetp && strcmp(targetp,"653x4621")) return;
  storage_decode_653x4621_x = callocplus(crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  test_decode_653x4621_x = aligned(storage_decode_653x4621_x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  if (crypto_decode_ITEMS*crypto_decode_ITEMBYTES > maxalloc) maxalloc = crypto_decode_ITEMS*crypto_decode_ITEMBYTES;
  storage_decode_653x4621_s = callocplus(crypto_decode_STRBYTES);
  test_decode_653x4621_s = aligned(storage_decode_653x4621_s,crypto_decode_STRBYTES);
  if (crypto_decode_STRBYTES > maxalloc) maxalloc = crypto_decode_STRBYTES;
  storage_decode_653x4621_x2 = callocplus(maxalloc);
  test_decode_653x4621_x2 = aligned(storage_decode_653x4621_x2,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  storage_decode_653x4621_s2 = callocplus(maxalloc);
  test_decode_653x4621_s2 = aligned(storage_decode_653x4621_s2,crypto_decode_STRBYTES);

  for (long long offset = 0;offset < 1;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("decode_653x4621 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_decode_653x4621();++impl)
      forked(test_decode_653x4621_impl,impl);
    ++test_decode_653x4621_x;
    ++test_decode_653x4621_s;
    ++test_decode_653x4621_x2;
    ++test_decode_653x4621_s2;
  }
  free(storage_decode_653x4621_s2);
  free(storage_decode_653x4621_x2);
  free(storage_decode_653x4621_s);
  free(storage_decode_653x4621_x);
}
#undef crypto_decode_STRBYTES
#undef crypto_decode_ITEMS
#undef crypto_decode_ITEMBYTES

