---
myst:
  html_meta:
    keywords: LaTeX, composition, texte, listes, énumérations, 
              espaces entre les items, espaces dans une liste, 
              économiser l'espace
---

# Comment modifier les espacements dans les listes ?

Dans son livre 
*[LaTeX, a Document Preparation System](/1_generalites/documentation/livres/documents_sur_latex)*, 
Leslie Lamport cite les divers paramètres pour la mise en page de listes 
(comme `\topsep`, `\itemsep` et `\parsep`, présentés dans la question 
"[](/3_composition/texte/listes/creer_sa_liste_personnalisee)") 
mais il ne mentionne pas qu'ils sont définis automatiquement 
dans les listes standard de LaTeX :

- chaque liste exécute en effet une commande `\@list⟨niveau⟩` (le `⟨niveau⟩` 
  apparaissant comme un chiffre romain minuscule) ;
- et la commande `\@listi` est généralement réinitialisée lorsque la taille 
  de la police est modifiée.

En conséquence, il est assez difficile pour l'utilisateur de contrôler 
l'espacement des listes. Bien sûr, la vraie bonne réponse est d'utiliser 
une classe de document conçue avec un espacement de liste plus resserré, 
mais de telles classes sont difficiles à trouver.

Certaines extensions fournissent un contrôle de l'espacement des listes, 
mais elles traitent rarement le cas de la séparation de la liste 
avec le texte environnant (définie par `\topsep`).

Voici ici un exemple pour observer l'espacement par défaut.

```
\documentclass{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \pagestyle{empty}           % N'affiche pas de numéro de page

\begin{document}
Voici la liste des merveilles du monde antique :
\begin{itemize}
   \item la statue chryséléphantine de Zeus, à Olympie ;
   \item le colosse de Rhodes, sur l'île du même nom ;
   \item les jardins suspendus de Babylone, en Mésopotamie ;
   \item le phare d'Alexandrie, en Égypte ;
   \item la grande pyramide de Gizeh, en Égypte ;
   \item l'Artémision à Éphèse, en Ionie ;
   \item le mausolée d'Halicarnasse, en Carie.
\end{itemize}
\end{document}
```


## Avec des commandes de base

Vous pouvez écrire des listes compactes avec le code suivant (s'inspirant 
de l'extension <ctanpkg:mdwlist> citée ci-après). Ce code définit 
un [nouvel environnement](/2_programmation/macros/commande_newenvironment),
basé sur `itemize`, dans lequel les longueurs `\itemsep` et `\parsep` 
sont mises à zéro.

```
\documentclass{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \pagestyle{empty}           % N'affiche pas de numéro de page

  \newenvironment{itemize*}%  % Définition d'un nouvel environnement
    {\begin{itemize}%
      \setlength{\itemsep}{0pt}%
      \setlength{\parskip}{0pt}}%
    {\end{itemize}}

\begin{document}
Voici la liste des merveilles du monde antique :
\begin{itemize*}
   \item la statue chryséléphantine de Zeus, à Olympie ;
   \item le colosse de Rhodes, sur l'île du même nom ;
   \item les jardins suspendus de Babylone, en Mésopotamie ;
   \item le phare d'Alexandrie, en Égypte ;
   \item la grande pyramide de Gizeh, en Égypte ;
   \item l'Artémision à Éphèse, en Ionie ;
   \item le mausolée d'Halicarnasse, en Carie.
\end{itemize*}
\end{document}
```

Une alternative est proposée par Martin Boyer sous forme d'une extension, 
[itemsep](../../../_static/old_extensions/itemsep.sty). Son code définit 
les commandes :
- `\noitemsep` pour supprimer tout espacement vertical entre les items 
  des environnements `itemize`, `enumerate` et `description` ;
- `\doitemsep` pour les remettre.

Mark Wooding propose également une autre solution de réduction de l'espace
entre les items de toutes les listes (en manipulant la commande `\@listI`
qui stocke la valeur de réinitialisation de `\@listi`):

```{noedit}
\makeatletter
\toks@\expandafter{\@listI}
\edef\@listI{\the\toks@\setlength{\parsep}{1pt}}
\makeatother
```


## Avec l'extension <ctanpkg:babel>

L'extension <ctanpkg:babel> avec son option `french` procède 
à des réglages adaptés à la typographie française. Elle diminue en particulier
l'espace entre les éléments d'une liste.

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page

\begin{document}
Voici la liste des merveilles du monde antique :
\begin{itemize}
   \item la statue chryséléphantine de Zeus, à Olympie ;
   \item le colosse de Rhodes, sur l'île du même nom ;
   \item les jardins suspendus de Babylone, en Mésopotamie ;
   \item le phare d'Alexandrie, en Égypte ;
   \item la grande pyramide de Gizeh, en Égypte ;
   \item l'Artémision à Éphèse, en Ionie ;
   \item le mausolée d'Halicarnasse, en Carie.
\end{itemize}
\end{document}
```

Si vous ne souhaitez pas avoir cette diminution d'espacement, 
l'extension <ctanpkg:babel> propose des options de paramétrage pour annuler
ce réglage typographique, à placer en argument de la commande `\frenchbsetup`
de <ctanpkg:babel> :
- `StandardListSpacing=true` restaure l'espacement par défaut des listes ;
- `StandardLists=true` restaure l'ensemble de la mise en forme par défaut 
  des listes, espacements compris (et retrouver, par exemple, l'affichage 
  du premier exemple de cette page). 

Ces options de paramétrage peuvent en particulier faire 
cohabiter <ctanpkg:babel> avec des extensions modifiant le format des listes.
La [documentation](texdoc:frenchb-doc) de l'option `french` de <ctanpkg:babel>
présente d'ailleurs quelques autres options sur ce sujet.

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page
  \frenchbsetup{StandardListSpacing=true}

\begin{document}
Voici la liste des merveilles du monde antique :
\begin{itemize}
   \item la statue chryséléphantine de Zeus, à Olympie ;
   \item le colosse de Rhodes, sur l'île du même nom ;
   \item les jardins suspendus de Babylone, en Mésopotamie ;
   \item le phare d'Alexandrie, en Égypte ;
   \item la grande pyramide de Gizeh, en Égypte ;
   \item l'Artémision à Éphèse, en Ionie ;
   \item le mausolée d'Halicarnasse, en Carie.
\end{itemize}
\end{document}
```


## Avec la classe <ctanpkg:memoir>

La classe <ctanpkg:memoir> ne fournit pas de listes compactes,  mais offre 
à l'utilisateur un contrôle sur l'espacement des listes en utilisant 
les commandes `\firmlist` et `\tightlist` (et leurs versions étoilées). 
Ceci est détaillé dans la section 8.6 du manuel de <ctanpkg:memoir>.

:::{todo} Ajouter un exemple.
:::


## Avec l'extension <ctanpkg:mdwlist>

L'extension <ctanpkg:mdwlist> propose une commande `\makecompactlist` 
pour les définitions de liste par les utilisateurs. Elle s'en sert d'ailleurs
pour définir des versions compactées des environnements courants : 
`itemize*`, `enumerate*` et `description*`.

```
\documentclass{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \pagestyle{empty}           % N'affiche pas de numéro de page
  \usepackage{mdwlist} 

\begin{document}
Voici la liste des merveilles du monde antique :
\begin{itemize*}
   \item la statue chryséléphantine de Zeus, à Olympie ;
   \item le colosse de Rhodes, sur l'île du même nom ;
   \item les jardins suspendus de Babylone, en Mésopotamie ;
   \item le phare d'Alexandrie, en Égypte ;
   \item la grande pyramide de Gizeh, en Égypte ;
   \item l'Artémision à Éphèse, en Ionie ;
   \item le mausolée d'Halicarnasse, en Carie.
\end{itemize*}
\end{document}
```


## Avec l'extension <ctanpkg:paralist>

L'extension <ctanpkg:paralist> fournit plusieurs environnements 
pour le compactage de listes :

- `asparaenum` présente chaque élément comme s'il s'agissait d'un paragraphe
  introduit par son numéro dans la liste (ce qui économise de l'espace 
  si les éléments listés sont longs) ;
- `compactenum` donne le même genre de liste compacte que celles fournies 
  dans <ctanpkg:expdlist> et <ctanpkg:mdwlist> ;
- `inparaenum` produit une liste "dans le paragraphe", c'est-à-dire 
  sans saut de ligne entre les éléments, ce qui conduit 
  à un large gain d'espace si les éléments listés sont courts.

L'extension modifie par ailleurs les étiquettes de l'environnement `enumerate` 
comme le fait l'extension <ctanpkg:enumerate> (vous pouvez ici consulter
la question 
"[](/3_composition/texte/listes/changer_la_numerotation_d_enumerate)").


Enfin, cette extension fournit également des environnements similaires 
à ceux présentés pour `enumerate` pour `itemize` (`asparaitem` et autres) 
ou `description` (`asparadesc` et autres).


## Avec l'extension <ctanpkg:multenum>

L'extension <ctanpkg:multenum> offre une mise en forme plus structurée 
que celle de l'environnement `inparaenum` de l'extension <ctanpkg:paralist>,
pour ne pas dire une forme tabulée. Vous définissez en effet 
une grille fictive sur laquelle les entrées de liste doivent être disposées, 
et les éléments de liste respecteront ces positions dans la grille. 
L'effet est un peu comme celui des touches "tabulation" sur les machines 
à écrire traditionnelles. Cette extension a été conçue par créer des listes
de réponses dans les annexes d'un livre.


## Avec l'extension <ctanpkg:expdlist>

L'extension <ctanpkg:expdlist>, parmi ses nombreux contrôles d'apparence 
de l'environnement `description`, propose un paramètre de compactage 
dénommé `\compact` (qui doit figurer dans l'argument facultatif 
de l'environnement).

Cette extension n'est cependant pas compatible avec l'option `french` 
de <ctanpkg:babel>.

```
\documentclass{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \pagestyle{empty}           % N'affiche pas de numéro de page
  \usepackage{expdlist}
  
\begin{document}
Voici la liste des merveilles du monde antique :
\begin{description}
   \item[la statue chryséléphantine de Zeus] à Olympie ;
   \item[le colosse de Rhodes] sur l'île du même nom ;
   \item[les jardins suspendus de Babylone] en Mésopotamie ;
   \item[le phare d'Alexandrie] en Égypte ;
   \item[la grande pyramide de Gizeh] en Égypte ;
   \item[l'Artémision] à Éphèse en Ionie ;
   \item[le mausolée d'Halicarnasse] en Carie.
\end{description}

Et voici cette liste avec l'espacement modifié :
\begin{description}[\compact]
   \item[la statue chryséléphantine de Zeus] à Olympie ;
   \item[le colosse de Rhodes] sur l'île du même nom ;
   \item[les jardins suspendus de Babylone] en Mésopotamie ;
   \item[le phare d'Alexandrie] en Égypte ;
   \item[la grande pyramide de Gizeh] en Égypte ;
   \item[l'Artémision] à Éphèse en Ionie ;
   \item[le mausolée d'Halicarnasse] en Carie.
\end{description}
\end{document}
```


## Avec l'extension <ctanpkg:enumitem> 

Les extensions <ctanpkg:expdlist>, <ctanpkg:mdwlist> et <ctanpkg:paralist> 
offrent toutes d'autres fonctionnalités pour la configuration de la liste. 
Les approches décrites ci-dessous devraient être évitées si vous avez besoin 
de l'une de ces extensions pour un autre objectif de configuration de liste.

Afin d'obtenir plus de flexibilité (y compris la manipulation de `\topsep`), 
l'extension <ctanpkg:enumitem> permet d'ajuster les paramètres de la liste 
en utilisant un format *`⟨clé⟩=⟨valeur⟩`*. Par exemple, pour supprimer 
les espaces avant et après la liste, vous pourriez écrire :

```{noedit}
\usepackage{enumitem}
...
\begin{enumerate}[topsep=0pt, partopsep=0pt]
\item ...
\item ...
\end{enumerate}
```

De manière similaire, voici un exemple de définition de l'espacement 
entre les éléments et entre les paragraphes dans les éléments :

```{noedit}
\usepackage{enumitem}
...
\begin{enumerate}[itemsep=2pt,parsep=2pt]
\item ...
\item ...
\end{enumerate}
```

L'extension <ctanpkg:enumitem> permet également de manipuler le format 
de l'étiquette d'une manière plus "basique" (et donc plus flexible) que 
l'extension <ctanpkg:enumerate> vue à la question
"[](/3_composition/texte/listes/changer_la_numerotation_d_enumerate)".


## Avec l'extension <ctanpkg:savetrees>

Le nec plus ultra en matière de compactage (de toutes sortes) est offert 
par l'extension <ctanpkg:savetrees> (littéralement "Sauvez des arbres"). 
Le compactage des listes fait parti des solutions retenues mais elles sont 
assez radicales. Si vous avez des contraintes de mise en forme, 
ne l'utilisez pas.


:::{sources}
- [How to adjust list spacing](faquk:FAQ-complist)
:::